/*
  File: SectionInput.c
  Author: David Meyers
          Cheryl Buck
  Last Modified: 15 March 1990
  Purpose: Input routines for Contours package.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <TypeDefinitions.h>
#include <Contours.h>
#include <ParsingFunctions.h>

extern char *malloc();
extern void free();
extern void exit();

/**************************************************************************/

/*
  Read from the input file any adjacency information associated with a contour.
  Reads the adjacency information, allocates memory and builds a circularly
  sorted list of adjacent contours.
 */
void GetAdjList(AdjacentPtr, FilePtr)
 Adjacency **AdjacentPtr;
 FILE *FilePtr;
{
 if(GetLeftBracket(FilePtr))
  {
   *AdjacentPtr = (Adjacency *) malloc(sizeof(Adjacency));
   (*AdjacentPtr)->next = *AdjacentPtr;
   (*AdjacentPtr)->prev = *AdjacentPtr;
   GetIdentifier(FilePtr, (*AdjacentPtr)->Name);
   (void) fscanf(FilePtr, "%d %d",
		 &((*AdjacentPtr)->LocalIndex),
		 &((*AdjacentPtr)->AdjacentIndex));
   if(!GetRightBracket(FilePtr))
    {
     (void) fprintf(stderr,
		    "GetAdjList: expecting right bracket, not found\n");
     exit(-1);
    }
  }
 
 while(GetLeftBracket(FilePtr))
  {
   Adjacency *temp, *InsertPointer;
   
   temp = (Adjacency *) malloc(sizeof(Adjacency));
   GetIdentifier(FilePtr, temp->Name);
   (void) fscanf(FilePtr,"%d %d",
		 &(temp->LocalIndex), &(temp->AdjacentIndex));

   InsertPointer = *AdjacentPtr;

   if((*AdjacentPtr)->LocalIndex > temp->LocalIndex)
    /* insert at head of list */
    {
     temp->next = *AdjacentPtr;
     temp->prev = (*AdjacentPtr)->prev;
     (*AdjacentPtr)->prev->next = temp;
     (*AdjacentPtr)->prev = temp;
     *AdjacentPtr = temp;
     temp = NULL;
    }
   else /* insert not at head of list */
    {
     while((temp->LocalIndex > InsertPointer->next->LocalIndex) &&
	   (InsertPointer->next != *AdjacentPtr))
      {
       InsertPointer = InsertPointer->next;
      }
     /*
       temp->LocalIndex is > InsertPointer->LocalIndex here and we are
       either at wraparound point of the circular list or temp->LocalIndex is
       less than InsertPointer->next->LocalIndex
      */
     temp->next = InsertPointer->next;
     temp->prev = InsertPointer;
     temp->prev->next = temp;
     temp->next->prev = temp;
     InsertPointer->next = temp;
     temp = NULL;
     InsertPointer = NULL;
    }
   
   if(!GetRightBracket(FilePtr))
    {
     (void) fprintf(stderr,
		    "GetAdjList: expecting right bracket, not found\n");
     exit(-1);
    }
  }
 if(!GetRightBracket(FilePtr))
  {
   (void) fprintf(stderr,
		  "GetAdjList: expecting right bracket, not found\n");
   exit(-1);
  }
}

/**************************************************************************/

/*
  Reads from the input file any information about possible aliases for a
  contour. Aliases can arise when two contours merge into one in an adjacent
  section, or when two contours have different names in adjacent sections but
  represent a slice through the same generalized cylinder. This function builds
  an array of alias names for the current contour. When tiling this contour
  we check the adjacent section for a contour with the same name as this
  contour or one with a name matching one of the aliases of this contour. If
  no match is found, then the cylinder represented by this contour is assumed 
  to end at this section. This may be a BUG which we want to do something about
   in the future.
 */
void GetAlias(TheContour, FilePtr)
 Contour **TheContour;
 FILE *FilePtr;
{
 int i, n;
 
 (void) fscanf(FilePtr,"%d", &n);
 (*TheContour)->NAliases = n;
 (*TheContour)->Aliases =  malloc(80 * sizeof(char) * (unsigned) n);
 for (i=0; i<n; i++)
  {
   if(GetLeftBracket(FilePtr))
    {
     (void) fscanf(FilePtr,"%s", &((*TheContour)->Aliases[i*80]));
     if(!GetRightBracket(FilePtr))
      {
       (void)fprintf(stderr,
		     "GetAlias: expecting Right Bracket, not found\n");
       exit(-1);
      }
    }
   else
    {
     (void) fprintf(stderr,
		    "GetAlias: expecting Left Bracket, not found\n");
     exit(-1);
    }
  }
 if(!GetRightBracket(FilePtr))
  {
   (void) fprintf(stderr,
		  "GetAlias: expecting Right Bracket, not found\n");
   exit(-1);
  }
}

/**************************************************************************/

static void ReadContour(FilePtr, TheContour)
 FILE *FilePtr;
 Contour *TheContour;
{
 int i, Npts;
 REAL x, y, z;
 char Identifier[20];

 GetIdentifier(FilePtr, TheContour->Name);
 (void) fscanf(FilePtr, "%d", &Npts);
 TheContour->ContourPoints = NewPoints(Npts);
 if(NULL == TheContour)
  {
   (void) fprintf(stderr, "Insufficient memory, ReadContour\n");
   exit(-1);
  }

 TheContour->Visited = FALSE;

 for(i=0; i<Npts; i++)
  {
   ReadPoint(FilePtr, &x, &y, &z);
   TheContour->ContourPoints->P[i].x = x;
   TheContour->ContourPoints->P[i].y = y;
   TheContour->ContourPoints->P[i].z = z;
  }
 TheContour->AdjContPrev = NULL;
 TheContour->AdjContNext = NULL;
 TheContour->Aliases = NULL;
 TheContour->NAliases = 0;
 /*
   The following gets the ADJACENT and ALIAS information. The ALIAS information
   may only be present if the ADJACENT information is present. Otherwise
   the current routines will not work. This is likely a **BUG** since we might 
   like to allow a contour to match one of a different name without actually
   merging with yet another contour.
  */

 while (GetLeftBracket(FilePtr))
  {
   GetIdentifier(FilePtr, Identifier);
   if(!strcmp("ADJPREVIOUS", Identifier))
    {
     GetAdjList(&(TheContour->AdjContPrev), FilePtr);
    }
   else if (!strcmp("ADJNEXT", Identifier))
    {
     GetAdjList(&(TheContour->AdjContNext), FilePtr);
    }
   else if (!strcmp("ALIAS", Identifier))
    {
     GetAlias(&(TheContour), FilePtr);
    }
   else
    {
     (void) fprintf(stderr,
		    "ReadContour: expecting ADJPREVIOUS, ADJNEXT, ALIAS\n");
     exit(-1);
    }
  }
 if(!GetRightBracket(FilePtr))
  {
   (void) fprintf(stderr,"ReadContour: expecting Right bracket, not found\n");
   exit(-1);
  }
}

/**************************************************************************/

extern void ReadSection(FilePtr, TheSection)
 FILE *FilePtr;
 Section *TheSection;
{
 char Identifier[10];
 int i;

 if (!GetLeftBracket(FilePtr))
  {
   if(feof(FilePtr))
    {
     TheSection->TheContours = NULL;
     TheSection->NContours = 0;
     return;
    }
   else
    {
     (void) fprintf(stderr,
		    "Improper format: ReadSection expecting left bracket\n");
     exit(-1);
    }
  }
 GetIdentifier(FilePtr, Identifier);
 if(strcmp("SECTION", Identifier))
  {
   (void) fprintf(stderr,"Improper format: ReadSection expecting SECTION\n");
   exit(-1);
  }
 GetIdentifier(FilePtr, TheSection->Name);

 TheSection->TheIndex = NULL;

 (void) fscanf(FilePtr, "%d", &(TheSection->NContours));
 TheSection->TheContours = 
  (Contour *) malloc((unsigned)TheSection->NContours * sizeof(Contour));

 for(i=0; i<TheSection->NContours; i++)
  {
   if (GetLeftBracket(FilePtr))
    {
     GetIdentifier(FilePtr, Identifier);
     if(!strcmp("CONTOUR", Identifier))
      {
       ReadContour(FilePtr, &(TheSection->TheContours[i]));
      }
     else
      {
       (void) fprintf(stderr, "CONTOUR expected, not found: ReadSection\n");
       exit(-1);
      }
    }
   else
    {
     (void) fprintf(stderr,
		    "Left curly bracket expected, not found: ReadSection\n");
     exit(-1);
    }
  }
 if(!GetRightBracket(FilePtr))
  {
   if(feof(FilePtr))
    {
     (void) fprintf(stderr,
		    "Unexpected end of file encountered in ReadSection\n");
     exit(-1);
    }
   else
    {
     (void) fprintf(stderr, "Uninterpretable input, ReadSection\n");
     exit(-1);
    }
  }
}

/**************************************************************************/

extern void ClearSection(TheSection)
 Section *TheSection;
{
 int i;

 for (i=0; i<TheSection->NContours; i++)
  {
   Adjacency *temp1, *temp2;

   free(TheSection->TheContours[i].Aliases);
   FreePoints(TheSection->TheContours[i].ContourPoints);
   temp1 = TheSection->TheContours[i].AdjContPrev;
   if(NULL != temp1)
    {
     temp1 = temp1->next;
     while(TheSection->TheContours[i].AdjContPrev != temp1)
      {
       temp2 = temp1->next;
       free((char *) temp1);
       temp1 = temp2;
      }
     free((char *) TheSection->TheContours[i].AdjContPrev);
    }

   temp1 = TheSection->TheContours[i].AdjContNext;
   if(NULL != temp1)
    {
     temp1 = temp1->next;
     while(TheSection->TheContours[i].AdjContNext != temp1)
      {
       temp2 = temp1->next;
       free((char *) temp1);
       temp1 = temp2;
      }
     free((char *) TheSection->TheContours[i].AdjContNext);
    }
  }
 TheSection->NContours = 0;
 TheSection->Name[0] = '\n';
 TheSection->Last = 0;
 if (NULL != TheSection->TheIndex)
    free((char *) TheSection->TheIndex);
 free((char *) TheSection->TheContours);
}

/**************************************************************************/
