/*
  File: Tiler.c
  Author: David Meyers
  Last Modified: 30 August 1990
  Purpose: The Chop one triangle at a time method. Version for use in surfaces
           from contours system. Optimizing of initial triangulation has now
	   been included.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <TypeDefinitions.h>
#include <Aggregates.h>
#include <WingEdge.h>
#include <WETraversal.h>
#include <EulerOps.h>

#define EPSILON (1.0e-16)

extern char *malloc();
extern void free();
extern void exit();

typedef struct LElt {
  struct LElt *next;
  WingEdgeNode *element;
 } ListElement;


/**************************************************************************/

void OutputTriangle(P1, P2, P3, Previous, TriangleNumber)
 WEVertexNode *P1, *P2, *P3;
 int Previous;
 int *TriangleNumber;
{
 if(!Previous)
  (void)fprintf(stdout," t%d = [%s, %s, %s];\n",
		*TriangleNumber, P1->PointName, P2->PointName, P3->PointName);
 else
  (void)fprintf(stdout," t%d = [%s, %s, %s];\n",
		*TriangleNumber, P1->PointName, P3->PointName, P2->PointName);
 *TriangleNumber = *TriangleNumber + 1;
}

/**************************************************************************/

void OutputShell(TheShell, Previous, TriangleNumber)
 WEFaceNode *TheShell;
 int Previous, *TriangleNumber;
{
 WEFaceNode *CurrentFace;
 WEVertexNode *P1, *P2, *P3;

 CurrentFace = TheShell;
 do /* while CurrentFace != TheShell */
  {
   P1 = CurrentFace->Vertices;
   P2 = NextVertexCCW(CurrentFace, P1);
   P3 = NextVertexCCW(CurrentFace, P2);
   OutputTriangle(P1, P2, P3, Previous, TriangleNumber);
   CurrentFace = CurrentFace->next;
  } while (CurrentFace != TheShell);
}

/**************************************************************************/

 /*
  Does the path A -> B -> C turn to the left?

  Watch out for colinear points - you may get a random answer.
  Since we are asking for "LeftTurns" - make sure it REALLY turns
  left...  Note that we may miss a VERY SLIGHT, legitimate, LeftTurn.
 */
int LeftTurn (A, B, C)
 WEVertexNode *A, *B, *C;
  {
   return     ( ( ((A->x - B->x)*(C->y - B->y))
                 -((A->y - B->y)*(C->x - B->x))
                )
               < -EPSILON   /* 0.0 ? */
              );
  }

/**************************************************************************/

 /*
   This version of Shrink has been modified from the DCEL version of Ken
   Sloan to work with the Wing Edge data Structure. We opt to make this a
   function which returns TRUE if the swap should be made and FALSE if not,
   rather than doing the swap internally, as was done in the original version.

   Shrink is the absolute HOT SPOT - so we"ve optimized as much as
   is seemly.  Given a pair of triangles ABC and BAD, we want to know if
   the pair ADC, BCD would be better.  If it is,  we return TRUE.
   
   To determine which local configuration is best, we calculate

    Rabc = the square of the radius of the circle through ABC
    Rbad = the square of the radius of the circle through BAD

    Radc = the square of the radius of the circle through ADC
    Rbcd = the square of the radius of the circle through BCD
    
    Rabc and Rbad are associated with the current configuration;
    Radc and Rbcd are associated with the proposed configuration.

    The calculation of the  R"s depends on the observations that:

                            2
               (   abc    )
        4*r*r= (-------   )
               (a cross c )

    so, we get:
               2 2 2              2
     Rabc =   a c e  / (a cross c)

               2 2 2              2
     Racd =   a d f  / (a cross d)

               2 2 2              2 
     Rbad =   b d e  / (b cross d)

               2 2 2              2
     Rbcd =   b c f  / (b cross c)

   where
        a is the edge connecting A and C
        b is the edge connecting B and D
        c is the edge connecting C and B
        d is the edge connecting D and A  
        e is the existing diagonal  connecting A and B
        f is the candidate diagonal connecting C and D

  Of the four circles, the smallest one is guaranteed to EXCLUDE the fourth 
  point - that is the configuration we want.  So, we first pick the minimum
  from each pair ([Rabc, Rabd] and [Racd, Rbcd]) and  compare the minima.
  if min(Rabc,Rabd) > min(Racd, Rbcd)  SWAP else DO NOT SWAP.

  BE CAREFUL: Rho is negative for colinear or "backwards" triangles;
              negative Rhos should compare GREATER than positive
              This should only happen when one of the already created
              triangles is linear...so we add an extra check.

  Finally, this criterion is not valid when the quadrilateral ADBC is not
  convex. So, we first check the angles DAC and CBD: 
  If (a cross d) and (b cross c) are both positive
   all is well.  If either (a cross d) of (b cross c) are negative,
   all of the above is moot, and no swap is possible.  
  Here, it is important that:
        
         a is the DIRECTED edge from A to C
         b is the DIRECTED edge from B to D
         c is the DIRECTED edge from C to B
         d is the DIRECTED edge from D to A

  Rabc, Rbad, Radc and Rbcd need to be calculated - we do it ourself, for speed.

 */
int Shrink(SharedEdge)
 WingEdgeNode *SharedEdge;
 {
  WEVertexNode *A, *B, *C, *D;
  double vAx, vAy, vBx, vBy, vCx, vCy, vDx, vDy;      /* vertex coordinates */
  double ax, ay, bx, by, cx, cy, dx, dy, ex, ey, fx, fy;/* edge vector deltas */
  double ax2, ay2, bx2, by2, cx2, cy2, dx2, dy2, ex2, ey2, fx2, fy2;
  double aCROSSd, bCROSSc, aCROSSc, bCROSSd;
  double Rabc, Radc, Rbad, Rbcd, R1, R2;

  A = SharedEdge->Org;       B = SharedEdge->Dest;

  C = NextVertexCW(SharedEdge->Left, A);
  D = NextVertexCCW(SharedEdge->Right, A);

  vAx = A->x; vAy = A->y;
  vBx = B->x; vBy = B->y;
  vCx = C->x; vCy = C->y;
  vDx = D->x; vDy = D->y;

  ax = vCx-vAx; ay = vCy-vAy;      
  bx = vDx-vBx; by = vDy-vBy;
  cx = vBx-vCx; cy = vBy-vCy;
  dx = vAx-vDx; dy = vAy-vDy;

  aCROSSd = ax*dy - ay*dx;
  bCROSSc = bx*cy - by*cx;
  aCROSSc = ax*cy - ay*cx;
  if(aCROSSc < 0.0) aCROSSc = -1.0 * aCROSSc;
  bCROSSd = bx*dy - by*dx;
  if (bCROSSd < 0.0) bCROSSd = -1.0 * bCROSSd;
  if ((aCROSSd > 0.0) && (bCROSSc > 0.0))
   { /* ADBC is convex */
                                  ax2 = ax*ax; ay2 = ay*ay;
                                  bx2 = bx*bx; by2 = by*by;
                                  cx2 = cx*cx; cy2 = cy*cy;
                                  dx2 = dx*dx; dy2 = dy*dy;
    fx = vCx-vDx; fy = vCy-vDy; fx2 = fx*fx; fy2 = fy*fy;
    ex = vBx-vAx; ey = vBy-vAy; ex2 = ex*ex; ey2 = ey*ey;

    if (aCROSSd < EPSILON) aCROSSd = EPSILON; /* fault protection */
    if (bCROSSc < EPSILON) bCROSSc = EPSILON; /* fault protection */
    if (aCROSSc < EPSILON) aCROSSc = EPSILON; /* fault protection */
    if (bCROSSd < EPSILON) bCROSSd = EPSILON; /* fault protection */

    Radc = (ax2+ay2)*(dx2+dy2)*(fx2+fy2)/(aCROSSd*aCROSSd);
    Rbcd = (bx2+by2)*(cx2+cy2)*(fx2+fy2)/(bCROSSc*bCROSSc);
    if (Radc < Rbcd)  R2 = Radc; else R2 = Rbcd;

    Rabc = (ax2+ay2)*(cx2+cy2)*(ex2+ey2)/(aCROSSc*aCROSSc);
    Rbad = (bx2+by2)*(dx2+dy2)*(ex2+ey2)/(bCROSSd*bCROSSd);
    if (Rabc < Rbad)
     if (Rabc >= 0.0)  R1 = Rabc; else R1 = Rbad;
    else if (Rbad >= 0.0)  R1 = Rbad; else R1 = Rabc;

    if (R1 > R2)
     return(TRUE);
    else
     return(FALSE);
   }   /* ADBC is convex */
  return(FALSE);
 } /* Shrink */

/**************************************************************************/

 /*
   InsideTriangle decides if a point P is Inside, or OnEdge
   of the triangle defined by A, B, C.

   If P is on an edge, V1 and V2 identify that edge
 */
void InsideTriangle (A, B, C, P, Inside, OnEdge)
 WEVertexNode *A, *B, *C, *P;
 int *Inside, *OnEdge;
 {
  double ax, ay, bx, by, cx, cy, apx, apy, bpx, bpy, cpx, cpy;
  double cCROSSap, bCROSScp, aCROSSbp, DOT;
  int OnEdgeAB, OnEdgeBC, OnEdgeCA;

  *Inside = FALSE; *OnEdge = FALSE;
  OnEdgeAB = FALSE; OnEdgeBC = FALSE; OnEdgeCA = FALSE;
  ax = C->x - B->x;  ay = C->y - B->y;
  bx = A->x - C->x;  by = A->y - C->y;
  cx = B->x - A->x;  cy = B->y - A->y;
  apx= P->x - A->x;  apy= P->y - A->y;
  bpx= P->x - B->x;  bpy= P->y - B->y;
  cpx= P->x - C->x;  cpy= P->y - C->y;

  aCROSSbp = ax*bpy - ay*bpx;
  cCROSSap = cx*apy - cy*apx;
  bCROSScp = bx*cpy - by*cpx;

  *Inside = ((aCROSSbp>0.0) && (bCROSScp>0.0) && (cCROSSap>0.0));

  if (!(*Inside))
   {
    if ((EPSILON > aCROSSbp) && (aCROSSbp > -EPSILON))
     {
      DOT = (ax*bpx)+(ay*bpy);
      OnEdgeBC = ((0.0 <= DOT) && (DOT <= (ax*ax)+(ay*ay)));
     }

    if ((EPSILON > bCROSScp) && (bCROSScp > -EPSILON))
     {
      DOT  = (bx*cpx)+(by*cpy);
      OnEdgeCA = ((0.0 <= DOT) && (DOT <= (bx*bx)+(by*by)));
     }

    if ((EPSILON > cCROSSap) && (cCROSSap > -EPSILON))
     {
      DOT = (cx*apx)+(cy*apy);
      OnEdgeAB = ((0.0 <= DOT) && (DOT <= (cx*cx)+(cy*cy)));
     }
    if ((OnEdgeAB) || (OnEdgeBC) || (OnEdgeCA))
     *OnEdge = TRUE;
   }
 } /* InsideTriangle */


/**************************************************************************/

 /*
   Assumes that P1, P2, P3 are three consecutve points of a polygon. Tests the
   remaining points to find out if any are inside the triangle P1P2P3.
  */
int NoPointsInside(ThePoly, P1, P2, P3)
 WEFaceNode *ThePoly;
 WEVertexNode *P1, *P2, *P3;
{
 WEVertexNode *Current;
 int Inside, OnEdge;

 Current = NextVertexCCW(ThePoly, P3);
 while(P1 != Current)
  {
   InsideTriangle(P1, P2, P3, Current, &Inside, &OnEdge);
   if(Inside || OnEdge)
    return(FALSE);
   else
    Current = NextVertexCCW(ThePoly, Current);
  }
 return(TRUE);
}

/***************************************************************************/

int IsTriangle(ThePoly)
 WEFaceNode *ThePoly;
{
 int VertexCount = 0;
 int Result;
 WingEdgeNode *CurrentEdge;

 CurrentEdge = ThePoly->Edges;
 do
  {
   VertexCount++;
   CurrentEdge = NextEdgeCCWFace(ThePoly, CurrentEdge);
  } while(VertexCount < 3);
 if(CurrentEdge == ThePoly->Edges)
  Result = TRUE;
 else
  Result = FALSE;
 return(Result);
}

/**************************************************************************/

int ConvexQuad(SharedEdge)
 WingEdgeNode *SharedEdge;
{
 WEVertexNode *P1, *P2, *P3, *P4;

 if((NULL != SharedEdge->Left) && (NULL != SharedEdge->Right))
  {
   P1 = SharedEdge->Org;
   P2 = NextVertexCCW(SharedEdge->Right, P1);
   P3 = SharedEdge->Dest;
   P4 = NextVertexCCW(SharedEdge->Left, P3);

   /* 
     this convoluted mess is because we don't know for sure that we are 
     looking from above in the z sense when we find the order for the 4 points
    */
   if( (LeftTurn(P1, P2, P3) && LeftTurn(P2, P3, P4) && 
	LeftTurn(P3, P4, P1) && LeftTurn(P4, P1, P2))
      ||
       (LeftTurn(P1, P4, P3) && LeftTurn(P4, P3, P2) &&
	LeftTurn(P3, P2, P1) && LeftTurn(P2, P1, P4))   )
    return(TRUE);
   else
    return(FALSE);
  }
 else
  return(FALSE);
}

/**************************************************************************/

void TilePoly(ThePoly)
 WEFaceNode *ThePoly;
{
 WingEdgeNode *E1, *E2, *CurrentEdge;
 WEVertexNode *P1, *P2, *P3;

 CurrentEdge = ThePoly->Edges;
 while(!IsTriangle(ThePoly))
  {
   E1 = CurrentEdge;
   E2 = NextEdgeCCWFace(ThePoly, E1);

   if(ThePoly == E1->Left)
    {
     P1 = E1->Org; P2 = E1->Dest;
    }
   else
    {
     P1 = E1->Dest; P2 = E1->Org;
    }
   if(ThePoly == E2->Left)
    P3 = E2->Dest;
   else
    P3 = E2->Org;

   if(LeftTurn(P1, P2, P3) && NoPointsInside(ThePoly, P1, P2, P3))
    {
     CurrentEdge = NextEdgeCWFace(ThePoly, CurrentEdge);
     if(ThePoly == E1->Left)
      P1 = E1->Org;
     else
      P1 = E1->Dest;
     if(ThePoly == E2->Left)
      P2 = E2->Dest;
     else
      P2 = E2->Org;
     MakeEdgeFace(P1, P3, ThePoly, CurrentEdge, E2);
     if(IsTriangle(ThePoly))
      ThePoly = NextFaceCCW(P1, ThePoly);
    }
   else
    CurrentEdge = NextEdgeCCWFace(ThePoly, CurrentEdge);
  }
}

/**************************************************************************/

WingEdgeNode *RemoveEdge(TheList)
 ListElement **TheList;
{
 ListElement *RemovedElement;
 WingEdgeNode *TheEdge = NULL;

 RemovedElement = *TheList;
 if(NULL != RemovedElement)
  {
   *TheList = RemovedElement->next;
   RemovedElement->next = NULL;
   TheEdge = RemovedElement->element;
   RemovedElement->element = NULL;
   free((char *) RemovedElement);
  }
 return(TheEdge);
}

/**************************************************************************/

void InsertEdge(TheList, TheEdge)
 ListElement **TheList;
 WingEdgeNode *TheEdge;
{
 ListElement *NewElement, *CurrentElement;
 int Found = FALSE;

 CurrentElement = *TheList;
 while((NULL != CurrentElement) && (!Found))
  {
   if(CurrentElement->element == TheEdge)
    Found = TRUE; /* edge already in the list */
   else
    CurrentElement = CurrentElement->next;
  }

 if(!Found)
  {
   /* the edge is not in the list, now add it at the head of the list */
   NewElement = (ListElement *) malloc(sizeof(ListElement));
   if(NULL == NewElement)
    {
     (void)fprintf(stderr,"InsertEdge: no memory!\n");
     exit(-1);
    }
   NewElement->element = TheEdge;
   NewElement->next = *TheList;
   (*TheList) = NewElement;
  }
}

/**************************************************************************/

void InitializeSuspects(SuspectList, TheTiling)
 ListElement **SuspectList;
 WEFaceNode *TheTiling;
{
 WEFaceNode *CurrentFace;
 WingEdgeNode *CurrentEdge;

 CurrentFace = TheTiling;
 do /* while CurrentFace != TheTiling */
  {
   CurrentEdge = CurrentFace->Edges;
   do /* while CurrentEdge != CurrentFace->Edges */
    {
     if(ConvexQuad(CurrentEdge))
      InsertEdge(SuspectList, CurrentEdge);
     CurrentEdge = NextEdgeCCWFace(CurrentFace, CurrentEdge);
    } while(CurrentEdge != CurrentFace->Edges);
   CurrentFace = CurrentFace->next;
  } while(CurrentFace != TheTiling);
}

/**************************************************************************/

void OptimizeTiling(TheTiling)
 WEFaceNode **TheTiling;
{
 ListElement *SuspectList = NULL;
 WingEdgeNode *CurrentEdge, *V1CCW, *V2CCW, *V1CW, *V2CW;
 WEVertexNode *V1, *V2;

 InitializeSuspects(&SuspectList, *TheTiling);
 while(NULL != SuspectList)
  {
   CurrentEdge = RemoveEdge(&SuspectList);
   if(Shrink(CurrentEdge))
    {
     V1 = NextVertexCCW(CurrentEdge->Right, CurrentEdge->Org);
     V1CCW = CurrentEdge->OCW; V1CW = CurrentEdge->OCCW;
     V2 = NextVertexCW(CurrentEdge->Left, CurrentEdge->Org);
     V2CCW = CurrentEdge->DCW; V2CW = CurrentEdge->DCCW;
     *TheTiling = KillEdgeFace(CurrentEdge);
     MakeEdgeFace(V1, V2, *TheTiling, V1CCW, V2CCW);
     if(ConvexQuad(V1CCW))
      InsertEdge(&SuspectList, V1CCW);
     if(ConvexQuad(V1CW))
      InsertEdge(&SuspectList, V1CW);
     if(ConvexQuad(V2CCW))
      InsertEdge(&SuspectList, V2CCW);
     if(ConvexQuad(V2CW))
      InsertEdge(&SuspectList, V2CW);
    }
  }
}

/**************************************************************************/

void DiscardEdge(TheEdge)
 WingEdgeNode *TheEdge;
{
 WingEdgeNode *DCW, *DCCW, *OCW, *OCCW;
 DCW = TheEdge->DCW; DCCW = TheEdge->DCCW;
 OCW = TheEdge->OCW; OCCW = TheEdge->OCCW;
 TheEdge->DCW = NULL; TheEdge->DCCW = NULL;
 TheEdge->OCW = NULL; TheEdge->OCCW = NULL;
  if(NULL != TheEdge->Left)
  {
   TheEdge->Left->Edges = NULL;
   TheEdge->Left->Vertices = NULL;
   TheEdge->Left->next = NULL;
   TheEdge->Left->prev = NULL;
   (void)free((char *) TheEdge->Left);
   TheEdge->Left = NULL;
  }
 if(NULL != TheEdge->Right)
  {
   TheEdge->Right->Edges = NULL;
   TheEdge->Right->Vertices = NULL;
   TheEdge->Right->next = NULL;
   TheEdge->Right->prev = NULL;
   (void)free((char *) TheEdge->Right);
   TheEdge->Right = NULL;
  }
 if(NULL != TheEdge->Org)
  {
   TheEdge->Org->Edges = NULL;
   (void)free((char *) TheEdge->Org);
  }
 if(NULL != TheEdge->Dest)
  {
   TheEdge->Dest->Edges = NULL;
   (void)free((char *) TheEdge->Dest);
  }
 (void)free((char *) TheEdge);
 if(NULL != DCW)
  DiscardEdge(DCW);
 if(NULL != DCCW)
  DiscardEdge(DCCW);
 if(NULL != OCW)
  DiscardEdge(OCW);
 if(NULL != OCCW)
  DiscardEdge(OCCW);
}

/**************************************************************************/

extern void TileSaddle(TheSection, C1, StartC1, EndC1, C2, StartC2, EndC2,
		Previous, TriangleNumber)
 Section *TheSection;
 int C1, StartC1, EndC1;
 int C2, StartC2, EndC2;
 int Previous; /* Needed to draw saddle tiles with proper orientation */
 int *TriangleNumber;
{
 WEFaceNode *ThePolygon = NULL;
 WEVertexNode *FirstVertex = NULL, *CurrentVertex = NULL;
 WingEdgeNode *CurrentEdge = NULL, *FirstEdge = NULL;
 ContourDescriptor *C1Desc = NULL, *C2Desc = NULL;
 int NC1, NC2;
 int C1Pts, C2Pts;
 int i, j;

 C1Desc = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));
 C2Desc = (ContourDescriptor *) malloc(sizeof(ContourDescriptor));

 C1Pts = TheSection->TheContours[C1].ContourPoints->n;
 C2Pts = TheSection->TheContours[C2].ContourPoints->n;

 if(EndC1 >= StartC1)
  NC1 = EndC1 + 1 - StartC1; 
 else
  NC1 = EndC1 + C1Pts + 1 - StartC1;

 if(EndC2 >= StartC2)
  NC2 = EndC2 + 1 - StartC2; 
 else
  NC2 = EndC2 + C2Pts + 1 - StartC2;

 C1Desc->Index = C1;            C2Desc->Index = C2; 
 C1Desc->Npts = NC1;            C2Desc->Npts = NC2; 
 C1Desc->FirstPoint = StartC1;  C2Desc->FirstPoint = StartC2;
 C1Desc->LastPoint = EndC1;     C2Desc->LastPoint = EndC2;
 (void)strcpy(C1Desc->SectionName, TheSection->Name);
 (void)strcpy(C2Desc->SectionName, TheSection->Name);
 (void)strcpy(C1Desc->Name, TheSection->TheContours[C1].Name);
 (void)strcpy(C2Desc->Name, TheSection->TheContours[C2].Name);

 /*
   A polygon is constructed from the points in the adjacency region. The order
   of the points is assumed to be counterclockwise as viewed from the positive
   z axis. If the order is reversed, problems will rear their ugly heads. This
   is a BUG to be worked out in the future. We construct a polygon using the
   Wing Edge data structure. The initial product has an "interior" and 
   "exterior" face. Appropriate selection of the face to be retained results in
   a counterclocwise specified polygon. What we are doing is selecting which 
   side of the face to view the polygon from. This is done with the 
   KillFaceMakeBoundary routine.
  */

 for(i=StartC1, j=0; j<NC1; i=(i+1)%C1Pts, j++)
  {
   if(NULL == ThePolygon)
    {
     MakeVertexFaceShell(&ThePolygon, &FirstVertex);
     CurrentVertex = FirstVertex;
    }
   else if(NULL == CurrentEdge)
    {
     MakeEdgeVertex(ThePolygon, CurrentVertex, CurrentVertex->Edges);
     CurrentEdge = CurrentVertex->Edges;
     FirstEdge = CurrentEdge;
     CurrentVertex = NextVertexCCW(ThePolygon, CurrentVertex);
    }
   else
    {
     MakeEdgeVertex(ThePolygon, CurrentVertex, CurrentEdge);
     CurrentVertex = NextVertexCCW(ThePolygon, CurrentVertex);
     CurrentEdge = NextEdgeCCWFace(ThePolygon, CurrentEdge);
    }
   (void)strcpy(CurrentVertex->PointName, PointName(TheSection, j, C1Desc));
   CurrentVertex->x = TheSection->TheContours[C1].ContourPoints->P[i].x;
   CurrentVertex->y = TheSection->TheContours[C1].ContourPoints->P[i].y;
  }
 for(i=StartC2, j=0; j<NC2; i=(i+1)%C2Pts, j++)
  {
   if(NULL == CurrentEdge) /* the last section may have only had one point */
    {
     MakeEdgeVertex(ThePolygon, CurrentVertex, CurrentVertex->Edges);
     CurrentEdge = CurrentVertex->Edges;
     FirstEdge = CurrentEdge;
     CurrentVertex = NextVertexCCW(ThePolygon, CurrentVertex);
    }
   else
    {
     MakeEdgeVertex(ThePolygon, CurrentVertex, CurrentEdge);
     CurrentVertex = NextVertexCCW(ThePolygon, CurrentVertex);
     CurrentEdge = NextEdgeCCWFace(ThePolygon, CurrentEdge);
    }
   (void)strcpy(CurrentVertex->PointName, PointName(TheSection, j, C2Desc));
   CurrentVertex->x = TheSection->TheContours[C2].ContourPoints->P[i].x;
   CurrentVertex->y = TheSection->TheContours[C2].ContourPoints->P[i].y;
  }
 MakeEdgeFace(CurrentVertex, FirstVertex, ThePolygon, CurrentEdge, FirstEdge);

 /*
   The next step selects an orientation from which the polygon is viewed.
   Choosing the Right face of FirstEdge as the interior of the polygon 
   gives us the proper orientation, in this case.
  */
 ThePolygon = FirstEdge->Right;
 KillFaceMakeBoundary(FirstEdge->Left);

 TilePoly(ThePolygon);
 OptimizeTiling(&ThePolygon);
 OutputShell(ThePolygon, Previous, TriangleNumber);
 DiscardEdge(ThePolygon->Edges);
}
