/*********************************************************************\
* 								      *
* File: OptimalArea.c						      *
* Purpose: Form the optimal tiling between two contours based on the  *
*          minimize surface area metric.			      *
* Author:  James Painter, David Meyers				      *
* Last Modified: 21 November 1989    				      *
* 								      *
\*********************************************************************/

#include <math.h>
#include <stdio.h>
#include <TypeDefinitions.h>
#include "BitArray.h"
#include "Contours.h"
#include "Vector.h"
#include "TorGraph.h"
extern char *malloc();

extern REAL SurfaceArea();

static GraphType *G = NULL;


void ReleaseCostGraph()
 {
  DisposeGraph( G );
  G = NULL;
 }
  
BitArray OptimalArea ( Contour1, Contour2, start1, start2 )
/* Returns a direction array representing a triangulation between two
   contours.
   The direction array is a binary array of n+m bits (where n and m are
   the number of points in the top and bottom contours respectively)
   indicating whether the next arc should be along the top contour
   (bit set) or along the bottom contour (bit clear).  The starting
   point on each contour are also returned as outputs 
*/
 PointsType *Contour1,  *Contour2;
 int        *start1, *start2;  /* Starting points on contours */
 {
  BitArray  result;
  PathType  *P;
  int i, index1, index2, nArcs;
  REAL MinCost, MaxCost;

  /*
  ** Alocate storage for the graph.
  */
  if (G == NULL)
   {
    if ( (
          G=MakeCostGraph( Contour1, Contour2, SurfaceArea,
                           &MinCost, &MaxCost)
          ) == NULL
       )
         return (BitArray) 0;
   }

  /*
  ** Find the optimal path in the graph.
  */
  (void) BtlnckAllPaths ( G, &P, *start1, *start2 );

  /*
  ** Construct the bit array describing the path for the caller.
  */
  nArcs = G->Rows + G->Columns;
  if (! (result = NewBitArray( (unsigned) nArcs ) ) )
   {
    (void) fprintf(stderr, " OptimalArea: No Memory!\n");
    DisposeGraph( G );
    G = NULL;
    DisposePath( P );
    return (BitArray) 0;
   }

  index1 = P->StartR;
  index2 = P->StartC;
  for ( i=0; i<nArcs; i++ )
   {
    if (P->LeavesColInRow[index2] == index1)
     {
      ClearBit( i, result );
      index2++;
      index2 %= (2*G->Columns);
     }
    else if (P->LeavesRowInCol[index1] == index2)
     {
      SetBit( i, result );
      index1++;
      index1 %= (2*G->Rows);
     }
    else
     {
      (void) fprintf( stderr, "Path error in OptimalArea.\n" );
     }
   }
  DisposePath ( P );
  *start1 = P->StartR;
  *start2 = P->StartC;
  *start1 %= G->Rows;
  *start2 %= G->Columns;
  ReleaseCostGraph();
  return result;
 }
