/****************************************************************************
*
*                         The Universal VESA VBE
*
*                   Copyright (C) 1993 Kendall Bennett.
*                           All rights reserved.
*
* Filename:     $RCSfile: _s3.c $
* Version:      $Revision: 1.1 $
*
* Language:     ANSI C
* Environment:  IBM PC (MS DOS)
*
* Description:  C language support routines for S3 SuperVGA's. Contains
*               detection code, mode translation tables and chipset
*               names.
*
* $Id: _s3.c 1.1 1993/09/19 01:26:26 kjb release $
*
* Revision History:
* -----------------
*
* $Log: _s3.c $
* Revision 1.1  1993/09/19  01:26:26  kjb
* Initial revision
*
****************************************************************************/

/* Mode translation table to determine the values to place into
 * AX and BX in order to initialise a particular video mode
 * via int 10h.
 */

short S3Modes[] = {     0x4F02, 0x102,  /* 800x600 16 color     */
                        0x4F02, 0x104,  /* 1024x768 16 color    */
                        0x4F02, 0x106,  /* 1280x1024 16 color   */
                        0, 0,           /* 640x350 256 color    */
                        0, 0,           /* 640x400 256 color    */
                        0x4F02, 0x101,  /* 640x480 256 color    */
                        0x4F02, 0x103,  /* 800x600 256 color    */
                        0x4F02, 0x105,  /* 1024x768 256 color   */
                        0x4F02, 0x107,  /* 1280x1024 256 color  */
                        0, 0,           /* 320x200 32k color    */
                        0, 0,           /* 640x350 32k color    */
                        0, 0,           /* 640x400 32k color    */
                        0x4F02, 0x301,  /* 640x480 32k color    */
                        0, 0,           /* 800x600 32k color    */
                        0, 0,           /* 1024x768 32k color   */
                        0, 0,           /* 1280x1024 32k color  */
                        0, 0,           /* 320x200 64k color    */
                        0, 0,           /* 640x350 64k color    */
                        0, 0,           /* 640x400 64k color    */
                        0x4F02, 0x111,  /* 640x480 64k color    */
                        0x4F02, 0x114,  /* 800x600 64k color    */
                        0x4F02, 0x117,  /* 1024x768 64k color   */
                        0x4F02, 0x11A,  /* 1280x1024 64k color  */
                        0, 0,           /* 320x200 16m color    */
                        0, 0,           /* 640x350 16m color    */
                        0, 0,           /* 640x400 16m color    */
                        0x4F02, 0x112,  /* 640x480 16m color    */
                        0, 0,           /* 800x600 16m color    */
                        0, 0,           /* 1024x768 16m color   */
                        0, 0,           /* 1280x1024 16m color  */
                        };

typedef enum {
    grS3_911,               /* S3 86c911 SuperVGA chip                  */
    grS3_924,               /* S3 86c924 SuperVGA chip                  */
    grS3_80x,               /* S3 86c801/86c805 SuperVGA chip           */
    grS3_928,               /* S3 86c928 SuperVGA chip                  */
    } S3_chipsets;

/* Names of chip revision id's. */

char *S3Names[] = {
    "86c911",
    "86c924",
    "86c801/805",
    "86c928",
    };

void setS3BytesPerLine(int mode)
/****************************************************************************
*
* Function:     setS3BytesPerLine
* Parameters:   mode    - S3 video mode number
*
* Description:  Obtains the bytes per line value from the S3 BIOS for
*               the specific video mode, and updates the values for this
*               mode.
*
****************************************************************************/
{
    ModeInfoBlock   modeInfo;
    union REGS      regs;
    struct SREGS    sregs;

    sregs.es = FP_SEG(&modeInfo);
    regs.x.di = FP_OFF(&modeInfo);
    regs.x.ax = 0x4F01;
    regs.x.cx = mode;
    int86x(0x10,&regs,&regs,&sregs);
    if (regs.x.ax != 0x004F)
        unsupportedMode(mode,S3Modes);  /* Call failed, so remove mode  */
    else
        setBytesPerLine(mode,modeInfo.BytesPerScanLine);
}

bool findS3(int *superVGA,int *chipID,int *memory,int *dac,int *pageFlip)
/****************************************************************************
*
* Function:     findS3
* Parameters:   superVGA    - ID of the SuperVGA card
*               chipID      - Internal chip ID number
*               memory      - Amount of memory of card
*               dac         - Type of DAC installed
*               pageFlip    - True if page flipping supported
* Returns:      True if card was detected
*
* Description:  Detects the presence of S3 based SuperVGA's.
*
****************************************************************************/
{
    int     oldLock,val;

    oldLock = rdinx(CRTC,0x38);     /* Read old value of reg lock       */
    wrinx(CRTC,0x38,0);             /* Lock S3 registers                */
    if (tstinx(CRTC,0x35,0x0F))     /* Check for locked bank reg        */
        goto NoS3;                  /* Reg was writeable, not S3        */
    wrinx(CRTC,0x38,0x48);          /* Unlock S3 registers              */
    if (!tstinx(CRTC,0x35,0x0F))    /* Check for unlocked bank reg      */
        goto NoS3;                  /* Reg was not writeable, not S3    */

    /* We have an S3 based SuperVGA */

    *superVGA = grSVGA_S3;
    *pageFlip = true;

    /* Determine the chipset type */

    val = rdinx(CRTC,0x30);         /* Read chip id register            */
    *chipID = grS3_911;             /* Assume we have an 86c911         */
    if ((val & 0xF0) == 0x90)
        *chipID = grS3_928;         /* We have an 86c928                */
    else if ((val & 0xF0) == 0xA0)
        *chipID = grS3_80x;         /* We have an 86c801/805            */
    else if (val == 0x82)
        *chipID = grS3_924;         /* We have an 86c924                */

    /* Determine the amount of memory on board */

    val = rdinx(CRTC,0x36) & 0xE0;  /* Read Config reg 1 bits 7-5       */
    *memory = 1024;                 /* Assume 1Mb to begin with         */
    if (val & 0x20)
        *memory = 512;              /* Bit 5 set for 512k memory        */
    if (*chipID >= grS3_80x) {      /* Extra check for 80x/928's        */
        switch (val) {
            case 0x40:
                *memory = 3072;
                break;
            case 0x80:
                *memory = 2048;
                break;
            case 0x00:
                *memory = 4096;
                break;
            }
        }
    wrinx(CRTC,0x38,0);             /* Lock extended registers          */

    /* Modify the bytes per line values for the S3 video card modes. We
     * need to do this since the values depend upon the current
     * operating environment and chipset.
     */

    setS3BytesPerLine(vbe_800x600x16);
    setS3BytesPerLine(vbe_1024x768x16);
    setS3BytesPerLine(vbe_1280x1024x16);
    setS3BytesPerLine(vbe_640x480x256);
    setS3BytesPerLine(vbe_800x600x256);
    setS3BytesPerLine(vbe_1024x768x256);
    setS3BytesPerLine(vbe_640x480x64k);
    setS3BytesPerLine(vbe_800x600x64k);
    setS3BytesPerLine(vbe_1024x768x64k);
    setS3BytesPerLine(vbe_1280x1024x64k);
    setS3BytesPerLine(vbe_640x480x16m);

    /* On the 911 and 924, the VESA 32k color mode is not available, so
     * we use the enhanced one instead. We can't directly use this mode
     * on the 80x/928's without special code to gain direct access to
     * the video memory, but the 80x/928's support the VESA mode anyway.
     */

    if (*chipID < grS3_80x)
        setBytesPerLine(vbe_640x480x32k,2048);
    else {
        setS3BytesPerLine(vbe_640x480x32k);
        S3Modes[25] = 0x110;        /* Use the VESA mode for 80x/928    */
        }

    return true;

NoS3:
    wrinx(CRTC,0x38,oldLock);       /* Write old value of reg lock      */
    return false;
}
