/****************************************************************************
*
*                         The Universal VESA VBE
*
*                   Copyright (C) 1993 Kendall Bennett.
*                           All rights reserved.
*
* Filename:     $RCSfile: detect.c $
* Version:      $Revision: 1.2 $
*
* Language:     ANSI C
* Environment:  IBM PC (MS DOS)
*
* Description:  Module to autodetect the type of installed video card. This
*               has now been written entirely in C to aid in portability
*               and maintainability.
*
*               No apologies are made for the liberal use of gotos in the
*               detection code. This code is complicated and hardware
*               specific, and was never meant to be elegant :-)
*
*               MUST be compiled in the large memory model.
*
* $Id: detect.c 1.2 1993/09/24 05:22:02 kjb release $
*
* Revision History:
* -----------------
*
* $Log: detect.c $
* Revision 1.2  1993/09/24  05:22:02  kjb
* Fixed a number of bugs.
*
* Revision 1.1  1993/09/19  01:26:00  kjb
* Initial revision
*
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <string.h>
#include "univbe.h"

#pragma hdrstop

/*----------------------------- Implementation ----------------------------*/

/* Index table entry structure. This is used to fill in most of the
 * transient mode information for all the different video modes when
 * requested by an application. Each entry also contains the values used
 * to actually initialise this mode by calling int 10 (these will be
 * set to zero if the mode is not available).
 *
 * Note also that some of the value may be changed and will be recalculated
 * before the information is finally passed to the resident portion of
 * the TSR.
 */

typedef struct {
    short   VBEMode;            /* Internal VBE mode number             */
    short   BytesPerScanLine;   /* Bytes per logical scanline           */
    short   XResolution;        /* Horizontal pixel resolution          */
    short   YResolution;        /* Vertical pixel resolution            */
    char    XCharSize;          /* Character cell width                 */
    char    YCharSize;          /* Character cell height                */
    char    NumberOfPlanes;     /* Number of memory planes              */
    char    BitsPerPixel;       /* Total bits per pixel                 */
    char    NumberOfBanks;      /* Number of memory banks (CGA style)   */
    char    MemoryModel;        /* Memory model type                    */
    char    BankSize;           /* Banks size (CGA style)               */
    char    NumberOfImagePages; /* Number of spare image pages          */
    short   AXVal;              /* Value to store in AX for mode set    */
    short   BXVal;              /* Value to store in BX for mode set    */
    } IndexEntry;

typedef enum {
    memTXT      = 0,            /* Text mode memory model               */
    memCGA      = 1,            /* CGA style mode                       */
    memHGC      = 2,            /* Hercules graphics style mode         */
    memPL       = 3,            /* 16 color VGA style planar mode       */
    memPK       = 4,            /* Packed pixel mode                    */
    memX        = 5,            /* Non-chain 4, 256 color (ModeX)       */
    memRGB      = 6,            /* Direct color RGB mode                */
    memYUV      = 7,            /* Direct color YUV mode                */
    } memModels;

/* Index table used to lookup the values for a particular video mode
 * given the video mode number. This will be filled in with the correct
 * values and then copied in it's entirety to the resident portion
 * of the TSR.
 */

#define COLORS16    4,4,1,memPL,0,0,0,0
#define COLORS256   1,8,1,memPK,0,0,0,0
#define COLORS32K   1,15,1,memRGB,0,0,0,0
#define COLORS64K   1,16,1,memRGB,0,0,0,0
#define COLORS16M   1,24,1,memRGB,0,0,0,0

IndexEntry IndexTable[] = {
    {0x102,100,800,600,8,8,COLORS16},           /* 800x600 16 color     */
    {0x104,128,1024,768,8,8,COLORS16},          /* 1024x768 16 color    */
    {0x106,160,1280,1024,8,8,COLORS16},         /* 1280x1024 16 color   */
    {0x11C,640,640,350,8,8,COLORS256},          /* 640x350 256 color    */
    {0x100,640,640,400,8,8,COLORS256},          /* 640x400 256 color    */
    {0x101,640,640,480,8,8,COLORS256},          /* 640x480 256 color    */
    {0x103,800,800,600,8,8,COLORS256},          /* 800x600 256 color    */
    {0x105,1024,1024,768,8,8,COLORS256},        /* 1024x768 256 color   */
    {0x107,1280,1280,1024,8,8,COLORS256},       /* 1280x1024 256 color  */
    {0x10D,640,320,200,8,8,COLORS32K},          /* 320x200 32k color    */
    {0x11D,1280,640,350,8,8,COLORS32K},         /* 640x350 32k color    */
    {0x11E,1280,640,400,8,8,COLORS32K},         /* 640x400 32k color    */
    {0x110,1280,640,480,8,8,COLORS32K},         /* 640x480 32k color    */
    {0x113,1600,800,600,8,8,COLORS32K},         /* 800x600 32k color    */
    {0x116,2048,1024,768,8,8,COLORS32K},        /* 1024x768 32k color   */
    {0x119,2560,1280,1024,8,8,COLORS32K},       /* 1280x1024 32k color  */
    {0x10E,640,320,200,8,8,COLORS64K},          /* 320x200 64k color    */
    {0x11F,1280,640,350,8,8,COLORS64K},         /* 640x350 64k color    */
    {0x120,1280,640,400,8,8,COLORS64K},         /* 640x400 64k color    */
    {0x111,1280,640,480,8,8,COLORS64K},         /* 640x480 64k color    */
    {0x114,1600,800,600,8,8,COLORS64K},         /* 800x600 64k color    */
    {0x117,2048,1024,768,8,8,COLORS64K},        /* 1024x768 64k color   */
    {0x11A,2560,1280,1024,8,8,COLORS64K},       /* 1280x1024 64k color  */
    {0x10F,960,320,200,8,8,COLORS16M},          /* 320x200 16m color    */
    {0x121,1920,640,350,8,8,COLORS16M},         /* 640x350 16m color    */
    {0x122,1920,640,400,8,8,COLORS16M},         /* 640x400 16m color    */
    {0x112,1920,640,480,8,8,COLORS16M},         /* 640x480 16m color    */
    {0x115,2400,800,600,8,8,COLORS16M},         /* 800x600 16m color    */
    {0x118,3072,1024,768,8,8,COLORS16M},        /* 1024x768 16m color   */
    {0x11B,3840,1280,1024,8,8,COLORS16M},       /* 1280x1024 16m color  */
    {-1,0,0,0,0,0,0,0,0,0,0,0,0}                /* Terminate the list   */
    };

short validModes[31];                           /* Table of valid modes */
short oldVBEModes[50];                          /* Pointer to old modes */

void unsupportedMode(int VBEMode,short *modeTable)
/****************************************************************************
*
* Function:     unsupportedMode
* Parameters:   VBEMode     - Internal VBE mode number
*               modeTable   - Chipset mode translation table
*
* Description:  Removes support for the specified VBE mode number from
*               the mode tables. This routine can be called from the
*               SuperVGA detection modules to remove specific modes from
*               operation depending on the video configuration.
*
****************************************************************************/
{
    int     i;

    for (i = 0; IndexTable[i].VBEMode != -1; i++) {
        if (IndexTable[i].VBEMode == VBEMode) {
            modeTable[i*2] = modeTable[i*2 + 1] = 0;
            return;
            }
        }
}

void setBytesPerLine(int VBEMode,int bytesPerLine)
/****************************************************************************
*
* Function:     setBytesPerLine
* Parameters:   VBEMode         - Internal VBE mode number
*               bytesPerLine    - New value for bytes per line
*
* Description:  Sets the logical bytes per scanline value for a specific
*               video mode. This can be called from the SuperVGA detection
*               to adjust the bytes per line values for non-standard
*               widths.
*
****************************************************************************/
{
    int     i;

    for (i = 0; IndexTable[i].VBEMode != -1; i++) {
        if (IndexTable[i].VBEMode == VBEMode) {
            /* We have found the entry to use, however we must also
             * ensure that the value is valid. Some BIOS return bad
             * values for this (the STB WIND/X S3 card does). If the
             * value is smaller than required, simply double it.
             */

            if (bytesPerLine < IndexTable[i].BytesPerScanLine)
                bytesPerLine *= 2;
            IndexTable[i].BytesPerScanLine = bytesPerLine;
            return;
            }
        }
}

static int oldMode,old50Lines,oldx,oldy;

void saveMode(void)
/****************************************************************************
*
* Function:     saveMode
*
* Description:  Saves the current video mode to be restored at a later
*               date.
*
****************************************************************************/
{
    union REGS  regs;

    regs.h.ah = 0x0F;
    int86(0x10,&regs,&regs);            /* Save previous video mode     */
    oldMode = regs.h.al & 0x7F;
    regs.h.ah = 0x3;
    int86(0x10,&regs,&regs);            /* Save previous cursor pos     */
    oldx = regs.h.dl;
    oldy = regs.h.dh;
    regs.x.ax = 0x1130;
    regs.h.bh = 0x00;
    regs.h.dl = 0x00;
    int86(0x10,&regs,&regs);
    old50Lines = (regs.h.dl == 49);
}

void restoreMode(void)
/****************************************************************************
*
* Function:     restoreMode
*
* Description:  Restores the previous video mode, without clearing the
*               display screen. This routine and the above routine can
*               be used to check for specific features.
*
****************************************************************************/
{
    union REGS  regs;
    uchar       *biosMode = FP(0x40,0x49);

    regs.h.ah = 0x0;
    regs.h.al = oldMode | 0x80; /* Set old mode - don't clear display   */
    int86(0x10,&regs,&regs);
    *biosMode &= 0x7F;          /* Remove clear bit in BIOS data area   */
    if (old50Lines) {
        regs.x.ax = 0x1112;
        regs.h.bl = 0x00;
        int86(0x10,&regs,&regs);
        }
    regs.h.ah = 0x02;
    regs.h.dl = oldx;
    regs.h.dh = oldy;
    regs.h.bh = 0x00;
    int86(0x10,&regs,&regs);    /* Restore cursor position              */
}

int checkForVESAVBE(void)
/****************************************************************************
*
* Function:     checkForVESAVBE
* Returns:      Version of VESA VBE BIOS (0 if none)
*
* Description:  Determines if a VESA VBE compatible BIOS is out there, and
*               returns the version number of it if it is.
*
****************************************************************************/
{
    VgaInfoBlock    info;
    union REGS      regs;
    struct SREGS    sregs;
    short           *p,i;

    sregs.es = FP_SEG(&info);
    regs.x.di = FP_OFF(&info);
    regs.x.ax = 0x4F00;
    int86x(0x10,&regs,&regs,&sregs);
    if (regs.x.ax != 0x004F)
        return false;
    if (strncmp(info.VESASignature,"VESA",4) != 0)
        return false;

    for (i = 0,p  = info.VideoModePtr; *p != -1; p++) {
        if (*p != 0)
            oldVBEModes[i++] = *p;
        }
    oldVBEModes[i] = -1;
    return info.VESAVersion;
}

bool isVGA(void)
/****************************************************************************
*
* Function:     isVGA
* Returns:      True if the card is a VGA compatible card.
*
* Description:  Attempts to determine if the video card is VGA compatible
*               or not.
*
****************************************************************************/
{
    union REGS  regs;

    regs.x.ax = 0x1A00;
    int86(0x10,&regs,&regs);
    if (regs.h.al != 0x1A)
        return false;               /* PS2 style BIOS is out there      */

    /* Check for active or inactive VGA color video card installed      */

    if (regs.h.bl == 0x08 || regs.h.cl == 0x07)
        return true;

    return false;
}

void dacToPel(void)
/****************************************************************************
*
* Function:     dacToPel
*
* Description:  Forces the DAC into PEL mode.
*
****************************************************************************/
{
    inp(0x3C8);
}

int dacToCommand(void)
/****************************************************************************
*
* Function:     dacToCommand
* Returns:      Last value read from register
*
* Description:  Enters the command mode of HiColor DAC's
*
****************************************************************************/
{
    dacToPel();
    inp(0x3C6);
    inp(0x3C6);
    inp(0x3C6);
    return inp(0x3C6);
}

int findDAC(void)
/****************************************************************************
*
* Function:     findDAC
* Returns:      Id of the Video DAC installed.
*
* Description:  Determines the type of video DAC installed in the system.
*
****************************************************************************/
{
    int     x,y,oldPel,oldCommand,dac = grVGADAC;

	/* Save the state of the PEL and command registers */

	dacToCommand();
	oldCommand = inp(0x3C6);
	dacToPel();
	oldPel = inp(0x3C6);

	/* Test for the SS24 TrueColor DAC first */

	dacToPel();
	x = inp(0x3C6);
	do {
		y = x;
		x = inp(0x3C6);
		} while (x != y);       /* Wait for the same value twice        */
	if (dacToCommand() == 0x8E) {
		dac = grTCDAC;         	/* We have an SS24 DAC                  */
		goto FoundDAC;
		}
	for (x = 0; x < 8; x++)
		if (inp(0x3C6) == 0x8E) {
			dac = grTCDAC;		/* We have an SS24 DAC                  */
			goto FoundDAC;
			}

	dacToPel();
    x = oldCommand ^ 0xFF;
    outp(0x3C6,x);
    dacToCommand();
    y = inp(0x3C6);
    if (y != x) {
        /* The command register is present and working, so we must have
         * a HiColor or TrueColor DAC of some type.
         */

        dac = grHCDAC;          /* Default to HiColor 15 bit dac        */
        dacToCommand();
        x = oldCommand ^ 0x60;
        outp(0x3C6,x);
        dacToCommand();
        if ((x & 0xE0) == (inp(0x3C6) & 0xE0)) {
            /* We cannot tell the difference between the Sierra 32k/64k
             * DAC's, so we assume that it is a 16 bit dac so that we can
             * at least run the 16 bit modes. This can be overridden from
             * the command line in the few cases it fails.
             */

            dac = grHC2DAC;
            x = inp(0x3C6);
            dacToPel();
            if (x == inp(0x3C6))
                dac = grTCDAC;
            }

        dacToCommand();
        outp(0x3C6,oldCommand); /* Restore old command register         */
        }

FoundDAC:
	dacToPel();
	outp(0x3C6,oldPel);         /* Restore old PEL register             */
	return dac;
}

/* Include SuperVGA detection code and mode tables. One source file is
 * assigned to each of the specific video chipsets supported to aid
 * in maintainability and extensibility.
 */

#pragma warn -par

#include "_ati.c"
#include "_ahead.c"
#include "_chips.c"
#include "_everex.c"
#include "_genoa.c"
#include "_oak.c"
#include "_paradis.c"
#include "_trident.c"
#include "_video7.c"
#include "_tseng.c"
#include "_ncr.c"
#include "_s3.c"
#include "_acumos.c"
#include "_avance.c"
#include "_mxic.c"
#include "_primus.c"
#include "_realtek.c"
#include "_cirrus.c"

#include "_tables.c"

void computeModeInfo(int superVGA,int memory,int dac,bool pageFlip,
    int VBEcompliant)
/****************************************************************************
*
* Function:     computeModeInfo
* Parameters:   SuperVGA    - SuperVGA to install for
*               memory      - Amount of memory on board
*               dac         - Type of DAC on board
*               pageFlip    - True if extended page flipping is available
*               VBEcompliant- True if old BIOS is VBE compliant
*
* Description:  Computes the correct values for the mode information
*               table above, such as the number of image pages and
*               moves the values used to initialise the mode into the
*               mode table. Once this is done we then pass the mode
*               information block to the resident portion of the TSR to
*               be stored in memory.
*
****************************************************************************/
{
    ModeInfoBlock   modeInfo;
    union REGS      regs;
    struct SREGS    sregs;
    int             i,maxBits;
    bool            found;
    long            size,memsize = memory * 1024L;
    short           *modeTable,*validTable = validModes,*p;
    IndexEntry      *entry;

    modeTable = ModeTables[superVGA];   /* Get correct mode table       */

    if (dac == grVGADAC)                /* Determine maximum bit depth  */
        maxBits = 8;
    else if (dac == grHCDAC)
        maxBits = 15;
    else if (dac == grHC2DAC)
        maxBits = 16;
    else maxBits = 24;

    for (entry = &IndexTable[0]; entry->VBEMode != -1; entry++,modeTable += 2) {
        /* For each mode in the index table, compute the correct number
         * of available video pages and filter out those modes that are
         * not supported by the DAC or the amount of memory on board. We
         * also build the table of valid modes as we go.
         */

        if (entry->BitsPerPixel > maxBits)
            continue;                   /* Not supported by DAC         */

        /* Compute the size of each video page rounded to 64k           */

        size = (long)entry->BytesPerScanLine * (long)entry->YResolution;
        if (entry->MemoryModel == memPL)
            size *= 4;
        size = (size + 0xFFFFL) & 0xFFFF0000L;
        if (size > memsize)
            continue;                   /* Too little memory for mode   */

        if (pageFlip)
            entry->NumberOfImagePages = (memsize / size) - 1;

        for (i = 0; removedModes[i] != -1; i++)
            if (removedModes[i] == entry->VBEMode)
                break;
        if (removedModes[i] == entry->VBEMode)
            continue;               /* Mode was removed in .ini file    */

        entry->AXVal = modeTable[0];    /* Copy initialisation values   */
        entry->BXVal = modeTable[1];
        if (entry->AXVal != 0 || entry->BXVal != 0)
            *validTable++ = entry->VBEMode; /* Mode is available        */
        }
    *validTable = -1;                   /* Terminate valid modes table  */

    if (VBEcompliant) {
        /* The underlying BIOS was VBE compliant to begin with, so copy
         * any modes that the old BIOS knows about that we dont handle
         * into the mode table (such as extended text modes).
         */

        for (p = oldVBEModes; *p != -1; p++) {
            for (found = i = 0; validModes[i] != -1; i++) {
                if (*p == validModes[i]) {
                    found = true;
                    break;
                    }
                }
            if (!found) {
                /* Make sure the mode is really valid by calling
                 * the mode information routine, which will fail if it
                 * is not available. Some VESA BIOSes list the modes
                 * even if they aren't available.
                 */

                sregs.es = FP_SEG(&modeInfo);
                regs.x.di = FP_OFF(&modeInfo);
                regs.x.ax = 0x4F01;
                regs.x.cx = *p;
                int86x(0x10,&regs,&regs,&sregs);

                for (i = 0; removedModes[i] != -1; i++)
                    if (removedModes[i] == *p)
                        break;

                if (removedModes[i] != *p &&
                        regs.x.ax == 0x004F &&
                        (modeInfo.ModeAttributes & 0x1) &&
                        modeInfo.MemoryModel == memTXT) {
                    *validTable++ = *p;
                    *validTable = -1;
                    }
                }
            }
        }

    /* Copy the mode information to the resident portion */

    copyModeInfo(IndexTable,sizeof(IndexTable));
    copyValidModes(validModes,sizeof(validModes));
}

/* Define a macro to simplify the method to check for each video card   */

#define check(card)                                                     \
    if (find##card(&_superVGA,&_chipID,&_memory,dac,pageFlip))          \
        goto found

void detect(int *superVGA,int *chipID,int *memory,int *dac,int *pageFlip,
    int *VESAcompliant)
/****************************************************************************
*
* Function:     detect
* Parameters:   superVGA        - Place to return the type of SuperVGA
*               chipID          - Place to return the chipID
*               memory          - Place to return the memory
*               dac             - Place to return the dac
*               pageFlip        - True if card supports page flipping
*               VESAcompliant   - Version of VESA compliant BIOS (0 for none)
*
* Description:  Autodetects the type of installed video card. First we
*               determine if a VGA card is installed, and then determine
*               what type of SuperVGA is installed, and whether the SuperVGA
*               already has a VESA VBE compliant BIOS.
*
*               The values passed in to this routine should be -1 if the
*               value is to be autodetected, otherwise the value will
*               be forced to the specific value required.
*
****************************************************************************/
{
    int _superVGA,_chipID,_memory;

    if (!isVGA()) {
        *superVGA = grSVGA_NONE;
        return;
        }

    *VESAcompliant = checkForVESAVBE();

    /* Auto detect the type of Video DAC present */

    if (*dac == -1) *dac = findDAC();

    /* Now check for the presence of all the supported SuperVGA adapters.
     * The order that the detection routines are called in below is
     * important, since some cards will answer to a number of tests.
     */

    check(Chips);
    check(Paradise);
    check(Video7);
    check(Genoa);
    check(Everex);
    check(Trident);
    check(ATI);
    check(Ahead);
    check(NCR);
    check(S3);
    check(Avance);
    check(MXIC);
    check(Cirrus);
    check(Acumos);
    check(Tseng);
    check(RealTek);
    check(Primus);
    check(Oak);

    *superVGA = grSVGA_NONE;
    return;

found:
    /* Save detected value for values that have not been forced from
     * the command line.
     */

    if (*superVGA == -1)    *superVGA = _superVGA;
    if (*chipID == -1)      *chipID = _chipID;
    if (*memory == -1)      *memory = _memory;

    computeModeInfo(*superVGA,*memory,*dac,*pageFlip,*VESAcompliant);
}

char *getSuperVGAName(int SuperVGA)
/****************************************************************************
*
* Function:     getSuperVGAName
* Parameters:   SuperVGA    - SuperVGA to retrieve name for
* Returns:      Pointer to the string describing the SuperVGA
*
* Description:  Obtains a string describing the SuperVGA given it's
*               number.
*
****************************************************************************/
{
    if (FIRST_SVGA <= SuperVGA && SuperVGA <= LAST_SVGA)
        return SuperVGANames[SuperVGA];
    else
        return "Invalid SuperVGA id!!";
}

/* Table of Video DAC names */

static char *DACNames[] = {
    "Normal DAC",
    "HiColor 15 bit DAC",
    "HiColor 16 bit DAC",
    "TrueColor 24 bit DAC",
    };

char *getDACName(int dac)
/****************************************************************************
*
* Function:     getDACName
* Parameters:   dac     - Video DAC id
* Returns:      Pointer to the string describing the DAC
*
****************************************************************************/
{
    if (0 <= dac && dac <= 3)
        return DACNames[dac];
    else
        return "Unknown DAC";
}

char *getChipsetName(int SuperVGA,int chipset)
/****************************************************************************
*
* Function:     getChipsetName
* Parameters:   SuperVGA    - SuperVGA id
*               chipset     - Chipset id
* Returns:      Pointer to the string describing the chipset. NULL if there
*               is no revision name.
*
****************************************************************************/
{
    if (FIRST_SVGA <= SuperVGA && SuperVGA <= LAST_SVGA) {
        if (chipsetNames[SuperVGA] == NULL)
            return NULL;
        else
            return chipsetNames[SuperVGA][chipset];
        }
    else
        return "Invalid SuperVGA id!!";
}
