/******************************************************************************
*******************************************************************************
**
** EXR0R3 - A ring 0/ring 3 IFS skeleton
** Copyright (C) 1993 by Andre Asselin
**
** R3STUBS.C - Ring 3 stubs
**
** History:
** 6/29/93 - created
**
*******************************************************************************
******************************************************************************/

#include "r3inc.h"


// Debug routines
static char *FormatCDFSI(struct cdfsi *pcdfsi);
static char *FormatCDFSD(struct cdfsd *pcdfsd);
int debug = 1;

/******************************************************************************
** Name of IFS
******************************************************************************/

const char FS_NAME[] = "EXR0R3";


/******************************************************************************
*******************************************************************************
**
** Primary entry points
**
*******************************************************************************
******************************************************************************/





/*-----------------------------------------------------------------------------
--
-- Volume management
--
-----------------------------------------------------------------------------*/


/******************************************************************************
**
** FS_ATTACH - Attach or detach a drive or device
**
** Parameters
** ----------
** unsigned short flag                  indicates attaching/detaching
**   values:
**     FSA_ATTACH               attach drive/device
**     FSA_DETACH               detach drive/device
**     FSA_ATTACH_INFO          return info on attached drive/device
** char *pDev                           drive or device that is being attached/detached
** struct vpfsd *pvpfsd                 pointer to FSD dependant volume parameters
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** void *pParm                          pointer to FSD dependant attachment info
** unsigned short far *pLen             length of area pointed to by pParm
**
******************************************************************************/

#pragma argsused
short int FSD_ATTACH(unsigned short flag, char *pDev, struct vpfsd *pvpfsd, 
                     struct cdfsd *pcdfsd, void *pParm, 
                     unsigned short *pLen)
{
   int rc;
   FILESTATUS3 buf;
   struct cdfsd *temp;

   temp = flag == FSA_ATTACH ? 0 : pcdfsd;
   if (debug)
      printf("FS_ATTACH: flag=%d, pDev='%s', %s, pParm='%s', pLen=%d\n",
             flag, pDev, FormatCDFSD(temp), pParm, *pLen);

   // Make sure it's a drive we're asking about
   if (pDev[0] == '\\')
      return ERROR_NOT_SUPPORTED;

   switch (flag) {
   case FSA_ATTACH:
      // check to see if pParm points to a directory
      rc = DosQueryPathInfo((char *)pParm, FIL_STANDARD, &buf, sizeof(buf));
      if (rc != NO_ERROR) {
         printf("FS_ATTACH error: DosQueryPathInfo=%d\n", rc);
         return rc;
      }

      if (buf.attrFile != FILE_DIRECTORY) {
         printf("FS_ATTACH error: DosQueryPathInfo reports attr=0x%x\n",
                buf.attrFile);
         return ERROR_INVALID_PARAMETER;
      }

      // Save the directory in the root CDFSD
      pcdfsd->dir = (char *)malloc(strlen((char *)pParm));
      pvpfsd->dir = pcdfsd->dir;

      if (pcdfsd->dir == NULL) {
         printf("FS_ATTACH error: pcdfsd->dir=NULL\n");
         return ERROR_NOT_ENOUGH_MEMORY;
      } 

      strcpy(pcdfsd->dir, (char *)pParm);
      if (pcdfsd->dir[strlen(pcdfsd->dir) - 1] == '\\')
         pcdfsd->dir[strlen(pcdfsd->dir) - 1] = '\0';
 
      break;

   case FSA_DETACH:
      // free the allocated memory
      // For now, don't do this so that we can detach drives that were
      // attached by another invocation of the FSD.
//      free(pcdfsd->dir);
      break;

   case FSA_ATTACH_INFO:
      // return the directory
      if (*pLen < strlen(pcdfsd->dir))
         return ERROR_INSUFFICIENT_BUFFER;

      *(USHORT *)pParm = strlen(pcdfsd->dir) + 1;
      strcpy((char *)pParm+sizeof(USHORT), pcdfsd->dir);
      break;
   }

   return NO_ERROR;
}


/******************************************************************************
**
** FS_FSINFO - Get/Set file system information
**
** Parameters
** ----------
** unsigned short flag                  indicates function to perform
**   values:
**     INFO_RETREIVE                    retrieve information
**     INFO_SET                         set information
** struct vpfsd *pvpfsd                 pointer to FSD dependant volume parameters
** char far *pData                      pointer to data buffer
** unsigned short cbData                length of data buffer
** unsigned short level                 type of information to return
**
******************************************************************************/

#pragma argsused
short int FS_FSINFO(unsigned short flag, struct vpfsd *vpfsd, char *pData,
                    unsigned short cbData, unsigned short level)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FLUSHBUF - Flush buffers for a specified volume
**
** Parameters
** ----------
** struct vpfsd *pvpfsd                 pointer to FSD dependant volume parameters
** unsigned short flag                  indicates whether to discard or retain cache
**   values:
**     FLUSH_RETAIN     retain cached information
**     FLUSH_DISCARD    discard cached information
**
******************************************************************************/

#pragma argsused
short int FS_FLUSHBUF(struct vpfsd *vpfsd, unsigned short flag)
{
   return ERROR_NOT_SUPPORTED;
}



/*-----------------------------------------------------------------------------
--
-- Directory management
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_CHDIR - Change current directory
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     CD_EXPLICIT      creating a new current directory
**     CD_VERIFY        verifying a current directory
**     CD_FREE          freeing an instance of a current directory
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pDir                           pointer to directory to change to
** unsigned short iCurDirEnd            offset to the end of the current directory in pDir
**
******************************************************************************/

#pragma argsused
short int FS_CHDIR(unsigned short flag, struct cdfsi *pcdfsi, 
                   struct cdfsd *pcdfsd, char *pDir, 
                   unsigned short iCurDirEnd)
{
   char buf[300];
   FILESTATUS3 buf2;
   int rc;
   int i;

   if (debug) {
      printf("FS_CHDIR: flag=%d, iCurDirEnd=%d, %s ", flag, iCurDirEnd, 
             FormatCDFSD(pcdfsd));
      if (flag != CD_FREE)
         printf("pDir='%s', %s\n", pDir, FormatCDFSI(pcdfsi));
      else
         printf("\n");
   }

   switch(flag) {
   case CD_EXPLICIT:
      // pDir is of the form: R:\dir\dir\
      // we use pDir[2] to skip the drive part
      sprintf(buf, "%s%s", pcdfsd->dir, &pDir[2]);
      rc = DosQueryPathInfo(buf, FIL_STANDARD, &buf2, sizeof(buf2));
      if (rc != NO_ERROR) {
         printf("FS_CHDIR error: DosQueryPathInfo=%d\n", rc);
         return rc;
      }
      if (buf2.attrFile != FILE_DIRECTORY) {
         printf("FS_CHDIR error: DosQueryPathInfo reports attr=%d\n", 
                buf2.attrFile);
         return ERROR_PATH_NOT_FOUND;
      }
      break;

   case CD_VERIFY:
      // cdi_curdir is of the form: R:\dir\dir\
      // we use cdi_curdir[2] to skip the drive part
      sprintf(buf, "%s%s", pcdfsd->dir, &pcdfsi->cdi_curdir[2]);
      rc = DosQueryPathInfo(buf, FIL_STANDARD, &buf2, sizeof(buf2));
      if (rc != NO_ERROR) {
         printf("FS_CHDIR error: DosQueryPathInfo=%d\n", rc);
         return rc;
      }
      if (buf2.attrFile != FILE_DIRECTORY) {
         printf("FS_CHDIR error: DosQueryPathInfo reports attr=%d\n", 
                buf2.attrFile);
         return ERROR_PATH_NOT_FOUND;
      }
      break;

   case CD_FREE:
      break;
   }

   return NO_ERROR;
}


/******************************************************************************
**
** FS_MKDIR - Make a new directory
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to directory name to create
** unsigned short iCurDirEnd            offset to the end of the current directory 
**                                      in pName
** char *pEABuf                         pointer to EAs to attach to new directory
** unsigned short flags                 0x40 = directory is non 8.3 filename
** unsigned long *oError                offset where error occurred in FEA list
**
******************************************************************************/

#pragma argsused
short int FS_MKDIR(struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                   char *pName, unsigned short iCurDirEnd, char *pEABuf, 
                   unsigned short flags, unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_RMDIR - Delete directory
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to directory name to delete
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
**
******************************************************************************/

#pragma argsused
short int FS_RMDIR(struct cdfsi *pcdfsi, struct cdfsd  *pcdfsd,
                   char *pName, unsigned short iCurDirEnd)
{
   return ERROR_NOT_SUPPORTED;
}



/*-----------------------------------------------------------------------------
--
-- File management
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_CHGFILEPTR - Change current location in file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** long offset                          signed offset
** unsigned short type                  indicates seek type
**   values:
**     CFP_RELBEGIN     move pointer relative to begining of file
**     CFP_RELCUR       move pointer relative to current position in file
**     CFP_RELEND       move pointer relative to end of file
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_CHGFILEPTR(struct sffsi *psffsi, struct sffsd *psffsd,
                        long offset, unsigned short type, 
                        unsigned short IOflag)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_CLOSE - Close an open file
**
** Parameters
** ----------
** unsigned short type                  indicates close type
**   values:
**     FS_CL_ORDINARY   this is not the final close of the file
**     FS_CL_FORPROC    this is the final close of the file for the process
**     FS_CL_FORSYS     this is the final close of the file for the whole system
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int FS_CLOSE(unsigned short type, unsigned short IOflag,
                   struct sffsi *psffsi, struct sffsd *psffsd)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_COMMIT - Commit a file to disk
**
** Parameters
** ----------
** unsigned short type                  indicates commit type
**   values:
**     FS_COMMIT_ONE    commit this one file
**     FS_COMMIT_ALL    commit all files
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int FS_COMMIT(unsigned short type, unsigned short IOflag,
                    struct sffsi *psffsi, struct sffsd *psffsd)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_COPY - Copy a file
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     DCPY_EXISTING    if destination file exists, replace it
**     DCPY_APPEND      source file should be appended to destination file
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pSrc                           pointer to source filename
** unsigned short iSrcCurDirEnd         offset to the end of the current directory in pSrc
** char *pDst                           pointer to destination filename
** unsigned short iDstCurDirEnd         offset to the end of the current directory in pDst
** unsigned short nameType              0x40 = destination is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int FS_COPY(unsigned short flag, struct cdfsi *pcdfsi,
                  struct cdfsd *pcdfsd, char *pSrc, 
                  unsigned short iSrcCurDirEnd, char *pDst,
                  unsigned short iDstCurDirEnd, unsigned short nameType)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_DELETE - Delete a file
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pFile                          pointer to filename to delete
** unsigned short iCurDirEnd            offset to the end of the current directory in pFile
**
******************************************************************************/

#pragma argsused
short int FS_DELETE(struct cdfsi *pcdfsi, struct cdfsd *pcdfsd, char *pFile, 
                    unsigned short iCurDirEnd)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FILEATTRIBUTE - Get/Set DOS file attributes
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     FA_RETRIEVE      retrieve attribute
**     FA_SET           set attribute
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short *pAttr                pointer to the attribute
**
******************************************************************************/

#pragma argsused
short int FS_FILEATTRIBUTE(unsigned short flag, struct cdfsi *pcdfsi,
                           struct cdfsd *pcdfsd, char *pName,
                           unsigned short iCurDirEnd, unsigned short *pAttr)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FILEINFO - Get/Set file information
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     FI_RETRIEVE      retrieve information
**     FI_SET           set information
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** unsigned short level                 level of information to get/set
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** unsigned long *oError                offset where error occurred in GEA/FEA
**                                      list (EA calls only)
**
******************************************************************************/

#pragma argsused
short int FS_FILEINFO(unsigned short flag, struct sffsi *psffsi,
                      struct sffsd *psffsd, unsigned short level, char *pData, 
                      unsigned short cbData, unsigned short IOflag, 
                      unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_MOVE - Move/rename a file
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pSrc                           pointer to source filename
** unsigned short iSrcCurDirEnd         offset to the end of the current directory in pSrc
** char *pDst                           pointer to destination filename
** unsigned short iDstCurDirEnd         offset to the end of the current directory in pDst
** unsigned short flags                 0x40 = destination is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int FS_MOVE(struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                  char *pSrc, unsigned short iSrcCurDirEnd,
                  char *pDst, unsigned short iDstCurDirEnd,
                  unsigned short flags)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_NEWSIZE - Change size of file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** unsigned long len                    new length of file
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_NEWSIZE(struct sffsi *psffsi, struct sffsd *psffsd,
                     unsigned long len, unsigned short IOflag)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_OPENCREATE - Open or create a new file
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** unsigned long openmode               sharing and access mode
** unsigned short openflag              action to take when file exists/doesn't exist
** unsigned short *pAction              returns the action that the IFS took
** unsigned short attr                  OS/2 file attributes
** char *pEABuf                         pointer to EAs to attach to new file
** unsigned short *pfgenFlag            flags returned by the IFS
**   values:
**     FOC_NEEDEAS      indicates there are critical EAs associated with the file
** unsigned long *oError                offset where error occurred in FEA list
**
******************************************************************************/

#pragma argsused
short int FS_OPENCREATE(struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                        char *pName, unsigned short iCurDirEnd,
                        struct sffsi *psffsi, struct sffsd *psffsd,
                        unsigned long openmode, unsigned short openflag,
                        unsigned short *pAction, unsigned short attr,
                        char *pEABuf, unsigned short *pfgenFlag,
                        unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_PATHINFO - get/set file information by filename
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     PI_RETRIEVE      retrieve information
**     PI_SET           set information
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short level                 level of information to get/set
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned long *oError                offset where error occurred in GEA/FEA
**                                      list (EA calls only)
**
******************************************************************************/

#pragma argsused
short int FS_PATHINFO(unsigned short flag, struct cdfsi *pcdfsi,
                      struct cdfsd *pcdfsd, char *pName, 
                      unsigned short iCurDirEnd, unsigned short level, 
                      char *pData, unsigned short cbData, 
                      unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_READ - read data from a file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** char *pData                          pointer to buffer
** unsigned short *pLen                 length of buffer
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_READ(struct sffsi *psffsi, struct sffsd *psffsd,
                  char *pData, unsigned short *pLen, unsigned short IOflag)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_WRITE - write data to a file
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** char *pData                          pointer to buffer
** unsigned short *pLen                 length of buffer
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int FS_WRITE(struct sffsi *psffsi, struct sffsd *psffsd,
                   char *pData, unsigned short *pLen, unsigned short IOflag)
{
   return ERROR_NOT_SUPPORTED;
}



/*-----------------------------------------------------------------------------
--
-- Directory management
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_FINDCLOSE - End a directory search
**
** Parameters
** ----------
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
**
******************************************************************************/

#pragma argsused
short int FS_FINDCLOSE(struct fsfsi *pfsfsi, struct fsfsd *pfsfsd)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FINDFIRST - Begin a new directory search
**
** Parameters
** ----------
** struct cdfsi *pcdfsi                 pointer to FSD independant current directory
** struct cdfsd *pcdfsd                 pointer to FSD dependant current directory
** char *pName                          pointer to filename mask
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short attr                  attribute mask
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short *pcMatch              maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
** unsigned long *oError                offset where error occurred in GEA
**                                      list (EA levels only)
**
******************************************************************************/

#pragma argsused
short int FS_FINDFIRST(struct cdfsi *pcdfsi, struct cdfsd *pcdfsd,
                       char *pName, unsigned short iCurDirEnd,
                       unsigned short attr, struct fsfsi *pfsfsi,
                       struct fsfsd *pfsfsd, char *pData,
                       unsigned short cbData, unsigned short *pcMatch,
                       unsigned short level, unsigned short flags,
                       unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FINDFROMNAME - Restart directory search
**
** Parameters
** ----------
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short *pcMatch              maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned long position               position in directory to restart search from
** char *pName                          pointer to filename to restart search from
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
** unsigned long *oError                offset where error occurred in GEA
**                                      list (EA levels only)
**
******************************************************************************/

#pragma argsused
short int FS_FINDFROMNAME(struct fsfsi *pfsfsi, struct fsfsd *pfsfsd,
                          char *pData, unsigned short cbData,
                          unsigned short *pcMatch, unsigned short level,
                          unsigned long position, char *pName,
                          unsigned short flags, unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FINDNEXT - Continue directory search
**
** Parameters
** ----------
** struct fsfsi *pfsfsi                 pointer to FSD independant search record
** struct fsfsd *pfsfsd                 pointer to FSD dependant search record
** char *pData                          pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short *pcMatch              maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned short flag                  indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
** unsigned long *oError                offset where error occurred in GEA
**                                      list (EA levels only)
**
******************************************************************************/

#pragma argsused
short int FS_FINDNEXT(struct fsfsi *pfsfsi, struct fsfsd *pfsfsd,
                      char *pData, unsigned short cbData, 
                      unsigned short *pcMatch, unsigned short level,
                      unsigned short flag, unsigned long *oError)
{
   return ERROR_NOT_SUPPORTED;
}



/*-----------------------------------------------------------------------------
--
-- Miscellaneous Functions
--
-----------------------------------------------------------------------------*/


/******************************************************************************
**
** FS_INIT - Initialize the IFS
**
** Parameters
** ----------
** char *szParm                         pointer to command line parameters
** unsigned long pDevHlp                pointer to DevHlp entry point
** unsigned long *pMiniFSD              pointer to data passed between the
**                                      mini-FSD and the IFS
**
******************************************************************************/

#pragma argsused
short int FS_INIT(char *szParm, void (*pDevHlp)(void),
                  unsigned long *pMiniFSD)
{
   return NO_ERROR;
}


/******************************************************************************
**
** FS_PROCESSNAME - Canonicalize a filename
**
** Parameters
** ----------
** char *pNameBuf                       filename to canonicalize
**
******************************************************************************/

#pragma argsused
short int FS_PROCESSNAME(char *pNameBuf)
{
   if (debug)
      printf("FS_PROCESSNAME: '%s'\n", pNameBuf);
   return NO_ERROR;
}


/******************************************************************************
*******************************************************************************
**
** Locking support entry points
**
*******************************************************************************
******************************************************************************/



/******************************************************************************
**
** FS_CANCELLOCKREQUEST - unlock a range
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** void *pLockRange                     range to unlock
**
******************************************************************************/

#pragma argsused
short int FS_CANCELLOCKREQUEST(struct sffsi *psffsi, struct sffsd *psffsd,
                               struct filelock *pLockRange)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FILELOCKS - lock a range
**
** Parameters
** ----------
** struct sffsi *psffsi                 pointer to FSD independant file instance
** struct sffsd *psffsd                 pointer to FSD dependant file instance
** void *pUnLockRange                   range to unlock
** void *pLockRange                     range to lock
** unsigned long timeout                time in milliseconds to wait
** unsigned long flags                  flags
**   values:
**     0x01                     sharing of this file region is allowed
**     0x02                     atomic lock request
**
******************************************************************************/

#pragma argsused
short int FS_FILELOCKS(struct sffsi *psffsi, struct sffsd *psffsd,
                       struct filelock *pUnLockRange, struct filelock *pLockRange,
                       unsigned long timeout, unsigned long flags)
{
   return ERROR_NOT_SUPPORTED;
}



/******************************************************************************
*******************************************************************************
**
** UNC entry point
**
*******************************************************************************
******************************************************************************/



/******************************************************************************
**
** FS_VERIFYUNCNAME - Check if the IFS controls the server in question
**
** Parameters
** ----------
** unsigned short flag                  flags
**   values:
**     VUN_PASS1                pass 1 poll
**     VUN_PASS2                pass 2 poll
** char *pName                          pointer to server in UNC format
**
******************************************************************************/

#pragma argsused
short int FS_VERIFYUNCNAME(unsigned short flag, char *pName)
{
   return ERROR_NOT_SUPPORTED;
}


static char *FormatCDFSI(struct cdfsi *pcdfsi) {
   static char buf[300];
 
   sprintf(buf, "pcdfsi.end=%d, pcdfsi.flags=%x, pcdfsi.curdir='%s'",
           pcdfsi->cdi_end, pcdfsi->cdi_flags, pcdfsi->cdi_curdir);

   return buf;
}


static char *FormatCDFSD(struct cdfsd *pcdfsd) {
   static char buf[300];

   if (pcdfsd == NULL)
      sprintf(buf, "pcdfsd=NULL");
   else
      sprintf(buf, "pcdfsd.dir='%s'", pcdfsd->dir);

   return buf;
}
