/***************************************
  $Revision: 1.16 $

  IP handling (ip). iproutines.h  - header file for conversions routines.
                                    defines data structures for IP module.

  Status: NOT REVUED, TESTED

  Design and implementation by: Marek Bukowy

  ******************/ /******************
  Copyright (c) 1999                              RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  ***************************************/


#ifndef _IP_H
#define _IP_H

#include <glib.h>
#include <erroutines.h>

#include <sys/types.h>

/*+ the key type (for ascii keys - tells what it was before it was 
    converted into prefixes in smart_conv() +*/

typedef enum {
  IPK_UNDEF = 0,
  IPK_RANGE,
  IPK_PREFIX,
  IPK_IP
} ip_keytype_t;

/*+ the space type +*/
typedef enum {
  IP_V4 = 1,
  IP_V6
} ip_space_t;  


typedef unsigned int   ip_limb_t; /* the limb must be at least 32 bit wide */
#ifdef _LINUX
typedef u_int64_t      ip_v6word_t;
#else
typedef uint64_t       ip_v6word_t;
#endif
/* should use 64bit for ipv6:   
   u_int64_t (bsd,linux)
   uint64_t (solaris)
*/
#define IPLIMBNUM (16/sizeof(ip_limb_t))

/*+ address structure +*/
typedef struct {
  ip_limb_t  words[IPLIMBNUM];    /*+ 32/128 bit ip addr. SUBJECT TO CHANGE +*/
                             
  ip_space_t space;       /*+ MUST NOT BE char ! prefixes are compared with 
			    memcmp, so there may be absolutely no unitialised 
			    bytes  +*/
} ip_addr_internal_t;

/*+ prefix structure +*/
typedef struct {
  unsigned                bits;		/*+ length in bits. +*/
  ip_addr_internal_t      ip;	        /*+ the IP of the prefix +*/        
} ip_prefix_internal_t;
 
/*+ range structure +*/
typedef struct {
  ip_addr_internal_t         begin;        /*+ IP where the range begins. +*/
  ip_addr_internal_t         end;          /*+ IP where it ends +*/
} ip_range_internal_t;

#if 0/* #ifndef IP_IMPL  -- set this to see accesses to structure members */
/* hide the internals */
typedef struct {char a[sizeof(ip_addr_internal_t)];}    ip_addr_t;
typedef struct {char a[sizeof(ip_range_internal_t)];}   ip_range_t;
typedef struct {char a[sizeof(ip_prefix_internal_t)];}  ip_prefix_t;
#else
typedef ip_addr_internal_t   ip_addr_t;
typedef ip_range_internal_t  ip_range_t;
typedef ip_prefix_internal_t ip_prefix_t;
#endif 


/*+ 
  stores size/span of an allocation 
  SUBJECT TO CHANGE: will be bigger for IPv6 
  +*/
typedef unsigned int  ip_rangesize_t; 

/*+  the length of a string that should be able to hold a prefix / range
    when used with b2a functions.
+*/
#define IP_ADDRSTR_MAX 20 /* XXX watch out for IPv6 !! */
#define IP_PREFSTR_MAX 24
#define IP_RANGSTR_MAX 48

/*+ 
  IP expansion mode - for use with t2b functions, they control
  whether the input is supposed to be fully expanded or contain shortcuts
  (eg. enabling saying 0/0 instead 0.0.0.0/0)
  +*/
typedef enum {
  IP_PLAIN = 1,
  IP_EXPN
} ip_exp_t;

/* prototypes */
/* text to binary */
er_ret_t IP_addr_t2b(ip_addr_t *ipptr, char *addr, ip_exp_t expf);
er_ret_t IP_pref_t2b(ip_prefix_t *prefptr, char *prefstr, ip_exp_t expf);
er_ret_t IP_rang_t2b(ip_range_t *rangptr, char *rangstr, ip_exp_t expf);
er_ret_t IP_revd_t2b(ip_prefix_t *prefptr, char *prefstr, ip_exp_t expf);
/* convenience (or call it backward compatibility) macros */

/*+ the e2b macros assume fully expanded text  +*/
#define IP_addr_e2b(a,b) IP_addr_t2b(a,b,IP_PLAIN)
#define IP_pref_e2b(a,b) IP_pref_t2b(a,b,IP_PLAIN)
#define IP_rang_e2b(a,b) IP_rang_t2b(a,b,IP_PLAIN)
#define IP_revd_e2b(a,b) IP_revd_t2b(a,b,IP_PLAIN)

/*+ the a2b macros will fully expand an address. 
  The details depend on the individual functions. +*/
#define IP_addr_a2b(a,b) IP_addr_t2b(a,b,IP_EXPN)
#define IP_pref_a2b(a,b) IP_pref_t2b(a,b,IP_EXPN)
#define IP_rang_a2b(a,b) IP_rang_t2b(a,b,IP_EXPN)
#define IP_revd_a2b(a,b) IP_revd_t2b(a,b,IP_EXPN)

/* text fragments to binary */
er_ret_t IP_addr_f2b_v4(ip_addr_t *addrptr, char *adrstr);
er_ret_t IP_rang_f2b_v4(ip_range_t *rangptr, char *beginstr,  char *endstr);
er_ret_t IP_pref_f2b_v4(ip_prefix_t *prefptr, char *prefixstr, 
			char *lengthstr);
er_ret_t IP_addr_f2b_v6(ip_addr_t *addrptr, char *msbstr, char *lsbstr );
er_ret_t IP_pref_f2b_v6(ip_prefix_t *prefptr, char *msbstr, char *lsbstr, 
			char *lengthstr);

er_ret_t IP_addr_b2a(ip_addr_t *binaddr, char *ascaddr, int strmax );
er_ret_t IP_pref_b2a(ip_prefix_t *prefptr, char *ascaddr, int strmax);
er_ret_t IP_rang_b2a(ip_range_t *rangptr, char *ascaddr, int strmax);
er_ret_t IP_rang_classful(ip_range_t *rangptr, ip_addr_t *addrptr);
er_ret_t IP_pref_2_rang( ip_range_t *rangptr, ip_prefix_t *prefptr );
  
/* utility functions: testers/converters */
int  IP_addr_bit_get(ip_addr_t *binaddr, int bitnum);
void IP_addr_bit_set(ip_addr_t *binaddr, int bitnum, int bitval);
int  IP_addr_cmp(ip_addr_t *ptra, ip_addr_t *ptrb, int len);
int  IP_sizebits(ip_space_t spc_id);
void IP_pref_bit_fix( ip_prefix_t *prefix );
int IP_addr_in_pref(ip_addr_t *ptra, ip_prefix_t *prefix);
int IP_addr_in_rang(ip_addr_t *ptra, ip_range_t *rangptr);
er_ret_t IP_smart_conv(char *key, int justcheck, int encomp, 
		       GList **preflist, ip_exp_t expf, ip_keytype_t *keytype);
er_ret_t IP_smart_range(char *key, ip_range_t *rangptr, ip_exp_t expf, 
			ip_keytype_t *keytype);


ip_rangesize_t IP_rang_span( ip_range_t *rangptr );
er_ret_t IP_addr_s2b(ip_addr_t *addrptr, void *addr_in, int addr_len);

/* accessor functions */
unsigned IP_addr_b2_space(ip_addr_t *addrptr);
unsigned IP_pref_b2_space(ip_prefix_t *prefix);
unsigned IP_rang_b2_space(ip_range_t *myrang);

unsigned IP_addr_b2v4_addr(ip_addr_t *addrptr);
ip_v6word_t IP_addr_b2v6_hi(ip_addr_t *addrptr); 
ip_v6word_t IP_addr_b2v6_lo(ip_addr_t *addrptr);

unsigned IP_pref_b2_space(ip_prefix_t *prefix);
unsigned IP_pref_b2_len(ip_prefix_t *prefix);
#define IP_pref_b2v4_len(prefix) IP_pref_b2_len(prefix)
#define IP_pref_b2v6_len(prefix) IP_pref_b2_len(prefix)

unsigned IP_pref_b2v4_addr(ip_prefix_t *prefix);
void IP_addr_b2v4(ip_addr_t *addrptr, unsigned *address);
void IP_pref_b2v4(ip_prefix_t *prefptr, 
		   unsigned int *prefix, 
		   unsigned int *prefix_length);
#define IP_revd_b2v4(a,b,c) IP_pref_b2v4(a,b,c)
void IP_pref_b2v6(ip_prefix_t *prefptr, 
		  ip_v6word_t *high, 
		  ip_v6word_t *low, 
		  unsigned int *prefix_length);
#define IP_revd_b2v6(a,b,c,d) IP_pref_b2v6(a,b,c,d)
void IP_rang_b2v4(ip_range_t *myrang,
		  unsigned *begin, 
		  unsigned *end);

/******** constructing from raw values **********/
er_ret_t IP_addr_v4_mk(ip_addr_t *addrptr, unsigned addrval);
er_ret_t IP_addr_v6_mk(ip_addr_t *addrptr,
		       ip_v6word_t high, ip_v6word_t low);
er_ret_t IP_pref_v4_mk(ip_prefix_t *prefix,
		       unsigned prefval, unsigned preflen);
er_ret_t IP_rang_v4_mk(ip_range_t *rangptr, 
		       unsigned addrbegin, unsigned addrend);
/* a2v4 functions to convert the ascii to binary, and
  then set the raw values at the pointers provided. */
er_ret_t IP_pref_a2v4(char *avalue, ip_prefix_t *pref,
		      unsigned *prefix, unsigned *prefix_length);
er_ret_t IP_pref_a2v6(char *avalue, ip_prefix_t *pref,
		      ip_v6word_t *high, ip_v6word_t  *low,
		      unsigned *prefix_length);
er_ret_t IP_revd_a2v4(char *avalue, ip_prefix_t *pref,
		      unsigned int *prefix, unsigned int *prefix_length);
er_ret_t IP_addr_a2v4(char *avalue,ip_addr_t *ipaddr, unsigned int *address);
er_ret_t IP_rang_a2v4(char *rangstr, ip_range_t *myrang,
		      unsigned int *begin_in, unsigned int *end_in);

/* decompose/find encompasssing prefix */
void IP_rang_encomp(ip_range_t *rangptr);
unsigned IP_rang_decomp(ip_range_t *rangptr, GList **preflist);

/*
  this is to define a constant struct for comparisons.
*/
#ifdef IP_IMPL
const ip_addr_t IP_ADDR_UNSPEC={{0,0,0,0},0}; /* unlikely to be real :-)
					       as there is no space 0 
					       bonus: it's a natural state after 
					       initializing to 0 */
#else
extern ip_addr_t IP_ADDR_UNSPEC;
#endif

#endif /* _IP_H */
