FUNCTION getWaterViscosity( Model, N, temperature ) RESULT(viscosity)

  USE DefUtils
  IMPLICIT NONE
  !------------ external variables ---------------------------
  TYPE(Model_t) :: Model
  INTEGER :: N
  REAL(KIND=dp) :: temperature, viscosity
  !------------ internal variables----------------------------
  TYPE(ValueList_t), POINTER :: Material
  REAL (KIND=dp), ALLOCATABLE :: viscA(:), viscB(:), viscC(:), ReferenceVisc(:)
  REAL(KIND=dp) :: Z
  INTEGER :: nMax,i,j,body_id,material_id,elementNodes,nodeInElement,istat
  LOGICAL :: FirstTime = .TRUE., GotIt, IsCelsius
  CHARACTER(LEN=MAX_NAME_LEN) :: SolverName
  !------------ remember this -------------------------------
  SAVE FirstTime, viscA, viscB, viscC, ReferenceVisc, SolverName

  !------------ allocations and initialization (first visit) ----
  IF (FirstTime) THEN
     WRITE(SolverName, '(A)') 'getWaterViscosity'
     nMax = Model % MaxElementNodes
     ALLOCATE(viscA(nMax), viscB(nMax), viscC(nMax), ReferenceVisc(nMax), &
          STAT=istat)
     IF ( istat /= 0 ) THEN
        CALL FATAL(SolverName,'Memory allocation error, Aborting.')
     END IF
     FirstTime = .FALSE.
     CALL INFO(SolverName,'Memory allocation done', level=3)
  END IF
  
  !-------------------------------------------
  ! get element properties
  !-------------------------------------------   
  IF ( .NOT. ASSOCIATED(Model % CurrentElement) ) THEN
     CALL FATAL(SolverName, 'Model % CurrentElement not associated')
  END IF
  ! ---- get the Material for the current Element
  Material => GetMaterial( ) ! Model % CurrentElement is the default argument,
                             ! so this is equal to:
! Material => GetMaterial( Model % CurrentElement )
  IF (.NOT.ASSOCIATED(Material)) THEN
     CALL FATAL(SolverName, 'No Material found')
  END IF
  ! ---- inquire the corresponding node in the Element
  GotIt = .FALSE.
  DO nodeInElement=1,Model % CurrentElement % Type % NumberOfNodes
     IF ( N == Model % CurrentElement % NodeIndexes(nodeInElement) ) &  
          GotIt = .TRUE.
     IF (GotIt) EXIT
  END DO
  IF (.NOT.GotIt) THEN
     CALL FATAL(SolverName, 'Point in Element not found')
  END IF


PRINT *, "alive"
  ! ---- read in values
  viscA(1:elementNodes) = ListGetReal( Material,'Visc Paramater A',&
        elementNodes, Model % CurrentElement % NodeIndexes, GotIt )
  IF (.NOT. GotIt) THEN
     viscA(nodeInElement) = -1.704_dp
     CALL WARN(SolverName, "<Visc Paramater A> not found - setting to -1.704")
  END IF
  viscB(1:elementNodes) = ListGetReal( Material,'Visc Paramater B',&
        elementNodes, Model % CurrentElement % NodeIndexes, GotIt )
  IF (.NOT. GotIt) THEN
     viscA(nodeInElement) = -5.306_dp
     CALL WARN(SolverName, "<Visc Paramater B> not found - setting to -5.306")
  END IF
  viscC(1:elementNodes) = ListGetReal( Material,'Visc Paramater C',&
        elementNodes, Model % CurrentElement % NodeIndexes, GotIt )
  IF (.NOT.GotIt) THEN
     viscA(nodeInElement) = 7.003_dp
     CALL WARN(SolverName, "<Visc Paramater C> not found - setting to 7.003")
  END IF
  ReferenceVisc(1:elementNodes) = ListGetReal( Material,'Reference Viscosity',&
        elementNodes, Model % CurrentElement % NodeIndexes, GotIt )
  IF (.NOT.GotIt) THEN
     ReferenceVisc(nodeInElement) =  1.788d-03
     CALL WARN(SolverName, "<Reference Viscosity> not found - setting to 7.003")
  END IF
  IsCelsius = ListGetLogical( Material,'Celsius', GotIt )
  IF (.NOT.GotIt) THEN
      IsCelsius = .FALSE.
     CALL WARN(SolverName,&
          "<Celsius> not found - assuming all temperatures to be given in Kelvin")
  END IF
  IF (IsCelsius) THEN
     IF (Temperature < -273.15_dp) &
          CALL WARN(SolverName, "Temperature is below absolute zero")
     Z = 273.0_dp/(Temperature + 273.15_dp)
  ELSE
     IF (Temperature < 0.0_dp) &
          CALL WARN(SolverName, "Temperature is below absolute zero")
     Z = 273.0_dp/Temperature
  END IF
  viscosity = ReferenceVisc(nodeInElement) * &
       EXP(viscA(nodeInElement) + viscB(nodeInElement) * Z &
       + viscC(nodeInElement) * (Z**2))
END FUNCTION getWaterViscosity
