/*
	Authors		: Sreedhar Sivakumaran
				  John Plevyak
	Notes		:

		Main program which calls a set of test routines for
		the DTM library.
*/

#include <stdio.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <string.h>

#include <dtmint.h>
#include <debug.h>

#include "test.h"



/*
	MACRO DEFINITIONS
*/

#define	USAGE	"Usage: %s port-name [-iOI] [testname1[,arg1[,arg2]][:n]  ...]\
[all[:n]]\n\
		where n is the optional number of iterations\n\
		-i	interactive mode\n\
		-O	output side of test (default)\n\
		-I	input side of basic tests\n\
		-P	input side of performance tests (obsolete)\n\
	available tests:\n"
#define	STR_MATCH		0
#define	CHAR_ITER		':'
#define	CHAR_ARGS		','

#ifdef FALSE
#undef FALSE
#endif
#define	FALSE			0
#ifdef TRUE
#undef TRUE
#endif
#define	TRUE			1
#ifndef NULL
#define	NULL			0
#endif
#define	SIZE(x)	(sizeof(x)/sizeof((x)[0]))


/*
	TYPE DEFINITIONS
*/
typedef struct { 
	char	*name;
	char	*description;
	int		(*funcOutput)();
	int		(*funcInput)();
} Test;


/*
	STATIC DATA
*/

static Test tests[] = {
	{	"all",	"all tests",			NULL, NULL },
	{	"wr",	"Write/Read",			DTMtestWriteRead, DTMtestInput },
	{	"cb",	"Callback tests",		DTMtestCallback, DTMtestCallbackInput },
	{	"pr",	"Performance tests",	DTMtestPerform, DTMtestPerfInput },
#if 0
	{	"cd",	"Create/Destroy", 		DTMtestCreateDestroy,  DTMtestInput},
	{	"sw",	"Switching connection",	DTMtestSwitch, DTMtestInput }, 
#endif
	{	NULL, NULL, NULL }		/* END MARKER */
};

/*
	GLOBAL DATA
*/

int interactiveMode = FALSE;
int quietMode = FALSE;


/*
	STATIC FUNCTIONS
*/

int do_test( testNumber, iterations, portName, arg1, arg2, arg3, arg4 )
	int		testNumber;
	int		iterations;
	char *	portName;
	int		arg1;
	int		arg2;
	int		arg3;
	int		arg4;
{
	char	iterStr[20];
	int		i;
	int		success = 1;

	for ( i = 1 ; i <= iterations ; i++ ) {
		sprintf( iterStr, ":%d", i);
		if ( !quietMode ) {
			fprintf( stdout, 
				"%%Test: Beginning %s%s test\n", tests[testNumber].description,
			(iterations == 1 ) ? "" : iterStr  );
		}
		if ( testNumber == 0 ) {		/* all */
			Test	* test = &tests[1];	/* skip all */
			while ( test->name != NULL ) {
				if ( !test->funcOutput( portName, arg1, arg2, arg3, arg4 ) ) {
					fprintf( stderr, "%%Test: %s test fails\n",
						test->description );
					success = 0;
					goto Lbreak2;
				}
				test++;
			}
		} else {
			if ( !tests[testNumber].funcOutput( portName, arg1, arg2, 
					arg3, arg4 )) {
				fprintf( stderr, "%%Test: %s test fails\n",
					tests[testNumber].description );
				success = 0;
				break;
			}
		}
	}
 Lbreak2:
	if ( !quietMode ) {
		fprintf( stdout, "%%Test: Ending %s test%s\n", 
			tests[testNumber].description, (iterations == 1 ) ? "" : "s" );
	}
	return success;
}

int find_test( testName )
char *	testName;
{
	Test	*test = &tests[0];
	while ( test->name != NULL ) {
		if ( strcmp( test->name, testName ) == STR_MATCH ) {
			return (test - tests);
		}
		test++;
	}
	fprintf( stdout, "%%Test ERROR- test %s not found", testName );
	return -1;
}

usage( progName) 
char	* progName;
{
	Test	*test = &tests[0];
	fprintf( stdout, USAGE, progName );
	while ( test->name != NULL ) {
		fprintf( stdout, "\t\t%s\t%s\n", test->name, test->description);
		test++;
	}
	exit( 1 );
}


/*
	GLOBAL FUNCTIONS
*/

main( argc, argv )
	int		argc ;
	char	*argv[] ;
{
	char	*portName[ MAX_PORT_NAMES ] ;
	int		portNameCount = 0;
	int		ok = TRUE;
	int		fOutput = TRUE;
	int		fPerform = FALSE;

	if( argc <= 1 ) usage( argv[0] ); 

	/*
		No port names begining with a dash
	*/
	if ((*++argv)[0] == '-' ) usage( --argv );
	--argv;
	while ( *++argv && ( (*argv)[0] != '-' )) 
		portName[ portNameCount++] =  *argv;
	portName[ portNameCount ] = NULL;
	--argv;

	while ( *++argv && (*argv)[0] == '-' ) {
		while ( *(++(*argv)) ) {
			switch ( **argv )  {
				case 'q' : quietMode = TRUE;
					break;
				case 'i' : interactiveMode = TRUE;
					break;
				case 'I' : fOutput = FALSE;
					break;
				case 'P' : 
					fOutput = FALSE;
					fPerform = TRUE;		
					break;
				case 'O' : fOutput = TRUE;
					break;
				case 'd' : uDTMdbg = DTM_DBG_MSG;
					break;
				default:
					usage( argv[0] );
			}
		}
	}
	--argv;

	if ( !quietMode ) {
		fprintf( stdout, "%%Test: Starting %s testing session\n",
			fOutput ? "Output" : "Input" );
	}

	if ( !fOutput ) {
		if ( fPerform )
			ok = !(DTMtestPerfInput( portName ) == DTMERROR );
		else {
			if ( !*++argv ) {
				ok = DTMtestInput( portName );
			} else {
				while ( ok && *argv ) {
					int		testNumber;
					if ( (testNumber = find_test( *argv )) == -1 ) {
						ok = FALSE;
						break;
					}
					if ( !tests[testNumber].funcInput( portName )) ok = FALSE;
					argv++;
				}	
			}
		}
	} else {
		/*
			OUTPUT
		*/
		while ( ok && *++argv ) {
			int		iterations = 1;
			int		testNumber;
			char	*iterStr = strchr( *argv, CHAR_ITER );
			char	*argStr = strchr( *argv, CHAR_ARGS );
			int		arg1 = 0, arg2 = 0;
			int		arg3 = 0, arg4 = 0;
			
			/*
				Accept up to four arguments
			*/
			if ( argStr != NULL ) {
				arg1 = atoi( ++argStr );
				*(--argStr) = '\0';
				argStr = strchr( ++argStr, CHAR_ARGS ); 
				if ( argStr != NULL ) {
					arg2 = atoi( ++argStr );
					*(--argStr) = '\0';
					argStr = strchr( ++argStr, CHAR_ARGS ); 
					if ( argStr != NULL ) {
						arg3 = atoi( ++argStr );
						*(--argStr) = '\0';
						argStr = strchr( ++argStr, CHAR_ARGS ); 
						if ( argStr != NULL ) {
							arg4 = atoi( ++argStr );
							*(--argStr) = '\0';
						}	
					}	
				}	
			}
			if ( iterStr != NULL ) {
				iterations = atoi( ++iterStr );
				*(--iterStr) = '\0';
			}
			if ( (testNumber = find_test( *argv )) == -1 ) {
				ok = FALSE;
				break;
			}
			if ( testNumber == 0 ) /* all */ {
				for ( testNumber = 0;testNumber < SIZE(tests);testNumber++ ) { 
					if (!do_test( testNumber, iterations, portName[0], 
								arg1, arg2, arg3, arg4 )) {
						ok = FALSE;
						break;
					}
				}
			} else {
				if (!do_test( testNumber, iterations, portName[0], 
						arg1, arg2, arg3, arg4 )) 
					ok = FALSE;
			}
		}
	}
	exit( !ok ); 
}
