static char rcsid[] = "expiredb.c,v 1.21 1996/01/04 03:46:43 duane Exp";
/*
 *  expiredb.c - Deletes any entries that are older than the TTL.
 *
 *  Usage: expiredb [-log logfile] db
 *
 *  Exit code = 0 if no objects were expired, or 1 if objects expired,
 *  of 2 on error.
 *
 *  Darren Hardy, hardy@cs.colorado.edu, August 1994
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <gdbm.h>
#include "util.h"
#include "url.h"
#include "template.h"

/* Local functions */
static void usage();

/* Local Variables */
static GDBM_FILE dbf;
static int exit_code = 0;

static void usage()
{
	fprintf(stderr, "Usage: expiredb [-log logfile] db\n");
	exit(2);
}

int main(argc, argv)
     int argc;
     char *argv[];
{
	FILE *fp = stderr;
	datum d, k, nk;
	Template *t;
	AVPair *avp;
	time_t ttl, update_time;

	exit_code = 0;
	setbuf(stderr, NULL);
	init_log3("expiredb", NULL, stderr);
	if (argc < 2)
		usage();

	if (!strcmp(argv[1], "-log")) {
		if (argc < 4)
			usage();
		argv += 2;
		if ((fp = fopen(argv[0], "a+")) == NULL) {
			log_errno(argv[0]);
			exit(2);
		}
	}
	setbuf(fp, NULL);
	init_log3("expiredb", fp, stderr);
	Log("Deleting expired objects from the Gatherer's database...\n");

	dbf = gdbm_open(argv[1], 0, GDBM_WRITER, 0644, NULL);
	if (dbf == NULL) {
		errorlog("gdbm_open: %s: %s\n", argv[1],
		    gdbm_strerror(gdbm_errno));
		log_errno(argv[1]);
		exit(2);
	}
	/* 
	 *  Now, for each key in the database, check to see if the
	 *  TTL has expired.  If it has, then delete the object.
	 */
	k = gdbm_firstkey(dbf);
	while (k.dptr != NULL) {
		nk = gdbm_nextkey(dbf, k);
		d = gdbm_fetch(dbf, k);
		if (d.dptr == NULL) {
			errorlog("gdbm_fetch: %s: %s: %s\n", k.dptr,
			    argv[1], gdbm_strerror(gdbm_errno));
			gdbm_close(dbf);
			exit(exit_code == 1 ? 1 : 2);
		}
		/* Parse the object to find it's update time */
		init_parse_template_string(d.dptr, d.dsize);
		if ((t = parse_template()) == NULL) {
			errorlog("Corrupt SOIF object: %s\n", k.dptr);
			free(d.dptr);
			gdbm_close(dbf);
			exit(exit_code == 1 ? 1 : 2);
		}
		finish_parse_template();
		free(d.dptr);

		/* Find out the UPDATE-TIME of the object */
		if ((avp = extract_AVPair(t->list, T_UPDATE)) == NULL) {
			Log("WARNING: %s has no %s attribute.\n",
			    t->url, T_UPDATE);
			goto clean_up;
		}
		update_time = atoi(avp->value);
		if (update_time < 1)
			goto clean_up;

		/* Find out the TTL of the object */
		ttl = 0;
		if ((avp = extract_AVPair(t->list, T_TTL)) != NULL)
			ttl = atoi(avp->value);
		if (ttl < 1)
			ttl = 6 * MONTH;	/* default TTL */

		/* If the object has expired, then delete it */
		if (time(NULL) > (update_time + ttl)) {
			Log("Expired: %s\n", t->url);
			exit_code = 1;
			if (gdbm_delete(dbf, k)) {
				errorlog("gdbm_delete: %s: %s: %s\n",
				    k.dptr, argv[1],
				    gdbm_strerror(gdbm_errno));
			}
		}
	      clean_up:	/* Clean up, and go onto the next item */
		free_template(t);
		free(k.dptr);
		k = nk;
	}
	gdbm_close(dbf);
	exit(exit_code == 1 ? 1 : 0);
}
