/* $Id: topology.c,v 1.3 1995/05/31 20:42:44 dante Exp $ */
#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#if defined (HAVE_SYSLOG_H)
#  include <syslog.h>
#endif /* !SYS_SYSLOG */
#include <floodd.h>
#include <statistics.h>
#include <dynamic_string.h>

/* A few function declarations */
char *
build_topology_definition (SiteInfo **sites, char *topology_matrix);

char *
build_group_definition (Group *group);

char *
build_site_definitions (SiteInfo **sites);

char *
trivial_topology (Group *group);

char *
run_external_topology_generator (Group *group);

char **
cmd_from_args (char *line);

/* Flag to use when we have a request to run a topology estimation, but
 * we are already doing it.
 */

generate_topology (void **args)
{
  char *group_name = args[0];
  Group *group;
  DataBlock *datablock;
  string_declare (update_string);
  char *site_definitions;
  char *group_definition;
  char *topology_definition;
  struct timeval time;

  group = group_by_name (group_name);
  if (group == NULL)
    {
      syslog (LOG_ERR, "No group name `%s'\n", group->name);
      return;
    }

  /* Check to see that we are still the master site */
  if (group->master_site != whoami)
    return;

  /* Eliminate any old site that we haven't heard from in a while */
  /* don't purge sites if this is our first time */
  if (group->topology != NULL)
    purge_sites ();


  time = timeval_current ();

  /* Set to the event to NULL so we can't reschedule it */
  group->topology_event = NULL;

  if (debug)
    printf ("Initiating topology estimate for `%s'\n", group->name);

  if (debug)
    printf ("%d\n", group->group_sites->count);

  site_definitions =  build_site_definitions (group->sites);
  group_definition =  build_group_definition (group);

  /* If we don't have enough sites, we need to generate a trivial topology */
  if (group->group_sites->count <= 3)
    topology_definition = 
      trivial_topology (group);
  else
    topology_definition = 
      run_external_topology_generator (group);

  if (topology_definition == NULL)
    goto topology_finish;

  /* Now build the complete topology update string */
  string_init (update_string, 64);

  /* Set up a header specifing the group  */
  string_append_string (update_string, "(:topology-update \n");

  string_append_string_field (update_string,
			      " ", 
			      ":group-name", group->name, 
			      "\n");
  
  string_append_int_field (update_string,
			   " ", 
			   ":version", time.tv_sec, 
			   "\n");

  string_append_string (update_string, site_definitions);
  string_append_string (update_string, group_definition)

  if (topology_definition != NULL)
    string_append_string (update_string, topology_definition);


  if (debug)
    printf ("The topology-update is:\n%s", update_string);

  datablock =  datablock_new (TYPE_TOPOLOGY, whoami->id, 
			      update_string, update_string_length + 1);

 topology_finish:
  string_free (update_string);
  xfree (site_definitions);
  xfree (group_definition);
  xfree (topology_definition);

  /* put us back on the event queue */
  {
    struct timeval how_soon;
    int reschedule = 0;

    /* The funky logic is so that all rescheduling takes place in
     * reschedule_topology ()
     */
    if (group->topology_event == PLEASE_RUN_TOPOLOGY)
      reschedule = 1;

    how_soon = variable_period (group->update_period);
    group->topology_event =
      event_post (generate_topology, how_soon, group_name,
		  END_OF_ARGS);
  
    if (reschedule != 0)
      reschedule_topology (group);
  }

  if (datablock != NULL)
    {
      dispatch_datablock (datablock);
      datablock_free (datablock);
    }
}

/* For use with a few sites (<= 3) */
char *
trivial_topology (Group *group)
{
  static char *trivial = "1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 ";
  char *topology;

  topology = build_topology_definition (group->sites, trivial);
  return (topology);
}

char *
run_external_topology_generator (Group *group)
{
  struct stat statbuf;
  int input_file = -1;
  int output_file;
  char *topology_result = NULL;
  int i, j;
  int n;
  float rtt;
  float bandwidth;
  int exit_code;

  string_declare (command);
  string_declare (output_filename);
  string_declare (output_string);

  string_init (output_string, 128);
  string_append_int (output_string, group->topology_connectivity);
  string_append_string (output_string, "\n");
  string_append_int (output_string, group->group_sites->count);
  string_append_string (output_string, "\n");

  /* Generate a string containing the input for the topology
   * generator program 
   */

  for (i = 0; group->sites[i]; i++)
    {
      /* Output a triangular matrix */
      SiteInfo *site;

      site = group->sites[i];

      for (j = 0; group->sites[j] != NULL; j++)
	{
	  Statistics *stat;

	  stat = 
	    LIST_FIND (group->sites[j]->estimates, &group->sites[i]->id,
		       statistics_by_id);

	  if (stat == NULL)
	    {
	      rtt = -1.0;
	      bandwidth = -1.0;
	    }
	  else
	    {
	      rtt = stat->round_trip_time;
	      bandwidth = stat->bandwidth;
	    }
	  
	  string_append_float (output_string, bandwidth);
	  string_append_string (output_string, " ");
	  string_append_float (output_string, rtt);
	  string_append_string (output_string, "\t");
	}
      string_append_string (output_string, "\n");
    }

  if (debug)
    printf ("%s\n", output_string);

  /* Now open the file and output the stuff */
  string_init (output_filename, 16);
  string_append_string (output_filename, "/tmp/");
  string_append_string (output_filename, group->name);
  string_append_string (output_filename, ".dat");

  output_file = open (output_filename, O_WRONLY | O_CREAT | O_TRUNC, 0777);

  if (output_file == -1)
    {
      syslog (LOG_ERR, "Could not open temporary file `%s'for estimator: %m",
	      output_filename);

      goto cleanup;
    }

  if (write (output_file, output_string, output_string_length) <= 0)
    {
      syslog (LOG_ERR, "Could write to temporary file `%s' for estimator: %m",
	      output_filename);
      string_free (output_string);
      close (output_file);
      goto cleanup;
    };

  close (output_file);
  string_free (output_string);


  /* Now start the estimate process */
  string_init (command, 10);
  string_append_string (command, group->topology_generator);
  string_append_string (command, " < ");
  string_append_string (command, output_filename);
  string_append_string (command, " > ");

  /* Now append the output suffix to the filename to create 
   * the output filename.
   * Sorry, I realize this is tacky...
   */
  string_append_string (output_filename, ".output");

  string_append_string (command, output_filename);

#ifdef NEVER
  string_append_string (command, " &");

  /* we do this to avoid popen which seemed to be causing a lot of problems 
   * We execute the system command which should return quickly since we 
   * toss the job in the background.
   * when then spin and wait on the topology results which we hope are written 
   * atomically.  This whole piece of code should be fixed, but lets do 
   * that when when we know this parts works.
   */
#endif
  exit_code = system (command);

#ifdef NEVER
  exit_code = (exit_code >> 8) & 0xff;

  if (exit_code != 0)
    {
      syslog (LOG_ERR, "Topology calculation failed, `%s'\n", command);
      goto cleanup;
    }
#endif
#ifdef NEVER
  {      
    int stat_result;

    retries = 6;
    do 
      {

	stat_result = 
	if (stat_result != 0)
	  perror ("stat");

	retries--;
      } while (statbuf.st_size == 0 && retries >= 0);
  }

  if (retries < 0)
    goto cleanup;
#endif

  if (stat (output_filename, &statbuf) == -1 )
    goto cleanup;

  input_file = open (output_filename, O_RDONLY);
  if (input_file >= 0)
    {
      char *buffer;
      buffer = xmalloc (statbuf.st_size + 1);
      n = read (input_file, buffer, statbuf.st_size);
      if (n <= 0)
	{
	  topology_result = NULL;
	}
      else
	{
	  buffer[n] = '\0';
	  topology_result = build_topology_definition (group->sites, buffer);
	}
      xfree (buffer);
    }
  else
    topology_result = NULL;


 cleanup:
  if (input_file != -1)
    close (input_file);

  string_free (command);
  string_free (output_filename);
  string_free (output_string);
  return (topology_result);
}

char *
build_site_definitions (SiteInfo **sites)
{
  int i;
  string_declare (data);
  string_declare (address);

  string_init (data, 128);
  string_init (address, 17);

  /* Build up the sitelist */
  string_append_string (data, " (:site-list\n (");
  for (i = 0; sites[i] != NULL; i++)
    {
      if (i != 0)
	string_append_string (data, "  ");

      string_append_string (data, "(:site-define");
      string_append_string_field (data, 
				  "\n   ", 
				  ":site-name", sites[i]->name, 
				  "");

      string_reset (address);
      string_append_address (address, sites[i]->id.addr.bytes);

      string_append_string_field (data, 
				  "\n    ", 
				  ":inet-address", address, 
				  "");

      string_append_string_field (data, 
				  "\n    ", 
				  ":hostname", sites[i]->host, 
				  "");
      string_append_int_field (data, 
			       "\n    ", 
			       ":client-port", sites[i]->client_port, 
			       "");

      string_append_int_field (data, 
			       "\n    ", 
			       ":data-port", sites[i]->data_port, 
			       "");
      
      string_append_float_field (data, 
				 "\n    ",
				 ":longitude",  sites[i]->longitude,
				 "");

      string_append_float_field (data, 
				 "\n    ",
				 ":lattitude",  sites[i]->lattitude,
				 ")");

    }
  string_append_string (data, "))");

  string_free (address);

  return (data);
}

/* Build up a list of group definitions so the master can set the 
 * group parameters when updating.
 */
char *
build_group_definition (Group *group)
{
  string_declare (data);

  string_init (data, 128);

  string_append_string (data, "(:group-define");
  string_append_string_field (data, 
			      "\n   ", 
			      ":group-name", group->name, 
			      "");

  string_append_int_field (data, 
			      "\n   ", 
			      ":ping-period", group->ping_period,
			      "");

  string_append_int_field (data, 
			      "\n   ", 
			      ":bandwidth-period", group->bandwidth_period,
			      "");

  string_append_int_field (data, 
			      "\n   ", 
			      ":estimates-period", group->estimates_period,
			      "");

  string_append_int_field (data, 
			      "\n   ", 
			      ":join-period", group->join_period,
			      "");

  string_append_int_field (data, 
			      "\n   ", 
			      ":update-period", group->update_period,
			      "");

  string_append_string_field (data, 
			      "\n   ", 
			      ":master-site", group->master_site->name,
			      "");

  string_append_string (data, "))");
  return (data);
}

/* Given a sitelist and a string representing and adajency matrix, 
 * build a lispy topology string.
 */
char *
build_topology_definition (SiteInfo **sites, char *raw_topology)
{  
  SiteID *site_id;
  char **words;
  int site_count;
  int i, j;
  char **cmd_args_from (char *);
  string_declare (result);

  string_init (result, 128);

  string_append_string (result, " (:topology ");
  /* Get the number of sites */
  for (site_count = 0; sites[site_count]; site_count++);
  
  /* Get data word by word.  Each line gives the neighbors of that node */
  words = cmd_args_from (raw_topology);
  
  /* The first word will indicate the number of nodes, and the reset the
   * adjacentcy matrix.
   */
  for (i = 0; i < site_count; i++)
    {
      /* Output basic site information */
      site_id = &sites[i]->id;
      string_append_string (result, "\n  (");
	    
      string_append_string (result, "(:site-id ");
      string_append_siteid (result, site_id);
      string_append_string (result, ")");
      
      string_append_string (result, "\n   (:neighbors ");
      for (j = 0; j < site_count; j++)
	{
	  /* Ignore ourselves */
	  if (i == j)
	    continue;
	  /* Now output the neighbors for that site */

	  if (atoi(words[1 + i * site_count + j]) != 0)
	    {
	      site_id = &sites[j]->id;
	      string_append_siteid (result, site_id);
	      string_append_string (result, " ");
	    }
	}
      string_append_string (result, "))");
    }
  string_append_string (result, "))\n");
  cmd_args_free (words);
  return (result);
}

/* Reschedule topology for sometime soon */
reschedule_topology (Group *group)
{
  if (group->topology_event == NULL)
    group->topology_event = PLEASE_RUN_TOPOLOGY;
  else if (group->topology_event != NULL 
	   && group->topology_event != PLEASE_RUN_TOPOLOGY)
    reschedule_now (group->topology_event);
}

