/* $Id: threads.c,v 1.6 1995/07/25 20:08:48 dante Exp $ */
#include <stdio.h>
#include <sys/wait.h>
#include <floodd.h>
#include <datablock.h>
#include <dynamic_string.h>
extern int debug;

DataBlock *bandwidth_block;

#if defined (HAVE_RANDOM)
extern long random ();
#else
#define srandom srand
#define random rand
#endif /* !HAVE_RANDOM */

/* We use the following global variables to reschedule events when
 * we need to (using event_reschedule()).
 */

Event *ping_event;
Event *bandwidth_event;
Event *estimates_event;

struct timeval variable_period (int period);
char *estimates_for_group (Group *group);

int
ping_site (char **args)
{
  char *group_name = args[0];
  Group *group;

  SiteInfo *site = NULL;
  Message ping;
  struct timeval how_soon;
  int i;

  /* for fast start */
  int ping_period, *current_period, *ping_count;	

/* Randomly choose a site to ping */

  group = group_by_name (group_name);
  if (group == NULL)
    return;

  /* Post another ping event on the event queue */
  ping_period = (int) group->ping_period;

  if (group->group_sites->count > 1)
    {
      /* First check to see that we have not sent ping to all sites */

      /* If we find an untapped site, choose, otherwise clear the bits */
      for (i = 0; group->sites[i] != NULL; i++)
	if (group->sites[i]->ping_sent == 0 && group->sites[i] != whoami)
	  goto choose;
      
      for (i = 0; group->sites[i] != NULL; i++)
	if (group->sites[i] != whoami)
	  group->sites[i]->ping_sent = 0;

    choose:
      do
       {
          site = group->sites[random () % group->group_sites->count];
       } while (site == whoami || site->ping_sent == 1);
      
      if (site != NULL)
	{
	  site->ping_sent = 1;
	  ping.type = htonl (TYPE_PING);
	  ping.sender = whoami->id;
	  ping.id.site = whoami->id;
	  ping.id.time = timeval_current ();
	  if (debug)
	    printf ("pinging `%s' %d.%d\n", 
		    site->name, ping.id.time.tv_sec, ping.id.time.tv_usec);
	  site_send_message (site, &ping, sizeof (ping));
	}
    }
    else 
      goto reschedule;

  /* modify ping period according to the fast start algorithm */
  current_period = &group->ping_current_period;
  ping_count = &group->ping_count;

  /* initial state for group or a new site joining*/
  if (*current_period == 0 || group->group_sites->count > group->ping_old_sites) {
     /* if some sites join, then restart pinging process */
     *current_period = ping_period / 16;
     *ping_count = 0;
  }
  /* update site count either more or less */
  group->ping_old_sites = group->group_sites->count;

  if (*current_period < ping_period) {
     ping_period = *current_period;
     if (++(*ping_count) >= site_list->count) {
	*ping_count = 0;
         /* slow down by double the ping period */
	*current_period += *current_period;
	if (*current_period > group->ping_period)
		*current_period = group->ping_period;
     }
  }

 reschedule:
  how_soon = variable_period (ping_period);

  ping_event =  event_post (ping_site, how_soon, group_name, END_OF_ARGS);
  return (0);
}

int
send_ack (SiteID *siteid, MessageID *id)
{
  Message ack;
  SiteInfo *site;

  site = site_by_id (siteid);
  if (site == NULL)
    return;
  
  if (debug)
    {
      printf ("Sending ACK: (%s:%u:%u)", siteid_to_string (&id->site),
	      ntohl (id->time.tv_sec),
	      ntohl (id->time.tv_usec));
      printf (" -> %s\n", siteid_to_string (siteid));
    }

  ack.type = htonl (TYPE_ACK);
  ack.sender = whoami->id;
  ack.id = *id;
  site_send_message (site, &ack, sizeof (ack));
}


int
bandwidth_estimate_some_site (char **args)
{
  char *group_name = args[0];
  Group *group;
  SiteInfo *site = NULL;
  struct timeval how_soon;
  int i;

  /* for fast start */
  int bw_period, *current_period, *bw_count;	

  group = group_by_name (group_name);
  if (group == NULL)
    return;

  /* Post another bandwidth estimation event on the event queue */
  bw_period = (int) group->bandwidth_period;

  /* Randomly choose a site to estimate bandwidth */
  if(group->group_sites->count > 1)
    {
      /* If we find an untapped site, choose, otherwise clear the bits */
      for (i = 0; group->sites[i] != NULL; i++)
	if (group->sites[i]->bw_estimate_sent == 0 && group->sites[i] != whoami)
	  goto choose;
      
      for (i = 0; group->sites[i] != NULL; i++)
	if (group->sites[i] != whoami)
	  group->sites[i]->bw_estimate_sent = 0;

    choose:
      do
	{
	  site = group->sites[random () % group->group_sites->count];
	} while (site == whoami || site->bw_estimate_sent == 1);

      if (site->status != STATUS_DOWN)
	{
    
	  if (debug)
	    printf ("Sending Bandwidth estimator to %s\n",
		    site->name);

	  datablock_reference (bandwidth_block);
	  queue_insert (site->datablocks_to_send, bandwidth_block);
	}
    }
    else 
      goto reschedule;


  /* modify ping period according to the fast start algorithm */
  current_period = &group->bw_current_period;
  bw_count = &group->bw_count;

  /* initial state for group or a new site joining*/
  if (*current_period == 0 || group->group_sites->count > group->bw_old_sites) {
     /* if some sites join, then restart pinging process */
     *current_period = bw_period / 16;
     *bw_count = 0;
  }
  /* update site count either more or less */
  group->bw_old_sites = group->group_sites->count;

  if (*current_period < bw_period) {
     bw_period = *current_period;
     if (++(*bw_count) >= site_list->count) {
	*bw_count = 0;
         /* slow down by double the ping period */
	*current_period += *current_period;
	if (*current_period > group->bandwidth_period)
		*current_period = group->bandwidth_period;
     }
  }

 reschedule:

  how_soon = 
    variable_period (bw_period);

  bandwidth_event =
    event_post (bandwidth_estimate_some_site, how_soon, group_name, 
		END_OF_ARGS);
  return (0);
}

void *
event_thread (void)
{
  struct timeval next;
  
  initialize_events ();

  event_process_queue ();
 
  while (1)
    {
      next = event_next_time ();
      if (next.tv_sec > 5)
	next.tv_sec = 5;

      IO_process (&next);

      event_process_queue ();
    }
  return (NULL);
}

/* Send statistics out */
send_estimates (char **args)
{
  char *group_name = args[0];
  Group *group;

  /* Send bunch of statistics structures */
  DataBlock *datablock;
  Statistics **stats;
  struct timeval how_soon;
  string_declare (data);
  int i;

  group = group_by_name (group_name);
  if (group == NULL)
    return;

  if (debug)
    printf ("Sending estimates\n");

  estimates_for_group (group);

  /* put us back on the event queue */
  how_soon = variable_period (group->estimates_period);

  estimates_event =
    event_post (send_estimates, how_soon, group_name, END_OF_ARGS);
}

char *
estimates_for_group (Group *group)
{
  Statistics *stat;
  DataBlock *datablock;
  string_declare (data);
  int i;

  string_init (data, 128);

  string_append_string (data, group->name);
  string_append_string (data, "\n");
  for (i = 0; group->sites[i]; i++)
    {
      stat = 
	LIST_FIND (whoami->estimates, &group->sites[i]->id, statistics_by_id);

      if (stat != NULL)
	{
	  string_append_siteid (data, &(stat->id));
	  string_append_string (data, " ");
	  string_append_int    (data, stat->round_trip_time);
	  string_append_string (data, " ");
	  string_append_int    (data, stat->bandwidth);
	  string_append_string (data, "\n");
	}
    }

  if (data_length != 0)
    {
      datablock = 
	datablock_new (TYPE_ESTIMATES, whoami->id, NULL, 0);

      if (datablock != NULL)
	{
	  if (datablock_set_data (datablock, data, data_length + 1) != NULL)
	    flood_datablock_to_group (group, datablock, NULL);
	  else
	    string_free (data);

	  datablock_free (datablock);
	}
    }
  else
    string_free (data);
}

status_sweep (char **args)
{
  char *group_name = args[0];
  Group *group;
  struct timeval how_soon;
  int i;

  if (debug)
    printf ("Status sweep\n");

  group = group_by_name (group_name);
  if (group == NULL)
    return;

  for (i = 0; sites[i] != NULL; i++)
    if (sites[i]->status > 0)
      sites[i]->status--;

  purge_logs ();

  /* Eliminate any old site that we haven't heard from in a while */
  purge_sites ();

  how_soon = variable_period (3600);

  event_post (status_sweep, how_soon, group->name, END_OF_ARGS);
}

/* Issue join requests */
/* Join protocol - try and keep this up to date!
 * For each group:
 *  1) Attempt to contact the `master' site.
 *  2) Attempt to contact any other site.
 *  3) repeat 1 and 2 until we get a topology.
 * We pass the group name rather than the group structure so that we
 * can recover simply and gracefully if the group is deleted.
 */

join_a_group (char **args)
{
  char *name = args[0];
  Group *group;
  SiteInfo *site;
  SiteInfo *master_site;
  string_declare(join_request);
  string_declare(address);
  int timeout;
  int i;

  group = group_by_name (name);
  if (group == NULL)
    {
      xfree (name);
      return;
    }

  group->join_event = NULL;

  /* Bail out if we are the group master */
  if (group->master_site == whoami)
    return;

#ifdef NEVER
   /* Hmm.... We would like to avoid issuing a new join, but how do
    * we know when the master site has come back up?  The master site
    * should probably send out a message when it need to be reinitialized.
    * in the mean time, we will leave this portion out...
    */

  /* If we have a topology, then we are done */
  if (group->neighbors != NULL && group->neighbors[0] != NULL)
    {
      goto repost_join;
    }
#endif

  /* Get some values we might need for last in case the group
   * changes during a send operation.
   */
  timeout = group->join_period / 10;
  master_site = group->master_site;

  /* If we are the master sites and we have no neighbors, don't
   * do anything.  If we are the master site and we have neighbors, 
   * we should be sociable and tell the other group members that 
   * we are here.  It might be that we have crashed and are 
   * starting up again, so others in the group might wished to be
   * notified asap that we are alive.
   */
      
  if (group->master_site == whoami)
    goto repost_join;
            
  /* Make a request with the latest parameters info and try to join */
  string_init (join_request, 128);
  string_init (address, 17);
  
  /* This was yanked from topology.c:generate-topology() */
  /* this should be macroized or librariyized */
  string_append_string (join_request, "set \""); 
  string_append_string (join_request, "(:site-define");
  string_append_string_field (join_request, 
			      "   ",
			      ":site-name", whoami->name, 
			      "");
  
  string_reset (address);
  string_append_address (address, whoami->id.addr.bytes);

  string_append_string_field (join_request, 
			      "    ", 
			      ":inet-address", address, 
			      "");
  
  string_append_string_field (join_request, 
			      "    ", 
			      ":hostname", whoami->host, 
			      "");
  string_append_int_field (join_request, 
			   "    ", 
			   ":client-port", whoami->client_port, 
			   "");

  string_append_int_field (join_request, 
			   "    ", 
			   ":data-port", whoami->data_port, 
			   "");
      
  string_append_float_field (join_request, 
			     "    ",
			     ":longitude",  whoami->longitude,
			     "");

  string_append_float_field (join_request, 
			     "    ",
			     ":lattitude",  whoami->lattitude,
			     ")");
      
  /* Now add the group groups we wish to join */
  for (i = 0; groups[i] != NULL; i++)
    {
      string_append_string (join_request, "(:group-join ");
      
      string_append_string_field (join_request, 
				  " ",
				  ":group-name",
				  group->name,
				  "");
      string_append_string (join_request, "(:site-id ");
      string_append_siteid (join_request, &whoami->id);
      string_append_string (join_request, "))\"\n");
    }

  /* Send off the request to the master site if we are not 
   * the master site 
   */
  if (group->master_site != NULL && group->master_site != whoami)
    {
      send_string_to_client_interface (group->master_site, join_request);
    }

  /* Send a join requst off to at least one other site 
   * Pick a site at random -  (if there is more than one site)
   * We could make the choosing of the sites a bit more efficient, but
   * who cares.  It's random anyway.
   */
  site = NULL;
  if (site_list->count > 1)
    {
      do 
	{
	  site = sites[random () % site_list->count];
	}
      while (site == whoami);
	  
      if (site != master_site)
	send_string_to_client_interface (site, join_request);
    }
  string_free (join_request);
  string_free (address);
  /* Set up a join for some time in the future 
   * If we don't yet have a topology (i.e. we are not in a group)
   * Issue a join request every minute.
   */

 repost_join:
  {
    struct timeval how_soon;

    if (group->topology == NULL)

      how_soon = variable_period (60);
    else
      how_soon = variable_period (group->join_period);

    group->join_event = 
      event_post (join_a_group, how_soon, name, END_OF_ARGS);
  }
}

/* A periodic thread */
join_groups ()
{
  int i;
  struct timeval how_soon;

  how_soon.tv_sec = 5;
  how_soon.tv_usec = 0;

  for (i = 0; groups[i] != NULL; i++) 
    {
      groups[i]->join_event = 
	event_post (join_a_group, how_soon, strdup (groups[i]->name), 
		    END_OF_ARGS);
      how_soon.tv_sec += 5;
    }
}

/* Adjust some things to know that we have head from a site */

void
siteinfo_up (SiteInfo *site)
{
  int i;
  struct timeval current;

  if (site->status < STATUS_MAX)
    site->status++;

  current = timeval_current ();
  site->last_heard_from = current.tv_sec;

  /* If the status went from STATUS_DOWN to up, then we 
   * need to fire up a sender connection for this site in case any
   * datablocks are waiting.
   */
  if (site->sender_fd == NOTSET && !queue_empty (site->datablocks_to_send))
    {
      connection_write_start (site);
      /* If this site was the master site, then we need to reschedule
       * the join process. 
       */
      for (i = 0; groups[i] != NULL; i++)
	{
	  if (groups[i]->master_site == site)
	    reschedule_now (groups[i]->join_event);
	}
    }
}

initialize_events ()
{
  struct timeval how_soon;
  void free();
  int i;

  /* Create one block to use for bandwidth estimation */
  bandwidth_block =
    datablock_new (TYPE_BANDWIDTH, whoami->id, NULL, 0);

  /* Put random bytes in the bw/exchange datablock */
  for (i = 0; i < bandwidth_block->length; i++)
    bandwidth_block->data[i] = random() % 8;

  bandwidth_block->length = bandwidth_size;
  bandwidth_block->data = xmalloc (bandwidth_block->length);
  memset (bandwidth_block->data, 0, bandwidth_block->length);

  /* Initialize per group threads */
  for (i = 0; groups[i] != NULL; i++)
    {
      /* Ping other sites, do the first one quickly */
      how_soon = variable_period (groups[i]->ping_period/60);
      ping_event =
	event_post (ping_site, how_soon, 
		    strdup (groups[i]->name), END_OF_ARGS);
  
      /* Estimate bandwidth to other sites, do the first one quickly */
      how_soon = variable_period (groups[i]->bandwidth_period/60);
      bandwidth_event = 
	event_post (bandwidth_estimate_some_site, how_soon, 
		    strdup (groups[i]->name), END_OF_ARGS);
      
      /* Send out the estimates collected */
      how_soon = variable_period (groups[i]->estimates_period);
      estimates_event =	
	event_post (send_estimates, how_soon, 
		    strdup (groups[i]->name), END_OF_ARGS);

      /* Start up the status sweeper */
      how_soon = timeval_from_seconds (groups[i]->estimates_period);
      event_post (status_sweep, how_soon, 
		  strdup (groups[i]->name), END_OF_ARGS);
    }
}

/* Given a period in seconds, return a timeval structure that is a random
 * variable.  We want to avoid synchronous behavior.
 */

struct timeval 
variable_period (int period)
{
  struct timeval time;

  time = timeval_from_seconds ((int) (((double) period / 2.0 + 
				       ((unit_variate () * (double) period)
					/ 2.0))));
  if (debug)
    printf ("Variable Period: (%d) %u.%u\n", period, time.tv_sec, time.tv_usec);

  return (time);
}

