/* $Id: parse.c,v 1.2 1995/07/25 20:07:49 dante Exp $ */
/* Simple lispy parsing routines */
#include <stdio.h>
#include <string.h>
#include <xmalloc.h>

/* They simply return strings and tokens in string form */
	
static char *delimiters = "()";

token_set_delimiters (char *new_delimiters)
{
  delimiters = new_delimiters;
}

/* Skips to the beginning of the next line */
char *
skip_to_next_line (char *string)
{
  if (string == NULL)
    return (NULL);

  while (*string && *string != '\n')
    string++;
  if (*string)
    string++;
  return (string);
}

/* Skip over any white space. Assumes that we are starting on a space */
char *
skip_space (char *string)
{
  if (string == NULL)
    return (NULL);

  /* skip over comments */
  while  (*string == ';' || *string == '#')
    string = skip_to_next_line (string);

  while (*string && isspace (*string))
    {
      if (*string == ';' || *string == '#')
	string = skip_to_next_line (string);
      else
	string++;
    }
  return (string);
}

/* skip over a quoted string */
char *
skip_quoted_string (char *string)
{
  int escaped = 0;

  if (string == NULL)
    return;

  if (*string != '\"')
    return (NULL);

  /* skip quote character */
  string++;

  while (*string != '\"' || escaped == 1)
    {
      /* If we don't find a closing quote, then we have an error */
      if (*string == '\0')
	return (NULL);

      if (*string == '\\')
	escaped = 1;
      else
	escaped = 0;

      string++;
    }
  /* skip trailing '"' */
  string++;
  return (string);
}

/* skips over a token.  Assumes we are starting on a token 
 * A token is either a sequence of non-space characters, or 
 * a quoted string (e.g. "foo bar")
 */
char *
skip_token (char *string)
{
  if (string == NULL)
    return (NULL);

  if (*string == '\0')
    return (string);

  if (strchr (delimiters, *string))
    {
      string++;
      return (string);
    }

  /* skip over comments
   * Quotes don't count in comments
   */
  while  (*string == ';' || *string == '#')
    string = skip_to_next_line (string);

  while (*string && !isspace (*string) && !strchr (delimiters, *string))
    {
      if (*string == '\"')
	{
	  string = skip_quoted_string (string);
	  if (string == NULL)
	    return (NULL);
	}
      else if (*string == ';' || *string == '#')
	string = skip_to_next_line (string);
      else 
	string++;
    }
  return (string);
}

/* Advance to the next token.  
 * Should add a simple thing for checking for quoted strings.
 * If a token begins with a '#' or ';' then we skip to the end of line
 */
char *
next_token (char *string)
{
  if (string == NULL)
    return (NULL);

  string = skip_token (string);
  string = skip_space (string);
  return (string);
}

/* Take current token and return it as a string */
char *
return_token (char *string)
{
  char *begin;
  char *end;
  char *result;
  int length;

  if (string == NULL)
    return (NULL);

  /* skip any leading space */
  begin = skip_space (string);
  end = skip_token (begin);

  if (end == NULL)
    return (NULL);

  /* Check to see if the string is quoted */
  if (*begin == '\"')
    {
      begin++;
      end--;
/*      if (*end == '\"')
	end--;
*/
    }

  if (end == NULL)
    return (NULL);
  else
    length = end - begin;

  if (length < 0)
    return (NULL);

  result = xmalloc (length + 1);
  strncpy (result, begin, length);
  result[length] = '\0';
  return (result);
}


/* Skip whitespace before we skip a list 
 *  If the first thing after the whitespace is not a '('
 * then skip till we find a closing ')' for this level of the list.
 */
char *
skip_list (char *string)
{
  int level;

  string = skip_space (string);

  if (string == NULL)
    return (NULL);

  while (*string == ')')
    string = next_token (string);

  string = next_token (string);

  level = 1;
  
  while (level != 0 && string != NULL && *string != '\0')
    {
      if (*string == '(')
	level++;
      if (*string == ')')
	level--;
      string = next_token (string);
    }
  return (string);
}

char *
next_list (char *string)
{
  if (string == NULL)
    return (NULL);

  string = skip_space (string);
  string = skip_list (string);
  string = skip_space (string);
  return (string);
}

/* Return a list.  A list is defined as beginning with the token '(' and 
 * ending with a ')'.  A list may contain sublists.  i.e. lisp.
 */
char *
return_list (char *string)
{
  char *begin;
  char *end;
  char *result;
  int length;

  if (string == NULL)
    return (NULL);

  begin = skip_space (string);
  end = skip_list (begin);

  /* trim trailing space */
  while (end && isspace (*end) && end != begin)
    end--;

  if (end == NULL)
    length = strlen (begin);
  else
    length = end - begin;
  result = xmalloc (length + 1);
  strncpy (result, begin, length);
  result[length] = '\0';
  return (result);
}
