/* $Id: group.c,v 1.5 1995/07/25 20:05:47 dante Exp $ */
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#if defined (HAVE_SYSLOG_H)
#  include <syslog.h>
#endif /* !SYS_SYSLOG */

#include <netdb.h>
#include <list.h>
#include <event.h>
#include <floodd.h>
#include <dynamic_string.h>
#include <xmalloc.h>
extern int debug;
/*
 * Groups
 */


extern Group **groups;

#ifndef NOTSET
#define NOTSET -1
#endif

Group *
group_new (char *name)
{
  Group *group;

  group = xmalloc (sizeof (Group));
  group->name = name;
  group->sites = NULL;
  group->group_sites = list_new (NULL, 0);
  group->master_site = NULL;

  group->estimates_period = NOTSET;
  group->update_period = NOTSET;
  group->ping_period = NOTSET;
  group->bandwidth_period = NOTSET;
  group->join_period = 3600; /* default to every hour */
  group->max_size = NOTSET;

  group->neighbor_list = list_new (NULL, NULL);
  group->neighbors = (SiteID **)group->neighbor_list->list;

  group->old_neighbor_list = list_new (NULL, NULL);
  group->old_neighbors = (SiteID **)group->old_neighbor_list->list;
  group->old_topology_version = TOPOLOGY_NOTSET;

  group->allow_list = list_new (NULL, NULL);
  group->deny_list = list_new (NULL, NULL);
  group->topology_version = TOPOLOGY_NOTSET;
  group->topology_generator = strdup ("topology-generator");
  group->topology_connectivity = 2;
  group->topology = NULL;

  group->join_event = NULL;
  group->topology_event = NULL;

  /* for fast start */
  group->ping_current_period = 0;
  group->ping_count = 0;
  group->ping_old_sites = 0;
  group->bw_current_period = 0;
  group->bw_count = 0;
  group->bw_old_sites = 0;

  /* for fast start */
  group->ping_current_period = 0;
  group->ping_count = 0;
  group->ping_old_sites = 0;
  group->bw_current_period = 0;
  group->bw_count = 0;
  group->bw_old_sites = 0;

  return (group);
}

Group *
group_by_name (char *name)
{
  int i;

  if (name == NULL || groups == NULL)
    return (NULL);

  for (i = 0; groups[i] != NULL; i++)
    if (strcmp (name, groups[i]->name) == 0)
      return (groups[i]);
  return (NULL);
}


/* Update the topology by taking the new_topology list and
 * creating the neighbors list for each site
 */
void
group_topology_update (Group *group, int version, char *topology)
{
  SiteID  site_id;
  SiteInfo *site;
  char *token;
  char *neighbor_list;
  void *tmp;

  if (topology == NULL)
    return;
 
  /* Save an ascii version of the topology_mesg */
  if (group->topology != NULL)
    xfree (group->topology);

  group->topology = (char *)strdup (topology);

  /* Swap old and new topology lists, and clean out the old one */
  tmp = group->neighbor_list;
  group->neighbor_list = group->old_neighbor_list;
  group->old_neighbor_list = tmp;

  /* Remove the old neighbors, but don't delete the list */
  LIST_FREE_ELEMENTS (group->neighbor_list, xfree);
  
  topology = next_token (strstr (topology, ":topology "));
  
  neighbor_list = return_list (topology);
  while  (neighbor_list != NULL && *neighbor_list != '\0') 
    {
      token = next_token (strstr (neighbor_list, ":site-id"));
      if (token == NULL)
	{
	  if (debug)

	    syslog (LOG_DEBUG, "Error parsing neighbors: `%s'\n", 
		    neighbor_list); 
	  break;
	}
      
      scan_siteid (token, &site_id);
      site = site_by_id (&site_id);

      if (site != NULL)
	group_site_add (group, site);

      if (site == whoami)
	{
	  SiteID *id;

	  token = (char *)strstr (topology, ":neighbors");
	  token = next_token (token);
	  
	  while (*token != '\0' && *token != ')')
	    {
	      /* The first id was the site_id that this pertains to */
	      id  = xmalloc (sizeof (*id));
	      scan_siteid (token, id);
	      list_insert (group->neighbor_list, id);
	      token = next_token (token);
	    }
	}
      topology = next_list (topology);
      xfree (neighbor_list);
      neighbor_list = return_list (topology);
    }
  xfree (neighbor_list);
  group->old_topology_version = group->topology_version;
  group->topology_version = version;
  group->neighbors = (SiteID **)group->neighbor_list->list;
  group->old_neighbors = (SiteID **)group->old_neighbor_list->list;
}

void
group_site_add (Group *group, SiteInfo *siteinfo)
{

  /* First, make sure that we can join the group */
  if (!group_allowed (group, siteinfo))
    {
      syslog (LOG_INFO, "Site `$s'not allowed in group `%s'\n",
	      siteid_to_string (&siteinfo->id), group->name);
      return;
    }

  /* don't add anything twice */
  if (list_find (group->group_sites, siteinfo, NULL) == NULL)
    {
      if (group->max_size == NOTSET ||
	  group->group_sites->count < group->max_size)
	{
	  group->sites = list_insert (group->group_sites, siteinfo);

	  /* Since we added a new site to the group, we need to recompute
	   * the topology
	   */
	  reschedule_topology (group);
	}
      else
	{
	  syslog (LOG_INFO, "Site `$s'not allowed in group `%s': Max size '%d' exceeded\n",
		  siteid_to_string (&siteinfo->id), 
		  group->name, 
		  group->max_size);
	  return;
	}
    }
}

void
group_site_remove (Group *group, SiteInfo *siteinfo)
{
  list_delete (group->group_sites, siteinfo);
  group->sites = (SiteInfo **)group->group_sites->list;
}

void
group_sites_print (Group *group, FILE *file)
{
  SiteInfo **sitep;

  for (sitep = group->sites; sitep != NULL && *sitep != NULL; sitep++)
    siteinfo_print (*sitep, file);
}

/* Rules for admission.
 * 1) If the accept list is non empty, first check to see if 
 * a site is on the accept list.  If we do not match the accept list
 * deny access.
 * 2) If the deny list is non empty, then check to see that the site
 * is not explicitly on the deny list.  If it is, deny access.
 * 3) allow access.
 */

int
group_allowed (Group  *group, SiteInfo *site)
{
  extern int mask_compare (void *, void *);

  access_list_update ();

  if (group->allow_list->count > 0)
    {
      if (!list_find (group->allow_list, site->id.addr.bytes, mask_compare))
	return (0);
    }
  
  if (list_find (group->deny_list, site->id.addr.bytes, mask_compare))
    return (0);
  
  return (1);
}

/*
 * Basic bread and butter routines.  Initialization and main, push a few
 * housekeeping functions.
 */

group_initialize (Group *group)
{
  struct timeval how_soon;
  int generate_topology (char *groupname);

  if (group->sites == NULL)
    {
      if (debug)
	syslog (LOG_ERR, "No sites read for group `%s'\n", group->name);
      exit (1);
    }

  if (group->ping_period == NOTSET)
    group->ping_period = 60;

  /* If the estimate period is not already set, set it to be
   *  2 * group->groups_count * ping_period
   */
  if (group->estimates_period == NOTSET) 
    {
      if (group->group_sites->count > 0)
	group->estimates_period = 
	  2 * group->ping_period * group->group_sites->count;
      else
	group->estimates_period = 10 * group->ping_period;
    }

  
  if (group->update_period == NOTSET)
    group->update_period = 1.5 * group->estimates_period;

  /* Start up the topology estimator for this group */
  if (group->master_site == whoami && 
      (group->topology_event == NULL ||
       group->topology_event == PLEASE_RUN_TOPOLOGY))
    {
      how_soon = variable_period (group->update_period);
      group->topology_event =
	event_post (generate_topology, how_soon, strdup (group->name), 
		    END_OF_ARGS);
    }
}


