/* $Id: event.c,v 1.3 1995/07/25 20:05:05 dante Exp $ */
#include <stdio.h>
#include <stdarg.h>
#include <event.h>
#include <timeval.h>
#include <xmalloc.h>

/*
 * Actions
 */

Action *
va_action_new (int (*func)(), va_list ap)
{
  Action *action;
  void *p;
  int max_arg = 0;

  action = xmalloc (sizeof (Action));
  memset (action, 0, sizeof (Action));

  action->action = func;

  /* grab the value arguments */
  p = va_arg (ap, void *);

  while (p != END_OF_ARGS && p != FREE_FUNCS)
    {
      if  ((action->args_count + 1) > max_arg)
	{
	  max_arg += 5;
	  action->args = 
	    xrealloc (action->args, max_arg * sizeof (void *));
	}
      action->args[action->args_count] = p;
      action->args_count++;
      p = va_arg (ap, void *);
    }

  max_arg = 0;

  /* Now grab the free functions */
  if (p != END_OF_ARGS)
    {
      p = va_arg (ap, void *);
      
      while (p !=END_OF_ARGS && p != FREE_FUNCS)
	{
	  if  ((action->delete_funcs_count + 1) > max_arg)
	    {
	      max_arg += 5;
	      action->delete_funcs = 
		xrealloc (action->delete_funcs, max_arg * sizeof (void *));
	    }
	  action->delete_funcs[action->delete_funcs_count] = p;
	  action->delete_funcs_count++;
	  p = va_arg (ap, void *);
	}
    }
  return (action);
}

/* The format looks like 
 * func, action_new (func, arg1, arg2, ... {END_OF_ARGS | FREE_FUNCS free_func1, ... END_OF_ARGS})
 */

Action *
action_new (int (*func)(), ...)
{
  Action *action;
  va_list ap;

  va_start (ap, func);
  action = va_action_new (func, ap);
  va_end (ap);
  return (action);
}

void
action_free (Action *action)
{
  int i;
  /* run free funcs */
  for (i = 0; i < action->delete_funcs_count; i++)
    {
      if (action->delete_funcs[i] != NULL)
	action->delete_funcs[i](action->args[i]);
    }
  xfree (action->args);
  xfree (action);
}

int
action_execute (Action *action, int arg_num, void *data)
{
  if (arg_num != NOTSET)
    action->args[arg_num] = data;
  return (action->action (action->args));
}

/*
 * Events & Event  Queues
 */

/* Event Queue */
int events_count = 0;
int events_max = 0;
Event **events = NULL;

Event *
va_event_new (int (*action) (), struct timeval time, va_list ap)
{
  Event *event;

  event = xmalloc (sizeof (Event));
  event->action = va_action_new (action, ap);
  event->time = time;
  return (event);
}

Event *
event_new (int (*action) (), struct timeval time, ...)
{
  va_list ap;
  Event *event;

  va_start (ap, time);
  event = va_event_new (action, time, ap);
  va_end (ap);
  return (event);
}

Event *
event_free (Event *event)
{
  if (event->action)
    action_free(event->action);
  xfree (event);
}

void
event_insert (Event *event)
{
  int i;

  if (events_count >= events_max)
    {
      events_max += 5;
      events =
	xrealloc (events, (events_max + 1) * sizeof (Event *));
    }
  /* Insert into head */
  events_count++;
  i = events_count - 1;
  
  while (i > 0 &&
	 timeval_compare (&event->time, &events[(i - 1) / 2]->time) < 0)
    {
      events[i] = events[(i - 1) / 2];
      i = (i - 1) / 2;
    }
  events[i] = event;

}

/* Create a new event structure and put it on the event queue.
 * The timeval passed is a taken as relative to the current time.
 */

Event *
event_post (int (*action) (), struct timeval how_soon, ...)
{
  Event *event;
  va_list ap;

  va_start (ap, how_soon);
  event =
    va_event_new (action, timeval_add (timeval_current (), how_soon), ap);
  event_insert (event);
  va_end (ap);
  return (event);
}

struct timeval
event_next_time ()
{
  struct timeval time;
  if (events_count == 0)
    {
      time.tv_sec = - (int) ~0x0;
      time.tv_usec = 0;
    }
  else
    time = timeval_subtract (events[0]->time, timeval_current ());

  return (time);
}

/* Heapify the heap starting with the given element */
void
heapify (int first, int last)
{
  Event *tmp;
  int left, right, smallest;

  smallest = first;

  /* Heapify */
  left = first * 2 + 1;
  right = first * 2 + 2;

  if (left < events_count &&
      (timeval_compare (&events[left]->time, &events[first]->time) < 0))
    smallest = left;
  else
    smallest = first;

  if (right < events_count &&
      (timeval_compare (&events[right]->time,
			    &events[smallest]->time) < 0))
    smallest = right;
  if (smallest != first)
    {
      tmp = events[first];
      events[first] = events[smallest];
      events[smallest] = tmp;
    
      heapify (smallest, last);
    }
}

Event *
event_next ()
{
  Event *result;

  if (events_count == 0)
    return (NULL);

  result = events[0];

  events[0] = events[events_count - 1];
  events_count--;

  heapify (0, events_count);
  return (result);
}

/* Process events on the event queue if their time is due */
event_process_queue ()
{
  struct timeval next;
  
  /* Do we have events in the queue */
  if (events == NULL)
    return;

  /* Time for next event? */
  next = event_next_time ();

  while (next.tv_usec == 0 && next.tv_sec == 0)
    {
      Event *event;
      event = event_next ();
      action_execute (event->action, NOTSET, NULL);
      event_free (event);
      next = event_next_time ();
    }
}

Event *
event_reschedule (Event *event, struct timeval time)
{
  int sooner;
  int i;

  /* If the event is on the top of the heap.  We probably cannot
   * safely reschedule since it is probably in progress
   * 
   * Well we shouldn't be doing anything on the top of the heap since it
   * should have already been deleted from the heap.
   */
#ifdef NEVER
  if (events[0] == event)
    {
      event->time = time;
      return (NULL);
    }
#endif

  /* If the new time is sooner than the old time, bubble the event up the
   * heap
   */

  if (timeval_compare (&event->time, &time) < 0)
    sooner = 0;
  else
    sooner = 1;

  event->time = time;

  /* Find the event in the heap */
  for (i = 0; i < events_count;  i++)
    if (event == events[i])
      break;

  if (sooner)
    {
      /* bubble up */
      while (i > 0 &&
	     timeval_compare (&event->time, &events[(i - 1) / 2]->time) < 0)
	{
	  events[i] = events[(i - 1) / 2];
	  i = (i - 1) / 2;
	}
      events[i] = event;
    }
  else
    {
      /* push down */
      /* Found out where we are */
      for (i = 0; i < events_count; i++)
	if (events[i] == event)
	  {
	    heapify (i, events_count);
	    break;
	  }
    }
  return (event);
}


/* Reschedule an event for now! 
 */

Event *
reschedule_now (Event  *event)
{
  struct timeval now;

  now.tv_sec = 0;
  now.tv_usec = 0;
  
  return (event_reschedule (event, now));
}


/*
 * Routines for printing out the event queue in html.
 */

#include <bprintf.h>
void
bprint_event_record (BprintfBuffer *buffer, Event *event)
{

  extern ping_site ();
  extern bandwidth_estimate_some_site ();
  extern generate_topology ();
  extern send_estimates ();
  extern join_a_group ();
  extern status_sweep ();

  bprintf (buffer, "<TR><TD>");
  if (event->action->action == ping_site)
    {
      bprintf (buffer, "<STRONG>ping_site()</STRONG>");
    }
  else if (event->action->action == bandwidth_estimate_some_site)
    {
      bprintf (buffer, "<STRONG>bandwidth_estimate_some_site()</STRONG>");
    }
  else if (event->action->action == generate_topology)
    {
      bprintf (buffer, "<STRONG>generate_topology(%s)</STRONG>",
	       event->action->args[0]);
    }
  else if (event->action->action == send_estimates)
    {
      bprintf (buffer, "<STRONG>send_estimates()</STRONG>");
    }
  else if (event->action->action == join_a_group)
    {
      bprintf (buffer, "<STRONG>join_a_group(%s)</STRONG>", 
	       event->action->args[0]);
    }
  else if (event->action->action == status_sweep)
    {
      bprintf (buffer, "<STRONG>status_sweep()</STRONG>");
    }
  else
    {
      bprintf (buffer, "<STRONG>0x%x()</STRONG>", event->action->action);
    }
  bprintf (buffer, "</TD><TD>%s</TD></TR>", ctime (&event->time.tv_sec));
}

void
event_queue_html (BprintfBuffer *buffer)
{
  int i;
  struct timeval now;
  now = timeval_current ();

  bprintf (buffer, "<H2>Event Queue</H2>\n");
  bprintf (buffer, "<STRONG>Current time: %s</STRONG>", ctime (&now.tv_sec));

  /* Print out an event queue table */

  bprintf (buffer, "<TABLE BORDER>\n<TR><TH>Event</TH><TH>Time</TH></TR>\n");
  for (i = 0; i < events_count; i++)
    {
      bprint_event_record (buffer, events[i]);
    }
  bprintf (buffer, "</TABLE>\n");
}
