/* $Id: datablock.c,v 1.4 1995/07/25 20:04:37 dante Exp $ */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/time.h>
#include <sys/types.h>
#include <floodd.h>

extern int debug;
extern SiteInfo *whoami;
int datablocks_total_bytes = 0;
int datablocks_max_bytes = 1024 * 1024 * 4; /* 4MB by default */

/* For debugging */
int datablock_allocate_negative = 0;
int datablock_set_negative = 0;
int datablock_free_negative = 0;
#define CHUNK (1024 * 8)

/*
 * Data Block manipulation
 */

/* A global sequence number for numbering datablock sent from this node */
int global_sequence_number = 0;

DataBlock *
datablock_new (int type, SiteID siteid, void *data, int length)
{
  DataBlock *datablock;

  datablock = xmalloc (sizeof (DataBlock));
  memset (datablock, 0, sizeof (DataBlock));

  /* Set up the header */

  datablock->header.data.length = length;
  datablock->header.type = type;
  datablock->header.sender = siteid;
  datablock->header.id.site = siteid; 
  datablock->header.id.time = timeval_current ();

  datablock->refcount = 1;
  datablock->length = length;
  datablock->data = NULL;

  if (datablock_allocate (datablock) == NULL)
    {
      datablock->length = 0;
      datablock->header.data.length = 0;
      datablock_free (datablock);
      return (NULL);
    }

  if (data != NULL)
    memcpy (datablock->data, data, length);
  else 
    if (datablock->data != NULL)
      memset (datablock->data, 0, length);
  
  return (datablock);
}

datablock_space (int size)
{
  int requested = size + datablocks_total_bytes;

  if (size < 0)
    return (datablocks_max_bytes);
    
  if (requested < datablocks_max_bytes)
    return (datablocks_max_bytes - requested);
  else
    return (0);
}

datablock_set_max_bytes (int size)
{
  datablocks_max_bytes = size;
}

/* Given a datablock, that has only a header, attempt to 
 * allocate memory for the payload.
 */

DataBlock *
datablock_allocate (DataBlock *datablock)
{

  /* If a negative datablock size is requested, then obviously we have an
   * error.  Return null and increment a count so we can check it with
   * a debugger.
   */

  if (datablock->header.data.length < 0)
    {
      datablock_allocate_negative++;
      return (NULL);
    }
  if (datablock->header.data.length > datablocks_max_bytes)
    return (NULL);

  if ((datablock->header.data.length + datablocks_total_bytes) > 
      datablocks_max_bytes)
    purge_datablocks (0);
  
  if ((datablock->header.data.length + datablocks_total_bytes) > 
      datablocks_max_bytes)
    purge_datablocks (1);


  if ((datablock->header.data.length + datablocks_total_bytes) > 
      datablocks_max_bytes)
    return (NULL);

  if (datablock->header.data.length > 0)
    {
      datablock->data = (char *) malloc (datablock->header.data.length);
      if (datablock->data == NULL)
	return (NULL);
    }

  datablock->length = datablock->header.data.length;

  datablocks_total_bytes += datablock->length;

  return (datablock);
}

/* Given an existing datablock, assign previously allocated memory to it */
DataBlock *
datablock_set_data (DataBlock *datablock, void *data, int length)
{
  if (length < 0)
    {
      datablock_set_negative++;
      return (NULL);
    }

  if ((length + datablocks_total_bytes ) > datablocks_max_bytes)
    return (NULL);

  if (datablock->data != NULL)
    {
      if (datablock->length > 0 && datablock->length < datablocks_total_bytes)
	datablocks_total_bytes = datablocks_total_bytes - datablock->length;
      xfree (datablock->data);
    }
  datablock->header.data.length = length;
  datablock->data = data;
  datablock->length = length;
  datablocks_total_bytes += length;

  return (datablock);
}

void
datablock_free (DataBlock *datablock)
{
  if (datablock == NULL)
    return;

  if (datablock->refcount <= 0)
    {
      if (debug)
	fprintf (stderr, "datablock freed twice!!!\n");
      dump_core ();
    }

  datablock->refcount--;
  if (datablock->refcount <= 0)
    {
      xfree (datablock->data);
      if (datablock->header.data.length >= 0)
	datablocks_total_bytes = 
	  datablocks_total_bytes - datablock->header.data.length;
      else
	{
	  datablock_free_negative++;
	}
      if (datablocks_total_bytes < 0)
	{
	  abort ();
	}
      xfree (datablock);
    }
}

datablock_reference (DataBlock *datablock)
{
  datablock->refcount++;
}

/*
 * Read a data block
 */

/* Do a non-blocking read of a datablock from a file descriptor 
 * The state parameter keeps track of what has been read so far.
 *
 * In this case it is just a pointer to an integer that says how many
 * bytes we have read.
 */

int
datablock_read (DataBlock **datablock, int fd, int *bytes_read)
{
  Message message;
  SiteInfo *site;
  int total;
  int n;
  int first_read = 1;
  /* Get message header */
  if (*bytes_read < sizeof (message))
    {
      Message header;

      /* Hmmm we are not really reentrant at this point, we really need to
       * grab the header on the first try.
       */
      /* Make sure things are in network byte order */
      n = read (fd, 
		&((char *)&header)[*bytes_read], 
		sizeof (header) - *bytes_read);
      
      first_read = 0;
      memcpy (&((char *)&message)[*bytes_read], 
	      &((char *)&header)[*bytes_read],
	      n);

      /* We bail out if we get a zero bytes since we only get here after
       * a select() which means that something is ready.
       */
      if (n == 0 )
	return (-1);

      if (n == -1 && 
	  (errno == EWOULDBLOCK || errno == EAGAIN || errno == EINPROGRESS))
	return (0);

      if (n == -1)
	{
	  if (debug)
	    fprintf (stderr, "errno = %d\n", errno);
	  if (debug)
	    perror ("datablock_read.read() == -1");
	  return (-1);
	}

      *bytes_read += n;
      if (*bytes_read < sizeof (message))
	return (-1);

      message.data.length = ntohl (header.data.length);
      message.type = ntohl (header.type);

      if (message.type != TYPE_BANDWIDTH && logged_message (&message.id))
	{
	  siteinfo_ack (&message.id);
	  return (-1);
	}

      if (message.data.length < 0)
	{
	  if (debug)
	    printf ("reading datablock length < 0. error..\n");
	  return (-1);
	}

      if (debug)
	printf ("Reading datablock of %d bytes\n", message.data.length);

#ifdef INSIDE_CONTROL
      {
	extern int sw_data_receive;
        if (sw_data_receive) return (-1);
      }
#endif /* INSIDE_CONTROL */

      *datablock = 
	datablock_new (TYPE_UNKNOWN, message.id.site, NULL, message.data.length);

      if (*datablock == NULL)
	return (-1);

      (*datablock)->header = message;

      /* Determine the site */
      site = site_by_id (&(*datablock)->header.sender);

      if (site == NULL)
	send_unknown_reply (&(*datablock)->header.sender);
      else
	siteinfo_up (site);
    }

  /* loop and read the data */
  total = (*bytes_read - sizeof (message));
  while (*bytes_read < ((*datablock)->length + sizeof (message)))
    {
      int bytes;
      if (((*datablock)->length - total) < CHUNK)
	bytes = (*datablock)->length - total;
      else
	bytes = CHUNK;

      n = read (fd, &(*datablock)->data[total], bytes);

      if (n != -1)
	{
	  *bytes_read += n;
	  total += n;
	}

      if (n == 0 /*&& errno != EAGAIN*/)
	{
	  if (first_read && errno != EINPROGRESS)
	    {
	      if (debug)
		{
		  printf ("errno = %d\n", errno);
		  perror ("datablock_read.read == 0");
		}
	      return (-1);
	    }
	  return (0);
	}
      if (n == -1 && (errno == EWOULDBLOCK 
		      || errno == EAGAIN 
		      || errno == EINPROGRESS))
	return (0);

      if (n == -1)
	{
	  if (debug)
	    fprintf (stderr, "errno = %d\n", errno);
	  if (debug)
	    perror ("datablock_read.read == -1");
	  return (-1);
	}
      first_read = 0;
    }

  return (*bytes_read);
}

/* Write a datablock to a non-block socket.  We keep track of how many bytes 
 * we have written.
 */

int
datablock_write (DataBlock *datablock, int fd, int *bytes_written,
		 int just_data)
{
  int length;
  int total;
  int n;
  static Message header;

  if (!just_data && *bytes_written < sizeof (Message))
    {

      /* Make sure things are in network byte order */
      memcpy (&header, &datablock->header, sizeof (header));
      header.data.length = htonl (datablock->length);
      header.type = htonl (datablock->header.type);
      memcpy  (&header.sender, &whoami->id, sizeof (header.sender));

      n = write (fd, 
		 &((char *)&header)[*bytes_written], 
		 sizeof (header) - *bytes_written);

      if ((n == 0 || n == -1) && (errno == EWOULDBLOCK || errno == EAGAIN))
	return (0);

      if (n == -1)
	return (-1);
      *bytes_written += n;
      if (*bytes_written < n)
	return (0);
    }

  /* Write the datablock in 4 K chunks */
  if (just_data)
    total = *bytes_written;
  else
    total = *bytes_written - sizeof (Message);

  while (total < datablock->length)
    {
      if ((datablock->length - total) < CHUNK)
	length = datablock->length - total;
      else
	length = CHUNK;

      n = write (fd, &datablock->data[total], length);
      if (n == -1 && (errno != EWOULDBLOCK && errno != EAGAIN))
	return (-1);
	  
      if (n <= 0)
	return (0);
      *bytes_written += n;
      total += n;
    }

  return (*bytes_written);
}


int
datablock_compare (DataBlock *a, DataBlock *b)
{
  return (messageid_compare (&a->header.id, &b->header.id));
}
