/*
 * config.c - runtime configuration
 *
 * $Log: config.c,v $
 * Revision 0.14  1994/05/17  13:38:15  reinpost
 * nothing of importance added
 *
 * Revision 0.14  1994/05/17  13:38:15  reinpost
 * nothing of importance added
 *
 *
 * Revision 0.10  1994/03/11  13:36:26  reinpost
 * some renamings in config variables
 * a reorganization of tasks and slightly different interface to cache.c
 * some small modifications in variable adjustments
 *
 * Revision 0.9  1994/03/02  21:23:12  reinpost
 * aux routines moved to util.c
 *
 * some new variables
 *
 * Revision 0.8  1994/02/25  20:20:04  reinpost
 * using dynamic string allocation now
 * improved mainiatiability with auxiliary routimes
 * added a simple time specification syntax
 * which is used to make Timeout and both expiration times configurable
 * modified interface to outside world because these routines can't
 * write to logfiles yet
 * proxy support added
 * and one or two small things i guess
 *
 * Revision 0.6  1994/02/17  21:24:47  reinpost
 * *** empty log message ***
 *
 * Revision 0.5  1994/02/17  10:20:29  reinpost
 * global vars are now malloc'ed
 * maintainability was improved by introducing aux routines
 *
 * Revision 0.4  1994/02/10  10:57:38  reinpost
 * no longer abort if the configuration file could not be accessed ...
 * some changes in the layout of error messages
 * a new read_parameters() routine
 *
 * Revision 0.3  1994/02/01  19:41:31  reinpost
 * initial version - code taken from cache.c and httpd 1.0a4 originally
 *
 */

static char rcsid[] =
  "$Id: config.c,v 0.14 1994/05/17 13:38:15 reinpost Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <pwd.h>

/* for remote hostname lookup */
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include "system.h"

#include "constants.h"
#include "util.h"
#include "log.h"
#include "error.h"

#include "config.h"

/* *--- CONFIGURATION VARIABLES --- */

/* the global variables to be set */

char *cache_version = (char *)NULL;

char *cache_root = (char *)NULL;
char *cache_prefix = (char *)NULL;
char *mime_types_confname = (char *)NULL; 
char *expire_confname = (char *)NULL; 
char *transfer_logfile = (char *)NULL;
char *error_logfile = (char *)NULL;
char *db_dir = (char *)NULL;
char *default_mime_type = (char *)NULL;
char *server_maintainer = (char *)NULL;
int cache_queries;
int translate_escaped;
int timeout;
int a_expires;
int m_expires;
int cache_threshold;

/* auxiliary routines */

static int set_string_variable(char **var,char *req_key,char *key,char *value)
/* assume the variable is NULL if not yet specified */
/* 1 if the variable was set, 0 otherwise */
{
  if (!(strcasecmp(req_key,key)))
  {
    if (*var)
    {
      /* variable was set before */
      add_warning("duplicate specification for configuration variable:",
	key,-1);
    }
    else
    {
      *var = strdup(value);
      return(1);
    }
  }
  return(0);
}

static int confstrtobool(char *value,int default_value)
{
  int result = stringtobool(value);
  if (result < 0)
  {
    if (value && *value)
      add_warning("incorrect boolean configuration value (not 0 or 1):",
	value, -1);
    result = default_value;
  }
  free(value);
  return(result);
}

static int confstrtonum(char *value,int default_value)
{
  int result = stringtoint(value);

  if (result == -1)  /* 0 and -1 are always incorrect results! */
  {
    if (value && *value)
      add_warning("incorrect numeral configuration value:",
	value, -1);
    result = default_value;
  }
  free(value);
  return(result);
}

static int confstrtotime(char *value,int default_value)
{
  int result = stringtotime(value);
  if (result < 0)
  {
    if (value && *value)
      add_warning("incorrect time specification for configuration value:",
	value, -1);
    result = default_value;
  }
  free(value);
  return(result);
}

static int conf_found;

int config_file_found()
{
  return(conf_found);
}

/* tmp strings, will be turned into ints */
static char *tmp_cache_queries = (char *)NULL;
static char *tmp_translate_escaped = (char *)NULL;
static char *tmp_timeout = (char *)NULL;
static char *tmp_a_exp = (char *)NULL;
static char *tmp_m_exp = (char *)NULL;
static char *tmp_threshold = (char *)NULL;

static char *config_table[] =
{
  /* this is safe, assuming sizeof(char *) = sizeof(char **) */
  (char *)&cache_root,		"CacheRoot",		DEFAULT_CACHE_ROOT,
  (char *)&cache_prefix,	"CachePrefix",		DEFAULT_CACHE_PREFIX,
  (char *)&db_dir,		"CacheDbmDir",		DEFAULT_DB_DIR,
  (char *)&transfer_logfile,	"CacheLog",	       DEFAULT_TRANSFER_LOGFILE,
  (char *)&error_logfile,	"CacheError",		DEFAULT_ERROR_LOGFILE,
  (char *)&mime_types_confname,	"CacheMimeConf",	DEFAULT_MIME_CONF,
  (char *)&expire_confname,	"CacheExpireConf",	DEFAULT_EXPIRE_CONF,
  (char *)&default_mime_type,	"CacheDefaultMimeType",	DEFAULT_MIME_TYPE,
  (char *)&server_maintainer,	"CacheServerMaintainer", (char *)NULL,
  (char *)&cache_version,	"CacheVersion",		(char *)NULL,
  (char *)&tmp_cache_queries,	"CacheQueries",		(char *)NULL,
  (char *)&tmp_translate_escaped, "CacheTranslateEscaped", (char *)NULL,
  (char *)&tmp_timeout,		"CacheTimeout",		(char *)NULL,
  (char *)&tmp_a_exp,		"CacheAccessExpires",	(char *)NULL,
  (char *)&tmp_m_exp,		"CacheModificationExpires", (char *)NULL,
  (char *)&tmp_threshold,	"CacheOnlyAfter",	(char *)NULL,

  (char *)NULL  /* terminator */
};

static int try_set_string_variable(char *key, char *value)
{
  int i;
  for (i = 0; config_table[i]; i += 3)
  {
    if (set_string_variable(
	  (char **)config_table[i],config_table[i+1],key,value))
      return(1);
  }
  return(0);
}

static void set_to_default_if_n(char **var, char *default_value)
{
  if (!*var && default_value)  /* var is not yet set, default is nontrivial */
  {
    *var = strdup(default_value);
  }
}

static void set_remaining_to_defaults()
{
  int i;

  for (i = 0; config_table[i]; i += 3)
  {
    set_to_default_if_n((char **)config_table[i],config_table[i+2]);
  }
}
    
/* the main routine */
 
void read_config()
/* reads the configuration parameters */
/* logfiles are not open yet so we can't write messages on error */
/* using the add_warning() instead */
{
  char l[MAX_STRING_LEN+1];
  char key[MAX_STRING_LEN+1];
  char value[MAX_STRING_LEN+1];
  FILE *f;
  struct stat finfo;

  /* open the configuration file; on failure, forget about the rest */

  if ((stat(CACHE_CONF,&finfo) == -1) || !(f = fopen(CACHE_CONF,"r")))
  {
    conf_found = 0;
    add_warning("configuration file not found:",CACHE_CONF,-1);
  }
  else
  {
    /* actually read it */

    conf_found = 1;
  
    /* test if the config table has a proper format */
    /* this is superfluous on any machine I can imagine */
    if (sizeof(char *) != sizeof(char **))
    {
      add_warning("Lagoon bug:","improper format for configuration table",-2);
      return;
    }
  
    /* try to find the configuration attributes in the config file */
  
    while (!(cfg_getline(l,256,f)))
    {
      int valid_line;  /* did this line actually cause a variable to be set? */
  
      skipspace(l);
      cfg_getword(key,l);
      skipspace(l);
      cfg_getword(value,l);
    
      /* set all configuration variable values, as strings */
    
      valid_line = *key && try_set_string_variable(key,value);
    }
    fclose(f);
  }
  /* every variable must have a value now, but (char *)NULL for unspecified */

  set_remaining_to_defaults();

  cache_queries =
     confstrtobool(tmp_cache_queries,DEFAULT_CACHE_QUERIES);
  translate_escaped =
    confstrtobool(tmp_translate_escaped,DEFAULT_TRANSLATE_ESCAPED);
  timeout =
    confstrtotime(tmp_timeout,DEFAULT_TIMEOUT);
  a_expires =
    confstrtotime(tmp_a_exp,DEFAULT_A_EXP);
  m_expires =
    confstrtotime(tmp_m_exp,DEFAULT_M_EXP);
  cache_threshold =
    confstrtotime(tmp_threshold,DEFAULT_THRESHOLD);

  /* every config variable except cache_version and server_maintainer */
  /* is supposed to have a non-NULL, dynamically allocated value now */
  /* but it is not necessarily the final value */
}

/* --- ENVIRONMENT PARAMETERS --- */

/* parameters directly taken from environment variables */
char *env_server_software;
char *env_server_name;
char *env_gateway_interface;
char *env_server_protocol;
char *env_server_port;
char *env_request_method;
char *env_http_accept;
char *env_path_info;
char *env_path_translated;
char *env_script_name;
char *env_query_string;
char *env_remote_host;
char *env_remote_addr;
char *env_remote_user;
char *env_auth_type;
char *env_content_type;
char *env_content_length;

/* derived parameters */
int env_assbackwards = 0;
char *env_url;		/* with query and anchor stripped off */
char *env_url_with_query;	/* with anchor stripped off */
char *env_url_anchor;	/* #name part to refer to a specific place in a doc */
int env_refresh = 0;
int env_its_proxy = 0;
 
void read_parameters()
/* read the script parameters from environment variables */
/* using add_warning() to report errors */
{
  /* read parameter values from environment variables */
  /* be polite about env. vars missing altogether - see my_getenv in util.h */
  /* e.g. we will regard the empty string as representing no query at all */
  /* as a consequence, a trailing ? by itself is omitted from the URL   */
  /* if the difference between "" and NULL in env vars is meaningful,   */
  /* this will have to be changed in later versions                     */

  env_server_software = strdup(my_getenv("SERVER_SOFTWARE"));
  env_server_name = strdup(my_getenv("SERVER_NAME"));
  env_gateway_interface = strdup(my_getenv("GATEWAY_INTERFACE"));
  env_server_protocol = strdup(my_getenv("SERVER_PROTOCOL"));
  env_server_port = strdup(my_getenv("SERVER_PORT"));
  env_request_method = strdup(my_getenv("REQUEST_METHOD"));
  env_http_accept = strdup(my_getenv("HTTP_ACCEPT"));
  env_path_info = strdup(my_getenv("PATH_INFO"));
  env_path_translated = strdup(my_getenv("PATH_TRANSLATED"));
  env_script_name = strdup(my_getenv("SCRIPT_NAME"));
  env_query_string = strdup(my_getenv("QUERY_STRING"));
  env_remote_host = strdup(my_getenv("REMOTE_HOST"));
  env_remote_addr = strdup(my_getenv("REMOTE_ADDR"));
  env_remote_user = strdup(my_getenv("REMOTE_USER"));
  env_auth_type = strdup(my_getenv("AUTH_TYPE"));
  env_content_type = strdup(my_getenv("CONTENT_TYPE"));
  env_content_length = strdup(my_getenv("CONTENT_LENGTH"));
}

static void derive_implied_parameters()
{
 /* derive the values of derived parameters:
  * env_assbackwards, env_url, env_url_with_query, env_url_anchor,
  * env_its_proxy, env_refresh
  */

  env_assbackwards = strcmp(env_server_protocol,"HTTP/1.0");
  /* this should be 1 iff the request asked for HTTP 0.X instead of 1.0 */

  if (env_path_info[0] == '/')
    env_url = strdup(&env_path_info[1]);  /* skip leading '/' */
  else
    env_url = strdup(env_path_info);
  /* env_url will be modified below */

  /* see if we have a proxy request */
  if (!strncmp(env_url,PROXY_PREFIX,strlen(PROXY_PREFIX)))
  {
    /* remember: request for untranslated links; and remove prefix from url */
    char *old = env_url;
    env_url = strdup(&env_url[strlen(PROXY_PREFIX)]);
    free(old);

    env_its_proxy = 1;
  }
  else
  {
    env_its_proxy = 0;
  }

  /* see if we have a forced refresh */
  if (!strncmp(env_url,REFRESH_PREFIX,strlen(REFRESH_PREFIX)))
  {
    /* remember: request for a forced refresh; and remove prefix from url */
    char *old = env_url;
    env_url = strdup(&env_url[strlen(REFRESH_PREFIX)]);
    free(old);

    env_refresh = 1;
  }
  else
  {
    env_refresh = 0;
  }

  /*
   * extract the env_url_anchor
   *
   * the proper syntax, judging from
   *   http://info.cern.ch/hypertext/WWW/Addressing/URL/5_BNF.html
   * is: env_url?query#anchor, not env_url#anchor?query
   * which is in line with the expected semantics (however, I haven't
   * found any specifics on them yet)
   * so in case there is a query_string, take the env_url_anchor off it,
   * otherwise, take it off the url
   *
   * NOTE: ALL THIS URL PARSING STUFF SHOULD BE MOVED TO html.c SOMETIME
   *
   * NOTE2: this parse is very crude, we don't take special chars into account
   */

  {
    char *s = (*env_query_string ? env_query_string : env_url);
    int on_hash = rind(s,'#');
    if (on_hash == -1)
    {
      /* no #anchor was found */
      env_url_anchor = strdup("");
    }
    else
    {
      s[on_hash] = '\0';
      env_url_anchor = s[on_hash+1];
    }
  }

  /* repair URLs of the form http://stringwithoutslash by adding the slash */

  if (!strncasecmp(env_url,"http://",7))
  {
    char *s;
    /* move s to next slash if present */
    for (s = &env_url[7]; *s && *s != '/'; ++s);
    if (!*s)  /* there is none; add one */
    {
      s = env_url;
      env_url = (char *)malloc(strlen(env_url)+2);
      strcpy(env_url,s);
      add_warning("missing slash added to URL","",0);
      strcat(env_url,"/");
      free(s);
    }
  }

  /* make an url with appended query string */
  if (*env_query_string)
  {
    env_url_with_query = (char *)malloc(strlen(env_url)+strlen(env_query_string)+2);
    strcpy(env_url_with_query,env_url);
    strcat(env_url_with_query,"?");
    strcat(env_url_with_query,env_query_string);
  }
  else
  {
    env_url_with_query = strdup(env_url);
  }
}

static void fix_cache_version()
{
  if (!cache_version)
  {
    char scratch[MAX_STRING_LEN+1];
    strcpy(scratch,RCS_VERSION);
    if (strlen(scratch) < 16)
    {
      add_warning("incorrect RCS version indication:",scratch,-1);
      cache_version = strdup("0.0");
    }
    else
    {
      /* this is of the format '$Revision: 0.14 $' */
      scratch[strlen(scratch)-2] = '\0';
      cache_version = strdup(&scratch[11]);
    }
  }
}

static void fix_cache_prefix()
/*
 * add a trailing slash if not there;
 * then complete the cache_prefix parameter by prefixing http://host:port/
 * unless it starts with "http:" (no full parse of relative URL is done!)
 */
{
  if (cache_prefix[strlen(cache_prefix)-1] != '/')
  {
    char *bare_prefix = cache_prefix;
    cache_prefix = (char *)malloc(strlen(bare_prefix+2));
    strcpy(cache_prefix,bare_prefix);
    strcat(cache_prefix,"/");
    free(bare_prefix);
  }

  if (strncmp(cache_prefix,"http:",5))
  {
    char scratch[HUGE_STRING_LEN+1];

    sprintf(scratch,"http://%s:%s/%s",env_server_name,env_server_port,
        (*cache_prefix == '/' ? &cache_prefix[1] : cache_prefix));
    free(cache_prefix);
    cache_prefix = strdup(scratch);
  }
}

static void fix_server_name()
/* doesn't always work yet */
{
  /* repair servername: make it supply the full name */
  /* try to guess the server's domain, because we don't really know */

  char my_hostname[MAX_STRING_LEN+1];
  char my_domainname[MAX_STRING_LEN+1];
  add_warning("server name as provided:",env_server_name,0);
  if ((rind(env_server_name,'.') == -1)
    /* $SERVER_NAME does not contain domain information */
    && !gethostname(my_hostname,MAX_STRING_LEN)
    /* my own hostname could be found */
    && !getdomainname(my_domainname,MAX_STRING_LEN)
    /* and my own domain name could be found */
    && my_hostname && *my_hostname && my_domainname && *my_domainname
    /* just to make sure */
    && !strcmp(my_hostname,env_server_name))
    /* and the hostname is equal to the server's hostname */
  {
    env_server_name = (char *)malloc
      (strlen(my_hostname)+strlen(my_domainname)+2);
    sprintf(env_server_name,"%s.%s",my_hostname,my_domainname);
  }
  /* otherwise, try to live with it */
  /* in that case, the cache won't work when used from a different */
  /* Internet domain */
  add_warning("server name as completed:",env_server_name,0);
}

static void fix_server_maintainer()
/* assumes env_server_name has already been fixed */
{
  if (!server_maintainer || !*server_maintainer)
  {
    /* no server maintainer was supplied; try to use my own */
    char *my_userid;

    /* use my own userid */
    my_userid = getpwuid(getuid())->pw_name;
    /* or invent one */
    if (!my_userid || !*my_userid)
    {
      my_userid = "nobody";
      add_warning
	("username could not be found to derive a suitable",
	 "CacheServerMaintainer value; trying 'nobody'",-1);
    }

    /* now turn this into an email address */
    if (env_server_name && (rind(env_server_name,'.') != -1))
    /* there is a server name with domain info */
    {
      server_maintainer = (char *)malloc
	(strlen(my_userid)+strlen(env_server_name)+2);
      sprintf(server_maintainer,"%s@%s",my_userid,env_server_name);
    }
    else
    {
      add_warning
	("no email address for the server maintainer","could be derived",-1);
      server_maintainer = strdup("<server maintainer unknown>");
    }
  }
}

void finish_global_variables()
/* uses add_warning() to report errors */
{
  fix_cache_version();
  fix_cache_prefix();
  fix_server_name();
  fix_server_maintainer();
  derive_implied_parameters();
}
