########################## -*- Mode: Perl -*- ##########################
##
## File             : BIBTEX.pm
##
## Description      : BibTeX converter for tagged format
##
#
# Copyright (C) 1995, 1996 Ulrich Pfeifer, Norbert Goevert
#
# This file is part of SFgate.
#
# SFgate is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# SFgate is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SFgate; see the file COPYING.  If not, write to
# the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
#
##
## Author           : Norbert Goevert
## Created On       : Wed Mar 27 11:10:08 1996
##
## Last Modified By : Norbert Goevert
## Last Modified On : Wed Feb  5 17:52:13 1997
##
## $State: Exp $
##
## $Id: BIBTEX.pm,v 5.1.1.2 1997/02/17 12:55:12 goevert Exp goevert $
##
## $Log: BIBTEX.pm,v $
## Revision 5.1.1.2  1997/02/17 12:55:12  goevert
## patch10: missing comma added
##
## Revision 5.1.1.1  1996/12/23 12:48:41  goevert
## patch6: introduced new order orl
##
## Revision 5.1  1996/11/05 16:54:47  goevert
## *** empty log message ***
##
## Revision 5.0.1.2  1996/05/23 17:06:53  goevert
## patch13: removed no strict 'vars';
##
## Revision 5.0.1.1  1996/05/13 11:26:53  goevert
## patch1:
##
########################################################################


package SFgate::Converter::BIBTEX;


use strict;


## #####################################################################
## convert
## #####################################################################
## interface for SFgate to convert one document
##
## (string) $text:     document to convert
## (string) $headline: headline of document to convert
##
## By default every document is printed with its corresponding wais
## headline. If you want to derive another headline just change
## the $headline variable according to your needs.
##
## HTML special characters (&, <, >) should be encoded in text not
## being HTML code. To do this you can use the function
## &::encode_entities, e.g. $ntext = &::encode_entities($ntext)
## encodes the special chars in $ntext.
##
## returns:
## - string: the converted document ($ntext)
## - string: headline of converted document ($headline)
##
sub convert
{
    my($text, $headline) = @_;
    ## local variable
    my $order = &get_orderurl($text);
    $order = '<A HREF="http://bib.informatik.uni-dortmund.de/journals/htbin/orderform?' . $order . '">';
    
    $text =~ s/^\f\n//;
    $text = &::encode_entities($text);
    $text = &scan($text);
    $text =~ s!\n  title =     (.)([^\}]*)!\n  title =     $1$order$2</A>!;
    $text = "<PRE>\n$text</PRE>\n";
    
    return($text, $headline);
}


## #####################################################################
## helper subroutines
## #####################################################################

sub get_orderurl
{
    local($_) = shift;
    my($urlprefix) = @_;

    my(%parms, $tmp);
    

    ($parms{CK}) = /^CK: (.*)/m;
    ($parms{AU}) = /^AU: ((.*\n )*.*)/m;
    $parms{AU}   =~ s/\s+/ /g;
    ($parms{TI}) = /\nTI: ((.*\n )*.*)/m;
    $parms{TI}   =~ s/\s+/ /g;
    $parms{TI}   =~ s/\.$//;
    ($parms{JT}) = /^JT: (.*)/m;
    ($parms{PY}) = /^PY: (.*)/m;
    ($parms{VO}) = /^VO: (.*)/m;
    if (/^NO: (.*)\nNO: (.*)/m) {
        $parms{NO} = "$1/$2";
    }
    else {
        ($parms{NO}) = /^NO: (.*)/m;
    }
    ($parms{PP}) = /^PP: (.*)/m;
    ($parms{SC}) = /^SC: (.*)/m;
    ($parms{DT}) = /^DT: (.*)/m;
    ($parms{ED}) = /^ED: (.*)/m;

    my $request;
    foreach (keys %parms) {
        $request .= "&$_=" . &::encode($parms{$_});
    }
    $request =~ s/^&//;
    $request;
}


sub scan
{
    local($_) = @_;
    ## local variables
    my($ck, $au, $ti, $jt, $py, $vo, $no, $pp, $ab, $pa, $sc, $ed);

    # analyse
    
    s/\n    / /g;

    ($ck) = /^CK: (.*)/m;
    # forced fields
    ($au) = /^AU: (.*)/m; 
    ($ti) = /^TI: (.*)/m; 
    ($jt) = /^JT: (.*)/m;
    ($py) = /^PY: (.*)/m; 
    # optional fields
    ($vo) = /^VO: (.*)/m; 
    ($no) = /^NO: (.*)/m; 
    ($pp) = /^PP: (.*)/m; 
    # obsolete fields
    ($ab) = /^AB: (.*)/m; 
    ($pa) = /^PA: (.*)/m; 
    ($sc) = /^SC: (.*)/m; 
    ($ed) = /^ED: (.*)/m; 
    
    $au = join(' and ', split('; ', $au));

    # synthesize

    $_  = "\@Article{$ck,\n";
    $_ .= "  author =   " . &lines("{$au},") . "\n" if $au;
    $_ .= "  title =    " . &lines("{$ti},") . "\n";
    $_ .= "  journal =  " . &lines("{$jt},") . "\n";
    $_ .= "  year =      $py,\n";
    $_ .= "  volume  =   $vo,\n";
    $_ .= "  number =    $no,\n";
    $_ .= "  pages =     $pp,\n" if $pp;
    $_ .= "  seperator = {-------------------------------------------------------},\n";
    $_ .= "  abstract = " . &lines("{$ab},") . "\n" if $ab;
    $_ .= "  publisher = {$pa},\n" if $pa;
    $_ .= "  shortcut =  {$sc},\n";
    $_ .= "  entrydate = $ed\n";
    $_ .= "}\n";
    
    return ($_);
}

    
sub lines
{
    my($value) = @_;
    ## local variables
    local($_);
    my($col) = 13;
    ## return value
    my($result);
    
    for (split(/\s+/, $value)) {
        if ($col+length($_)+1 > 72) {
            $result .= "\n             ";
            $col = 13;
        }
        $result .= " $_";
        $col  += 1 + length($_);
    }

    return($result);
}


1;
