/*
 *	(c) Copyright 1990, Kim Fabricius Storm.  All rights reserved.
 *
 *	Database access and update
 */

#include "config.h"
#include "db.h"
#include <errno.h>
#include <ctype.h>
#include <string.h>

import char
    *master_directory, *db_directory, *db_data_directory, *news_directory;
import int db_data_subdirs;

export master_header master;
#ifdef MALLOC_64K_LIMITATION
export group_header **active_groups = NULL;
#else
export group_header *active_groups = NULL;
#endif
export group_header **sorted_groups = NULL;

export int  reread_groups_file = 0;  /* nnmaster -G */

export int  check_group_access = 0;

export data_header db_hdr;
export data_dynamic_data db_data;

export int32 db_read_counter = 0; 	/* articles read by db_read_art */
export int32 db_write_counter = 0;	/* articles written by db_write_art */

/*
 * Init access to a group
 */

export group_header *current_group = NULL;

export char group_path_name[FILENAME];
export char *group_file_name = NULL;

static char *group_position = NULL;
static article_number current_digest_article = 0;

static readactfile();
static readtimfile();

init_group(gh)
register group_header *gh;
{
    register char *p, *q;

    current_digest_article = 0;

    if (gh == NULL) return 0;
/*    if (gh->master_flag & M_IGNORE_GROUP) return 0;	/* OBS */
    if (gh == current_group) return 1;

    current_group = gh;

    if (gh->group_flag & G_FOLDER) {
	group_position = NULL;
	group_file_name = NULL;
	strcpy(group_path_name, gh->archive_file);
	return 1;
    }

#ifdef NNTP
    if (use_nntp && nntp_set_group(gh) < 0)
	return 0;
#endif /* NNTP */

    if (group_position == NULL)
	if (who_am_i == I_AM_MASTER || who_am_i == I_AM_EXPIRE)
	    group_position = group_path_name;
	else {
	    strcpy(group_path_name, news_directory);
	    group_position = group_path_name + strlen(group_path_name);
	    *group_position++ = '/';
	}

    for (p = group_position, q = gh->group_name; *q; q++)
	*p++ = (*q == '.') ? '/' : *q;

    if (who_am_i == I_AM_MASTER) {

	/* The master will chdir to the group's directory to get */
	/* better performance (can use relative path names). */

	*p++ = NUL;

	if (!use_nntp) {
	    if (chdir(news_directory) < 0)
		sys_error(news_directory);

	    if (chdir(group_path_name) < 0)
		return 0;
	}
	group_file_name = group_path_name;
	return 1;
    }

    /* client */
    if (gh->master_flag & M_NO_DIRECTORY) return 0;

    if (check_group_access && !use_nntp) {
	*p = NUL;
	if (file_exist(group_path_name, "dxr") == 0) return 0;
    }

    *p++ = '/';
    *p = NUL;
    group_file_name = p;
    return 1;
}

static char *mk_archive_file(gh, cp, logerr)
register group_header *gh;
register char *cp;
int logerr;
{
    char *name;

    while (*cp && (*cp == '>' || isspace(*cp))) cp++;

    name = cp;
    while (*cp && !isspace(*cp)) cp++;
    if (*cp) *cp++ = NUL;
    if (*name) {
	gh->archive_file = copy_str(name);

	if (*name == '/')
	    gh->master_flag |= M_AUTO_ARCHIVE;
	else {
	    gh->master_flag &= ~M_AUTO_ARCHIVE;
	    if (who_am_i == I_AM_MASTER)
		if (logerr)
		    log_entry('E', "GROUPS %s >%s: Full path required",
			      gh->group_name, name);
		else
		    printf("Error in GROUPS: %s >%s: Full path required\n",
			      gh->group_name, name);
	}
    }

    return cp;
}


/* db_parse_group(gh, trust_master) */

/*
 *	Init the groups data from active file.
 */

open_master(mode)
int mode;
{
    register group_header *gh;
    int trust_master;

    freeobj(sorted_groups);
    freeobj(active_groups);
    active_groups = NULL;
    sorted_groups = NULL;

    readactfile();		/* Read in the active file - count groups */
    readtimfile();		/* Read the newsgroup creation time file */

    db_expand_master();		/* uses count from readact() call! */

    Loop_Groups_Header(gh) {
	/* db_read_group opens a file per call; use db_init_group instead */
	db_init_group(gh, l_g_index);
    }

    sort_groups();
}

db_expand_master()
{
    master.free_groups = 20;

    active_groups = resizeobj(active_groups, group_header,
			      master.number_of_groups + master.free_groups);
    clearobj(active_groups + master.number_of_groups, group_header,
	     master.free_groups);
    sorted_groups = resizeobj(sorted_groups, group_header *,
			      master.number_of_groups + master.free_groups);
}

update_group(gh)
group_header *gh;
{
    group_number numg;

#if undef
    numg = master.number_of_groups;
    /* db_read_master(); $$$$ */
    master.number_of_groups = numg;
    if (master.db_lock[0]) return -3;
#endif

    db_read_group(gh);

    if (gh->master_flag & M_IGNORE_GROUP) return 0;
    if (gh->master_flag & M_BLOCKED) return -1;

    return 1;
}


static sort_gh(g1, g2)
group_header **g1, **g2;
{
    return strcmp((*g1)->group_name, (*g2)->group_name);
}


sort_groups()
{
    register group_header *gh;

    Loop_Groups_Header(gh)
	sorted_groups[l_g_index] = gh;

    quicksort(sorted_groups, master.number_of_groups, group_header *, sort_gh);

    s_g_first = 0;
    Loop_Groups_Sorted(gh)
	if (gh->group_name[0] != NUL) {
	    s_g_first = l_g_index;
	    break;
	}
}


group_header *lookup_no_alias(name)
char *name;
{
    register i, j, k, t;

    i = s_g_first; j = master.number_of_groups - 1;

    while (i <= j) {
	k = (i + j) / 2;

	if ( (t=strcmp(name, sorted_groups[k]->group_name)) > 0)
	    i = k+1;
	else
	if (t < 0)
	    j = k-1;
	else
	    return sorted_groups[k];
    }

    return NULL;
}

group_header *lookup(name)
char *name;
{
    register group_header *gh;
    group_header *gh_na;
    register int32 n, x;

    gh = lookup_no_alias(name);
    if (gh == NULL || (gh->master_flag & M_ALIASED) == 0) return gh;

    gh_na = gh;
    x = 16;
    do {
	if (--x == 0) {
	    log_entry('R', "Possible alias loop: %s", name);
	    return gh_na;
	}
	n = (int32)gh->data_write_offset;
	/* if alias info is unreliable, return original group
	   which will be ignored anyway */
	if (n < 0 || n >= master.number_of_groups) {
	    log_entry('R', "Bad aliasing of %s -> %d", gh->group_name, n);
	    return gh_na;
	}
	gh = ACTIVE_GROUP(n);
    } while (gh->master_flag & M_ALIASED);

    return gh;
}

art_collected(gh, art_num)
group_header *gh;
article_number art_num;
{
    return gh->first_db_article <= art_num && gh->last_db_article >= art_num;
}

#ifdef NETWORK_DATABASE

#define MASTER_FIELDS	5	/* + DB_LOCK_MESSAGE bytes */
#define	GROUP_FIELDS	9
#define	ARTICLE_FIELDS	10


typedef int32 net_long;


#ifdef NETWORK_BYTE_ORDER

#define net_to_host(buf, n)
#define host_to_net(buf, n)

#else

static net_to_host(buf, lgt)
register net_long *buf;
int lgt;
{
    while (--lgt >= 0) {
	*buf = ntohl(*buf);
	buf++;
    }
}

static host_to_net(buf, lgt)
register net_long *buf;
int lgt;
{
    while (--lgt >= 0) {
	*buf = htonl(*buf);
	buf++;
    }
}
#endif /* not NETWORK_BYTE_ORDER */
#endif /* NETWORK_DATABASE */

#define NWDB_MAGIC 0x00190000	/* NN#n <-> NW#n */

#include <hash.h>
#include <hdbm.h>
#include <newsoverview.h>

static struct novgroup *ngovp;
static struct novart *allarts;
static HASHTABLE *grouptbl, *acttbl, *timtbl;

extern long atol();
extern char *strsave();

static char *
grpnumtoname(num)
int num;
{
	char acount[30];

	if (grouptbl == NULL)
		readgroupsfile();
	(void) sprintf(acount, "%ld", num);
	return hashfetch(grouptbl, acount);
}

static
setgrpname(gh)
register group_header *gh;
{
	if (gh->group_name == NULL) {
		/*
		 * how bleedin' useful!
		 * turn gh->group_num into gh->group_name.
		 * remember to spoon-feed nn with gh->group_name_length.
		 */
		gh->group_name = grpnumtoname(gh->group_num);
		if (gh->group_name == NULL) {
			fprintf(stderr, "nn: can't map group %d to name\n",
				gh->group_num);
			nn_exit(1);
		}
	}
	gh->group_name_length = strlen(gh->group_name);
}

static
readactfile()
{
	register int count = 0;
	char actline[512];
	static FILE *actfp;
	extern char *news_lib_directory;

	if (acttbl != NULL)
		return;
	acttbl = hashcreate(3000, (unsigned (*)())NULL);
	if (acttbl == NULL) {
		fprintf(stderr, "nn: can't create hash table\n");
		nn_exit(1);
	}

	grouptbl = hashcreate(3000, (unsigned (*)())NULL);
	if (grouptbl == NULL) {
		fprintf(stderr, "nn: can't create group table\n");
		nn_exit(1);
	}

	actfp = fopen(relative(news_lib_directory, "active"), "r");
	if (actfp == NULL) {
		perror("active file");
		nn_exit(1);
	}
	for (; fgets(actline, sizeof actline, actfp) != NULL; count++) {
		char acount[30];
		char *line = strsave(actline);
		char *p = strchr(line, ' ');
		char *g;

		if (p != NULL)
			*p = '\0';

		g = strsave(line);

		if (!hashstore(acttbl, strsave(line), line)) {
			fprintf(stderr, "nn: active hashstore failed\n");
			nn_exit(1);
		}
		(void) sprintf(acount, "%ld", count);
		if (!hashstore(grouptbl, strsave(acount),  g)) {
		    fprintf(stderr, "nn: group hashstore failed\n");
		    nn_exit(1);
		}
		if (p != NULL)
			*p = ' ';
	}
	(void) fclose(actfp);

	/* init the master struct */
	bzero(&master, sizeof(master));
	master.number_of_groups = count;
}

static
readtimfile()
{
	char timline[512];
	static FILE *timfp;
	extern char *news_lib_directory;

	if (timtbl != NULL)
		return;
	timtbl = hashcreate(500, (unsigned (*)())NULL);
	if (timtbl == NULL) {
		fprintf(stderr, "nn: can't create hash table\n");
		nn_exit(1);
	}

	timfp = fopen(relative(news_lib_directory, "active.times"), "r");
	if (timfp == NULL)
	    return;		/* no great shakes if its missing */

	/* alt.fan.marla-thrift 736668095 netnews@ccc.amdahl.com */

	for (; fgets(timline, sizeof timline, timfp) != NULL;) {
		char acount[30];
		char *line = strsave(timline);
		char *p = strchr(line, ' ');
		char *g;

		if (p != NULL)
			*p = '\0';
		if (!hashstore(timtbl, strsave(line), line)) {
			fprintf(stderr, "nn: time hashstore failed\n");
			nn_exit(1);
		}
		if (p != NULL)
			*p = ' ';
	}
	(void) fclose(timfp);
}

/*
 * initialise *gh; this is much cheaper than calling db_read_group.
 */
db_init_group(gh, num)
register group_header *gh;
int num;
{
	register char *line;

	bzero(gh, sizeof(struct group_header));	/* tidy up the struct */
	gh->group_num = num;
	setgrpname(gh);
	gh->master_flag = M_VALID;
	if (strcmp(gh->group_name, "control") == 0)
		gh->master_flag |= M_CONTROL;
	/* these next two are subtle and we need to lie below */
/*	gh->first_db_article = 0; 		/* lowest # in ov. data */
/*	gh->last_db_article = 0;		/* highest # in ov. data */
	gh->first_a_article = 1; 		/* lowest # in active */
/*	gh->last_a_article = 0;			/* highest number in active */
/*	gh->index_write_offset = 0;		/* dunno */
/*	gh->data_write_offset = 0;		/* dunno */

	/* set the creation time */
	gh->creation_time = 1;                  /* group creation date (~epoch) */
	line = hashfetch(timtbl, gh->group_name);
	if (line != NULL)
	{
	    register char *p = strchr(line, ' ');

	    if (p != NULL)
		gh->creation_time = atol(p+1);
	}

	line = hashfetch(acttbl, gh->group_name);
	if (line != NULL) {
		register char *p = strchr(line, ' ');

		if (p == NULL)
			return;
		p++;
		gh->last_a_article = atol(p);
		p = strchr(p, ' ');
		if (p == NULL)
			return;
		p++;
		gh->first_a_article = atol(p);
	}
	gh->first_db_article = gh->first_a_article; /* lowest # in ov. data */
	gh->last_db_article = gh->last_a_article; /* highest # in ov. data */
}

/*
 * slurp up the overview data for this group into *gh.
 * this costs a file open and so should not be done frivolously.
 */
db_read_group(gh)
register group_header *gh;
{
	register struct novart *artp;
	register long maxart;

	/* db_init_group(gh, group_num???); $$$$ already done early at init time */

	if (ngovp != NULL)
		novclose(ngovp);		/* trash last group's data */

	ngovp = novopen(gh->group_name);
	if (ngovp == NULL) {
		printf("no overview data for group `%s'\n", gh->group_name);
		return;
	}
	allarts = novall(ngovp);
	if (allarts == NULL) {
/*
		printf("overview data inaccessible for group `%s'\n",
			gh->group_name);
 */
		return;
	}
	for (artp = allarts; artp != NULL; artp = artp->a_nxtnum)
		maxart = atol(artp->a_num);

	gh->first_db_article = atol(allarts->a_num); /* lowest # in ov. data */
	gh->last_db_article = maxart;		/* highest number in ov. data */
}

/*
 * fill in db_hdr and db_data from the overview data for the next
 * article in this group.  does weirdo nn encodings of header fields.
 */
db_read_art()
{
	register data_header *dhp = &db_hdr;
	register data_dynamic_data *ddp = &db_data;
	register struct novart *artp;
	int recnt = 0;

	if (ngovp == NULL)
		return 0;
	artp = novnext(ngovp);
	if (artp == NULL)
		return 0;			/* group exhausted */

	dhp->dh_number = atol(artp->a_num);
	/* printf("article #%ld\n", dhp->dh_number);	/* DEBUG */
	dhp->dh_date = pack_date(artp->a_date);	/* "encoded Date: filed" */
	dhp->dh_hpos = 0;			/* 1st hdr byte */
	dhp->dh_lpos = 1L<<30;			/* last article byte */
	dhp->dh_fpos = 0;			/* 1st article text byte */
	dhp->dh_lines = -1;			/* -1 == "unknown" */
	if (isascii(artp->a_lines[0]) && isdigit(artp->a_lines[0]))
		dhp->dh_lines = atoi(artp->a_lines);
	dhp->dh_replies = 0;			/* # of References: */
	if (artp->a_refs != NULL) {
		register char *p;

		for (p = artp->a_refs; *p != '\0'; p++)
			if (*p == '<')
				dhp->dh_replies++;
	}
	dhp->dh_cross_postings = 0; /* $$$$$$$ fix me */

	if (dhp->dh_number < 0) {
		current_digest_article = dhp->dh_number = -dhp->dh_number;
		ddp->dh_type = DH_DIGEST_HEADER;
	} else if (dhp->dh_number == 0) {
		dhp->dh_number = current_digest_article;
		ddp->dh_type = DH_SUB_DIGEST;
	} else {
		current_digest_article = 0;
		ddp->dh_type = DH_NORMAL;
	}

	dhp->dh_sender_length =
		pack_name(ddp->dh_sender, artp->a_from, NAME_LENGTH);
	dhp->dh_subject_length = pack_subject(ddp->dh_subject, artp->a_subj,
		&recnt, DBUF_SIZE);

	if (recnt)		/* 5/3/93 wolfgang@wsrcc.com */
	    dhp->dh_replies |= 0x80;

	db_read_counter++;
	return 1;
}

/* stubs */
close_master()
{
}

