#include "parms.h"
#include "structs.h"
#include "macros.h"
#include "header.h"
#include "acl.h"

#ifdef	RCSIDENT
static char rcsid[]="$Header: /g1/users/staff/gore/exp/notes/src/RCS/nfdirect.c,v 1.2 89/01/29 16:05:22 gore Exp $";
#endif	RCSIDENT

#define TO_DELETE -1

static void usage(), explain_stat();

static char *what_type[] = {"user", "group", "system"};

static char *what_mode[] = {
    "----", "r---", "--w-", "r-w-",
    "---d", "r--d", "--wd", "r-wd",
    "-a--", "ra--", "-aw-", "raw-",
    "-a-d", "ra-d", "-awd", "rawd"
};


/*
 * Provides command-line access to director options.
 *
 * Any normal user can run this program. It only allows changes to notesfiles
 * the user is a director in.
 *
 * Derived from Walter Underwood's "transplantation" of Ray Essick's nfaccess
 * program.
 *
 * 88/03/04	Original version 		Jacob Gore	
 *
 * 88/05/18	Added moderation support	Jacob Gore
 */


main (argc, argv)
    int     argc;
    char  **argv;
{
    void *calloc();

    struct io_f io;
    struct nfile *nfp;

    int verbose = FALSE;	/* If on, chats more */
    int quiet   = FALSE;	/* If on, doesn't even report errors */
    int turn_on;  /* 1 if '+', 0 if '-' */

    int ok;	   /* Temporary storage for */
    acl_h handle;  /*   returned results    */
    ace_t ace;

    /* stat_mask[0]: bits to turn off */
    /* stat_mask[1]: bits to turn on  */
    static short stat_mask[2] = {0,0};
    short old_stat;

    ace_t *Delmodes; /* to store mode entries to delete */
    int delmodec;
    ace_t *Addmodes; /* to store mode entries to add    */
    int addmodec;

    int argn, moden;  /* loop counters */

    int acl_changed, stat_changed, policy_changed;

    char *policy_fname;
    FILE *policy_file;
    struct note_f policy_note;
    struct hbuf policy_header;

    int replace_policy = FALSE;
    int delete_policy = FALSE;

    char *submission_address = NULL;

    int active_options = 0;

    acl_t *aclp;  /* The access control list */


    /* There has to be at least one argument (other than the program name) */
    if (argc < 2) {
	usage(argv[0]);
    }

    initenv();	/* common init code */

    /* Largest possible number of mode insertions in command line is argc-1 */
    Delmodes = (ace_t *)calloc(argc-1, sizeof(ace_t));
    delmodec = 0;
    Addmodes = (ace_t *)calloc(argc-1, sizeof(ace_t));
    addmodec = 0;

    /* Extract all notesfile names from the command line */
    for (argn=1; argn<argc; argn++) {

	if (argv[argn][0] == '+' || argv[argn][0] == '-') {

	    /* It's a parameter */
	    turn_on = argv[argn][0] == '+';

	    switch (forcelower(argv[argn][1])) {
	      case 'v': /* verbose mode on/off */
		verbose = turn_on;
		quiet = !verbose;
		break;

	      case 'q': /* quiet mode on/off */
		quiet = turn_on;
		verbose = !quiet;
		break;

	      case 'p':	/* It's a permission parameter */
		if (parsemode(&argv[argn][2],
			      &ace, turn_on, verbose)) {
		    if (quiet) {
			exit (1);  /* die a quiet death */
		    } else {
			printf("%s: Bad permission list in '%s'\n",
			       argv[0], argv[argn]);
			usage(argv[0]);
		    }
		}
		if (turn_on) {
		    ace_copy(&Addmodes[addmodec++], &ace);
		} else {
		    /* contradictory permissions are ignored */
		    ace_copy(&Delmodes[delmodec], &ace);
		    Delmodes[delmodec++].perms += PERMDELETED;
		}
		++active_options;
		break;

	      case 'a':
		stat_mask[turn_on] |= ANONOK;
		++active_options;
		break;

	      case 'o':
		stat_mask[turn_on] |= OPEN;
		++active_options;
		break;

	      case 'n':
		stat_mask[turn_on] |= NETWRKD;
		++active_options;
		break;

	      case 'm': /* Moderation... */
		stat_mask[turn_on] |= MODERATED; /* Ugly, but... */
		if (turn_on && argv[argn][2] != '\0') {
		    submission_address = &argv[argn][2];
		}
		++active_options;
		break;

	      case 'w': /* Policy note */
		if (turn_on) {
		    policy_fname = &argv[argn][2];
		    replace_policy = TRUE;
		} else {
		    delete_policy = TRUE;
		}
		++active_options;
		break;

	    }

	} else {
	    /* It's a notesfile name */
	    expand(argv[argn]);
	}
    }

    /* Have seen all command line arguments */

    if (nfile == NULL || active_options < 1) {
	if (quiet) {
	    exit(1);
	} else {
	    printf(
		  "%s: Need at least one option and at least one group name\n",
		   argv[0]
		  );
	    usage(argv[0]);
	}
    }

    for (nfp = nfile; nfp; nfp = nfp->n_next) {

	acl_changed = FALSE;
	stat_changed = FALSE;
	policy_changed = FALSE;

	if (init (&io, nfp->n_name) < 0) {		/* open */
	    if (!quiet) {
		printf ("%s: couldn't open\n", nfp->n_name);
	    }
	    continue;
	}

	if (!allow(&io, DRCTOK)) {			/* a director? */
	    if (!quiet) {
		printf ("Warning: you are not a director of %s\n",
			nfp->n_name);
	    }
	    closenf (&io);
	    continue;
	}

	if (delmodec + addmodec > 0) {
	    /* Handle permissions */
	    aclp = acl_open(&io);

	    /* Handle deletions first ... */
	    for (moden=0; moden < delmodec; moden++) {
		ok = acl_delete_ace(aclp, &Delmodes[moden]);
		if (!ok && !quiet) {
		    printf(
			 "Warning: could not delete permissions for %s '%s'\n",
			   what_type[Delmodes[moden].ptype],
			   Delmodes[moden].name);
		} else {
		    if (verbose && nfp==nfile) {
			printf("Deleted access list entry for %s '%s'\n",
			       what_type[Delmodes[moden].ptype],
			       Delmodes[moden].name);
		    }
		}
	    }

	    /* ... then the additions */
	    for (moden=0; moden < addmodec; moden++) {
		handle = acl_add(aclp, &Addmodes[moden]);
		if (handle == ACL_NOHANDLE && !quiet) {
		    printf("Warning: could not add permissions for %s '%s'\n",
			   what_type[Addmodes[moden].ptype],
			   Addmodes[moden].name);
		} else {
		    if (verbose && nfp==nfile) {
			printf("New mode for %s '%s': %s\n",
			       what_type[Addmodes[moden].ptype],
			       Addmodes[moden].name,
			       what_mode[Addmodes[moden].perms]);
		    }
		}
	    }

	    acl_changed = acl_was_changed(aclp);

	    /* Done with the access control list */
	    (void)acl_close(aclp);
	}

	/*
	 * Take care of the policy note.
	 *
	 * Positive prevails, so if there is a request to replace the
	 * policy note, don't bother deleting it.
	 */
	if (delete_policy && !replace_policy) {
	    delnote(&io, 0, LOCKIT);	/* Do I need to lock it? */
	    policy_changed = TRUE;	/* No status returned, assume OK */
	}

	if (replace_policy) {
	    if (policy_fname[0] == '\0') {
		if (quiet) {
		    exit(1);
		} else {
		    printf(
	      "%s: \"+w\" must be followed immediately by filename or \"-\"\n",
			   argv[0]
			  );
		    usage(argv[0]);
		}
	    }
	    if (policy_fname[0] == '-' && policy_fname[1] == '\0') {
		policy_file = stdin;
		if (verbose) {
		    printf("Getting policy note from standard input\n");
		}
	    } else {
		if ((policy_file = fopen(policy_fname, "r")) == NULL) {
		    if (!quiet) {
			printf("%s: could not open file '%s'\n",
			       argv[0], policy_fname);
		    }
		    exit(1);
		}
		if (verbose) {
		    printf("Getting policy note from '%s'\n", policy_fname);
		}
	    }

	    pagein(&io, policy_file, &policy_note.n_msg.m_addr);
	    if (policy_file != stdin) {
		fclose(policy_file);
	    }
	    genhdr(&io, &policy_header, 0);
	    strcpy(policy_note.n_title, "POLICY NOTE");
	    policy_note.n_stat = 0;
	    lock(&io, 'n');
	    puthdr(&io, &policy_note.n_msg, &policy_header);
	    putnote(&io, &policy_note, POLICY, ADDTIME);
	    unlock(&io, 'n');

	    policy_changed = TRUE;
	}

	if (stat_mask[0] || stat_mask[1]) {
	    /* Process status options */
	    lock(&io, 'n');
	    getdscr(&io, &io.descr);		/* get up to date descriptor */
	    old_stat = io.descr.d_stat;
	    io.descr.d_stat &= ~stat_mask[0];
	    io.descr.d_stat |=  stat_mask[1];	/* Positive prevails.  */
	    if (stat_changed = (old_stat != io.descr.d_stat)) {
		putdscr(&io, &io.descr);
	    }
	    unlock(&io, 'n');

	    if (verbose && stat_changed) {
		explain_stat(       old_stat, "Old status");
		explain_stat(io.descr.d_stat, "New status");
	    }

	    /* Set moderation address if necessary */
	    if (stat_mask[1] & MODERATED) {

		char old_subm_addr[ADDRLEN];
		char new_subm_addr[ADDRLEN];

		if (verbose) {
		    get_moderator(&io, old_subm_addr);
		}
		make_moderated(&io, submission_address);
		if (verbose) {
		    get_moderator(&io, new_subm_addr);
		    printf("Old submission address: %s\n", 
			   (old_subm_addr[0]=='\0') ? "(none)" :old_subm_addr);
		    printf("New submission address: %s\n", new_subm_addr);
		    stat_changed = TRUE;
		}
	    } else if (stat_mask[0] & MODERATED) {
		make_unmoderated(&io);
	    }

	}

	if (verbose && (acl_changed || stat_changed || policy_changed)) {
	    printf("Modified %s\n", nfp->n_name);
	}

	closenf(&io);
    }
    exit (0);
}

static void
explain_stat(a_stat, explanation)
    short a_stat;
    char *explanation;
{
    printf("%s: anonymous%s ok, %s, %snetworked, %smoderated\n",
	   explanation,
	   (a_stat & ANONOK)	? ""		: " not",
	   (a_stat & OPEN)	? "open"	: "closed",
	   (a_stat & NETWRKD)	? ""		: "not ",
	   (a_stat & MODERATED)	? ""		: "not "
	  );
}

static void
usage (name)						/* how to invoke */
    char   *name;
{
    fprintf(stderr,
	    "Usage: %s <options> <notesfile> [<notesfile>...]\n",
	    name);
    fprintf(stderr, "Options (in case of conflict, positive prevails):\n");
    fprintf(stderr,
	    "\t+v/-v  -- Verbose on/off    +n/-n  -- Networked/non-networked\n"
	   );
    fprintf(stderr,
	    "\t+q/-q  -- Quiet on/off      +a/-a  -- Anonymous OK/not OK\n"
	   );
    fprintf(stderr,
	    "\t                            +o/-o  -- Open/closed\n");
    fprintf(stderr,
	    "\t+m[submission-address]  -- Make the notesfile moderated\n");
    fprintf(stderr,
	    "\t\t(use last submission address if none given)\n");
    fprintf(stderr, "\t-m  -- Make the notesfile unmoderated\n");
    fprintf(stderr,
       "\t+wFile  -- Load policy note from File (\"-\" = \"standard input\")\n"
	   );
    fprintf(stderr,
       "\t-w      -- Delete the policy note\n");
    fprintf(stderr,
	   "\t+pT:Name=Perm  -- Set permissions for Name of type T to Perm\n");
    fprintf(stderr, "\t\tT is any word starting with one of:\n");
    fprintf(stderr, "\t\t\tu  -- Name is a user name\n");
    fprintf(stderr, "\t\t\tg  -- Name is a group name\n");
    fprintf(stderr, "\t\t\ts  -- Name is a system name\n");
    fprintf(stderr, "\t\tName is case-sensitive\n");
    fprintf(stderr, "\t\tPerm is a combination of:\n");
    fprintf(stderr, "\t\t\tr  -- read     a  -- answer\n");
    fprintf(stderr, "\t\t\tw  -- write    d  -- director\n");
    fprintf(stderr, "\t\t\tn  -- none\n");
    fprintf(stderr,
	    "\t-pT:Name  -- Remove entry Name of type T from the access list\n"
	   );
    exit (1);
}
