#include "parms.h"
#include "structs.h"
#include "stringpool.h"

/*
 * is_moderated - is this group moderated?  Non-0 if so; 0 otherwise.
 */
int
is_moderated(io)
    struct io_f *io;
{
    lock(io, 'n');
    getdscr(io, &io->descr);
    unlock(io, 'n');
    return io->descr.d_stat & MODERATED;
}

/*
 * get_moderator - fetches the submission address for the moderated group.
 *
 * buf is assumed to be of length ADDRLEN.
 */
void
get_moderator(io, buf)
    struct io_f *io;
    char *buf;
{
    sp_t sp;
    sp = sp_open(io);
    sp_get_string(sp, buf, SP_SUBMISN_ADDR, ADDRLEN);
    sp_close(sp);
}

#ifdef __STDC__
static char *substitute(char*, char, char*, char);
static void replace_char(char*, char, char);
#else
static char *substitute();
static void replace_char();
#endif __STDC__

/*
 * make_moderated
 *
 * Makes the group moderated and stores the submission address.  If 
 * submission_address is NULL, uses the last-saved submission address
 * (can be used to undo the preceding make_unmoderated operation).
 *
 * If the last submission address is NULL, the "Default-Moderator:" entry 
 * is retrieved from the config file.  Its value can be a mail destination
 * in any of the following forms:
 *
 *	user@site					(RFC-822 address)
 *	@gateway1,@gateway2,...,@gatewayN:user@site	(RFC-822 route-address)
 *	user%site%gatewayN%...%gateway2@gateway1	(RFC-733 route)
 *	gateway1!gateway2!...!gatewayN!site!user	(UUCP path)
 *
 * If there is a "?" in the address without a "\" in front of it, the "?" is
 * replaced by the group name with all dots in it changed to hyphens (this
 * corresponds to the treatment of the "BACKBONE" mailpath in News B 11).
 * Only the first occurance of the "?" without a "\" in front of it gets such
 * treatment.
 *
 * For example, if alt.sand-surfing is made moderated while no submission
 * address is available, and the line
 *
 *	Default-Moderator:	?@knowitall.hunga-dunga.edu
 *
 * appears in the config file, the moderation address for alt.sand-surfing
 * will be set to
 *
 *	alt-sand-surfing@knowitall.hunga-dunga.edu
 */
void
make_moderated(io, subm_addr)
    struct io_f *io;
    char *subm_addr;
{
    char mod_str[SP_STR_LEN];	/* To avoid access violations */
    sp_t string_pool;
    char old_subm_addr[ADDRLEN];
    int moderator_changed = FALSE;

    get_moderator(io, old_subm_addr);

    lock(io, 'n');
    getdscr(io, &io->descr);
    io->descr.d_stat |= MODERATED;
    putdscr(io, &io->descr);
    unlock(io, 'n');

    if (subm_addr == NULL) {
	/*
	 * If there was no old submission address, use the
	 * default address from the config file.
	 */
	if (old_subm_addr[0] == '\0') {

	    char hyphenated_nf[NNLEN];
	    safecpy(hyphenated_nf, io->nf, NNLEN);
	    replace_char(hyphenated_nf, '.', '-');
	    subm_addr = substitute(default_subm_addr, '?', hyphenated_nf,'\\');

	    moderator_changed = TRUE;
	}
    } else {
	moderator_changed = TRUE;
    }

    if (moderator_changed) {

	safecpy(mod_str, subm_addr, SP_STR_LEN);

	string_pool = sp_open(io);
	sp_put_string(string_pool, mod_str, SP_SUBMISN_ADDR);

	sp_close(string_pool);
    }
}

/*
 * make_unmoderated
 *
 * Makes the group unmoderated; leaves the submission address alone.
 */
void
make_unmoderated(io)
    struct io_f *io;
{
    lock(io, 'n');
    getdscr(io, &io->descr);
    io->descr.d_stat &= ~MODERATED;
    putdscr(io, &io->descr);
    unlock(io, 'n');
}

/*
 * Replaces the first occurance of target (a character) not following quote in
 * an original string with a replacement string.  Strips one level of quotes.
 * Resulting string is returned.
 */
static char *
#ifdef __STDC__
substitute(char *original, char target, char *replacement, char quote)
#else
substitute(original, target, replacement, quote)
    char *original;
    char target;
    char *replacement;
    char quote;
#endif
{
    int len_original, len_replacement, max_len_final;
    char *final;
    char *op, *rp, *fp;
    int oc;
    int done = FALSE;
    char *calloc();

    len_original    = strlen(original);
    len_replacement = strlen(replacement);
    max_len_final   = len_original + len_replacement;
	/* -1 for target_char, +1 for '\0' */

    final = calloc(max_len_final, sizeof(char));

    op = original;
    rp = replacement;
    fp = final;

    for (oc = 0; oc < len_original; oc++) {
	if (*op == quote) {
	    ++op; ++oc;
	    x((oc >= len_original),
	      "substitute: quote at end of string %s", original);
	    *fp++ = *op++;
	} else if (!done && *op == target) {
	    ++op;
	    strcat(fp, replacement);
	    fp += len_replacement;
	    done = TRUE;
	} else {
	    *fp++ = *op++;
	}
    }
    return final;
}

static void
#ifdef __STDC__
replace_char(char *string, char target, char replacement)
#else
replace_char(string, target, replacement)
    char *string;
    char target, replacement;
#endif
{
    if (string != NULL) {
	while (*string) {
	    if (*string == target) {
		*string++ = replacement;
	    } else {
		++string;
	    }
	}
    }
}
