/*
 * Duplicates the functionality of BSD access(2), but denies access if BSD
 * would have granted it merely because the effective gid has the right
 * privs.
 *
 * Jacob Gore, September 17, 1988.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <errno.h>

extern int errno;

int bsd_access(path, mode)
    char *path;
    int mode;
{
    struct stat st;
    gid_t rgid, egid;
    uid_t ruid, euid;
    int bsd_result;

    rgid = getgid();
    egid = getegid();
    ruid = getuid();
    euid = geteuid();

    if (stat(path, &st) == -1) {
	/*
	 * errno values in this case coincide
	 * with what access() would set
	 */
	return -1;
    }

    /* See what BSD thinks the access should be */
    bsd_result = access(path, mode);

    if (bsd_result == -1
	|| rgid == egid
	|| st.st_gid != egid) {
	/*
	 * Either the problem is irrelevant to this denial,
	 * or open(2) will be denied access no matter what
	 * we do here.
	 */
	return bsd_result;
    }
    
    if (st.st_uid == ruid && ((st.st_mode>>6) & mode)) {
	/*
	 * The uid was sufficient to give us access,
	 * so the gid did not come into play.
	 */
	return bsd_result;
    }
    if (st.st_gid == egid && ((st.st_mode>>3) & mode)) {
	/*
	 * BSD gave us access simply because the file
	 * is accessible by the effective gid.  This is
	 * the wrong behavior, so we correct it.
	 */
	errno = EACCES;
	return -1;
    }

    /*
     * All our worries were for nothing.  Return what BSD claimed was right.
     */
    return bsd_result;
}
