
#include "nc.h"

/* structure for binary header tree */
 /*
  * Newsclip(TM) Compiler Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip compiler for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed before compiling.
  * A short time evaluation of this product is also permitted.  See the file
  * 'Licence' in the library source directory for details.
  */

/* The header tree is maintained so that we may keep track of which header
 * items the user wants to parse at run time.  At the end of the compile,
 * we output a data structure with the header items desired in sorted order.
 */

typedef struct header_tree {
	char *hname;
	struct header_tree *left, *right;
	char *varname;	/* header variable name */
	char *delims;	/* delimiter code */
	dtype htype;	/* type of element */
	byte hflags;	/* flags, such as local variable flag */
	} *htree;


/*
 * Handle the definition of header items and header variables
 * Our output of the header items must be sorted for binary search,
 * so we might as well build them into a binary tree.  It doesn't really
 * matter if the tree gets really lopsided.
 */

select_header( hstring,  hidecl, hdelims )
char *hstring;		/* name of the header item */
nodep hidecl;		/* header variable declaration node */
char *hdelims;		/* delimiter for header array processing */
{
	symptr hsym;		/* symbol node */
	nodep idnode;		/* N_ID node */
	int hflags;

	if( !hidecl )
		return;
	if( !(idnode = kid0(hidecl)) )
		return;

	hsym = (symptr)kid0(idnode);

	hflags = isupper( hstring[0] ) ? T_DUALCASE : 0;

	lowercase( hstring );

	add_hitem( hstring, hsym->name, hdelims, hsym->type | hflags,
			hsym->sflags & SF_LOCAL );
}

/* the root of the header tree.  Newsgroups is always defined */

struct header_tree hroot = {
"newsgroups", 0, 0, "newsgroups", ", ", arrayof(T_NEWSGROUP), 0 
};

htree htree_root = (htree)0;

add_hitem( name, varname, delim, type, flags )
char *name;		/* name of header item */
char *varname;		/* variable name */
char *delim;		/* delimiter for array */
dtype type;		/* type of header variable */
byte flags;		/* flags of header symbol for printing */
{
	htree ourh;	/* our header item, as we scan through tree */
	htree *linkplace;	/* place to link in our new item */
	int res;		/* result of string compare */
	char *p;		/* name scanning pointer */

	lowercase( name );
	for( p = name; *p; p++ )
		if( !isgraph( *p ) || *p ==':' ) {
			parerror( "Invalid header field name %s", name );
			return;
			}

	ourh = &hroot;
	while( ourh ) {
		res = strcmp( ourh->hname, name );
		if( res > 0 )  {
			linkplace = &ourh->left;
			ourh = ourh->left;
			}
		 else if( res < 0 ) {
			linkplace = &ourh->right;
			ourh = ourh->right;
			}
		 else {
			parerror( "Header item '%s:' already defined", name );
			return;
			}
		}
	ourh = (htree)checkalloc( sizeof(struct header_tree) );

	ourh->hname = name;
	ourh->varname = varname;
	ourh->delims = delim;
	ourh->htype = type;
	ourh->hflags = flags;
	ourh->left = ourh->right = (htree)0;

	*linkplace = ourh;
	
}

/* list of predefined headers and the delimiters they need */
struct pdh {
	char *headname;
	char *delims;
	byte dcflag;		/* dual case flag */
	} pdhlist[] = {
{ "distribution",	", \t", 0 },
{ "followup-to",	", \t", 0 },
{ "keywords",		"S,", 0 },
{ "newsgroups",		", \t", 0 },
{ "path",		"!:@%\t", 0 },
{ "references",		" \t", T_DUALCASE },
{ "xref",		" \t", T_DUALCASE },
{ "message-id",		"", T_DUALCASE },
{ 0, 0, 0 }
};

/* create a header line for a referenced header symbol */

hcreate( name, type )
char *name;
dtype type;
{
	char *newname;			/* new buffer for modified name */
	char *p;			/* pointer for mapping */
	char *delim;			/* delimiter for array */
	int i;
	int dualcase;			/* case flag to add to type */

	/* copy over the name, then map underbar to dash */
	newname = allocstring( name );
	/* that alloc is a waste if the name is a duplicate */
	for( p = newname; *p; p++ )
		if( *p == '_' )
			*p = '-';

	/* set delim to null to start */

	delim = "";
	dualcase = 0;
	/* now check to see if there are predefined delimiters */
	for( i = 0; pdhlist[i].headname; i++ )
		if( strcmp( pdhlist[i].headname, newname ) == 0 ) {
			delim = pdhlist[i].delims;
			dualcase = pdhlist[i].dcflag;
			break;
			}
	/* we now have the header name and delimiters */
	add_hitem( newname, name, delim, type | dualcase, 0 );
}

static hcount = 0;		/* number of header items */

dump_header()
{
	cprintf( "struct hitem_list header_items[] = {\n" );
	hdump( &hroot );
	cprintf( "{ 0, 0, 0, 0 }\n};\n" );
	cprintf( "int num_headers = %d;\n", hcount );
}

/* recursive routine to dump header tree elements in sorted order */

hdump( ht )
htree ht;
{
	if( !ht )
		return;

	hdump( ht->left );

	cprintf( "{ \"%s\", %d, \"%s\", (datau *)&%s%s },\n", ht->hname,
		ht->htype, ht->delims, ht->hflags ? "U" : "", ht->varname );
	hcount++;
	hdump( ht->right );
}


