/*------------------------------------------------------------------------
       Name: eepmain.c

This program is intended to make it easier for users to manage the
hierarchy of their .newsrc file.

     Author: Paul Gillingwater, paul@actrix.gen.nz

Usage:
	eep [-p] [-!]

Options:
	-p:  Use a terse pointer on screen instead of a bar
	-!:  Set flag to disallow user shell out

------------------------------------------------------------------------*/
/* Software revision notes:

Dec 31, 1990:	A useful feature would be that when this software is
compiled without XBBS, it should have a command line switch that
causes it to only be used as a newsgroup selection mechanism.  Thus,
people who prefer rn or trn as their news reader, may find this
program a superior method for updating their .newsrc.

The first version of this software won't bother to  word wrap descriptions,
but will truncate them near the last column.

July 1991:  A busy year means this software has been neglected until
now.  I've hacked out the XBBS stuff, and made it work with two
different news description files.

Rebuilt data structure so that active and news descriptions are
stored in one unified structure, with more efficient use of storage
through array of pointers to malloc()ed areas.

Check for duplicates code has now been added.  July 13, 1991.

*************
Credit where credit is due.

Thanks must be given to Dave Taylor, for the pattern matching code
that I extracted from Elm, one of the nicest e-mail front-ends.

I cribbed the code for the binary chop search from ``Advanced C'' by
Herbert Schildt (Osborne McGraw-Hill, 1986).

Almost all the rest is my own ideas.

*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include "eep.h"

extern char     *malloc();
extern char     *getenv();

char	buffer[BUFSIZE], /* general purpose line buffer */
	tmp[BUFSIZE],
	match[BUFSIZE]; /* used to hold string for matching purposes */
char	s_hi[10], /* strings for numbers */
	s_lo[10],
	s_flag[5];
int	t_hi, t_lo;
char	t_flag;
char	*ptr;
int	mask,
	eepoint = 0,	/* -p flag for screen pointer type */
	noshell = 0,	/* -! flag to deny shell-out */
	uid,	/* real user id */
	write_active, /* flag used if we want to overwrite the active */
	counter,
	offset,	/* used in binary chop search */
	result;
FILE	*fnewsrc,
	*factive;

struct  actif *act[MAXLINES];  /* here's the main array */
struct	actif *topact[MAXDIST]; /* array for top level distributions only */

int	i_active,	/* index into actif arrays */
	c_active,	/* number of elements in act array */
	t_active,	/* number of elements in topact array */
	i_order,	/* index used for ordering .newsrc */
	really_active;	/* how many actually are active */

/* The levels array contains the Head pointers which control the
linked lists of newsgroups grouped by their "depth" in the
hierarchy, and sorted alphabetically.  E.g. alt, sci, soc are 
level 0, alt.pagan, sci.physics are level 1 and so on. */

struct actif *levels[MAXLEVELS];  /* keep track of levels with this array */
struct actif *aptr;	/* temporary pointer */

int	i_levels;	/* index into array */

/* Comparison functions for qsort.  These functiona are used for
sorting the array of pointers that point to our actif data structures. */

int	qcompare(item1,item2)	/* sort by name */
struct actif	**item1, **item2;
{
struct actif	*ptr1, *ptr2;
	ptr1 = (struct actif *) *item1;
	ptr2 = (struct actif *) *item2;
        return (strcmp(ptr1->name, ptr2->name));
}

int	icompare(item1,item2)	/* sort by index number */
struct actif	**item1, **item2;
{
struct actif	*ptr1, *ptr2;
	ptr1 = (struct actif *) *item1;
	ptr2 = (struct actif *) *item2;
        return (ptr1->index - ptr2->index);
}

main(argc, argv)
int	argc;
char	**argv;
{
int	ch;

	while ((ch = getopt(argc,argv,"ap")) != EOF) switch(ch) {
		case 'p':
			eepoint++;
			break;
		case '!':
			noshell++;
			break;
		case '?':
			fprintf(stderr,"usage: eep [-p] [-!]\n\n");
			fprintf(stderr,"-p means use terse pointer\n");
			fprintf(stderr,"-! means deny shell out\n");
			fprintf(stderr,"\nUse man eep for more info.\n");
			exit (2);
	}

	initial();	/* read in newsgroups, active and .newsrc */
	newsmain();
	for (i_active = 0; i_active < c_active; ++i_active) {
		if ((aptr = act[i_active]) == NULL) continue;
		if (aptr->name != NULL) free(aptr->name);
		free(aptr);
	}
	return(0);
}

/* Initialise the various chunks of memory and read in files. */

initial()
{
int	high,low,mid,	/* used for searching */
	warning = FALSE; /* if warning messages have been issued */

	uid = getuid();
	/* For systems with a fixed BBS login, we have this hack
	to deny shell access. */
	if (uid == SHELLDENY) noshell++;

	/* Now we create the data structure that will store
	our active file.  We will start by reading a file containing
	descriptions (/usr/lib/news/newsgroups), and sort it.  We 
	will also read a further file containing local additions
	(/usr/lib/news/newslocal) and sort that in too.  Then
	we'll read the active file, and try to match each line to
	a description.  If we can't, then we add it to the end with
	a null description.  Our next stage is to sort the resulting
	structure into alphabetical order by newsgroup.  Then we
	will build a series of linked lists, one for each level of
	the news hierarchy.  The final step is to read in the user's
	.newsrc, matching it to each newsgroup, and working out
	their high and low articles. */

	if ((factive = fopen(NEWSGROUPS, "r")) == NULL) {
		printf("unable to read ");
		printf(NEWSGROUPS);
		printf(CRLF);
	/* Make this file essential. */
		return(1);
	}

	c_active = 0;
	while (fgets(buffer,BUFSIZE,factive) != NULL) {
		/* ignore comment lines */
		if (buffer[0] == '#') continue;
		/* strip off CR and LF */
		while ((ptr = strchr(buffer,'\n')) != NULL)
			*ptr = '\000';
		while ((ptr = strchr(buffer,'\r')) != NULL)
			*ptr = '\000';
		/* Now allocate a chunk of memory for actif */
		if ((aptr = (struct actif *) malloc(
			sizeof(struct actif))) == NULL) {
			printf("Error while allocating memory!\n");
			return;
		}
		act[c_active] = aptr; /* record this pointer */
		/* allocate space for string + null byte */
		if ((aptr->name = malloc(strlen(buffer)+1)) 
			== NULL) {
			printf("Error while allocating memory!\n");
			return;
		}
		strcpy(aptr->name,buffer);

		/* Split off the desc part of the string (if present). 
		We do this by my favourite trick of searching for
		some white space, then putting a null there.  */

		if ((ptr = strchr(aptr->name,' ')) != NULL) {
			*ptr = '\000';
			ptr++;
		} else
		if ((ptr = strchr(aptr->name,'\t')) != NULL) {
			*ptr = '\000';
			ptr++;
		}
		/* Let's clean up the other chunk of the string.
		This simply means advancing the pointer past any
		leading white space. */
		while ((*ptr == ' ') || (*ptr == '\t')) ptr++;
		aptr->desc = ptr;
/* Diagnostic only
		if (c_active < 10) {
			sprintf(tmp,"%-25s: %-.67s\n",
				aptr->name,
				aptr->desc);
			printf(tmp);
		}
*/
	/* Now null out the other variables. */
		aptr->hi = 0;
		aptr->lo = 0;
		aptr->hilo = NULL;
		aptr->flag = '\000';
		aptr->status = '\000';
		aptr->index = 9999; /* sort at end by default */
		aptr->depth = (struct actif *) NULL;
		c_active++;
	}
	fclose(factive);

/*	Now do the same thing to the local newsgroups list.
	If we can't open it, don't worry, just carry on.  */

	if ((factive = fopen(NEWSLOCAL, "r")) != NULL) {
		while (fgets(buffer,BUFSIZE,factive) != NULL) {
			/* ignore comment lines */
			if (buffer[0] == '#') continue;
			/* strip off CR and LF */
			while ((ptr = strchr(buffer,'\n')) != NULL)
				*ptr = '\000';
			while ((ptr = strchr(buffer,'\r')) != NULL)
				*ptr = '\000';
			/* allocate space for string + null byte */
			if ((aptr = (struct actif *) malloc(
				sizeof(struct actif))) == NULL) {
				printf("Error while allocating memory!\n");
				return;
			}
			act[c_active] = aptr;
			if ((aptr->name = malloc(strlen(buffer)+1)) 
				== NULL) {
				printf("Error while allocating memory!\n");
				return;
			}
			strcpy(aptr->name,buffer);

			if ((ptr = strchr(aptr->name,' ')) != NULL) {
				*ptr = '\000';
				ptr++;
			} else
			if ((ptr = strchr(aptr->name,'\t')) != NULL) {
				*ptr = '\000';
				ptr++;
			}
			/* Let's clean up the other chunk of the string.
			This simply means advancing the pointer past any
			leading white space. */
			while ((*ptr == ' ') || (*ptr == '\t')) ptr++;
			aptr->desc = ptr;

		/* Now null out the other variables. */
			aptr->hi = 0;
			aptr->lo = 0;
			aptr->hilo = NULL;
			aptr->flag = '\000';
			aptr->status = '\000';
			aptr->index = 9999;
			aptr->depth = (struct actif *) NULL;
			aptr->order = (struct actif *) NULL;
			c_active++;
		}
		fclose(factive);
	}

	if (c_active == 0) {
		printf("Can't find any news group descriptions.  Sorry,\n");
		printf("but I just can't cope with this situation.\n");
		return(1);
	}

	qsort( act, (unsigned) c_active, 
		(unsigned) sizeof(act[0]), qcompare);

	/* Here we can start reading the active file, checking as we go...  */

	if ((factive = fopen(ACTIVEFILE, "r")) == NULL) {
		printf("unable to read ");
		printf(ACTIVEFILE);
		printf(CRLF);
		return(1);
	}
	really_active = 0;
	write_active = 1;  /* flag saying don't overwrite active yet */
	i_active = 0;
	while (fgets(buffer,BUFSIZE,factive) != NULL) {
		/* ignore comment lines */
		if (buffer[0] == '#') continue;
		/* strip off CR and LF */
		while ((ptr = strchr(buffer,'\n')) != NULL)
			*ptr = '\000';
		while ((ptr = strchr(buffer,'\r')) != NULL)
			*ptr = '\000';
		sscanf(buffer,"%s %s %s %s",
			match, s_hi, s_lo, s_flag);
		/* don't try to match a null string */
		if (strlen(match) == 0) continue;
		t_hi = atoi(s_hi);
		t_lo = atoi(s_lo);
		t_flag = s_flag[0];

/*	Here we begin a binary chop search, comparing the newsgroup
	name we have just read from the active file with the list of
	newsgroups read from the newsgroups file.  We cannot assume
	that the active file is in alphabetic order, but parts of it
	may be -- so let's try reading from the top before we do the
	binary search. If we don't find it, we should add this to the 
	bottom of the active data structure.
*/
		if (i_active <= c_active)	/* range check */
			aptr = act[i_active];
		if ((aptr != NULL) && (strcmp(match,aptr->name) == 0)) {
			aptr->hi = t_hi;
			aptr->lo = t_lo;
			aptr->flag = t_flag;
			if ((t_flag == 'y') ||
			    (t_flag == 'm'))
				really_active++;
			i_active++; /* advance index to next pointer */
			continue;
		}
		/* Binary chop! */
		low = 0;
		high = c_active - 1;
loop1:
		if (low <= high) {
			mid = (low+high)/2;
			aptr = act[mid];
			result = strcmp(match,aptr->name);
			if (result == 0) {
				aptr->hi = t_hi;
				aptr->lo = t_lo;
				aptr->flag = t_flag;
				if ((t_flag == 'y') ||
				    (t_flag == 'm'))
					really_active++;
				i_active = mid + 1;
				continue;
			} else
			if (result > 0) { /* after */
				low = mid+1;
				goto loop1;
			} else
			if (result < 0) { /* before */
				high = mid-1;
				goto loop1;
			}
		} else {
			printf(match);
			printf(" not found -- adding to active list");
			printf(CRLF);
			warning = TRUE;
			/* here's where we add it to the list */
			if ((aptr = (struct actif *) malloc(
				sizeof(struct actif))) == NULL) {
				printf("Error while allocating memory!\n");
				return;
			}
			act[c_active] = aptr; /* record this pointer */
			if ((aptr->name = malloc(strlen(match)+1)) 
				== NULL) {
				printf("Error while allocating memory!\n");
				return(-1);
			}
			strcpy(aptr->name,match);
			aptr->desc = NULL;
			aptr->hi = t_hi;
			aptr->lo = t_lo;
			aptr->flag = t_flag;
	/* Sort added newsgroups near the front, because we don't
	have a description yet. */
			aptr->index = 0;
#ifdef NEVER
			if (uid == 0) { /* allow root to change */
				write_active = 0;
				sprintf(tmp,"%s %d %d %c\n",
					match, t_hi, t_lo, t_flag);
				printf(tmp);
				printf("Please enter description (65): ");
				read(buffer,65);
				printf(CRLF);
				printf(CRLF);
				/* strip off CR and LF */
				while ((ptr = strchr(buffer,'\n')) != NULL)
					*ptr = '\000';
				while ((ptr = strchr(buffer,'\r')) != NULL)
					*ptr = '\000';
				if (strlen(buffer) > 0) {
				   if ((aptr->desc = 
				      malloc(strlen(buffer)+1)) == NULL) {
				      printf("Error while allocating memory");
				      printf(CRLF);
				      return;
				   }
				   strcpy(aptr->desc,buffer);
				}
			}
#endif /* NEVER */
			c_active++;
		}
	}
	fclose(factive); /* finished reading the active file */
	if ((uid == 0) && (write_active == 0)) {
		printf("I'm root, and I want to write the active file!");
		printf(CRLF);
	/* here goes the code to ask if root really wants to do this */
	}

	/* Sort them again in case we have added some */
	qsort( act, (unsigned) c_active, 
		(unsigned) sizeof(act[0]), qcompare);

	/* Now we can check for duplicates.  This is purely
	advisory, as it won't affect things later... we hope! */

	buffer[0] = '\0'; /* start empty */
	i_active = 0;
	while (i_active < c_active) {
		aptr = act[i_active];
		if ((aptr != NULL) && (aptr->name != NULL)) {
			if (strcmp(buffer,aptr->name) == 0) {
				printf("Duplicate entry found: ");
				printf(buffer);
				printf(CRLF);
			}
			strcpy(buffer,aptr->name);
		}
		i_active++;
	}
			
	/* Let's now build chains of pointers for each of the 
	hierarchies of news groups, taking our initial pointer
	from the array levels[]. */

	i_levels = 0;
	while (i_levels < MAXLEVELS) 
		levels[i_levels++] = (struct actif *) NULL; /* null array */

	/* Work backwards through the array, building the linked list.
	We also record the array index in each record.  If we were
	only accessing it as an array, this would be redundant, but
	we're using multiple linked lists of pointers as well.  
	We work backwards to ensure our chains are NULL terminated.  */

	i_active = c_active - 1;
	while (i_active >= 0) {
		aptr = act[i_active];
		/* count the number of dots in the newsgroup name */
		ptr = aptr->name;
		i_levels = 0;
		while ((ptr = strchr(ptr,'.')) != NULL) {
			i_levels++;
			ptr++;
		}
		/* add this to our linked list */
		if (i_levels < MAXLEVELS) {
			aptr->depth = levels[i_levels];
			levels[i_levels] = aptr;
		}
		i_active--;
	}
/* This code will be re-used later when parsing newsgroups */
	/* let's check the pointers in levels[] */
/*
	i_levels = 0;
	while (i_levels < MAXLEVELS) {
		if (levels[i_levels] != NULL) {
			aptr = levels[i_levels];
			while (aptr != NULL) {
				sprintf(tmp,"%-25s %-.67s\n",
					aptr->name,
					aptr->desc);
				printf(tmp);
				aptr = (struct actif *) aptr->depth;
			}
		}
		i_levels++;
	}
*/

	/* Now read in and match up our personal .newsrc
	   Get the $HOME from env. */

	if ((ptr = getenv("HOME")) != NULL)
		sprintf(tmp, "%s/.newsrc", ptr);
	else
		sprintf(tmp, ".newsrc");   /* default to current directory */

/* Maybe a future version should create the .newsrc? */

	if ((fnewsrc = fopen(tmp, "r")) == NULL) {
		printf("unable to read your ");
		printf(tmp);
		printf(CRLF);
		printf("please create it using the rn or trn news reader.\n\r");
		exit(1);
	}

/* Later on we will insert code in this next loop to keep track of
the original order in which the .newsrc file was found.  Note that
this should be via a linked list rather than through index numbers,
because it's easier to move chunks of newsgroups around simply by
adjusting pointers. */
		
	i_active = 0;
	/* i_order is used to keep track of the order in which
	we find newsgroups in the .newsrc.  Note that we start
	at 1 to allow 0 to be used to mark groups that have not
	been found in the newsgroups file, and are thus likely
	to be new -- this will sort them to the top of the list. */
	i_order = 1;
	while (fgets(buffer,BUFSIZE,fnewsrc) != NULL) {
		/* ignore comment lines */
		if (buffer[0] == '#') continue;
		/* strip off CR and LF */
		while ((ptr = strchr(buffer,'\n')) != NULL)
			*ptr = '\000';
		while ((ptr = strchr(buffer,'\r')) != NULL)
			*ptr = '\000';
		/* don't try to match a null string */
#ifdef DEBUG
		printf("[1]Looking for %s... \n",buffer);
#endif /* DEBUG */
		if (strlen(buffer) == 0) continue; 

		t_flag = ' ';  /* default to SPACE */
		if ((ptr = strchr(buffer,':')) != NULL) {
			t_flag = ':';
			*ptr = '\000'; /* null terminate newsgroup */
			ptr++;	/* point to rest of line (hilo) */
		} else if ((ptr = strchr(buffer,'!')) != NULL) {
			t_flag = '!';
			*ptr = '\000'; /* null terminate newsgroup */
			ptr++;	/* point to rest of line (hilo) */
		}
		/* advance past any whitespace */
		while ((*ptr == ' ') || (*ptr == '\t')) ptr++;

#ifdef DEBUG
		printf("[2]Looking for %s...i=%d, c=%d \n",
			buffer,i_active,c_active);
#endif /* DEBUG */
		if (i_active >= c_active) break; /* full up!! */
#ifdef DEBUG
		printf("[3]Looking for %s... \n",buffer);
#endif /* DEBUG */
		aptr = act[i_active]; /* range check */
		if ((aptr != NULL) && (strcmp(buffer,aptr->name) == 0)) {
#ifdef DEBUG
			printf("[1]found %s%c\n",aptr->name,t_flag);
#endif /* DEBUG */
			if ((aptr->hilo = malloc(strlen(ptr)+1)) 
				== NULL) {
				printf("Error allocating memory!\n");
				return(1);
			}
			strcpy(aptr->hilo,ptr);
			aptr->index = i_order++;	/* useful later */
			aptr->status = t_flag;
			continue;
		}
		/* Binary chop! */
		low = 0;
		high = c_active - 1;
loop2:
		if (low <= high) {
			mid = (low+high)/2;
			aptr = act[mid];
			result = strcmp(buffer,aptr->name);
#ifdef DEBUG
			printf("Target %s, hit %s\n",buffer,aptr->name);
#endif /* DEBUG */
			if (result == 0) {
#ifdef DEBUG
				printf("[2]found %s%c\n",aptr->name,t_flag);
#endif /* DEBUG */
				if ((aptr->hilo = malloc(strlen(ptr)+1)) 
					== NULL) {
					printf("Error allocating memory!\n");
					return(1);
				}
				strcpy(aptr->hilo,ptr);
				aptr->status = t_flag;
				aptr->index = i_order++;  /* useful later */
				i_active = mid + 1;
			/* This next hack is necessary to prevent the
			corner case where mid has pointed at the very
			last entry in the act strucutre. */
				if (i_active == c_active)  i_active = 0;
				continue;
			} else
			if (result > 0) { /* after */
				low = mid+1;
				goto loop2;
			} else
			if (result < 0) { /* before */
				high = mid-1;
				goto loop2;
			}
		} else {
			printf("bogus newsgroup: ");
			printf(buffer);
			printf(" found in your .newsrc -- ignored");
			printf(CRLF);
			warning = TRUE;
			continue;
		}
	}
	fclose(fnewsrc);

	/* Now let's sort this lot into the order that we originally
	read it from the .newsrc in.  New newsgroups will be forced
	to the top, making it easier for them to be spotted.  */

	qsort( act, (unsigned) c_active, 
		(unsigned) sizeof(act[0]), icompare);

	/* Now let's see if we can create a new .newsrc in $HOME */
	if ((ptr = getenv("HOME")) != NULL)
		sprintf(tmp, "%s/.newsrc.new", ptr);
	else	sprintf(tmp, ".newsrc.new");

	if ((fnewsrc = fopen(tmp, "w")) == NULL) {
	   printf("warning: cannot create new .newsrc -- check permissions\n");
	   warning = TRUE;
	}

	if (warning) {
		printf("\nPress ENTER to continue.");
		gets(buffer);
	}
}

/* end of eepmain.c */
