/*
 *	Copyright 1989 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 *
 *	Fully LISPis memory allocation scheme with garbage collection
 *	by   Matti Aarnio <mea@nic.funet.fi> 1996
 */

#include "malloc.h"
#ifndef	_LISTUTILS_H
#define	_LISTUTILS_H

/*
 * A LIST is a struct conscell with flags 0, the list is kept under dtpr.
 * A STRING is a struct conscell with flags !0, the string is kept under string.
 * The value of a LIST is a linked list of (struct conscell) of either type.
 * The value of a C variable must not be assumed to be either a LIST of STRING.
 * NIL is a LIST with null dtpr.
 */

struct conscell {
	struct conscell *next;
	union {
		struct conscell *u_dtpr;
		unsigned char	*u_string;
	} u;
#ifdef __alpha	/* Alpha doesn't take nicely stores into too small objects.
		   The hardware has store support only for 32/64/128-bit
		   objects. */
	int		flags;	/* if 0 is a list, otherwise a string */
	int		pflags;	/* if 1 x->prev->cdr == x */
#else
	short		flags;	/* if 0 is a list, otherwise a string */
	short		pflags;	/* if 1 x->prev->cdr == x */
#endif
	struct conscell *prev;	/* points to whichever dtpr points to here */
};

#define	NEWSTRING	0001	/* newly allocated string (free() when GC) */
#define	CONSTSTRING	0002	/* constant, mutually exclusive with above */
#define	QUOTEDSTRING	0004	/* quoted, don't apply IFS after expansion */
#define	NOQUOTEIFQUOTED	0010	/* if to be quoted, don't quote (for "$@") */
#define ELEMENT		0020	/* result of element expansion */
#define DSW_MARKER	0040	/* for garbage collection run.. */
#define DSW_BACKPTR	0100	/* ... this too (DSW == Deutch-Schorr-Waite) */

#define LIST(C)		((C)->flags == 0)
#define STRING(C)	((C)->flags != 0)
#define	ISCONST(C)	((C)->flags & CONSTSTRING)
#define	ISNEW(C)	((C)->flags & NEWSTRING)
#define	ISQUOTED(C)	((C)->flags & QUOTEDSTRING)
#define	ISDONTQUOTE(C)	((C)->flags & NOQUOTEIFQUOTED)
#define	ISELEMENT(C)	((C)->flags & ELEMENT)

#define	dtpr		u.u_dtpr
#define	string		u.u_string


#define	car(X)		(X)->dtpr
#define	cdr(X)		(X)->next
#define	caar(X)		car(car(X))
#define cadr(X)		car(cdr(X))
#define	cdar(X)		cdr(car(X))
#define	cddr(X)		cdr(cdr(X))
#define	caaar(X)	car(car(car(X)))
#define	cdaar(X)	cdr(car(car(X)))
#define	cadar(X)	car(cdr(car(X)))
#define	cddar(X)	cdr(cdr(car(X)))

#define grindef(T,L)	(fprintf(runiofp, T), s_grind(L,runiofp), putc('\n', runiofp))

/*
 * These macros make it easier to maintain the illusion of dealing with lists.
 */

/* YOU MUST BE VERY CAREFUL ABOUT CALLING MACROS FROM WITHIN MACROS!!! */
/* BE VERY CAREFUL ABOUT USING EXPRESSIONS WITH SIDEEFFECTS INSIDE MACRO ARGS */
/* #define newcell()	(struct conscell *)tmalloc(sizeof (struct conscell)) */
#define copycell(X)	(tmp = newcell(), *tmp = *(X), tmp)
/* nconc(list, list) -> old (,@list ,@list) */
#define nconc(X,Y)	(car(X) != 0 ? cdr(s_last(car(X))) = (Y) \
			       : (car(X) = (Y), (X)))
/* ncons(s-expr) -> new (s-expr) */
#define ncons(X)	(tmp = newcell(), car(tmp) = (X), \
    tmp->flags = 0, cdr(tmp) = 0, tmp->prev = 0, tmp->pflags = 0, tmp)
/* cons(s-expr, list) -> new (s-expr ,@list) */
#define cons(X,Y)	(tmp = ncons(X), cdar(tmp) = (Y), tmp)
/* s_push(s-expr, list) -> old (s-expr ,@list) */
#define s_push(X,Y)	(cdr(X) = car(Y), car(Y) = (X), (Y))
#define newstring(X)	(tmp = newcell(), tmp->string = (X), \
    tmp->flags = NEWSTRING, cdr(tmp) = 0, tmp->prev = 0, tmp->pflags = 0, tmp)
#define conststring(X)	(tmp = newcell(), tmp->string = (X), \
    tmp->flags = CONSTSTRING, cdr(tmp) = 0, tmp->prev = 0, tmp->pflags = 0, tmp)
#define NIL		ncons(0)


/* variables.c */
extern struct conscell *v_find(), *v_expand();
extern struct conscell *envarlist;

/* sh.c */
extern struct conscell *commandline;

/* listutils.c */
extern struct conscell *s_last(), *s_nth();
extern struct conscell *s_read(), *s_catstring(), *s_listify();
extern void s_grind(), s_set_prev();
extern struct conscell *s_pushstack(), *s_popstack();
#ifdef	MALLOC_TRACE
#ifndef	s_copy_tree
extern struct conscell *__s_copy_tree();
extern void __s_free_tree();
#define s_copy_tree(l)		__s_copy_tree((l), __FILE__, __LINE__)
#define s_free_tree(l)		__s_free_tree((l), __FILE__, __LINE__)
#endif	/* !s_copy_tree */
#else	/* !MALLOC_TRACE */
extern struct conscell *s_copy_tree();
extern void s_free_tree();
#endif	/* MALLOC_TRACE */

/* LISPic memory allocator, and other stuff.. */
extern struct conscell * newcell();
extern int cons_garbage_collect();
extern int consvar_register(/* const struct conscell* */);
extern void *consvar_mark();
extern void consvar_release(/* void* */);

#endif	/* _LISTUTILS_H */
