/*
 *	Copyright 1988 by Rayan S. Zachariassen, all rights reserved.
 *	This will be free software, but only when it is finished.
 */

/* LINTLIBRARY */

#include "hostenv.h"
#include "mailer.h"
#ifdef	USE_NDBM
#include <ndbm.h>
#include <sys/file.h>
#include "search.h"
#include "io.h"


/*
 * Search an NDBM format database for a key pair.
 */

struct conscell *
search_ndbm(sip)
	struct search_info *sip;
{
	DBM *db;
	datum val, key;
	struct conscell *tmp;
	struct spblk *spl;
	int retry, symid;
	extern int deferit;
	extern void v_set();

	if (sip->file == NULL)
		return NULL;
	retry = 0;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (db = (DBM *)spl->data) == NULL) {
reopen:
		db = dbm_open(sip->file, O_RDONLY, 0);
		if (db == NULL) {
			++deferit;
			v_set(DEFER, DEFER_IO_ERROR);
			fprintf(stderr, "search_ndbm: cannot open %s!\n",
					sip->file);
			return NULL;
		}
		if (spl == NULL)
			sp_install(symid, (u_char *)db, O_RDONLY, spt_files);
		else
			spl->data = (u_char *)db;
	}
	key.dptr = (char *)(sip->key);
	key.dsize = strlen((char *)(sip->key)) + 1;
	val = dbm_fetch(db, key);
	if (val.dptr == NULL) {
		if (!retry && dbm_error(db)) {
			dbm_close(db);
			++retry;
			goto reopen;
		}
		return NULL;
	}
	return newstring((u_char *)strnsave(val.dptr, val.dsize));
}

/*
 * Flush buffered information from this database, close any file descriptors.
 */

void
close_ndbm(sip)
	struct search_info *sip;
{
	DBM *db;
	struct spblk *spl;
	int symid;

	if (sip->file == NULL)
		return;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_modcheck);
	if (spl != NULL)
		sp_delete(spl, spt_modcheck);
	spl = sp_lookup(symid, spt_files);
	if (spl == NULL || (db = (DBM *)spl->data) == NULL)
		return;
	dbm_close(db);
	(void) sp_install(symid, (u_char *)NULL, 0, spt_files);
}


DBM *
open_ndbm(sip, flag, comment)
	struct search_info *sip;
	int flag;
	char *comment;
{
	DBM *db;
	struct spblk *spl;
	int symid;
	extern int deferit;
	extern void v_set();

	if (sip->file == NULL)
		return NULL;
	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_files);
	if (spl != NULL && flag == O_RDWR && spl->mark != O_RDWR)
		close_ndbm(sip);
	if (spl == NULL || (db = (DBM *)spl->data) == NULL) {
		db = dbm_open(sip->file, flag, 0);
		if (db == NULL) {
			++deferit;
			v_set(DEFER, DEFER_IO_ERROR);
			fprintf(stderr, "%s: cannot open %s!\n",
					comment, sip->file);
			return NULL;
		}
		if (spl == NULL)
			sp_install(symid, (u_char *)db, flag, spt_files);
		else
			spl->data = (u_char *)db;
	}
	return db;
}

/*
 * Add the indicated key/value pair to the database.
 */

int
add_ndbm(sip, value)
	struct search_info *sip;
	char *value;
{
	DBM *db;
	datum val, key;
	extern int deferit;
	extern void v_set();

	if ((db = open_ndbm(sip, O_RDWR, "add_ndbm")) == NULL)
		return EOF;
	key.dptr = (char *)(sip->key);
	key.dsize = strlen((char *)(sip->key)) + 1;
	val.dptr = value;
	val.dsize = strlen(value)+1;
	if (dbm_store(db, key, val, DBM_REPLACE) < 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr, "add_ndbm: cannot store (\"%s\",\"%s\")\n",
				sip->key, value);
		return EOF;
	}
	return NULL;
}

/*
 * Remove the indicated key from the database.
 */

int
remove_ndbm(sip)
	struct search_info *sip;
{
	DBM *db;
	datum key;
	extern int deferit;
	extern void v_set();

	if ((db = open_ndbm(sip, O_RDWR, "remove_ndbm")) == NULL)
		return EOF;
	key.dptr = (char *)(sip->key);
	key.dsize = strlen((char *)(sip->key)) + 1;
	if (dbm_delete(db, key) < 0) {
		++deferit;
		v_set(DEFER, DEFER_IO_ERROR);
		fprintf(stderr, "remove_ndbm: cannot remove \"%s\"\n",
				sip->key);
		return EOF;
	}
	return NULL;
}

/*
 * Print the database.
 */

void
print_ndbm(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	DBM *db;
	datum key, val;

	if ((db = open_ndbm(sip, O_RDONLY, "print_ndbm")) == NULL)
		return;
	for (key = dbm_firstkey(db); key.dptr != NULL; key = dbm_nextkey(db)) {
		val = dbm_fetch(db, key);
		if (val.dptr == NULL)
			continue;
		if (dbm_error(db))
			break;
		if (*val.dptr == '\0')
			fprintf(outfp, "%s\n", key.dptr);
		else
			fprintf(outfp, "%s\t%s\n", key.dptr, val.dptr);
	}
	(void) fflush(outfp);
}

/*
 * Print the uid of the owner of the database.  Note that for ndbm-style
 * databases there are several files involved so picking one of them for
 * security purposes is very dubious.
 */

void
owner_ndbm(sip, outfp)
	struct search_info *sip;
	FILE *outfp;
{
	DBM *db;
	struct stat stbuf;

	if ((db = open_ndbm(sip, O_RDONLY, "owner_ndbm")) == NULL)
		return;
	if (fstat(dbm_pagfno(db), &stbuf) < 0) {
		fprintf(stderr, "owner_ndbm: cannot fstat(\"%s\")!\n",
				sip->file);
		return;
	}
	(void) fprintf(outfp, "%d\n", stbuf.st_uid);
	(void) fflush(outfp);
}

int
modp_ndbm(sip)
	struct search_info *sip;
{
	DBM *db;
	struct stat stbuf;
	struct spblk *spl;
	int symid, rval;

	if (sip->file == NULL
	    || (db = open_ndbm(sip, O_RDONLY, "owner_ndbm")) == NULL)
		return 0;
	if (fstat(dbm_pagfno(db), &stbuf) < 0) {
		fprintf(stderr, "modp_ndbm: cannot fstat(\"%s\")!\n",
				sip->file);
		return 0;
	}

	symid = symbol((u_char *)sip->file);
	spl = sp_lookup(symid, spt_modcheck);
	if (spl != NULL) {
		rval = stbuf.st_mtime != (time_t)spl->data
			|| stbuf.st_nlink != (int)spl->mark;
	} else
		rval = 0;
	sp_install(symid, (u_char *)stbuf.st_mtime,
			  stbuf.st_nlink, spt_modcheck);
	return rval;
}
#endif	/* USE_NDBM */
