/* ************************************************************ *
 *								*
 *		CMD-MAILLISTS					*
 *								*
 *	A part of main piece of mail (and other) servers.	*
#
#  Copyright 1990-1993   Matti.Aarnio @ FUNET.FI
#  This software is free under similar rules as BSD copyrights.
#  (Definitely this is NOT Public Domain.  "Just" FREELY AVAILABLE.
#   Don't claim you did this..)
#  You can use this, but you shall not hold us liable for anything.
#  You must not use our name in marketing, in case you decide to
#  use this.  We do appreciate bug-reports  -> mailserver-owner@nic.funet.fi
#  for improving this piece of software.
 *								*
 * ************************************************************ */

#include <sys/param.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>

#if	defined(sun) && !defined(BSD)
# define BSD
#endif

#include "config.h"
#include "input.h"

extern char *strchr();
extern char *strrchr();
extern char *malloc();
extern void  free();
extern int  errno;
extern char *sys_errlist[];
extern void usage();
extern int  process_input();

extern int DebugState;


enum listopers {
  ML_ADDREPLACE,	/* 0 */
  ML_DELETE,		/* 1 */
  ML_LOCATE,		/* 2 */
  ML_LOCATEFUZZ		/* 3 */
  };
enum listmatchers {
  ML_EXACT,		/* 0 */
  ML_FUZZ,		/* 1 */
  ML_UNIQUE,		/* 2 */
  ML_NOTUNIQUE		/* 3 */
  };
enum listresults {
  ML_ADDED,		/* 0 */
  ML_DELETED,		/* 1 */
  ML_REPLACED,		/* 2 */
  ML_FUZZMATCH,		/* 3 */  /* only with ML_DELETE.. */
  ML_BADREALNAME,	/* 4 */
  ML_BADCMD,		/* 5 */
  ML_NOTFOUND,		/* 6 */
  ML_NOTALLOWED		/* 7 */
  };

typedef struct listitem {
	char *emailuser;
	char *emailhost;
	char *realname;
	char *userparams;
	struct listitem **next;
} listitem;

typedef struct hdritem {
	char *itemname;
	char *itemdata;
	struct hdritem *next;
} hdritem;

typedef struct HdrData {
	char	local_only;
	char	remote_only;
	char	Moderator[80];
	char	Moved[132];
	/* etc.. */
} HeaderData;

/*
  Sort within array of items,
  USABLE ONLY WHEN ALL ITEMS ARE FRESH IN
  MEMORY ARRAY AND NONE HAVE BEEN DELETED!

  TO BE SURE: CALL ONLY FROM  read_maillist()

  Creates `next' links.
*/

   /*
    *  Compare domains in reverse order, then userids, and lastly
    *  real names if multiple addresses are fed in for some reason..
    */

   static int
   compare_listitems(a,b)
   listitem **a, **b;
   {
	int rc;

	if ((rc = revdomaincmp((*a)->emailhost,(*b)->emailhost))) return rc;
	if ((rc = strcasecmp((*a)->emailuser,(*b)->emailuser))) return rc;
	return strcasecmp((*a)->realname,(*b)->realname);
   }

static void printlist __((listitem **list,FILE *outfile,const int interExter));

void
sort_maillist(list)
listitem **list;
{
	int cnt = 0;
	listitem **li = list;
	listitem **prev = list;

	if (!list) return; /* Should not be called... */

	if (DebugState)
	  printf("Starting sort_maillist()\n");

	while (*li && (*li)->emailuser != NULL) {
	  ++li;
	  ++cnt;
	}
	if (DebugState) {
	  li = list;
	  prev = list;
	  (*prev)->next = ++li;
	  (*li)->next = NULL;
	  while (*li && (*li)->emailuser) {
	    (*prev++)->next = li;
	    (*li++)->next = NULL;
	  }
	  printf("Items to sort: %d\n-----------------\n",cnt);
	  printlist(list,stdout,'I');
	  printf("--------------\n");
	  printf("qsort(list,%d,%d,compare_listitems);\n",cnt,sizeof(listitem*));

	}

	if (cnt>1) {
	  qsort(list,cnt,sizeof(listitem*),compare_listitems);
	  li = list;
	  prev = list;
	  (*prev)->next = ++li;
	  (*li)->next = NULL;
	  while (*li && (*li)->emailuser) {
	    (*prev++)->next = li;
	    (*li++)->next = NULL;
	  }
	}
}

static void
printlist(list,outfile,interExtern)
listitem **list;
FILE *outfile;
const int interExtern;
{
	int cnt = 0;
	int longest = 0;
	int targetcol = 40-1;
	int len =0;
	int col = 0;
	listitem **li = list;

	if (!list) return;	/* Should not be called like this... */

	while (li && *li) {
	  if ((*li)->emailuser)
	    len = strlen((*li)->emailuser) + strlen((*li)->emailhost) + 1;
	  if (len > longest)
	    longest = len;
	  ++cnt;
	  li = (*li)->next;
	}
	if (longest > targetcol)
	  longest = targetcol;
	li = list;
	while (li && *li) {
	  if (*li && (*li)->emailhost) {
	    /* Something to print.. */
	    if ((*li)->emailhost[0])
	      col = fprintf(outfile,"%s@%s",(*li)->emailuser,(*li)->emailhost);
	    else
	      col = fprintf(outfile,"%s",(*li)->emailuser);
	    if (col >= targetcol+1)
	      fputc(' ',outfile);
	    else
	      while (col < targetcol+1) {
		col += 8;
		fputc('\t',outfile);
	      }
	    fprintf(outfile,"%s",(*li)->realname);
	    if (interExtern == 'I') {
	      fprintf(outfile," %s",(*li)->userparams);
	    }
	    fprintf(outfile,"\n");
	  }
	  li = (*li)->next;
	}
}

static char *
create_user_params()
{
	static char timebuf[30];
	time_t tim;
	struct tm *TM;
	static int done_it = 0; /* Static data thru this processing run.. */

	if (done_it) return timebuf;

	time(&tim);
	TM = (struct tm *)gmtime(&tim);

	strftime(timebuf,sizeof(timebuf),"(%Y%j:00000000)",TM);
	done_it = 1;
	return timebuf;
}


static listitem **
read_maillist(fd)
FILE *fd;
{
	listitem **list = NULL;
	int itemcnt = 0;
	int itemsize = 0;
	listitem *item;
	char *linebuf = NULL;
	int  linesize = 0;
	char *emailuser  = NULL;
	char *emailhost  = NULL;
	char *realname   = NULL;
	char *userparams = NULL;
	char *s = NULL, *p;

	if (!fd) return NULL;	/* Should not be called... */

	if (DebugState)
	  fprintf(stdout,"DEBUG: Starting read_maillist()\n");

	linebuf = malloc(512); /* I hope I don't need to expand this... */
	linesize = 512;
	*linebuf = 0;

	while (!feof(fd) && !ferror(fd)) {
	  *linebuf = 0;
	  if (fgets(linebuf,linesize,fd) == NULL) break;
	  if (*linebuf == 0) break;

	  /* Zap trailing \n, and clean trailing whitespace */
	  s = strchr(linebuf,'\n');
	  if (s) *s = 0;
	  s = linebuf + strlen(linebuf);
	  while (s >= linebuf && (*s == ' ' || *s == '\t')) *s-- = 0;
	  if (*linebuf == 0) continue; /* Blank line ? */

	  /* Isolate - somehow - recipient name from address, should
	     parse address with real RFC822 parser, but... */
	  s = strchr(linebuf,'\t');
	  if (!s) s = strchr(linebuf,' ');
	  /* Right, points to 1st tab or space.
	     If NULL, no realname on line! */
	  if (s) {
	    while (*s == ' ' || *s == '\t')
	      *s++ = 0;
	    if (!*s) s = NULL;
	  }
	  if (!s) {
	    realname = "(No Realname Defined)";
	    userparams = create_user_params();
	  } else {
	    realname = s;
	    userparams = create_user_params();
	    if ((p = strchr(realname,')')) && *(p+1) != 0) {
	      /* Separating (Real Name) and (MAILSERVERPARAMS) */
	      *++p = 0;
	      s = strchr(p+1,'(');
	      if (s) {
		/* There is something -- the serverparams */
		userparams = s;
		p = strchr(s+1,')');
		if (p) *++p = 0;
	      }
	    }
	  }
	  /* Estabilish subscription time.. */
	  if (userparams[1] == '0' || *userparams == 0)
	    userparams = create_user_params();

	  /* Isolate parts of email address, before and after `@' */
	  emailuser = linebuf;
	  emailhost = strchr(emailuser,'@');
	  if (emailhost)
	    *emailhost++ = 0;
	  else
	    emailhost = "";

	  /*  Make `item' with freshly allocated parts.. */
	  item = (listitem *)xcalloc (sizeof(listitem));
	  item->emailuser = xsalloc(emailuser);
	  item->emailhost = xsalloc(emailhost);
	  item->realname  = xsalloc(realname);
	  item->userparams= xsalloc(userparams);
	  item->next	  = NULL;

	  if (itemsize == 0) {
	    itemsize += 8;
	    list = (listitem **)xcalloc(itemsize * sizeof(item));
	    if (DebugState)
	      printf("xcalloc(%d * %d)\n",itemsize,sizeof(item));
	  }
	  if (itemcnt+1 >= itemsize) {
	    itemsize += 8;
	    list = (listitem **)xrealloc(list,itemsize * sizeof(item));
	    /* Zero that claimed memory */
	    memset(&list[itemcnt],0,sizeof(item)*(itemsize-itemcnt));
	    if (DebugState)
	      printf("xrealloc(list,%d * %d)\n",itemsize,sizeof(item));
	  }
	  list[itemcnt] = item;
	  ++itemcnt;
	  list[itemcnt] = NULL;
	}

	if (DebugState)
	  fprintf(stdout,"DEBUG: Did successfull read on file. %d entries. Sorting it now!\n",itemcnt);

	/* Let SORTing take care of `next' links. */
	sort_maillist(list);

	if (DebugState)
	  fprintf(stdout,"DEBUG: File sorted.\n");

	return list;
}

static listitem **
find_mlentry(list,emailuser,emailhost,uniqueness,exactness)
listitem **list;
char *emailuser,*emailhost;
int *uniqueness;
int exactness;
{
	listitem **li = list;
	listitem **firstfound = NULL;
	int match = 0;
	int localhost = match_myaddr(emailhost);

	char userpart[128]; /* Should be enough for our purposes.. */
	char *s;
#ifdef BSD
	extern char *re_comp();
	extern int   re_exec();
#else
    	extern char	*regcmp();
	extern char	*regex();
#endif

	if (!list) return NULL;

	strncpy(userpart,emailuser,sizeof(userpart));
	userpart[sizeof(userpart)-1] = 0;
	s = strrchr(userpart,'!');
	if (s) { /* UUCP address !!! */
	  ++s;
	  strcpy(userpart,s);
	}
	s = strchr(userpart,'%');
	if (s) { /* %-hack */
	  *s = 0;  /* Zap it! */
	}
	lowerstr(userpart);
	if (exactness == ML_FUZZ) {
#ifdef BSD
	  s = re_comp(userpart);
#else
	  s = regcmp(userpart, 0);
#endif
	}

	while (li) {
	  if ((*li)->emailuser) {
	    /* Now we have an item to match.. */
	    if (exactness == ML_EXACT) {
	      match = (strcasecmp(emailuser,(*li)->emailuser) == 0 &&
		       strcasecmp(emailhost,(*li)->emailhost) == 0);

	      if (localhost && !match &&
		  match_myaddr((*li)->emailhost))
		match = (strcasecmp(emailuser,(*li)->emailuser) == 0);

	      if (match)
		if (!firstfound)
		  firstfound = li;
		else {
		  *uniqueness = ML_NOTUNIQUE;
		  return firstfound;
		}
	    } else if (exactness == ML_FUZZ) {
	      /* Fuzz match... */
	      /* Doesn't do exact match.  Not surprising actually...
		 This should be called once NO exact matches have been
		 found.. */
	      char *usr = malloc(strlen((*li)->emailuser)+1);
	      char *hst = malloc(strlen((*li)->emailhost)+1);
	      strcpy(usr,(*li)->emailuser);
	      strcpy(hst,(*li)->emailhost);
	      lowerstr(usr);
	      lowerstr(hst);
#ifdef BSD
	      match = re_exec(usr);
#else		
	      match = (regex(s, usr) != NULL);
#endif
	      if (match == 0)
#ifdef BSD
		match = re_exec(hst);
#else		
		match = (regex(s, hst) != NULL);
#endif
	      if (usr) free(usr);
	      if (hst) free(hst);

	      if (match > 0) {
		if (!firstfound)
		  firstfound = li;
		else {
		  *uniqueness = ML_NOTUNIQUE;
		  return firstfound;
		}
	      } else { /* About match quality.. */
	      }
	    } else { /* About exactness requirement of match */
	      /* What ??? */
	    }
	    /* Ok, let's try next */
	  }
	  li = (*li)->next;
	} /* while */
	if (firstfound) {
	  *uniqueness = ML_UNIQUE;
	  return firstfound;
	}
	return NULL;	/* Didn't find... */
}

static void
delete_mlentry(list,li)
listitem **list, **li;  /* list == begin of whole list, li == deletable item */
{
	register listitem **prev = list;

	/* Unlink this item from list, but if deleted
	   item is first on list, just replace it with
	   NULL entry with pointer forward */
	if (list != li)
	  while (prev && (*prev)->next != li)
	    prev = (*prev)->next;
	if (prev)
	  (*prev)->next = (*li)->next;

	/* Free data associated with it */
	if ((*li)->emailuser)
	  free ((*li)->emailuser);
	(*li)->emailuser = NULL;
	if ((*li)->emailhost)
	  free ((*li)->emailhost);
	(*li)->emailhost = NULL;
	if ((*li)->realname)
	  free ((*li)->realname);
	(*li)->realname  = NULL;
	if ((*li)->userparams)
	  free ((*li)->userparams);
	(*li)->userparams= NULL;
}

static void
freeup_list(list)
listitem **list;
{
	listitem **li = list;
	listitem **next;

	if (!list) return;	/* Should not be called like this... */

	while (li) {
	  if ((*li)->emailuser)
	    free ((*li)->emailuser);
	  (*li)->emailuser = NULL;
	  if ((*li)->emailhost)
	    free ((*li)->emailhost);
	  (*li)->emailhost = NULL;
	  if ((*li)->realname)
	    free ((*li)->realname);
	  (*li)->realname  = NULL;
	  if ((*li)->userparams)
	    free ((*li)->userparams);
	  (*li)->userparams= NULL;
	  next = (*li)->next;
	  free(*li);
	  *li = NULL;
	  li = next;
	}
	if (list)
	  free(list);
}


static
enum listresults
listoper(filepath,code,who,realname,oldnamep,outfile,HdrData)
char *filepath;
enum listopers code;
char *who;
char *realname;
char **oldnamep;
FILE *outfile;
HeaderData *HdrData;
{
	FILE *fd;
	char *s;
	listitem **list, **li, **item, *iteme, **li2;
	int cnt = 0;
	int rc;
	int uniqueness = 0;
	int deleted = 0;
	char *emailhost;
	char emailuser[256];
	int localhost;

	iteme = (listitem *)malloc(sizeof(listitem));
	memset (iteme,0,sizeof(listitem));
	item = (listitem **)malloc(sizeof(listitem*));
	*item = iteme;

	/* Check realname to be valid. */
	s = realname;
	if (s)
	  while (*s) {
	    switch (*s) {
	      case '.':	    case ',':	    case ':':	    case ';':
	      case ' ':	    case '-':	    case '_':	    case '+':
	      case '=':	    case '[':	    case '{':	    case ']':
	      case '}':	    case '|':
		  break;
	      case '(':
		  if (s == realname) break;	/* First char is ok	*/
		  *s = '[';			/* Change it to '['	*/
		  break;
	      case ')':
		  if (s[1] == 0) break;		/* Last char is ok	*/
		  *s = ']';			/* Change it to ']'	*/
		  break;
	      default:
		  if ((*s >= 'a' && *s <= 'z') ||
		      (*s >= 'A' && *s <= 'Z') ||
		      (*s >= '0' && *s <= '9'))
		    break;
		  return ML_BADREALNAME;
	    }
	    ++s;
	  }
	/* Name is ok, lets see what we can do.. */

	fd = fopen(filepath,"r+");
	if (!fd) {
	  fprintf(outfile,
		  "* List file `%s' Could not be opened!\n* Something is WRONG !\n",
		  filepath);
	  return -1;
	}

	strncpy(emailuser,who,sizeof emailuser);
	emailhost = strchr(emailuser,'@');
	if (emailhost)
	  *emailhost++ = 0;
	else
	  emailhost = "";

	localhost = match_myaddr(emailhost);

	if (HdrData->local_only && !localhost) {
	  if (code != ML_LOCATE && code != ML_LOCATEFUZZ)
	    fprintf(outfile,
		    "* This list is LOCAL-ONLY for subscriptions.\n");
	  return ML_NOTALLOWED;
	}
	if (HdrData->remote_only && localhost) {
	  if (code != ML_LOCATE && code != ML_LOCATEFUZZ)
	    fprintf(outfile,
		    "* This list is REMOTE-ONLY for subscriptions. (Test stuff..)\n");
	  return ML_NOTALLOWED;
	}
	if (HdrData->Moved[0] != 0 &&
	    code != ML_LOCATE && code != ML_LOCATEFUZZ) {
	  fprintf(outfile,
		  "* Moved %s\n",HdrData->Moved);
	  
	  return ML_NOTALLOWED;
	}

#ifdef BSD
	/* Grab exclusive locking for processing this file.. */
	flock(fileno(fd),LOCK_EX);
#endif
	list = read_maillist(fd);

	cnt = 0;
	
	/* Now processings.. */
	/* ================================================================ */
	if (code == ML_ADDREPLACE) {
	  /* Add/replace */
	  char *olduserparams = NULL;
	  uniqueness = -1;
	  li = find_mlentry(list,emailuser,emailhost,&uniqueness,ML_EXACT);
	  if (li) { /* Something found with userparams.. */
	    olduserparams = xsalloc((*li)->userparams);
	    if (oldnamep)
	      *oldnamep = xsalloc(olduserparams);
	  }
	  if (li && uniqueness == ML_UNIQUE) {
	    delete_mlentry(list,li);
	    deleted = 1;
	  } else if (li) {
	    /* Not unique, DUPLICATE !! */
	    li2 = li;
	    while (li2){
	      li = find_mlentry((*li2)->next,emailuser,emailhost,&uniqueness,ML_EXACT);
	      delete_mlentry(list,li2);
	      li2 = li;
	      deleted = 1;
	    }
	  } else {
	    /* Ok, not found it at all.  All is fine. */
	  }

	  /* Now, add me in.. */
	  li = list;
	  if (li) {
	    /* LIST is defined, something MAYBE deleted...
	       Find NULL entry in the list */
	    while (*li && (*li)->next && (*li)->emailuser )
	      li = (*li)->next;
	    if ((*li)->emailuser == NULL) {
	      /* Something deleted, item place is in array.. */
	      ; /* void routine.. */
	    } else if (*li) {
	      /* Item will get appended/joined to array */
	      (*li)->next = item;
	      li = item;
	    } else {
printf("DEBUG: Address array insert into place of NIL pointer.  Bug ??\n");
	      /* Nothing to be pointed to.. Add ITEM to array.. */
	      *li = iteme;
	      item = NULL;
	      iteme = NULL;
	    }
	  } else {
	    /* NO list!   item is the list! */
	    list = item;
	    item = NULL;
	    iteme = NULL;
	    li = list;
	  }
	    
	  (*li)->emailuser = xsalloc(emailuser);
	  (*li)->emailhost = xsalloc(emailhost);
	  (*li)->realname  = xsalloc(realname);
	  if (!olduserparams)
	    olduserparams = create_user_params();
	  (*li)->userparams= olduserparams;
	  /* Entry is in chain, just print it out and all is fine :-) */
	  rc = deleted ? ML_REPLACED : ML_ADDED;

	} else if (code == ML_DELETE) {
	  /* Delete */

	  uniqueness = -1;
	  rc = ML_NOTFOUND;
	  li = find_mlentry(list,emailuser,emailhost,&uniqueness,ML_EXACT);
	  if (li) { /* Something found with userparams.. */
	    if (oldnamep)
	      *oldnamep = xsalloc((*li)->userparams);
	  }
	  if (li && uniqueness == ML_UNIQUE) {
	    delete_mlentry(list,li);
	    deleted = 1;
	    rc = ML_DELETED;
	  } else if (li && *li) {
	    /* Not unique, DUPLICATE !! */
	    li2 = li;
	    while (li2){
	      delete_mlentry(list,li2);
	      li2 = find_mlentry((*li2)->next,emailuser,emailhost,&uniqueness,ML_EXACT);
	      deleted = 1;
	      rc = ML_DELETED;
	    }
	  } else {
	    /* Didn't find exact, lets try FUZZ! */
	    li = find_mlentry(list,emailuser,emailhost,&uniqueness,ML_FUZZ);
	    if (li) { /* Something found with userparams.. */
	      if (oldnamep)
		*oldnamep = xsalloc((*li)->userparams);
	    }
	    while (li) {
	      rc = ML_FUZZMATCH;
	      fprintf(outfile,"* Fuzzy match: `%s%s%s'  `%s'\n",
		      (*li)->emailuser,*((*li)->emailhost) ? "@" : "",
		      (*li)->emailhost,(*li)->realname);
	      li = find_mlentry((*li)->next,emailuser,emailhost,&uniqueness,ML_FUZZ);
	    }
	    goto quit_this; /* Don't update file -- FUZZ only*/
	  }
	} else if (code == ML_LOCATE) {
	  /* Do locate service.  Am I there ? */
	  uniqueness = -1;
	  li = find_mlentry(list,emailuser,emailhost,&uniqueness,ML_EXACT);
	  if (li && uniqueness == ML_UNIQUE) {
	    fprintf(outfile,"*\t%s%s%s  %s\n",
		    (*li)->emailuser,*((*li)->emailhost) ? "@" : "",
		    (*li)->emailhost,(*li)->realname);
	  } else if (li) {
	    /* Not unique, DUPLICATE !! */
	    li2 = li;
	    while (li2){
	      fprintf(outfile,"*\t%s%s%s  %s\n",
		      (*li2)->emailuser,*((*li2)->emailhost) ? "@" : "",
		      (*li2)->emailhost,(*li2)->realname);
	      li2 = find_mlentry((*li2)->next,emailuser,emailhost,&uniqueness,ML_EXACT);
	    }
	  } else {
	    /* Ok, not found it at all.  All is fine. */
	    fprintf(outfile,"*\tNot on this list.\n");
	  }
	  rc = 0;		/* Whatever, no fuzz.. */
	  goto quit_this;	/* Don't update file! It was only a review! */
	} else if (code == ML_LOCATEFUZZ) {
	  /* Do locate service.  Am I there ? */
	  uniqueness = -1;
	  li = find_mlentry(list,emailuser,emailhost,&uniqueness,ML_FUZZ);
	  if (li && uniqueness == ML_UNIQUE) {
	    fprintf(outfile,"*\tFuzzy Match: %s%s%s  %s\n",
		    (*li)->emailuser,*((*li)->emailhost) ? "@" : "",
		    (*li)->emailhost,(*li)->realname);
	  } else if (li) {
	    /* Not unique, DUPLICATE !! */
	    li2 = li;
	    while (li2){
	      fprintf(outfile,"*\tFuzzy Match: %s%s%s  %s\n",
		      (*li2)->emailuser,*((*li2)->emailhost) ? "@" : "",
		      (*li2)->emailhost,(*li2)->realname);
	      li2 = find_mlentry((*li2)->next,emailuser,emailhost,&uniqueness,ML_EXACT);
	    }
	  } else {
	    /* Ok, not found it at all.  All is fine. */
	    fprintf(outfile,"*\tNot on this list.\n");
	  }
	  rc = 0;		/* Whatever, no fuzz.. */
	  goto quit_this;	/* Don't update file! It was only a review! */
	} else {
	  fprintf(outfile,"* Unimplemented command (%d) for LISTOPER()! BUG!\n",
		  code);
	  rc = ML_BADCMD;
	  goto quit_this;
	}

	/* ================================================================ */

	/* Store file back in, and truncate it! */
	rewind(fd);
	printlist(list,fd,'I');
#ifdef BSD
	ftruncate(fileno(fd),ftell(fd));
#endif	

    quit_this:
#ifdef BSD
	flock(fileno(fd),LOCK_UN);
#endif
	fclose(fd);
	freeup_list(list);
	if (item) free(item);
	if (iteme) free(iteme);
	/* Free that exclusive lock */
	return rc;
}


/*
    FIND_LISTFILEPATH(argstr,nextargp)

    Search for filename of list given in 1st argument.
    Return pointer (nextargp) to begin of 2nd argument.
    If list is not found, return NULL, otherwise a pointer to
    internal STATIC buffer containing name of file.

    If  argstr == NULL, return path to LISTINDEX.
 */

char *
find_listfilepath(argstr,nextarg)
char *argstr;
char **nextarg;
{
	char *s;
	static char filepath[512];
	char *p = NULL, *q = NULL;
	FILE *lists = NULL;
	char listname[30],listline[250];

	/* dig up first token */
	s = pick_nextarg(argstr);
	*nextarg = s;

	strcpy(filepath,TRICKLE_HOME);
	strcat(filepath,"/maillists/");
	p = filepath + strlen(filepath);

	strcpy(p,"LISTINDEX");
	lists = fopen(filepath,"r");
	if (!lists) return NULL;
	if (argstr == NULL) {
	  fclose(lists);
	  return filepath;
	}
	*listline = 0;
	while (!feof(lists) && !ferror(lists)) {
	  *listline = 0;
	  q = fgets(listline,sizeof listline,lists);
	  if (q == NULL) break;
	  if (   *q == '#' || *q == '\n'
	      || *q == ';' || *q == '*'
	      || *q == ' ' || *q == '\t')
	    continue; /* Comment */
	  while (*q && *q!= ' ' && *q != '\t') ++q;
	  if (*q)  *q++ = 0;
	  /* while (*q && (*q==' ' || *q=='\t')) ++q; */
	  if (strcasecmp(listline,argstr)==0) { /* FOUND! */
	    strncpy(listname,listline,sizeof listname);
	    break;
	  }
	}
	fclose(lists);
	if (*listline == 0) return NULL;

	/* All parts of filename are at hand, form it and return a pointer...*/
	strcpy(p,listline);
	lowerstr(p);
	return filepath;
}


FILE *
iterate_next_listname(listidxfile,listname)
FILE *listidxfile;
char listname[30];
{
	char listline[512];
	char *q = NULL;

	*listline = 0;
	while (!feof(listidxfile) && !ferror(listidxfile)) {
	  *listline = 0;
	  q = fgets(listline,sizeof listline,listidxfile);
	  if (q == NULL) break;
	  if (*q == '#' || *q == '\n' ||
	      *q == ';' || *q == '*'  ||
	      *q == ' ' || *q == '\t')
	    continue;
	  while (*q && *q!= ' ' && *q != '\t') ++q;
	  if (*q)  *q++ = 0;
	  break;
	}
	if (*listline) {
	  strcpy(listname,listline);
	  return listidxfile;
	}
	if (listidxfile) {
	  fclose(listidxfile);
	}
	return NULL;
}

static HeaderData *
parse_HdrData(filepath,listname,outfile)
const char *filepath, *listname;
FILE *outfile;
{
	FILE *HF;
	char linebuf[200];
	static HeaderData HD;
	char *q = NULL;

	memset(&HD,0,sizeof HD);

	if (!(HF = fopen(filepath,"r"))) {
	  fprintf(outfile,"* List `%s' has configuration missing!  Report to Postmaster!\n",listname);
	  return &HD; /* Failed -- miserably.. */
	}
	while (!feof(HF) && !ferror(HF)) {
	  *linebuf = 0;
	  q = fgets(linebuf,sizeof linebuf,HF);
	  if (q == NULL) break;
	  if ((q = strchr(linebuf,'\n')) != NULL) *q = 0;
	  q = linebuf;
	  if (*q++ != '#' || *q++ != '+')
	    continue; /* Comment */
	  /* Ok, `#+' so far.. */
	  if (*q < 'A' || *q > 'z' || (*q > 'Z' && *q < 'a'))
	    continue;

	  /* Header options.. */
	  if (strcasecmp(q,"LOCAL-ONLY")==0)
	    HD.local_only = 1;
	  if (strcasecmp(q,"REMOTE-ONLY")==0)
	    HD.remote_only = 1;
	  if (strncasecmp(q,"MOVED",5)==0) {
	    strncpy(HD.Moved,q+6,sizeof(HD.Moved)-1);
	    HD.Moved[sizeof(HD.Moved)-1] = 0;
	  }
	}
	fclose(HF);
	return &HD;
}



static void
pick_namepart(str)
char *str;
{
	while (str && *str && *str != ')') ++str;
	if (str && *str) *str = 0;
}


int
do_subscribe(Cmd, argstr, Hdr, outfile)
struct command *Cmd;
char *argstr;
struct headers *Hdr;
FILE *outfile;
{
	char *nextarg = NULL;
	char *filepath = NULL;
	char *oldname = NULL;
	enum listresults rc = 0;
	char realname[256];
	int addjob = 0;
	char *s = NULL;
	char *realaddr = Hdr->Reply;
	HeaderData *HdrData = NULL;

	filepath = find_listfilepath(argstr,&nextarg);
	if (filepath == NULL) {
	  fprintf(outfile,"* List `%s' is not known to me.\nTry command  LIST\n",argstr);
	  return 1;
	}
	strcpy(realname,filepath);
	strcat(realname,".header");
	HdrData = parse_HdrData(realname,argstr,outfile);

	if (strcmp(Cmd->name,"ADD")==0) {
	  realaddr = nextarg;
	  nextarg = pick_nextarg(nextarg);
	  addjob = 1;
	}

	if (nextarg == NULL || *nextarg == 0) {
	  fprintf(outfile,
		  "* You must give %s real name as an argument after listname.\n",
		  addjob ? "recipients" : "your");
	  fprintf(outfile,"* See `HELP %s' for details.\n",Cmd->name);
	  return 1;
	}
	sprintf(realname,"(%s)",nextarg);
	if ((s = strchr(nextarg,'@')) != NULL) {
	  fprintf(outfile,"* `@' character in  `Real Name' ???  See HELP %s  for guidance!\n",Cmd->name);
	  fprintf(outfile,"* After all, this is no `info server'...\n");
	}

	oldname = NULL;
	rc = listoper(filepath,ML_ADDREPLACE,realaddr,realname,&oldname,outfile,HdrData);
	pick_namepart(oldname);
	switch (rc) {
	  case ML_ADDED:
	      fprintf(outfile,
		      "* %s been added to list `%s' with personal\n* name: `%s'\n",
		      addjob ? "Recipient has":"You have",argstr,nextarg);
	      break;
	  case ML_REPLACED:
	      fprintf(outfile,
		      "* %s real name on list `%s' is changed to:\n* `%s'%s%s%s\n",
		      addjob ? "Recipients":"Your",argstr,nextarg,
		      oldname ? ", old one was `":"",oldname?oldname+1:"",
		      oldname?"'":"");
	      break;
	  case ML_FUZZMATCH:
	      fprintf(outfile,"* -- FUZZMATCH -- IMPOSSIBLE RESULT ON ADD/REPLACE OPERATION. BUG!\n");
	      return 2;
	      break;
	  case ML_DELETED:
	      fprintf(outfile,"* -- DELETED -- IMPOSSIBLE RESULT ON ADD/REPLACE OPERATION. BUG!\n");
	      return 2;
	      break;
	  case ML_NOTFOUND:
	      fprintf(outfile,"* -- NOTFOUND -- IMPOSSIBLE RESULT ON ADD/REPLACE OPERATION. BUG!\n");
	      return 2;
	      break;
	  case ML_BADREALNAME:
	      fprintf(outfile,"* %s real name `%s' contains characters\n* that are not acceptable for use with SENDMAIL based UNIX systems.\n* See HELP %s.\n",
		      addjob?"Recipients":"Your",nextarg,Cmd->name);
	      return 1;
	  case ML_NOTALLOWED:
	      break;	/* listoper() reported the error */
	  default:
	      fprintf(outfile,"* Unknown reply code `%d' from listoper(add/replace) job!  Bug!\n",rc);
	      return 2;
	      break;
	}
	if (oldname) free(oldname);
	return 0;
}

int
do_unsubscribe(Cmd, argstr, Hdr, outfile)
struct command *Cmd;
char *argstr;
struct headers *Hdr;
FILE *outfile;
{
	char *nextarg = NULL;
	char *filepath = NULL;
	char *oldname = NULL;
	int rc = -1;
	char *realaddr = Hdr->Reply;
	int listdelete = 0;
	static int NotFoundSermon = 0;
	char realname[256];
	HeaderData *HdrData = NULL;

	filepath = find_listfilepath(argstr,&nextarg);

	if (strcmp(Cmd->name,"DELETE")==0) {
	  realaddr = nextarg;
	  nextarg = pick_nextarg(nextarg);
	  listdelete = 1;
	}

	if (filepath == NULL) {
	  if (strcmp(argstr,"*")== 0) {
	    char *next2arg;
	    char list[30];
	    char *argbuf = malloc(sizeof(list)+strlen(realaddr)+3);
	    int rc = 0;
	    char *listsfilename = find_listfilepath(NULL,&next2arg);
	    FILE *listsfile = fopen(listsfilename,"r");

	    if (!listdelete)
	      fprintf(outfile,"* %s *",Cmd->name);
	    else
	      fprintf(outfile,"* %s * %s",Cmd->name,realaddr);
	    fprintf(outfile," -- We iterate thru all lists and delete you from them all - if found.\n");
	    while ((listsfile = iterate_next_listname(listsfile,list))) {
	      if (!listdelete) {
		fprintf(outfile,"* Trying:  %s %s\n",Cmd->name,list);
		strcpy(argbuf,list);
	      } else {
		fprintf(outfile,"* Trying:  %s %s %s\n",Cmd->name,list,realaddr);
		sprintf(argbuf,"%s %s",list,realaddr);
	      }
	      rc |= do_unsubscribe(Cmd,argbuf,Hdr,outfile);
	    }
	    free(argbuf);
	    /* iterate_next_listname() closes listsfile once it is thru! */
	    return rc;
	  }
	  fprintf(outfile,
		  "* List `%s' is not known to me.\n* Try command  LIST\n",argstr);
	  return 1;
	}
	/* Ok, list is known to me.. */
	strcpy(realname,filepath);
	strcat(realname,".header");
	HdrData = parse_HdrData(realname,argstr,outfile);

	if( nextarg && *nextarg != 0)
	  fprintf(outfile,"* WARNING: Extra argument for %s ignored!\n",
		  Cmd->name);

	oldname = NULL;

	rc = listoper(filepath,ML_DELETE,realaddr,nextarg,oldname,outfile,HdrData);
	pick_namepart(oldname);
	switch (rc) {
	  case ML_ADDED:
	      fprintf(outfile,"* -- ADDED -- IMPOSSIBLE RESULT ON DELETE OPERATION. BUG!\n");
	      return 2;
	  case ML_REPLACED:
	      fprintf(outfile,"* -- REPLACED -- IMPOSSIBLE RESULT ON DELETE OPERATION. BUG!\n");
	      return 2;
	  case ML_FUZZMATCH:
	      fprintf(outfile,"* Match was Fuzzy, thus above listing of addresses is only result from this\n");
	      if (listdelete)
		fprintf(outfile,"* `%s %s %s' request.  Recipient has",
			Cmd->name,argstr,realaddr);
	      else
		fprintf(outfile,"* `%s %s' request.  You have",Cmd->name,argstr);
	      fprintf(outfile," NOT BEEN REMOVED FROM LIST!\n");
	      return 1;
	  case ML_DELETED:
	      if (listdelete)
		fprintf(outfile,"* User `%s' has",realaddr);
	      else
		fprintf(outfile,"* You have");
	      fprintf(outfile," been deleted from `%s' list\n",argstr);
	      break;
	  case ML_BADREALNAME:
	      fprintf(outfile,"* -- BADREALNAME -- IMPOSSIBLE RESULT ON DELETE OPERATION. BUG!\n");
	      return 2;
	  case ML_NOTFOUND:
	      if (listdelete)
		fprintf(outfile,"* User `%s' was",realaddr);
	      else
		fprintf(outfile,"* You were");
	      fprintf(outfile," not found from `%s' list,  not even with Fuzzy match.\n",argstr);
	      if (!NotFoundSermon++) {
		if (listdelete) {
		  fprintf(outfile,"* Check list with `REVIEW %s' if he/she/it is in with some other address,\n",argstr);
		} else {
		  fprintf(outfile,"* Check list with `REVIEW %s' if you are in with some other address,\n",argstr);
		}
		fprintf(outfile,"* and then use  `REPLYTO'-command  to mask up as that.\n");
		fprintf(outfile,"* Or `DELETE' command w/o need to mask up as anything.\n");
		fprintf(outfile,"* HOWEVER, DON'T ABUSE `REPLYTO' TO REMOVE OTHER PEOPLE AGAINST THEIR WILL!\n\n");
	      }
	      return 1;
	  case ML_NOTALLOWED:
	      break;	/* listoper() reported the error */
	  default:
	      fprintf(outfile,"* Unknown reply code `%d' from listoper(add/replace) job!  Bug!\n",rc);
	      return 2;
	}
	if (oldname) free(oldname);
	return 0;
}

int
do_listmaillists(Cmd, argstr, Hdr, outfile)
struct command *Cmd;
char *argstr;
struct headers *Hdr;
FILE *outfile;
{
	FILE *fd;
	char lastc = 0;
	char *filepath;
	char *s;

	if (*argstr != 0) {
	  fprintf(outfile,"* WARNING: Arguments to %s command are ignored!\n",
		  Cmd->name);
	  fprintf(outfile,"* After all, this is NOT BITNET LISTSERV !\n");
	}
	

	filepath = find_listfilepath(NULL,&s);
	if (filepath == NULL) {
	  fprintf(outfile,"** Can't find LIST OF LISTS.  Bug somewhere ?\n");
	  return 1;
	}
	fd = fopen(filepath,"r");
	while(!feof(fd) && !ferror(fd)) {
	  char line[512];
	  *line = 0;
	  if (fgets(line,sizeof line, fd)==0)	/* Get lines */
	    break;
	  if (*line == 0) break;
	  if (strncmp(line,"#+",2)==0)		/* Comment in file */
	    continue;
	  fputs(line,outfile);
	  lastc = line[ strlen(line)-1 ];
	}
	fclose(fd);
	if (lastc != '\n')
	  fputc('\n',outfile);

	return 0;
}


int
do_listreview(Cmd, argstr, Hdr, outfile)
struct command *Cmd;
char *argstr;
struct headers *Hdr;
FILE *outfile;
{
	char *nextarg;
	char *filepath;
	listitem **list;
	FILE *fd;


	if (*argstr == 0) {
	  fprintf(outfile,"%s  command requires name of list to be reviewed.\nUse command  LIST  to see list of lists we have locally.\n",Cmd->name);
	    return 1;
	}

	filepath = find_listfilepath(argstr,&nextarg);
	if (filepath == NULL) {
	  fprintf(outfile,"* List `%s' is not known to me.\n* Try command  LIST\n",
		  argstr);
	  return 1;
	}

	if( nextarg && *nextarg != 0)
	  fprintf(outfile,"* WARNING: Extra argument for REVIEW ignored!\n");

	fd = fopen(filepath,"r");
	if (!fd) {
	  fprintf(outfile,"* Can't open list file `%s'.\n* Something is WRONG!\n",
		  filepath);
	  return 2;
	};
	list = read_maillist(fd);
	fclose(fd);
	strcat(filepath,".header");
	fd = fopen(filepath,"r");
	if (!fd) {
	  fprintf(outfile,"* Can't open list file `%s'.\n* Something is WRONG!\n",
		  filepath);
	  return 2;
	};
	while (!feof(fd) && !ferror(fd)) {
	  char line[512];
	  if (fgets(line,sizeof line,fd) == NULL)	/* Get line.. */
	    break;
	  if (strncmp(line,"#+",2)==0)			/* Comment in header */
	    continue;
	  fputs(line,outfile);
	}
	fclose(fd);
	printlist(list,outfile,'E');

	return 0;
}

int
do_listrevme(Cmd, argstr, Hdr, outfile)
struct command *Cmd;
char *argstr;
struct headers *Hdr;
FILE *outfile;
{
	char *next2arg;
	char list[30];
	int rc = 0, rc2;
	char *listsfilename = find_listfilepath(NULL,&next2arg);
	FILE *listsfile = fopen(listsfilename,"r");

	fprintf(outfile,"* REVME -- We iterate thru all lists and locate you from them all - if found.\n");
	while ((listsfile = iterate_next_listname(listsfile,list))) {
	  char *listfilename = find_listfilepath(list,&next2arg);
	  char realname[256];
	  HeaderData *HdrData;

	  if (listfilename == NULL) break;

	  strcpy(realname,listfilename);
	  strcat(realname,".header");
	  HdrData = parse_HdrData(realname,list,outfile);
	  fprintf(outfile,"* Studying list: %s\n",list);
	  rc2 = listoper(listfilename,*argstr == '*' ? ML_LOCATEFUZZ : ML_LOCATE,Hdr->Reply,NULL,NULL,outfile,HdrData);
	  rc |= rc2;
	}
	/* iterate_next_listname() closes listsfile once it is thru! */
	return rc;
}
