#!/usr/local/bin/perl
# process.pl -- 
# SCCS Status     : @(#)@ process	3.67
# Author          : Johan Vromans
# Created On      : ***
# Last Modified By: Johan Vromans
# Last Modified On: Sat Jan  2 14:14:45 1993
# Update Count    : 672
# Status          : Going steady.

# This program processes mail messages, and enqueues requests for
# the mail server.
#
# For options and calling, see subroutine 'usage'.
#
$my_name = "process";
$my_version = "3.67";
#
################ Common stuff ################

$libdir = $ENV{"MSERVLIB"} || "/usr/local/lib/mserv";
unshift (@INC, $libdir);

################ Options handling ################

$opt_interactive = -t;
&options if @ARGV > 0 && $ARGV[0] =~ /^-+[^-]+/;
@ARGV = ("-") unless @ARGV > 0;
$trace_headers = 1 if defined $opt_trace_headers;
$interactive = $opt_interactive || defined $opt_i0;

################ More common stuff ################

# Require common here, so $opt_config can be used to select an
# alternate configuration file.
require "ms_common.pl";

################ Setting up ################

if ( $interactive ) {
    if ( defined $opt_i0 ) {
	# Attach STDOUT to STDIN.
	close (STDOUT);
	open (STDOUT, ">&0");
    }
    require "ctime.pl";
    print STDOUT ("$mserv_sender ($my_package) ready.\n");
    local ($t) = &ctime (time);
    chop ($t);
    print STDOUT ("Local time is $t.\n");
    select (STDOUT);
    $| = 1;
}
else {
    # All output goes to STDOUT, and will be mailed to the requestor.
    # Create a temp file to catch all.
    $tmpfile = &fttemp;
    open (STDOUT, ">" . $tmpfile) unless $opt_debug;
}
# Catch stderr also.
open (STDERR, ">&STDOUT");

# Motd.
&include ($notesfile);

$errflag = 0;
$didhelp = 0;
$needhelp = 0;

# Turn extensions into pattern.
($extpat = "(" . join("|", @exts) . ")") =~ s/\./\\./g;

# Search strategy.
$dofilesearch = 1 unless $dodirsearch || $doindexsearch;

require "$libdir/rfc822.pl";

# Defaults from RFC822 mail headers.
$h_from = $h_reply = "";

# Defaults from UUCP From_ header.
# Note that these will only be set if the host is existent and reachable,
# and the user name appears to be good-looking.
$h_uufrom = $h_uuhost = "";
@hdrs = () if $trace_headers;

if ( !$interactive ) {
    &start_read (shift(@ARGV)) ||
	&die ("Cannot read input [$!]\n");
}

# UUCP "From_" line...
if ( defined $rfc822'line_in_cache && $rfc822'line_in_cache =~ /^From (\S+) / ) {
    local ($try) = $1;
    local (@h);

    push (@hdrs, $rfc822'line_in_cache), chop $hdrs[0] if $trace_headers;

    print STDOUT ("Processing UUCP header...\n");

    $try = $1 . '!' . $try
	if $rfc822'line_in_cache =~ /remote from (\S+)$/; #';
	    
    # UUCP defaults...
    @h = split ('!', $try);

    # Sometimes the system name is prepended.
    shift (@h) if $h[0] eq $uucp_name;

    # For safety, we'll only accept good looking addresses.
    if ( @h == 2 && $h[1] =~ /^\w[-\w.]*$/ &&
	&check_uucp_name ($h[0], 1) ) {

	# We have a valid UUCP name, and a good looking user name.
	# We'll accept is as a default return address.
	($h_uuhost, $h_uufrom) = @h;
	$h_from = join ('!', @h);
	print STDOUT ("=> Return address (UUCP): \"$h_from\"\n");
	push (@hdrs, "=> Return address (UUCP): \"$h_from\"")
	    if $trace_headers;
    }
    else {
	&warning ("Unusable UUCP header", $rfc822'line_in_cache);	#');
	push (@hdrs, "=> WARNING: Unusable UUCP header") if $trace_headers;
    }
    undef $rfc822'line_in_cache;	#';
}

if ( !$interactive ) {
    # Scan RFC822 headers, extracting From: and Reply-To: info.
    print STDOUT ("Processing mail headers...\n");
    while ( $res = &read_header ) {
	last if $res == $rfc822'EMPTY_LINE;	#';
	push (@hdrs, $rfc822'line) if $trace_headers;	#');
	next unless $res == $rfc822'VALID_HEADER;	#';
	$rfc822'header =~ tr/[A-Z]/[a-z]/;		#';
	$h_from = $rfc822'contents if $rfc822'header eq "from";
	$h_reply = $rfc822'contents if $rfc822'header eq "reply-to";
    }

    # Preset sender info.
    $h_from = $h_reply if $h_reply;
    $v_sender = $h_from;
    &parse_addresses ($h_from);
    if ( @rfc822'addresses == 1 ) {		#'){
	$h_from = $rfc822'addresses[0];	#';
	$v_sender = $rfc822'addr_comments{$h_from} || $h_from;	#';
    }
}

# Setup defaults.
&reset;

if ( !$interactive ) {
    print STDOUT ("=> Default return address: \"$sender\"\n");

    # Check the sender against the list of system accounts.
    &validate_recipient ($sender, 2);

    push (@hdrs, "=> Return address: \"$sender\"") if $trace_headers;

    if ( $trace_headers && defined $trace_file && $trace_file ) {
	if (open (TRACE, ">>$trace_file")) {
	    if ( &locking (*TRACE, 1) == 1 ) {
		seek (TRACE, 0, 2);
		print TRACE (join ("\n", @hdrs), "\n\n");
		close (TRACE);
	    }
	}
    }

    print STDOUT ("\nProcessing message contents...\n\n");
    require "$libdir/pr_parse.pl";
    &command_loop;
    print STDOUT ("Your message has been processed.\n");
    close (STDIN);
}
else {
    require "$libdir/pr_parse.pl";
    &interactive_loop;
}

if ( $commands == 0 ) {
    print STDOUT ("No commands were found.\n");
    &help unless $interactive;
}
elsif ( $errflag ) {
    print STDOUT ("Number of errors detected = $errflag.\n",
		  "NO WORK WILL BE DONE.\n");
    &help unless $didhelp;
}
else {
    print STDOUT ("\n");

    # Be nice and forgiving
    eval { setpriority (0, $$, $nice) } if $nice;

    # Subroutines index_loop and work_loop are contained in separate
    # sources, since they may not always be needed. This speeds up
    # processing and cuts down memory resources.
    require "$libdir/pr_doindex.pl", &index_loop if @indexq > 0;
    &search_loop if @searchq > 0;
    if ( @workq > 0 ) {
	require "$libdir/pr_dowork.pl";
	&work_loop;
    }
    &help if $needhelp && !$didhelp;
}

&include ($hintsfile) 
    unless $didhelp || $opt_debug || $opt_nomail || $interactive;

print STDOUT ("\nMail Server finished.\n");

# Send confirmation message to recipient.
&confirm unless $interactive;

# Startup the queue run in the background.
&background_run ("$libdir/dorequest" . 
		 ($config_file ? " -config $config_file" : "") .
		 ($opt_trace ? " -trace" : "")) 
    if -s $queue && $auto_runrequest && !$opt_debug && !$opt_noqueue;

exit (0);

################ Subroutines ################

sub search {
    local ($request, $wantall) = @_;

    # This function returns an array of strings, each describing one
    # possibility. Each description is a NUL-joined string with fields:
    #   - the basename (used for sorting)
    #   - the size
    #   - the last modification date
    #   - the name of the library (LIB)
    #   - the part between library and basename
    #
    # If $wantall == TRUE, all possibilities are returned.
    # If $wantall == FALSE, one possibility is returned if the filesearch
    # (failing that, the directory search) locates exactly one file.
    # Otherwise, all possibilities are returned.

    local (@ret) = ();

    if ( $dofilesearch ) {
	foreach $lib ( @libdirs ) {
	    push (@ret, &filesearch ($lib, $request));
	}
    }

    if ( $dodirsearch && ($wantall || @ret != 1)) {
	require "$libdir/pr_dsearch.pl";
	foreach $lib ( @libdirs ) {
	    push (@ret, &dirsearch ($lib, $request));
	}
    }

    if ( $doindexsearch && ($wantall || @ret != 1)) {
	require "$libdir/pr_isearch.pl";
	if ( $indexfile =~ m|^/| ) {
	    local ($lib) = defined $indexlib ? $indexlib 
		: (&fnsplit($indexfile))[0];
	    push (@ret, &indexsearch ($indexfile, $lib, $request));
	}
	else {
	    foreach $lib ( @libdirs ) {
		push (@ret, &indexsearch ("$lib/$indexfile", $lib, $request));
	    }
	}
    }

    if ( $opt_debug || $opt_trace ) {
	@ret = reverse ( sort (@ret));
	print STDOUT ("=> Search queue:\n");
	local ($i) = 1;
	foreach $entry ( @ret ) {
	    local (@a) = &zu ($entry);
	    printf STDOUT ("  %3d: %s %s %s %s:%s:%s\n", $i, 
			   $a[0], $a[1], $a[2], $a[3], $a[4], $a[0]);
	    $i++;
	}
	@ret;
    }
    else {
	reverse ( sort (@ret));
    }
}

sub filesearch {

    local ($libdir, $request) = @_;

    # Locate an archive item $request in library $libdir.
    # Eligible items are in the format XXX or
    # XXX.EXT, where EXT is one of the known extensions.
    #
    # See "sub search" for a description of the return values.

    local (@retval);		# return value
    local (@a);			# to hold stat() result

    # Normalize the request. 
    # $tryfile will be the basename of the request.
    # $subdir holds the part between $libdir and $tryfile.
    local ($subdir, $tryfile) = &fnsplit ($request);
    $subdir .= "/" if $subdir && $subdir !~ m|/$|;
    $libdir .= "/" if $libdir && $libdir !~ m|/$|;

    print STDOUT ("Try file $libdir$subdir$tryfile...\n") if $opt_debug;

    # First attempt: see if the given file exists 'as is', with possible 
    # extensions

    foreach $ext ( "", @exts) {
	if ( -f $libdir.$subdir.$tryfile.$ext && -r _ ) {
	    @a = stat (_);
	    print STDOUT ("File $libdir$subdir$tryfile$ext (found)\n")
		if $opt_debug;
	    push (@retval, 
		  &zp ($tryfile.$ext, $a[7], $a[9], $libdir, $subdir));
	    last if $ext eq "";	# exact match prevails
	}
    }

    return @retval;
}

sub confirm {

    # Send the contents of the temp file to the requestor.

    # Close it, and reopen.
    close (STDOUT);
    open (MESSAGE, $tmpfile);

    if ( $opt_debug || $opt_nomail ) {
	open (MAILER, ">&STDERR");
    }
    else {
	open (MAILER, "|$sendmail '$recipient' $mserv_bcc");
    }

    print MAILER ("To: $recipient\n",
		  "Subject: Request by $v_sender\n");

    if ( defined @x_headers ) {
	foreach $hdr ( @x_headers ) {
	    print MAILER ($hdr, "\n");
	}
    }
    print MAILER ("\n");

    local ($inhdrs) = 1;
    while ( <MESSAGE> ) {

	# Include everything before the message contents.
	if ( $inhdrs ) {
	    print MAILER $_;
	    if ( $_ eq "Processing message contents...\n" ) {
		$inhdrs = 0;
		print MAILER "\n";
	    }
	    next;
	}

	# Suppress unrecognized stuff.
	if ( $reset > 1 ) {
	    $reset-- if /^=> Resetting/;
	    if ( $reset > 1 ) {
		print MAILER $' if /^Command: /;
	    }
	    else {
		print MAILER $_;
	    }
	}
	else {
	    print MAILER $_;
	}
    }
    close (MAILER);
    close (MESSAGE);

    # This aids in debugging...
    rename ($tmpfile, $tmpdir . "/mserv.last");
    unlink ($tmpfile);
}

sub discard {
    local ($msg) = @_;

    # Discard the job.
    # Do not attempt to send feedback except for a mailer error.
    # This is used when requests are received from someone on the 
    # 'black list'.

    print STDOUT ("\nREQUEST DISCARDED: ", $msg, "\n");
    close (STDOUT);

    # This aids in debugging...
    rename ($tmpfile, $tmpdir . "/mserv.last");
    unlink ($tmpfile);

    # The end of it all (silently)
    exit (0);
}

sub dolist {
    local ($list_type, $query, *found) = (@_);
    local ($entries) = 0;
    local ($name, $size, $date, $lib, $subdir); # elements of @found
    local ($prev);		# to suppress duplicates
    local (@tm);		# for time conversions

    $~ = "list_header";
    write;
    $~ = "list_format";
    $: = " /";		# break filenames at logical places
    $= = 99999;

    # have we found something?
    unless ( @found > 0 ) {
	$size = $date = "";
	$name = "***not found***";
	write;
	next;
    }

    $prev = "";
    foreach $found ( @found ) {

	($name, $size, $date, $lib, $subdir) = &zu ($found);

	# Avoid duplicates.
	next if $lib.$subdir.$name eq $prev;
	$prev = $lib.$subdir.$name;

	# Normalize size and date, if needed.
	$size = int(($size+1023)/1024) . "K" unless $size =~ /K$/;
	if ( $date =~ /^T/ ) {
	    $date = $';
	}
	else {
	    @tm = localtime ($date);
	    $date = sprintf("%02d/%02d/%02d", 
			    1900+$tm[5], $tm[4]+1, $tm[3]);
	}

	$name = $subdir.$name;
	write;
    }
}

sub search_loop {

    print STDOUT ("Search results:\n");

    foreach $query ( @searchq ) {

	local (@found);		# return from search

	# Locate them.
	@found = &search ($query, 1);

	# Print report.
	&dolist ("Search", $query, *found);

    }
    @searchq = ();
    print STDOUT ("\n");
}

sub reset {
    # Set defaults.
    @workq = ();
    @searchq = ();
    @indexq = ();
    $commands = 0;
    $errflag = 0;
    $method = '';
    @limits = defined $email ? @email_limits : @uucp_limits;
    $ftphost = '';

    # Who sent this mail?
    $sender = $h_from || "?";

    # Who gets the replies?
    $recipient = $sender;

    # Destination for email transfers.
    $destination = "";

    # Tally.
    $reset++;
}

sub errmsg {
  local (@msg) = @_;
  print STDOUT ('>>>>>>>> Error: ', shift(@msg), "\n");
  foreach $msg ( @msg ) {
      print STDOUT ('         ', $msg, "\n");
  }
  # Most parsing routines use 'return &errmsg...', so make sure it
  # errmsg returns a non-zero value.
  ++$errflag;
}

sub warning {
  local (@msg) = @_;
  print STDOUT ('>>>>>>>> Warning: ', shift(@msg), "\n");
  foreach $msg ( @msg ) {
      print STDOUT ('         ', $msg, "\n");
  }
  1;				# must be non-zero;
}

sub include {
    local ($file) = @_;
    local (*F);
    local ($ok) = 0;

    if ( $interactive ) {
	$ok = open (F, $file . 'i');
    }
    if ( $ok || ($ok = open (F, $file)) ) {
	while ( <F> ) {
	    print STDOUT;
	}
	close (F);
    }
    $ok;
}

# Pseudo-record pack/unpack
sub zp { join ("\0", @_); }
sub zu { split (/\0/, $_[0]); }

sub email_defaults {
    local ($dest) = @_;
    $method = "M";
    $destination = $dest;
    push (@workq, &zp ("M", $destination));
    &method_msg;
    @limits = @email_limits;
}

sub uucp_defaults {
    local ($uuhost, $uupath, $uunote) = @_;
    $uunote = $h_uufrom unless $uunote;
    $uuhost = $h_uuhost unless $uuhost;
    $uupath = "~uucp/receive/$h_uufrom" unless $uupath;

    if ( &check_uucp_name ($uuhost) &&
	&check_uucp_path ($uupath) ) {
	$method = "U";
	$uupath = $uuhost . '!' . $uupath;
	push (@workq, &zp ("U", $uupath, $uunote));
	&method_msg;
	@limits = @uucp_limits;
    }
}

sub method_msg {
    if ( $method eq 'U' ) {
	print STDOUT ("=> Transfer via UUCP to \"$uupath\"\n");
	print STDOUT ("=> (UUCP notification to \"$uunote\")\n");
    }
    elsif ( $method eq 'M' ) {
	print STDOUT ("=> Transfer via email to \"$destination\"\n");
    }
    else {
	&errmsg ("Please issue a MAIL or UUCP command first");
    }
}

sub ftp_defaults {

    # Setup FTP stuff. Check if allowed.

    ($ftphost) = @_;

    if ( $ftphost eq '' ) {
	&errmsg ("Missing FTP host name");
	return 0;
    }

    local ($prefer_uucp) = $prefer_uucp | $ftp_uucp_only;
    return 0 unless &setdefaults;

    if ( $ftp_uucp_only && $method ne 'U' ) {
	&errmsg ("FTP commands are only allowed when delivering via UUCP");
	print STDOUT ("         (Issue an UUCP command first)\n");
	$ftphost = '';
	return 0;
    }

    push (@workq, &zp ('G', 'O', $ftphost));
    print STDOUT ("=> FTP Connect to \"$ftphost\"\n");
    1;
}

sub setdefaults {

    local (@_);

    if ( $interactive && ! $method ) {
	&method_msg;
	return 0;
    }

    unless ( $recipient || $interactive ) {
	$recipient = $sender;
	print STDOUT ("=> Return address: \"$recipient\"\n");
    }

    unless ( $method ) {
	if ( defined $uucp && $prefer_uucp && $h_uufrom && $h_uuhost ) {
	    &uucp_defaults;
	    print STDOUT ("=> If delivery via UUCP is not desired, ",
			  "issue a MAIL command first\n");
	}
	elsif ( defined $email ) {
	    &email_defaults ($destination || $recipient);
	}
	elsif ( defined $uucp ) {
	    if ( $h_uufrom && $h_uuhost ) {
		&uucp_defaults;
	    }
	    else {
		&errmsg ("Please issue a UUCP command first");
		return 0;
	    }
	}

	unless ( $method ) {
	    &errmsg ("Sorry, can't transfer the requests to you",
		     "Issue a MAIL or UUCP command first");
	    return 0;
	}
    }
    1;
}

sub validate_recipient {
    local ($addr, $todo) = @_;

    # Validate a recipient name against the black list.
    # Values for $todo:
    #  0: return offending user name if invalid, otherwise return ''
    #  1: as 0, but also supply warning
    #  2: as 1, and discard job if configured to do so

    local ($user);

    return '' unless defined @black_list;
    return '' if $interactive;

    while ( ! defined $user ) {
	$addr = $', next if $addr =~ /^@[^:]+:/;	# @domain,domain:...
	$addr = $', next if $addr =~ /^[^!]+!/;		# host!...
	$addr = $`, next if $addr =~ /@[^@]+$/;		# ...@domain
	$user = $addr;
    }

    $addr = join ('!', @black_list);
    return '' if index ("!\U$addr\E!", "!\U$user\E!") < $[;

    if ( $todo >= 2 && ! $black_list_warning ) {
	&discard ("User \"$user\" access refused");
	# Not reached.
    }

    if ( $todo >= 1 ) {
	&warning ("User \"$user\" will be refused access in the future",
		  "Please use a user account instead of a system account");
    }

    # Return the offending user name, so caller can provide a message.
    return $user;
}

sub die {
    local ($msg) = "@_";
    print STDOUT ($msg, "\n");
    $sender = $sender || $mserv_owner || $mserv_bcc || "postmaster";
    $mserv_bcc = $mserv_owner;
    &confirm;
    exit (1);
}

sub background_run {
    local ($cmd) = @_;

    # Run $cmd in the background.

    local ($pid);

    if ( ($pid = fork) == 0 ) {

	# Child process. Disable signals.
	foreach $sig ( "HUP", "INT", "QUIT" ) {
	    $SIG{$sig} = "IGNORE";
	}

	# Fork another child to do the job.
	if ( fork == 0 ) {
	    # Execute command. No way to signal failure.
	    exec $cmd;
	    exit (0);
	}

    }

    # Wait for first child to complete. 
    # This assures that the signals are armed before the parent can do
    # harmful things.
    waitpid ($pid, 0);
}

sub check_uucp_name {
    local ($host, $silent) = @_;
    $host = $` if $host =~ /\.uucp/i;	# strip .UUCP
    return 1 if $host eq $h_uuhost; 	# already verified
    return 1 unless $uuname ne "";
    open ( UUNAME, $uuname . "|" );
    local (@hosts) = <UUNAME>;
    close (UUNAME);
    @found = grep ( /^$host$/, @hosts );
    return 1 if @found == 1;
    &errmsg ("Unknown UUCP system name: \"$host\"") unless $silent;
    $opt_debug;
}

sub check_uucp_path {
    local ($path) = @_;
    # $path should start with slash or tilde.
    return 1 if $path =~ /^[\/~]/;
    &errmsg ("Invalid UUCP path name: \"$path\"");
    0;
}

sub options {
    require "newgetopt.pl";
    local ($opt_noi, $opt_nointeractive);
    $opt_debug = $opt_trace = $opt_nomail = $opt_noqueue = $opt_help = 0;
    if ( !&NGetOpt ("config=s", "trace_headers", "interactive", "i0",
		    "nointeractive", "noi",
		    "debug", "trace", "noqueue", "nomail", "help")
	|| $opt_help
	|| (@ARGV > 0 && !($opt_debug || $opt_trace || $opt_nomail))) {
	&usage;
    }
    $config_file = $opt_config if defined $opt_config;
    $opt_interactive = 0 if defined $opt_noi || defined $opt_nointeractive;

}

sub usage {
    require "ms_common.pl";
    print STDERR <<EndOfUsage;
$my_package [$my_name $my_version]

Usage: $my_name [options] < mail-message

Options:
    -config XX	load this config file instead of ms_config.pl
    -help	this message
    -interactive interactively read commands, and execute them
    -nointeractive read an email message, even from terminal
    -noqueue	process message, but do not enter request in the queue
    -nomail	do not reply by email (testing only)
    -debug	for debugging
    -trace	for debugging
    -trace_headers	for debugging

'mail-message' should be RFC-822 conformant.
EndOfUsage
    exit (1);
}

format list_header =

     Date       Size  @<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
$list_type . ": " . $query
  ----------  ------  ----------------------------
.
format list_format =
  @<<<<<<<<< @>>>>>>  ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
$date, $size, $name
~~                      ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
$name
.

sub help {
    require 'pr_help.pl';
    &do_help;
    &include ($hintsfile) if $interactive;
}

sub add_help {
    # For user extensions, so they can give help too.
    local ($cmd, @text) = @_;
    @ext_help = () unless defined @ext_help;
    push (@ext_help, "+$cmd", @text);
}
    
1;
