#!/usr/bin/perl

$[ = 0;	# let arrays start at 0];

# BEGIN module rfc822_read
#
# Copyright 1989 Johan Vromans
#
# This software may be redistributed on the same terms as the 
# GNU Public Licence.
#
# Private variables:
#
	$rfc822_line_in_cache = "";
#
# Global variables
#
	$rfc822_version = "@(#)@ rfc822	1.2 - rfc822.pl";
#
	$rfc822_input = 0;
	$rfc822_line = "";
	$rfc822_header = "";
	$rfc822_contents = "";
	@rfc822_addresses = ();
	%rfc822_addr_comments = ();
#
# Global constants
#
	$rfc822_valid_header = 0;
	$rfc822_empty_line = 1;
	$rfc822_data_line = 2;
#
# Exported routines
#
#	rfc822_start_read	initializes this module
#
#		must be passed the filename to read from
#
#	rfc822_read_header	reads, and parses RFC822 header
#
#		returns $rfc822_valid_header if a valid RFC822 header was found.
#		$rfc822_header and $rfc822_contents contain the header and contents.
#		$rfc822_line contains the normalized header.
#		If called with a non-null argument: does not handle continuation lines.
#		(for reading the body of a message).
#
#		returns $rfc822_empty_line if an empty line was read.
#
#		returns $rfc822_data_line otherwise.
#		$rfc822_line contains the contents of the line.
#
#	rfc822_not_eof	the read-ahead systems notion of end-of-file
#
#	rfc822_parse_addresses	parses an address specification.
#
#		return addresses in @rfc822_addresses, the address
#		comments in %rfc822_addr_comments.
#

sub rfc822_start_read {
  local ($rfc822_file) = @_;
  if ( $rfc822_input != 0 ) {
    close (rfc822_input);
    $rfc822_input = 1;
  }
  if (open (rfc822_input, $rfc822_file)) {
    # read ahead to make eof work properly
    $rfc822_line_in_cache = <rfc822_input>;
    return 1;
  } else { return 0; }
}

sub rfc822_not_eof {
  return (($rfc822_line_in_cache ne "") || (!eof (rfc822_input)));
}

sub rfc822_read_header {

  local ($line);
  local ($body) = shift (@_);

  if ( $rfc822_line_in_cache ne "" ) {
    $line = $rfc822_line_in_cache;
    $rfc822_line_in_cache = "";
    chop ($line);
  } else {
    $line = <rfc822_input>;
    chop ( $line );
  }

  if ( $line =~ /^([-\w]+)\s*:\s*/ ) {
    $rfc822_header = $1;
    $rfc822_contents = $';			#';
  } else {
    $rfc822_line = $line;
    $rfc822_header = $rfc822_contents = "";
    return $rfc822_empty_line if $rfc822_line eq "";
    return $rfc822_data_line;
  }

  # handle continuation lines
  while ( ($rfc822_line_in_cache eq "") && (! eof) ) {
    $rfc822_line_in_cache = $line = <rfc822_input>;
    chop ($line);
    if ( !$body && ($line =~ /^\s+/) ) {
      $rfc822_contents = $rfc822_contents . " " . $';		#';
      $rfc822_line_in_cache = "";
    }
  }
  $rfc822_line = $rfc822_header . ": " . $rfc822_contents;
  return $rfc822_valid_header;
}

sub rfc822_parse_addresses {
  #
  # syntax:  address [, address ...]
  #
  #    address: addr [ ( comment ) ] | [ comment ] <addr>
  #

  local ($addr) = shift (@_);
  local ($left);
  local (@left);
  local ($right);
  local ($comment);

  @rfc822_addresses = ();
  %rfc822_addr_comments = ();

  # first break out the (...) comments
  while ( $addr =~ /\(([^)]*)\)/ ) {
    $right = $';
    $comment = $1;
    @left = split (/[ \t]+/, $`);
    if ( $#left >= 0 ) {
      # print "() match: \"", $left[$#left], "\" -> \"$1\"\n";
      unshift (@rfc822_addresses, pop (@left));
      $rfc822_addr_comments{$rfc822_addresses[0]} = $1;
    }
    if ( $right =~ /^\s*,\s*/ ) {
      $right = $';
    }
    $addr = join (" ", @left) . " " . $right;
    # print "todo: $addr\n";
  }

  # then split on commas, and handle each part separately
  @addr = split (/,/, $addr);

  while ( $#addr >= 0 ) {
    $addr = shift (@addr);
    # print "doing: \"$addr\"\n";
    $addr = $' if $addr =~ /^\s+/ ;
    $addr = $` if $addr =~ /\s+$/ ;
    next if $addr eq "";
    if ( $addr =~ /<([^>]+)>/ ) {
      # print "\"$addr\" matched: \"$`\"-\"$+\"-\"$'\"\n";
      unshift (@rfc822_addresses, $1);
      $rfc822_addr_comments{$1} = join (" ", split (/[ \t]+/, "$` $'"));
    }
    else {
      unshift (@rfc822_addresses, $addr);
      $rfc822_addr_comments{$addr} = "";
      # print "did: \"$addr\"\n";
    }
  }

}
# END module rfc822_read
