
/*
 * mhn.c -- display, list, or store the contents of MIME message
 *
 * $Id$
 */

#include <h/mh.h>
#include <fcntl.h>
#include <h/signals.h>
#include <h/md5.h>
#include <errno.h>
#include <setjmp.h>
#include <signal.h>
#include <zotnet/mts/mts.h>
#include <zotnet/tws/tws.h>
#include <h/mhn.h>

#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

static struct swit switches[] = {
#define	AUTOSW                  0
    { "auto", 0 },
#define	NAUTOSW                 1
    { "noauto", 0 },
#define	CACHESW                 2
    { "cache", 0 },
#define	NCACHESW                3
    { "nocache", 0 },
#define	CHECKSW                 4
    { "check", 0 },
#define	NCHECKSW                5
    { "nocheck", 0 },
#define	HEADSW                  6
    { "headers", 0 },
#define	NHEADSW                 7
    { "noheaders", 0 },
#define	LISTSW                  8
    { "list", 0 },
#define	NLISTSW                 9
    { "nolist", 0 },
#define	PAUSESW                10
    { "pause", 0 },
#define	NPAUSESW               11
    { "nopause", 0 },
#define	SIZESW                 12
    { "realsize", 0 },
#define	NSIZESW                13
    { "norealsize", 0 },
#define	SERIALSW               14
    { "serialonly", 0 },
#define	NSERIALSW              15
    { "noserialonly", 0 },
#define	SHOWSW                 16
    { "show", 0 },
#define	NSHOWSW                17
    { "noshow", 0 },
#define	STORESW                18
    { "store", 0 },
#define	NSTORESW               19
    { "nostore", 0 },
#define	VERBSW                 20
    { "verbose", 0 },
#define	NVERBSW                21
    { "noverbose", 0 },
#define	FILESW                 22	/* interface from show */
    { "file file", 0 },
#define	FORMSW                 23
    { "form formfile", 0 },
#define	PARTSW                 24
    { "part number", 0 },
#define	TYPESW                 25
    { "type content", 0 },
#define	RCACHESW               26
    { "rcache policy", 0 },
#define	WCACHESW               27
    { "wcache policy", 0 },
#define VERSIONSW              28
    { "version", 0 },
#define	HELPSW                 29
    { "help", 4 },

/*
 * switches for debugging
 */
#define	DEBUGSW                30
    { "debug", -5 },

/*
 * switches for moreproc/mhlproc
 */
#define	PROGSW                 31
    { "moreproc program", -4 },
#define	NPROGSW                32
    { "nomoreproc", -3 },
#define	LENSW                  33
    { "length lines", -4 },
#define	WIDTHSW                34
    { "width columns", -4 },

/*
 * switches for mhbuild
 */
#define BUILDSW                35
    { "build", -5 },
#define NBUILDSW               36
    { "nobuild", -7 },
#define	EBCDICSW               37
    { "ebcdicsafe", -10 },
#define	NEBCDICSW              38
    { "noebcdicsafe", -12 },
#define	RFC934SW               39
    { "rfc934mode", -10 },
#define	NRFC934SW              40
    { "norfc934mode", -12 },

/*
 * switches for viamail
 */
#define	VIAMSW                 41
    { "viamail mailpath", -7 },
#define	VIASSW                 42
    { "viasubj subject", -7 },
#define	VIAPSW                 43
    { "viaparm arguments", -7 },
#define	VIADSW                 44
    { "viadesc text", -7 },
#define	VIACSW                 45
    { "viacmnt text", -7 },
#define	VIAZSW                 46
    { "viadelay seconds", -8 },
#define	VIAFSW                 47
    { "viafrom mailpath", -7 },
    { NULL, 0 }
};

#define	NPARTS	50
#define	NTYPES	20

static struct swit caches[] = {
#define	CACHE_NEVER    0
    { "never", 0 },
#define	CACHE_PRIVATE  1
    { "private", 0 },
#define	CACHE_PUBLIC   2
    { "public", 0 },
#define	CACHE_ASK      3
    { "ask", 0 },
    { NULL, 0 }
};

extern int errno;

int debugsw  = 0;
int verbosw  = 0;

/*
 * variables for mhbuild (mhn -build)
 */
static int buildsw  = 0;
static int ebcdicsw = 0;
static int rfc934sw = 0;

static int autosw   = 0;
static int cachesw  = 0;
static int checksw  = 0;
static char *formsw = NULL;
static int headsw   = 1;
static int listsw   = 0;
static int nolist   = 0;
static int nomore   = 0;
static int npart    = 0;
static char *parts[NPARTS + 1];
static int pausesw  = 1;
static char *progsw = NULL;
static int rcachesw = CACHE_ASK;
static int serialsw = 0;
static int showsw   = 0;
static int sizesw   = 1;
static int storesw  = 0;
static int ntype    = 0;
static char *types[NTYPES + 1];
static int wcachesw = CACHE_ASK;

static int endian   = 0;

static pid_t xpid   = 0;
static int userrs   = 0;

static char *cache_public;
static char *cache_private;
static int cwdlen;
static char *cwd;
static char *errs = NULL;
static char *dir;
static char *tmp;

/* various formats for -list option */
#define	LSTFMT1		"%4s %-5s %-24s %5s %-36s\n"
#define	LSTFMT2a	"%4d "
#define	LSTFMT2b	"%-5s %-24.24s "
#define	LSTFMT2c1	"%5lu"
#define	LSTFMT2c2	"%4lu%c"
#define	LSTFMT2c3	"huge "
#define	LSTFMT2c4	"     "
#define	LSTFMT2d1	" %-36.36s"
#define	LSTFMT2d2	"\t     %-65.65s\n"

#define	NPARMS	10

/*
 * ABSTRACT TYPES FOR MHN
 */
typedef struct cefile  *CE;
typedef struct CTinfo  *CI;
typedef struct Content *CT;

/*
 * Structure for storing parsed elements
 * of the Content-Type component.
 */
struct CTinfo {
    char *ci_type;
    char *ci_subtype;
    char *ci_attrs[NPARMS + 2];
    char *ci_values[NPARMS];
    char *ci_comment;
    char *ci_magic;
};

/*
 * Structure for storing information related
 * to Content-Transfer-Encoding handling.
 */
struct cefile {
    char *ce_file;
    FILE *ce_fp;
    int	  ce_unlink;
};

/*
 * flags for Content->c_type
 */
#define	CT_UNKNOWN	0x00
#define	CT_APPLICATION	0x01
#define	CT_AUDIO	0x02
#define	CT_IMAGE	0x03
#define	CT_MESSAGE	0x04
#define	CT_MULTIPART	0x05
#define	CT_TEXT		0x06
#define	CT_VIDEO	0x07
#define	CT_EXTENSION	0x08

/*
 * flags for Content->c_encoding
 */
#define	CE_UNKNOWN	0x00
#define	CE_BASE64	0x01
#define	CE_QUOTED	0x02
#define	CE_8BIT		0x03
#define	CE_7BIT		0x04
#define	CE_BINARY	0x05
#define	CE_EXTENSION	0x06
#define	CE_EXTERNAL	0x07	/* for external-body */

/*
 * type for Init function (both type and encoding)
 */
typedef int (*InitFunc) (CT);

/*
 * types for Content-Type access functions
 */
typedef int (*ListFunc) (CT, int);
typedef int (*ShowFunc) (CT, int, int);
typedef int (*StoreFunc) (CT, char *);
typedef void (*FreeFunc) (CT);

/*
 * types for various encoding access functions
 */
typedef int (*OpenCEFunc) (CT, char **);
typedef void (*CloseCEFunc) (CT);
typedef int (*ListCEFunc) (CT);
typedef unsigned long (*SizeCEFunc) (CT);
typedef void (*FreeCEFunc) (CT, int);

/*
 * Primary structure for handling Content (Entity)
 */
struct Content {
    char *c_partno;		/* within multipart content          */
    char *c_vrsn;		/* Body-Version:                     */

    char *c_ctline;		/* Content-Type:                     */
    int	c_type;			/* internal form/flag for type       */
    int	c_subtype;		/* filled-in by c_ctinitfnx          */
    struct CTinfo c_ctinfo;	/* parsed elements of Content-Type   */

    void *c_ctparams;		/* content type specific data        */
    struct exbody *c_ctexbody;	/*   ..                              */

    char *c_showproc;		/* default, if not in profile        */
    char *c_termproc;		/* for charset madness...            */
    char *c_storeproc;		/* overrides profile entry, if any   */

    InitFunc  c_ctinitfnx;	/* parse content                     */
    ListFunc  c_ctlistfnx;	/* list content                      */
    ShowFunc  c_ctshowfnx;	/* show content                      */
    StoreFunc c_ctstorefnx;	/* store content                     */
    FreeFunc  c_ctfreefnx;	/* free content-specific structures  */

    char *c_celine;		/* Content-Transfer-Encoding:        */
    int	c_encoding;		/* internal form                     */
    CE c_cefile;		/* filled-in by encoding initfnx     */

    OpenCEFunc  c_ceopenfnx;	/* get a stream to decoded contents  */
    CloseCEFunc c_ceclosefnx;	/* release stream                    */
    ListCEFunc  c_celistfnx;	/* list decoding info                */
    SizeCEFunc  c_cesizefnx;	/* size of decoded contents          */
    FreeCEFunc  c_cefreefnx;	/* free encoding-specific structures */

    char *c_id;			/* Content-ID:                       */
    char *c_descr;		/* Content-Description:              */
    int	c_digested;		/* Content-MD5:                      */
    unsigned char c_digest[16];	/*   ..                              */
    FILE *c_fp;			/* read contents (stream)            */
    char *c_file;		/* read contents (file)              */
    int	c_unlink;		/* remove file when done?            */
    int	c_umask;		/* associated umask                  */
    long c_begin;		/* where content starts in file      */
    long c_end;			/* where content ends in file        */
    pid_t c_pid;		/* process doing display             */
    char *c_storage;		/* write contents (file)             */
    int	c_rfc934;		/* rfc934 compatibility              */
};

/*
 * The main list of contents to display
 */
static CT *cts = NULL;


/*
 * Structure for mapping types to their internal flags
 */
struct k2v {
    char *kv_key;
    int	  kv_value;
};


/*
 * Structures for TEXT messages
 */
#define	TEXT_UNKNOWN	0x00
#define	TEXT_PLAIN	0x01
#define	TEXT_RICHTEXT	0x02
#define TEXT_ENRICHED	0x03

#define	CHARSET_UNKNOWN	    0x00
#define CHARSET_UNSPECIFIED 0x01  /* only needed when building drafts */
#define	CHARSET_USASCII	    0x01
#define	CHARSET_LATIN	    0x02

struct text {
    int	tx_charset;
};

static struct k2v SubText[] = {
    { "plain",    TEXT_PLAIN },
    { "richtext", TEXT_RICHTEXT },  /* defined in RFC-1341    */
    { "enriched", TEXT_ENRICHED },  /* defined in RFC-1896    */
    { NULL,       TEXT_UNKNOWN }    /* this one must be last! */
};

static struct k2v Charset[] = {
    { "us-ascii",   CHARSET_USASCII },
    { "iso-8859-1", CHARSET_LATIN },
    { NULL,         CHARSET_UNKNOWN }  /* this one must be last! */
};


/*
 * Structures for MULTIPART messages
 */
#define	MULTI_UNKNOWN	0x00
#define	MULTI_MIXED	0x01
#define	MULTI_ALTERNATE	0x02
#define	MULTI_DIGEST	0x03
#define	MULTI_PARALLEL	0x04

struct part {
    CT mp_part;
    struct part *mp_next;
};

struct multipart {
    char *mp_start;
    char *mp_stop;
    struct part *mp_parts;
};

static struct k2v SubMultiPart[] = {
    { "mixed",       MULTI_MIXED },
    { "alternative", MULTI_ALTERNATE },
    { "digest",      MULTI_DIGEST },
    { "parallel",    MULTI_PARALLEL },
    { NULL,          MULTI_UNKNOWN }    /* this one must be last! */
};


/*
 * Structures for MESSAGE messages
 */
#define	MESSAGE_UNKNOWN	 0x00
#define	MESSAGE_RFC822	 0x01
#define	MESSAGE_PARTIAL	 0x02
#define	MESSAGE_EXTERNAL 0x03

struct partial {
    char *pm_partid;
    int	pm_partno;
    int	pm_maxno;
    int	pm_marked;
    int	pm_stored;
};

static struct k2v SubMessage[] = {
    { "rfc822",        MESSAGE_RFC822 },
    { "partial",       MESSAGE_PARTIAL },
    { "external-body", MESSAGE_EXTERNAL },
    { NULL,            MESSAGE_UNKNOWN }		/* this one must be last! */
};

/*
 * Structure for message/external bodies
 */
struct exbody {
    CT eb_parent;
    CT eb_content;
    char *eb_partno;
    char *eb_access;
    int	eb_flags;
    char *eb_name;
    char *eb_permission;
    char *eb_site;
    char *eb_dir;
    char *eb_mode;
    unsigned long eb_size;
    char *eb_server;
    char *eb_subject;
    char *eb_body;
};


/*
 * Structure for APPLICATION messages
 */
#define	APPLICATION_UNKNOWN	0x00
#define	APPLICATION_OCTETS	0x01
#define	APPLICATION_POSTSCRIPT	0x02

static struct k2v SubApplication[] = {
    { "octet-stream", APPLICATION_OCTETS },
    { "postscript",   APPLICATION_POSTSCRIPT },
    { NULL,           APPLICATION_UNKNOWN }	/* this one must be last! */
};


/*
 * Type for a compare function for qsort.  This keeps
 * the compiler happy.
 */
typedef int (*qsort_comp) (const void *, const void *);

#define	quitser	pipeser

/*
 * prototypes
 */
int SOprintf (char *, ...);   /* from termsbr.c */

/*
 * static prototypes
 */
void DisplayMsgHeader (CT);
static RETSIGTYPE pipeser (int);
static CT get_content (FILE *, char *, int);
static int get_ctinfo (char *, CT);
static int get_comment (CT, char **, int);
static int list_content (CT, int);
static void content_error (char *, CT, char *, ...);
static void flush_errors (void);
static RETSIGTYPE intrser (int);
static int show_content (CT, int, int);
static int show_content_aux (CT, int, int, char *, char *);
static int show_content_aux2 (CT, int, int, char *, char *, int, int, int, int, int);
static int store_content (CT, char *);
static int copy_some_headers (FILE *, CT);
static int make_intermediates (char *);
static void free_ctinfo (CT);
static void free_content (CT);
static int part_ok (CT, int);
static int type_ok (CT, int);
static int InitGeneric (CT);
static int InitText (CT);
static int show_text (CT, int, int);
static void free_text (CT);
static int InitMultiPart (CT);
static int list_multi (CT, int);
static int show_multi (CT, int, int);
static int show_multi_internal (CT, int, int);
static int show_multi_aux (CT, int, int, char *);
static int store_multi (CT, char *);
static void free_multi (CT);
static int InitMessage (CT);
static int show_message (CT, int, int);
static int list_partial (CT, int);
static int ct_compar (CT *, CT *);
static int store_partial (CT, char *);
static void free_partial (CT);
static int params_external (CT, int);
static int list_external (CT, int);
static int show_external (CT, int, int);
static int store_external (CT, char *);
static void free_external (CT);
static int InitApplication (CT);
static int list_application (CT, int);
static int init_encoding (CT, OpenCEFunc);
static int list_encoding (CT);
static void close_encoding (CT);
static unsigned long size_encoding (CT);
static void free_encoding (CT, int);
static int InitBase64 (CT);
static int openBase64 (CT, char **);
static void set_endian (void);
static int InitQuoted (CT);
static int openQuoted (CT, char **);
static int Init7Bit (CT);
static int open7Bit (CT, char **);
static int openExternal (CT, CT, CE, char **, int *);
static int InitFile (CT);
static int openFile (CT, char **);
static int InitFTP (CT);
static int openFTP (CT, char **);
static int InitMail (CT);
static int openMail (CT, char **);
static int find_cache (CT, int, int *, char *, char *);
static int find_cache_aux (int, char *, char *, char *);
static int find_cache_aux2 (char *, char *, char *);
static void cache_content (CT);
static int writeBase64aux (FILE *, FILE *);
static int readDigest (CT, char *);
static int via_mail (char *, char *, char *, char *, char *, int, char *);
static int via_post (char *, int);
static int pidcheck (int);


/*
 * Structures for mapping (content) types to
 * the functions to handle them.
 */

struct str2init {
    char *si_key;
    int	  si_val;
    InitFunc si_init;
};

static struct str2init str2cts[] = {
    { "application", CT_APPLICATION, InitApplication },
    { "audio",	     CT_AUDIO,	     InitGeneric },
    { "image",	     CT_IMAGE,	     InitGeneric },
    { "message",     CT_MESSAGE,     InitMessage },
    { "multipart",   CT_MULTIPART,   InitMultiPart },
    { "text",	     CT_TEXT,	     InitText },
    { "video",	     CT_VIDEO,	     InitGeneric },
    { NULL,	     CT_EXTENSION,   NULL },  /* these two must be last! */
    { NULL,	     CT_UNKNOWN,     NULL },
};

static struct str2init str2ces[] = {
    { "base64",		  CE_BASE64,	InitBase64 },
    { "quoted-printable", CE_QUOTED,	InitQuoted },
    { "8bit",		  CE_8BIT,	Init7Bit },
    { "7bit",		  CE_7BIT,	Init7Bit },
    { "binary",		  CE_BINARY,	NULL },
    { NULL,		  CE_EXTENSION,	NULL },	 /* these two must be last! */
    { NULL,		  CE_UNKNOWN,	NULL },
};

/*
 * NOTE WELL: si_key MUST NOT have value of NOTOK
 *
 * si_key is 1 if access method is anonymous.
 */
static struct str2init str2methods[] = {
    { "afs",         1,	InitFile },
    { "anon-ftp",    1,	InitFTP },
    { "ftp",         0,	InitFTP },
    { "local-file",  0,	InitFile },
    { "mail-server", 0,	InitMail },
    { NULL,          0, NULL }
};


int
main (int argc, char **argv)
{
    int f6 = 0, msgp = 0, msgnum, *icachesw;
    char *cp, *f1 = NULL, *f2 = NULL, *f3 = NULL;
    char *f4 = NULL, *f5 = NULL, *f7 = NULL, *file = NULL;
    char *folder = NULL, *maildir, buf[100], **ap;
    char **argp, *arguments[MAXARGS], *msgs[MAXARGS];
    struct msgs *mp;
    CT ct, *ctp;
    FILE *fp;

#ifdef LOCALE
    setlocale(LC_ALL, "");
#endif
    invo_name = r1bindex (argv[0], '/');

    /*
     * If using viamail, then foil search
     * of user profile/context
     */
    if (argv[1] && uprf (argv[1], "-via")) {
	if (context_foil (NULL) == -1)
	    done (1);
    }
    if ((cp = context_find (invo_name))) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    } else {
	ap = arguments;
    }
    copyip (argv + 1, ap);
    argp = arguments;

    while ((cp = *argp++)) {
	if (*cp == '-') {
	    switch (smatch (++cp, switches)) {
	    case AMBIGSW: 
		ambigsw (cp, switches);
		done (1);
	    case UNKWNSW: 
		adios (NULL, "-%s unknown", cp);

	    case HELPSW: 
		sprintf (buf, "%s [+folder] [msgs] [switches]", invo_name);
		print_help (buf, switches, 1);
		done (1);
	    case VERSIONSW:
		print_version(invo_name);
		done (1);

	    case AUTOSW:
		autosw++;
		continue;
	    case NAUTOSW:
		autosw = 0;
		continue;

	    case CACHESW:
		cachesw++;
		continue;
	    case NCACHESW:
		cachesw = 0;
		continue;

	    case RCACHESW:
		icachesw = &rcachesw;
		goto do_cache;
	    case WCACHESW:
		icachesw = &wcachesw;
	    do_cache: ;
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		switch (*icachesw = smatch (cp, caches)) {
		case AMBIGSW:
		    ambigsw (cp, caches);
		    done (1);
		case UNKWNSW:
		    adios (NULL, "%s unknown", cp);
		default:
		    break;
		}
		continue;

	    case CHECKSW:
		checksw++;
		continue;
	    case NCHECKSW:
		checksw = 0;
		continue;

	    case HEADSW:
		headsw++;
		continue;
	    case NHEADSW:
		headsw = 0;
		continue;

	    case LISTSW:
		listsw++;
		continue;
	    case NLISTSW:
		listsw = 0;
		continue;

	    case PAUSESW:
		pausesw++;
		continue;
	    case NPAUSESW:
		pausesw = 0;
		continue;

	    case SERIALSW:
		serialsw++;
		continue;
	    case NSERIALSW:
		serialsw = 0;
		continue;

	    case SHOWSW:
		showsw++;
		continue;
	    case NSHOWSW:
		showsw = 0;
		continue;

	    case SIZESW:
		sizesw++;
		continue;
	    case NSIZESW:
		sizesw = 0;
		continue;

	    case STORESW:
		storesw++;
		continue;
	    case NSTORESW:
		storesw = 0;
		continue;

	    case VERBSW: 
		verbosw++;
		continue;
	    case NVERBSW: 
		verbosw = 0;
		continue;

	    case PARTSW:
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		if (npart >= NPARTS)
		    adios (NULL, "too many parts (starting with %s), %d max",
			   cp, NPARTS);
		parts[npart++] = cp;
		continue;

	    case TYPESW:
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		if (ntype >= NTYPES)
		    adios (NULL, "too many types (starting with %s), %d max",
			   cp, NTYPES);
		types[ntype++] = cp;
		continue;

	    case FILESW:
		if (!(cp = *argp++) || (*cp == '-' && cp[1]))
		    adios (NULL, "missing argument to %s", argp[-2]);
		file = *cp == '-' ? cp : path (cp, TFILE);
		continue;

	    case FORMSW:
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		if (formsw)
		    free (formsw);
		formsw = getcpy (etcpath (cp));
		continue;

	    case DEBUGSW:
		debugsw++;
		continue;
    
	    /*
	     * Switches for moreproc/mhlproc
	     */
	    case PROGSW:
		if (!(progsw = *argp++) || *progsw == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case NPROGSW:
		nomore++;
		continue;

	    case LENSW:
	    case WIDTHSW:
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;

	    /*
	     * Switches for mhbuild
	     */
	    case BUILDSW:
		buildsw++;
		continue;
	    case NBUILDSW:
		buildsw = 0;
		continue;
	    case RFC934SW:
		rfc934sw = 1;
		continue;
	    case NRFC934SW:
		rfc934sw = -1;
		continue;
	    case EBCDICSW:
		ebcdicsw = 1;
		continue;
	    case NEBCDICSW:
		ebcdicsw = -1;
		continue;

	    /*
	     * Switches for viamail
	     */
	    case VIAMSW:
		if (!(f1 = *argp++))
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case VIASSW:
		if (!(f2 = *argp++))
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case VIAPSW:
		if (!(f3 = *argp++))
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case VIADSW:
		if (!(f4 = *argp++))
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case VIACSW:
		if (!(f5 = *argp++))
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    case VIAZSW:
		if (!(cp = *argp++) || *cp == '-')
		    adios (NULL, "missing argument to %s", argp[-2]);
		if (sscanf (cp, "%d", &f6) != 1 || f6 < 0)
		    f6 = 300;
		continue;
	    case VIAFSW:
		if (!(f7 = *argp++))
		    adios (NULL, "missing argument to %s", argp[-2]);
		continue;
	    }
	}
	if (*cp == '+' || *cp == '@') {
	    if (folder)
		adios (NULL, "only one folder at a time!");
	    else
		folder = path (cp + 1, *cp == '+' ? TFOLDER : TSUBCWF);
	} else {
	    msgs[msgp++] = cp;
	}
    }

    parts[npart] = NULL;
    types[ntype] = NULL;

    set_endian ();

    if (f1) {
	via_mail (f1, f2, f3, f4, f5, f6, f7);
	/* NOTREACHED */
    } else {
	if (f2 || f3 || f4 || f5 || f6 || f7)
	    adios (NULL, "missing -viamail \"mailpath\" switch");
    }

    /*
     * Check if we've specified an additional profile
     */
    if ((cp = getenv ("MHN"))) {
	if ((fp = fopen (cp, "r"))) {
	    m_readefs ((struct node **) 0, fp, cp, 0);
	    fclose (fp);
	} else {
	    admonish ("", "unable to read $MHN profile (%s)", cp);
	}
    }

    /*
     * Read the standard profile setup
     */
    if ((fp = fopen (cp = etcpath ("mhn.defaults"), "r"))) {
	m_readefs ((struct node **) 0, fp, cp, 0);
	fclose (fp);
    }

    /* Check for public cache location */
    sprintf (buf, "%s-cache", invo_name);
    if ((cache_public = context_find (buf)) && *cache_public != '/')
	cache_public = NULL;

    /* Check for private cache location */
    sprintf (buf, "%s-private-cache", invo_name);
    if (!(cache_private = context_find (buf)))
	cache_private = ".cache";
    cache_private = getcpy (m_maildir (cache_private));

    cwd = getcpy (pwd());
    cwdlen = strlen (cwd);

    /*
     * Check for storage directory.  If no profile
     * entry, then use current directory.
     */
    sprintf (buf, "%s-storage", invo_name);
    dir = getcpy ((cp = context_find (buf)) && *cp ? cp : cwd);
    tmp = cp && *cp ? concat (cp, "/", invo_name, NULL)
	: add (m_maildir (invo_name), NULL);

    if (!context_find ("path"))
	free (path ("./", TFOLDER));

    /*
     * Process a mhn composition file (mhn -build)
     */
    if (buildsw) {
	char *vec[MAXARGS];
	int vecp;

	if (showsw || storesw || cachesw)
	    adios (NULL, "cannot use -build with -show, -store, -cache");
	if (msgp < 1)
	    adios (NULL, "need to specify a %s composition file", invo_name);
	if (msgp > 1)
	    adios (NULL, "only one %s composition file at a time", invo_name);

	vecp = 0;
	vec[vecp++] = "mhbuild";

	if (ebcdicsw == 1)
	    vec[vecp++] = "-ebcdicsafe";
	else if (ebcdicsw == -1)
	    vec[vecp++] = "-noebcdicsafe";

	if (rfc934sw == 1)
	    vec[vecp++] = "-rfc934mode";
	else if (rfc934sw == -1)
	    vec[vecp++] = "-norfc934mode";

	vec[vecp++] = msgs[0];
	vec[vecp] = NULL;

	execvp ("mhbuild", vec);
	fprintf (stderr, "unable to exec ");
	_exit (-1);
    }

    /*
     * Process a mhn composition file (old MH style)
     */
    if (msgp == 1 && !folder && !npart && !cachesw
	&& !showsw && !storesw && !ntype && !file
	&& (cp = getenv ("mhdraft"))
	&& strcmp (cp, msgs[0]) == 0) {

	char *vec[MAXARGS];
	int vecp;

	vecp = 0;
	vec[vecp++] = "mhbuild";

	if (ebcdicsw == 1)
	    vec[vecp++] = "-ebcdicsafe";
	else if (ebcdicsw == -1)
	    vec[vecp++] = "-noebcdicsafe";

	if (rfc934sw == 1)
	    vec[vecp++] = "-rfc934mode";
	else if (rfc934sw == -1)
	    vec[vecp++] = "-norfc934mode";

	vec[vecp++] = cp;
	vec[vecp] = NULL;

	execvp ("mhbuild", vec);
	fprintf (stderr, "unable to exec ");
	_exit (-1);
    }

    if (file && msgp)
	adios (NULL, "cannot specify msg and file at same time!");

    /*
     * message is coming from file
     */
    if (file) {
	int stdinP;

	if ((cts = (CT *) calloc ((size_t) 2, sizeof(*cts))) == NULL)
	    adios (NULL, "out of memory");

	ctp = cts;
	if ((stdinP = (strcmp (file, "-") == 0))) {
	    char buffer[BUFSIZ];

	    file = add (m_tmpfil (invo_name), NULL);

	    if ((fp = fopen (file, "w+")) == NULL)
		adios (file, "unable to fopen for writing and reading");
	    chmod (file, 0600);
	    while (fgets (buffer, sizeof(buffer), stdin))
		fputs (buffer, fp);
	    fflush (fp);

	    if (ferror (stdin)) {
		unlink (file);
		adios ("stdin", "error reading");
	    }

	    if (ferror (fp)) {
		unlink (file);
		adios (file, "error writing");
	    }

	    fseek (fp, 0L, SEEK_SET);
	} else {
	    if ((fp = fopen (file, "r")) == NULL)
		adios (file, "unable to read");
	}

	if ((ct = get_content (fp, file, 1))) {
	    if (stdinP)
		ct->c_unlink = 1;

	    ct->c_fp = NULL;
	    if (ct->c_end == 0L) {
		fseek (fp, 0L, SEEK_END);
		ct->c_end = ftell (fp);
	    }
	    if (ct->c_ctinitfnx && (*ct->c_ctinitfnx) (ct) == NOTOK)
		free_content (ct);
	    else
		*ctp++ = ct;
	} else {
	    advise (NULL, "unable to decode %s", file);
	}

	fclose (fp);
	mp = NULL;
    } else {
	/*
	 * message(s) are coming from a folder
	 */
	if (!msgp)
	    msgs[msgp++] = "cur";
	if (!folder)
	    folder = getfolder (1);
	maildir = m_maildir (folder);

	if (chdir (maildir) == NOTOK)
	    adios (maildir, "unable to change directory to");
	if (!(mp = folder_read (folder)))
	    adios (NULL, "unable to read folder %s", folder);
	if (mp->hghmsg == 0)
	    adios (NULL, "no messages in %s", folder);

	/* parse all the message ranges/sequences/names and set SELECTED */
	for (msgnum = 0; msgnum < msgp; msgnum++)
	    if (!m_convert (mp, msgs[msgnum]))
		done (1);
	seq_setprev (mp);	/* set the previous-sequence */

	cts = (CT *) calloc ((size_t) (mp->numsel + 1), sizeof(*cts));
	if (cts == NULL)
	    adios (NULL, "out of memory");

	ctp = cts;
	for (msgnum = mp->lowsel; msgnum <= mp->hghsel; msgnum++) {
	    if (is_selected(mp, msgnum)) {
		char *msgnam;

		if ((fp = fopen (msgnam = m_name (msgnum), "r")) == NULL)
		    adios (msgnam, "unable to read message");

		if ((ct = get_content (fp, msgnam, 1))) {
		    ct->c_fp = NULL;
		    if (ct->c_end == 0L) {
			fseek (fp, 0L, SEEK_END);
			ct->c_end = ftell (fp);
		    }
		    if (ct->c_ctinitfnx && (*ct->c_ctinitfnx) (ct) == NOTOK)
			free_content (ct);
		    else
			*ctp++ = ct;
		} else {
		    advise (NULL, "unable to decode message %s", msgnam);
		}

		fclose (fp);
	    }
	}
    }

    if (!*cts)
	done (1);

    /* if no action is specified, -show */
    if (!listsw && !showsw && !storesw && !cachesw)
	showsw++;

    userrs = 1;
    SIGNAL (SIGQUIT, quitser);
    SIGNAL (SIGPIPE, pipeser);

    for (ctp = cts; *ctp; ctp++) {
	struct stat st;

	ct = *ctp;
	if (type_ok (ct, 1) && (ct->c_ctlistfnx || ct->c_ctstorefnx
		|| ct->c_ctshowfnx) && !ct->c_umask) {
	    if (stat (ct->c_file, &st) != NOTOK)
		ct->c_umask = ~(st.st_mode & 0777);
	    else
		ct->c_umask = ~m_gmprot();
	}
    }

    /*
     * There are several different places in which
     * we handle the listsw option.
     *
     * if (listsw && showsw)
     * user wants per-message listing,
     * so delay until showsw processing
     *
     * if (listsw && storesw && sizesw)
     * evaluating size will cause more work,
     * so delay until after storesw processing
     */

    /*
     * List the message content
     */
    if (listsw && !showsw && (!storesw || !sizesw)) {
	if (headsw)
	    printf (LSTFMT1, "msg", "part", "type/subtype", "size",
		    "description");

	for (ctp = cts; *ctp; ctp++) {
	    ct = *ctp;
	    if (type_ok (ct, 1) && ct->c_ctlistfnx) {
		umask (ct->c_umask);
		(*ct->c_ctlistfnx) (ct, 1);
		if (ct->c_fp) {
		    fclose (ct->c_fp);
		    ct->c_fp = NULL;
		}
		if (ct->c_ceclosefnx)
		    (*ct->c_ceclosefnx) (ct);
	    }
	}

	flush_errors ();
    }

    /*
     * Store the message content
     */
    if (storesw) {
	for (ctp = cts; *ctp; ctp++) {
	    ct = *ctp;
	    if (type_ok (ct, 1) && ct->c_ctstorefnx) {
		umask (ct->c_umask);
		(*ct->c_ctstorefnx) (ct, NULL);
		if (ct->c_fp) {
		    fclose (ct->c_fp);
		    ct->c_fp = NULL;
		}
		if (ct->c_ceclosefnx)
		    (*ct->c_ceclosefnx) (ct);
	    }
	}

	flush_errors ();
    }

    /*
     * Cache the message content
     */
    if (cachesw) {
	for (ctp = cts; *ctp; ctp++) {
	    ct = *ctp;
	    if (type_ok (ct, 1)) {
		cache_content (ct);
		if (ct->c_fp) {
		    fclose (ct->c_fp);
		    ct->c_fp = NULL;
		}
		if (ct->c_ceclosefnx)
		    (*ct->c_ceclosefnx) (ct);
	    }
	}

	flush_errors ();
    }

    /*
     * List the message content (2nd time around)
     */
    if (listsw && !showsw && storesw && sizesw) {
	if (headsw)
	    printf (LSTFMT1, "msg", "part", "type/subtype", "size",
		    "description");

	for (ctp = cts; *ctp; ctp++) {
	    ct = *ctp;
	    if (type_ok (ct, 1) && ct->c_ctlistfnx) {
		umask (ct->c_umask);
		(*ct->c_ctlistfnx) (ct, 1);
		if (ct->c_fp) {
		    fclose (ct->c_fp);
		    ct->c_fp = NULL;
		}
		if (ct->c_ceclosefnx)
		    (*ct->c_ceclosefnx) (ct);
	    }
	}

	flush_errors ();
	listsw = 0;
    }

    /*
     * Show the message content
     */
    if (showsw) {
	for (ctp = cts; *ctp; ctp++) {
	    sigset_t set, oset;

#ifdef WAITINT
	    int status;
#else
	    union wait status;
#endif
	    ct = *ctp;
	    if (!type_ok (ct, 0))
		continue;

	    umask (ct->c_umask);

	    /*
	     * List the message content (right before show)
	     */
	    if (listsw) {
		if (headsw)
		    printf (LSTFMT1, "msg", "part", "type/subtype", "size",
			    "description");

		if (ct->c_ctlistfnx)
		    (*ct->c_ctlistfnx) (ct, 1);
	    }

	    if (!ct->c_ctshowfnx) {
		if (ct->c_fp) {
		    fclose (ct->c_fp);
		    ct->c_fp = NULL;
		}
		if (ct->c_ceclosefnx)
		    (*ct->c_ceclosefnx) (ct);
		continue;
	    }

	    /*
	     * default form for showing headers of MIME messages
	     */
	    if (!formsw)
		formsw = getcpy (etcpath ("mhl.headers"));

	    /*
	     * Show the message headers, unless
	     * the name of format file is "mhl.null".
	     */
	    if (strcmp (formsw, "mhl.null"))
		DisplayMsgHeader(ct);
	    else
		xpid = 0;

	    /* Show the body of the message */
	    (*ct->c_ctshowfnx) (ct, 1, 0);
	    if (ct->c_fp) {
		fclose (ct->c_fp);
		ct->c_fp = NULL;
	    }
	    if (ct->c_ceclosefnx)
		(*ct->c_ceclosefnx) (ct);

	    /* block a few signals */
	    sigemptyset (&set);
	    sigaddset (&set, SIGHUP);
	    sigaddset (&set, SIGINT);
	    sigaddset (&set, SIGQUIT);
	    sigaddset (&set, SIGTERM);
	    SIGPROCMASK (SIG_BLOCK, &set, &oset);

	    while (wait (&status) != NOTOK) {
#ifdef WAITINT
		pidcheck (status);
#else
		pidcheck (status.w_status);
#endif
		continue;
	    }

	    /* reset the signal mask */
	    SIGPROCMASK (SIG_SETMASK, &oset, &set);

	    xpid = 0;
	    flush_errors ();
	}
    }

    /* Now free all the structures for the content */
    for (ctp = cts; *ctp; ctp++)
	free_content (*ctp);

    free ((char *) cts);
    cts = NULL;

    /* If reading from a folder, do some updating */
    if (mp) {
	context_replace (pfolder, folder);/* update current folder  */
	seq_setcur (mp, mp->hghsel);	  /* update current message */
	seq_save (mp);			  /* synchronize sequences  */
	context_save ();		  /* save the context file  */
    }

    done (0);
    /* NOTREACHED */
}


/*
 * Use the mhlproc to show the header fields
 */

void
DisplayMsgHeader (CT ct)
{
    pid_t child_id;
    int i, vecp;
    char *vec[8];

    vecp = 0;
    vec[vecp++] = r1bindex (mhlproc, '/');
    vec[vecp++] = "-form";
    vec[vecp++] = formsw;
    vec[vecp++] = "-nobody";
    vec[vecp++] = ct->c_file;

    /*
     * If we've specified -(no)moreproc,
     * then just pass that along.
     */
    if (nomore) {
	vec[vecp++] = "-nomoreproc";
    } else {
	if (progsw) {
	    vec[vecp++] = "-moreproc";
	    vec[vecp++] = progsw;
	}
    }
    vec[vecp] = NULL;

    fflush (stdout);

    for (i = 0; (child_id = vfork()) == NOTOK && i < 5; i++)
	sleep (5);

    switch (child_id) {
    case NOTOK:
	adios ("fork", "unable to");
	/* NOTREACHED */

    case OK:
	execvp (mhlproc, vec);
	fprintf (stderr, "unable to exec ");
	perror (mhlproc);
	_exit (-1);
	/* NOTREACHED */

    default:
	xpid = -child_id;
	break;
    }
}


static RETSIGTYPE
pipeser (int i)
{
    if (i == SIGQUIT) {
	unlink ("core");
	fflush (stdout);
	fprintf (stderr, "\n");
	fflush (stderr);
    }

    done (1);
    /* NOTREACHED */
}


/*
 * Allocate structure for message content and
 * read the Content-X headers.
 *
 * toplevel =  1   # we are at the top level of the message
 * toplevel =  0   # we are inside message type or multipart type
 *                 # other than multipart/digest
 * toplevel = -1   # we are inside multipart/digest
 */

static CT
get_content (FILE *in, char *file, int toplevel)
{
    int compnum, state;
    char buf[BUFSIZ], name[NAMESZ];
    CT ct;

    if (!(ct = (CT) calloc (1, sizeof(*ct))))
	adios (NULL, "out of memory");

    ct->c_fp = in;
    ct->c_begin = ftell (ct->c_fp) + 1;
    ct->c_file = add (file, NULL);

    /*
     * Read the content headers
     */
    for (compnum = 1, state = FLD;;) {
	switch (state = m_getfld (state, name, buf, sizeof(buf), in)) {
	case FLD:
	case FLDPLUS:
	case FLDEOF:
	    compnum++;

	    /* Get MIME-Version field */
	    if (!strcasecmp (name, VRSN_FIELD)) {
		int ucmp;
		char c, *cp, *dp;

		cp = add (buf, NULL);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    cp = add (buf, cp);
		}

		if (ct->c_vrsn) {
		    advise (NULL, "message %s has multiple %s: fields (%s)",
			    ct->c_file, VRSN_FIELD, dp = trimcpy (cp));
		    free (dp);
		    free (cp);
		    goto out;
		}

		ct->c_vrsn = cp;
		while (isspace (*cp))
		    cp++;
		for (dp = strchr(cp, '\n'); dp; dp = strchr(dp, '\n'))
		    *dp++ = ' ';
		for (dp = cp + strlen (cp) - 1; dp >= cp; dp--)
		    if (!isspace (*dp))
			break;
		*++dp = '\0';
		if (debugsw)
		    fprintf (stderr, "%s: %s\n", VRSN_FIELD, cp);

		if (*cp == '(' && get_comment (ct, &cp, 0) == NOTOK)
		    goto out;

		for (dp = cp; istoken (*dp); dp++)
		    continue;
		c = *dp, *dp = '\0';
		ucmp = !strcasecmp (cp, VRSN_VALUE);
		*dp = c;
		if (!ucmp)
		    admonish (NULL,
			      "message %s has unknown value for %s: field (%s)",
			      ct->c_file, VRSN_FIELD, cp);
		goto got_header;
	    }

	    /* Get Content-Type field */
	    if (!strcasecmp (name, TYPE_FIELD)) {
		char *cp;
		struct str2init *s2i;
		CI ci = &ct->c_ctinfo;

		cp = add (buf, NULL);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    cp = add (buf, cp);
		}

		/* Check if we've already seen a Content-Type header */
		if (ct->c_ctline) {
		    char *dp = trimcpy (cp);

		    advise (NULL, "message %s has multiple %s: fields (%s)",
			    ct->c_file, TYPE_FIELD, dp);
		    free (dp);
		    free (cp);
		    goto out;
		}

		/* Parse the Content-Type field */
		if (get_ctinfo (cp, ct) == NOTOK)
		    goto out;

		/*
		 * Set the Init function and the internal
		 * flag for this content type.
		 */
		for (s2i = str2cts; s2i->si_key; s2i++)
		    if (!strcasecmp (ci->ci_type, s2i->si_key))
			break;
		if (!s2i->si_key && !uprf (ci->ci_type, "X-"))
		    s2i++;
		ct->c_type = s2i->si_val;
		ct->c_ctinitfnx = s2i->si_init;
		goto got_header;
	    }

	    /* Get Content-Transfer-Encoding field */
	    if (!strcasecmp (name, ENCODING_FIELD)) {
		char *cp, *dp;
		char c;
		struct str2init *s2i;

		cp = add (buf, NULL);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    cp = add (buf, cp);
		}

		/*
		 * Check if we've already seen the
		 * Content-Transfer-Encoding field
		 */
		if (ct->c_celine) {
		    advise (NULL, "message %s has multiple %s: fields (%s)",
			    ct->c_file, ENCODING_FIELD, dp = trimcpy (cp));
		    free (dp);
		    free (cp);
		    goto out;
		}

		ct->c_celine = cp;	/* Save copy of this field */
		while (isspace (*cp))
		    cp++;
		for (dp = cp; istoken (*dp); dp++)
		    continue;
		c = *dp;
		*dp = '\0';

		/*
		 * Find the internal flag and Init function
		 * for this transfer encoding.
		 */
		for (s2i = str2ces; s2i->si_key; s2i++)
		    if (!strcasecmp (cp, s2i->si_key))
			break;
		if (!s2i->si_key && !uprf (cp, "X-"))
		    s2i++;
		*dp = c;
		ct->c_encoding = s2i->si_val;

		/* Call the Init function for this encoding */
		if (s2i->si_init && (*s2i->si_init) (ct) == NOTOK)
		    goto out;
		goto got_header;
	    }

	    /* Get Content-ID field */
	    if (!strcasecmp (name, ID_FIELD)) {
		ct->c_id = add (buf, ct->c_id);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    ct->c_id = add (buf, ct->c_id);
		}
		goto got_header;
	    }

	    /* Get Content-Description field */
	    if (!strcasecmp (name, DESCR_FIELD)) {
		ct->c_descr = add (buf, ct->c_descr);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    ct->c_descr = add (buf, ct->c_descr);
		}
		goto got_header;
	    }

	    /* Get Content-MD5 field */
	    if (!strcasecmp (name, MD5_FIELD)) {
		char *cp, *dp, *ep;

		cp = add (buf, NULL);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    cp = add (buf, cp);
		}

		if (!checksw) {
		    free (cp);
		    goto got_header;
		}

		if (ct->c_digested) {
		    advise (NULL, "message %s has multiple %s: fields (%s)",
			    ct->c_file, MD5_FIELD, dp = trimcpy (cp));
		    free (dp);
		    free (cp);
		    goto out;
		}

		ep = cp;
		while (isspace (*cp))
		    cp++;
		for (dp = strchr(cp, '\n'); dp; dp = strchr(dp, '\n'))
		    *dp++ = ' ';
		for (dp = cp + strlen (cp) - 1; dp >= cp; dp--)
		    if (!isspace (*dp))
			break;
		*++dp = '\0';
		if (debugsw)
		    fprintf (stderr, "%s: %s\n", MD5_FIELD, cp);

		if (*cp == '(' && get_comment (ct, &cp, 0) == NOTOK) {
		    free (ep);
		    goto out;
		}

		for (dp = cp; *dp && !isspace (*dp); dp++)
		    continue;
		*dp = '\0';

		readDigest (ct, cp);
		free (ep);
		ct->c_digested++;
		goto got_header;
	    }

#if 0
	    if (uprf (name, XXX_FIELD_PRF))
		advise (NULL, "unknown field (%s) in message %s",
			name, ct->c_file);
	    /* and fall... */
#endif

	    while (state == FLDPLUS)
		state = m_getfld (state, name, buf, sizeof(buf), in);

got_header:;
	    if (state != FLDEOF) {
		ct->c_begin = ftell (in) + 1;
		continue;
	    }
	    /* else fall... */

	case BODY:
	case BODYEOF:
	    ct->c_begin = ftell (in) - strlen (buf);
	    break;

	case FILEEOF:
	    ct->c_begin = ftell (in);
	    break;

	case LENERR:
	case FMTERR:
	    adios (NULL, "message format error in component #%d", compnum);

	default:
	    adios (NULL, "getfld() returned %d", state);
	}
	break;
    }

    /*
     * Check if we saw a Content-Type field.
     * If not, then assign a default value for
     * it, and the Init function.
     */
    if (!ct->c_ctline) {
	/*
	 * If we are inside a multipart/digest message,
	 * so default type is message/rfc822
	 */
	if (toplevel < 0) {
	    if (get_ctinfo ("message/rfc822", ct) == NOTOK)
		goto out;
	    ct->c_type = CT_MESSAGE;
	    ct->c_ctinitfnx = InitMessage;
	} else {
	    /*
	     * Else default type is text/plain
	     */
	    if (get_ctinfo ("text/plain", ct) == NOTOK)
		goto out;
	    ct->c_type = CT_TEXT;
	    ct->c_ctinitfnx = InitText;
	}
    }

    /*
     * Set up default handlers.  Some of these
     * may change when the Init function is called.
     */
    if (!ct->c_ctlistfnx)
	ct->c_ctlistfnx = list_content;
    if (!ct->c_ctshowfnx)
	ct->c_ctshowfnx = show_content;
    if (!ct->c_ctstorefnx)
	ct->c_ctstorefnx = store_content;

    /* Use default Transfer-Encoding, if necessary */
    if (!ct->c_celine) {
	ct->c_encoding = CE_7BIT;
	Init7Bit (ct);
    }

    return ct;

 out:
    free_content (ct);
    return NULL;
}


/*
 * Parse Content-Type line and fill in the
 * information of the CTinfo structure.
 */

static int
get_ctinfo (char *cp, CT ct)
{
    int	i;
    char *dp, **ap, **ep;
    char c;
    CI ci;

    ci = &ct->c_ctinfo;
    i = strlen (invo_name) + 2;

    /* store copy of Content-Type line */
    cp = ct->c_ctline = add (cp, NULL);

    while (isspace (*cp))	/* trim leading spaces */
	cp++;

    /* change newlines to spaces */
    for (dp = strchr(cp, '\n'); dp; dp = strchr(dp, '\n'))
	*dp++ = ' ';

    /* trim trailing spaces */
    for (dp = cp + strlen (cp) - 1; dp >= cp; dp--)
	if (!isspace (*dp))
	    break;
    *++dp = '\0';

    if (debugsw)
	fprintf (stderr, "%s: %s\n", TYPE_FIELD, cp);

    if (*cp == '(' && get_comment (ct, &cp, 1) == NOTOK)
	return NOTOK;

    for (dp = cp; istoken (*dp); dp++)
	continue;
    c = *dp, *dp = '\0';
    ci->ci_type = add (cp, NULL);	/* store content type */
    *dp = c, cp = dp;

    if (!*ci->ci_type) {
	advise (NULL, "invalid %s: field in message %s (empty type)", 
		TYPE_FIELD, ct->c_file);
	return NOTOK;
    }

    /* down case the content type string */
    for (dp = ci->ci_type; *dp; dp++)
	if (isalpha(*dp) && isupper (*dp))
	    *dp = tolower (*dp);

    while (isspace (*cp))
	cp++;

    if (*cp == '(' && get_comment (ct, &cp, 1) == NOTOK)
	return NOTOK;

    if (*cp != '/') {
	ci->ci_subtype = add ("", NULL);
	goto magic_skip;
    }

    cp++;
    while (isspace (*cp))
	cp++;

    if (*cp == '(' && get_comment (ct, &cp, 1) == NOTOK)
	return NOTOK;

    for (dp = cp; istoken (*dp); dp++)
	continue;
    c = *dp, *dp = '\0';
    ci->ci_subtype = add (cp, NULL);	/* store the content subtype */
    *dp = c, cp = dp;

    if (!*ci->ci_subtype) {
	advise (NULL,
		"invalid %s: field in message %s (empty subtype for \"%s\")",
		TYPE_FIELD, ct->c_file, ci->ci_type);
	return NOTOK;
    }

    /* down case the content subtype string */
    for (dp = ci->ci_subtype; *dp; dp++)
	if (isalpha(*dp) && isupper (*dp))
	    *dp = tolower (*dp);

magic_skip: ;
    while (isspace (*cp))
	cp++;

    if (*cp == '(' && get_comment (ct, &cp, 1) == NOTOK)
	return NOTOK;

    /*
     * Parse attribute/value pairs given with Content-Type
     */
    ep = (ap = ci->ci_attrs) + NPARMS;
    while (*cp == ';') {
	char *vp, *up;

	if (ap >= ep) {
	    advise (NULL,
		    "too many parameters in message %s's %s: field (%d max)",
		    ct->c_file, TYPE_FIELD, NPARMS);
	    return NOTOK;
	}

	cp++;
	while (isspace (*cp))
	    cp++;

	if (*cp == '(' && get_comment (ct, &cp, 1) == NOTOK)
	    return NOTOK;

	if (*cp == 0) {
	    advise (NULL,
		    "extraneous trailing ';' in message %s's %s: parameter list",
		    ct->c_file, TYPE_FIELD);
	    return OK;
	}

	for (dp = cp; istoken (*dp); dp++)
	    if (isalpha(*dp) && isupper (*dp))
		*dp = tolower (*dp);
	for (up = dp; isspace (*dp); )
	    dp++;
	if (dp == cp || *dp != '=') {
	    advise (NULL,
		    "invalid parameter in message %s's %s: field\n%*.*sparameter %s (error detected at offset %d)",
		    ct->c_file, TYPE_FIELD, i, i, "", cp, dp - cp);
	    return NOTOK;
	}

	vp = (*ap = add (cp, NULL)) + (up - cp);
	*vp = '\0';
	for (dp++; isspace (*dp); )
	    dp++;
	ci->ci_values[ap - ci->ci_attrs] = vp = *ap + (dp - cp);
	if (*dp == '"') {
	    for (cp = ++dp, dp = vp;;) {
		switch (c = *cp++) {
		    case '\0':
bad_quote: ;
		        advise (NULL,
				"invalid quoted-string in message %s's %s: field\n%*.*s(parameter %s)",
				ct->c_file, TYPE_FIELD, i, i, "", *ap);
			return NOTOK;

		    case '\\':
			*dp++ = c;
			if ((c = *cp++) == '\0')
			    goto bad_quote;
			/* else fall... */

		    default:
    			*dp++ = c;
    			continue;

		    case '"':
			*dp = '\0';
			break;
		}
		break;
	    }
	} else {
	    for (cp = dp, dp = vp; istoken (*cp); cp++, dp++)
		continue;
	    *dp = '\0';
	}
	if (!*vp) {
	    advise (NULL,
		    "invalid parameter in message %s's %s: field\n%*.*s(parameter %s)",
		    ct->c_file, TYPE_FIELD, i, i, "", *ap);
	    return NOTOK;
	}
	ap++;

	while (isspace (*cp))
	    cp++;

	if (*cp == '(' && get_comment (ct, &cp, 1) == NOTOK)
	    return NOTOK;
    }

    /*
     * Check if anything is left over
     */
    if (*cp) {
	advise (NULL, "extraneous information in message %s's %s: field\n%*.*s(%s)",
	    ct->c_file, TYPE_FIELD, i, i, "", cp);
    }

    return OK;
}


static int
get_comment (CT ct, char **ap, int istype)
{
    int i;
    char *bp, *cp;
    char c, buffer[BUFSIZ], *dp;
    CI ci;

    ci = &ct->c_ctinfo;
    cp = *ap;
    bp = buffer;
    cp++;

    for (i = 0;;) {
	switch (c = *cp++) {
	case '\0':
invalid:;
	advise (NULL, "invalid comment in message %s's %s: field",
		ct->c_file, istype ? TYPE_FIELD : VRSN_FIELD);
	return NOTOK;

	case '\\':
	    *bp++ = c;
	    if ((c = *cp++) == '\0')
		goto invalid;
	    *bp++ = c;
	    continue;

	case '(':
	    i++;
	    /* and fall... */
	default:
	    *bp++ = c;
	    continue;

	case ')':
	    if (--i < 0)
		break;
	    *bp++ = c;
	    continue;
	}
	break;
    }
    *bp = '\0';

    if (istype) {
	if ((dp = ci->ci_comment)) {
	    ci->ci_comment = concat (dp, " ", buffer, NULL);
	    free (dp);
	} else {
	    ci->ci_comment = add (buffer, NULL);
	}
    }

    while (isspace (*cp))
	cp++;

    *ap = cp;
    return OK;
}


#define empty(s) ((s) ? (s) : "")

/*
 * Generic method for listing content
 */

static int
list_content (CT ct, int toplevel)
{
    unsigned long size;
    char **ap, **ep;
    char *cp, buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    printf (toplevel > 0 ? LSTFMT2a : toplevel < 0 ? "part " : "     ",
	    atoi (r1bindex (empty (ct->c_file), '/')));
    sprintf (buffer, "%s/%s", empty (ci->ci_type), empty (ci->ci_subtype));
    printf (LSTFMT2b, empty (ct->c_partno), buffer);

    if (ct->c_cesizefnx && sizesw)
	size = (*ct->c_cesizefnx) (ct);
    else
	size = ct->c_end - ct->c_begin;

    /* find correct scale for size (Kilo/Mega/Giga/Tera) */
    for (cp = " KMGT"; size > 9999; size >>= 10)
	if (!*++cp)
	    break;

    switch (*cp) {
        case ' ':
	    if (size > 0 || ct->c_encoding != CE_EXTERNAL)
		printf (LSTFMT2c1, size);
	    else
		printf (LSTFMT2c4);
	    break;

	default:
	    printf (LSTFMT2c2, size, *cp);
	    break;

	case '\0':
	    printf (LSTFMT2c3);
    }

    if (ct->c_descr) {
	char *dp;

	dp = trimcpy (cp = add (ct->c_descr, NULL));
	free (cp);
	printf (LSTFMT2d1, dp);
	free (dp);
    }

    printf ("\n");

    if (verbosw && ci->ci_comment) {
	char *dp;

	dp = trimcpy (cp = add (ci->ci_comment, NULL));
	free (cp);
	sprintf (buffer, "(%s)", dp);
	free (dp);
	printf (LSTFMT2d2, buffer);
    }

    if (!debugsw)
	return OK;

    fflush (stdout);

    fprintf (stderr, "  partno \"%s\"\n", empty (ct->c_partno));

    if (ct->c_vrsn)
	fprintf (stderr, "  %s:%s\n", VRSN_FIELD, ct->c_vrsn);

    if (ct->c_ctline)
	fprintf (stderr, "  %s:%s", TYPE_FIELD, ct->c_ctline);
    fprintf (stderr,
	     "    type \"%s\"  subtype \"%s\"  comment \"%s\"  magic \"%s\"\n",
	     empty (ci->ci_type), empty (ci->ci_subtype),
	     empty (ci->ci_comment), empty (ci->ci_magic));
    for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++)
	fprintf (stderr, "      parameter %s=\"%s\"\n", *ap, *ep);
    fprintf (stderr, "    type 0x%x subtype 0x%x params 0x%x\n",
	     ct->c_type, ct->c_subtype, (unsigned int) ct->c_ctparams);

    fprintf (stderr, "     showproc \"%s\"\n", empty (ct->c_showproc));
    fprintf (stderr, "     termproc \"%s\"\n", empty (ct->c_termproc));
    fprintf (stderr, "     storeproc \"%s\"\n", empty (ct->c_storeproc));

    if (ct->c_celine)
	fprintf (stderr, "  %s:%s", ENCODING_FIELD, ct->c_celine);
    fprintf (stderr, "    encoding 0x%x params 0x%x\n",
	     ct->c_encoding, (unsigned int) ct->c_cefile);

    if (ct->c_id)
	fprintf (stderr, "  %s:%s", ID_FIELD, ct->c_id);
    if (ct->c_descr)
	fprintf (stderr, "  %s:%s", DESCR_FIELD, ct->c_descr);

    fprintf (stderr, "  fp 0x%x file \"%s\" begin %ld end %ld\n",
	     (unsigned int) ct->c_fp, empty (ct->c_file),
	     ct->c_begin, ct->c_end);

    if (ct->c_celistfnx)
	(*ct->c_celistfnx) (ct);

    return OK;
}

#undef empty


static void
content_error (char *what, CT ct, char *fmt, ...)
{
    va_list arglist;
    int	i;
    char *bp;
    char buffer[BUFSIZ];
    CI ci;

    bp = buffer;

    if (userrs && invo_name && *invo_name) {
	sprintf (bp, "%s: ", invo_name);
	bp += strlen (bp);
    }

    va_start (arglist, fmt);
    vsprintf (bp, fmt, arglist);
    bp += strlen (bp);
    ci = &ct->c_ctinfo;

    if (what) {
	char *s;

	if (*what) {
	    sprintf (bp, " %s: ", what);
	    bp += strlen (bp);
	}
	if ((s = strerror (errno)))
	    sprintf (bp, "%s", s);
	else
	    sprintf (bp, "Error %d", errno);
	bp += strlen (bp);
    }

    i = strlen (invo_name) + 2;
    sprintf (bp, "\n%*.*s(content %s/%s", i, i, "", ci->ci_type, ci->ci_subtype);
    bp += strlen (bp);
    if (ct->c_file) {
	sprintf (bp, " in message %s", ct->c_file);
	bp += strlen (bp);
	if (ct->c_partno) {
	    sprintf (bp, ", part %s", ct->c_partno);
	    bp += strlen (bp);
	}
    }
    sprintf (bp, ")");
    bp += strlen (bp);

    if (userrs) {
	*bp++ = '\n';
	*bp = '\0';

	errs = add (buffer, errs);
    } else {
	advise (NULL, "%s", buffer);
    }
}


static void
flush_errors (void)
{
    if (errs) {
	fflush (stdout);
	fprintf (stderr, "%s", errs);
	free (errs);
	errs = NULL;
    }
}

static	jmp_buf	intrenv;


static RETSIGTYPE
intrser (int i)
{
#ifndef RELIABLE_SIGNALS
    SIGNAL (SIGINT, intrser);
#endif

    putchar ('\n');
    longjmp (intrenv, DONE);
}


/*
 * Generic method for displaying content
 */

static int
show_content (CT ct, int serial, int alternate)
{
    char *cp, buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    /* Check for mhn-show-type/subtype */
    sprintf (buffer, "%s-show-%s/%s", invo_name, ci->ci_type, ci->ci_subtype);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_content_aux (ct, serial, alternate, cp, NULL);

    /* Check for mhn-show-type */
    sprintf (buffer, "%s-show-%s", invo_name, ci->ci_type);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_content_aux (ct, serial, alternate, cp, NULL);

    if ((cp = ct->c_showproc))
	return show_content_aux (ct, serial, alternate, cp, NULL);

    /* complain if we are not a part of a multipart/alternative */
    if (!alternate)
	content_error (NULL, ct, "don't know how to display content");

    return NOTOK;
}


/*
 * Parse the display string for displaying generic content
 */

static int
show_content_aux (CT ct, int serial, int alternate, char *cp, char *cracked)
{
    int fd, xstdin;
    int	xlist, xpause, xtty;
    char *bp, *file;
    char buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    if (!ct->c_ceopenfnx) {
	if (!alternate)
	    content_error (NULL, ct, "don't know how to decode content");

	return NOTOK;
    }

    file = NULL;
    if ((fd = (*ct->c_ceopenfnx) (ct, &file)) == NOTOK)
	return NOTOK;
    if (ct->c_showproc && !strcmp (ct->c_showproc, "true"))
	return (alternate ? DONE : OK);
    
    xlist  = 0;
    xpause = 0;
    xstdin = 0;
    xtty   = 0;

    if (cracked) {
	strcpy (buffer, cp);
	goto got_command;
    }
    buffer[0] = '\0';

    for (bp = buffer; *cp; cp++) {
	if (*cp == '%') {
	    switch (*++cp) {
	    case 'a':
		/* insert parameters from Content-Type field */
	    {
		char **ap, **ep;
		char *s = "";

		for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
		    sprintf (bp, "%s%s=\"%s\"", s, *ap, *ep);
		    bp += strlen (bp);
		    s = " ";
		}
	    }
	    break;

	    case 'd':
		/* insert content description */
		if (ct->c_descr) {
		    char *s;

		    strcpy (bp, s = trimcpy (ct->c_descr));
		    free (s);
		}
		break;

	    case 'e':
		/* exclusive execution */
		xtty = 1;
		break;

	    case 'F':
		/* %e, %f, and stdin is terminal not content */
		xstdin = 1;
		xtty = 1;
		/* and fall... */

	    case 'f':
		/* insert filename containing content */
		sprintf (bp, "%s", file);
		break;

	    case 'p':
		/* %l, and pause prior to displaying content */
		xpause = pausesw;
		/* and fall... */

	    case 'l':
		/* display listing prior to displaying content */
		xlist = !nolist;
		break;

	    case 's':
		/* insert subtype of content */
		strcpy (bp, ci->ci_subtype);
		break;

	    case '%':
		/* insert character % */
		goto raw;

	    default:
		*bp++ = *--cp;
		*bp = '\0';
		continue;
	    }
	    bp += strlen (bp);
	} else {
raw: ;
	*bp++ = *cp;
	*bp = '\0';
	}
    }

    /* use charset string to modify display method */
    if (ct->c_termproc) {
	char term[BUFSIZ];

	strcpy (term, buffer);
	sprintf (buffer, ct->c_termproc, term);
    }

got_command: ;
    return show_content_aux2 (ct, serial, alternate, cracked, buffer,
			      fd, xlist, xpause, xstdin, xtty);
}


/*
 * Routine to actually display the content
 */

static int
show_content_aux2 (CT ct, int serial, int alternate, char *cracked, char *buffer,
                   int fd, int xlist, int xpause, int xstdin, int xtty)
{
    pid_t child_id;
    int i;
    char *vec[4], exec[BUFSIZ + sizeof "exec "];
    CI ci = &ct->c_ctinfo;
    
    if (debugsw || cracked) {
	fflush (stdout);

	fprintf (stderr, "%s msg %s", cracked ? "storing" : "show",
		 ct->c_file);
	if (ct->c_partno)
	    fprintf (stderr, " part %s", ct->c_partno);
	if (cracked)
	    fprintf (stderr, " using command (cd %s; %s)\n", cracked, buffer);
	else
	    fprintf (stderr, " using command %s\n", buffer);
    }

    if (xpid < 0 || (xtty && xpid)) {
	if (xpid < 0)
	    xpid = -xpid;
	pidcheck(pidwait (xpid, NOTOK));
	xpid = 0;
    }

    if (xlist) {
	char prompt[BUFSIZ];

	if (ct->c_ctlistfnx) {
	    if (ct->c_type == CT_MULTIPART)
		list_content (ct, -1);
	    else
		(*ct->c_ctlistfnx) (ct, -1);

	    if (xpause && SOprintf ("Press <return> to show content..."))
		printf ("Press <return> to show content...");
	} else {
	    char *pp;

	    pp = prompt;
	    if (ct->c_descr) {
		sprintf (pp, "%s (", ct->c_descr);
		pp += strlen (pp);
	    }

	    sprintf (pp, "content %s/%s", ci->ci_type, ci->ci_subtype);
	    pp += strlen (pp);
	    if (ct->c_file) {
		sprintf (pp, " in message %s", ct->c_file);
		pp += strlen (pp);
		if (ct->c_partno) {
		    sprintf (pp, ", part %s", ct->c_partno);
		    pp += strlen (pp);
		}
	    }

	    if (ct->c_descr) {
		sprintf (pp, ")");
		pp += strlen (pp);
	    }

	    if (!xpause)
		printf ("%s\n", prompt);
	    else
		if (SOprintf ("Press <return> to show %s...", prompt))
		    printf ("Press <return> to show %s...", prompt);
	}

	if (xpause) {
	    int	intr;
	    SIGNAL_HANDLER istat;

	    istat = SIGNAL (SIGINT, intrser);
	    if ((intr = setjmp (intrenv)) == OK) {
		fflush (stdout);
		prompt[0] = 0;
		read (fileno (stdout), prompt, sizeof(prompt));
	    }
	    SIGNAL (SIGINT, istat);
	    if (intr != OK) {
		(*ct->c_ceclosefnx) (ct);
		return (alternate ? DONE : NOTOK);
	    }
	}
    }

    sprintf (exec, "exec %s", buffer);

    vec[0] = "/bin/sh";
    vec[1] = "-c";
    vec[2] = exec;
    vec[3] = NULL;

    fflush (stdout);

    for (i = 0; (child_id = vfork ()) == NOTOK && i < 5; i++)
	sleep (5);
    switch (child_id) {
	case NOTOK:
	    advise ("fork", "unable to");
	    (*ct->c_ceclosefnx) (ct);
	    return NOTOK;

	case OK:
	    if (cracked)
		chdir (cracked);
	    if (!xstdin)
		dup2 (fd, 0);
	    close (fd);
	    execvp ("/bin/sh", vec);
	    fprintf (stderr, "unable to exec ");
	    perror ("/bin/sh");
	    _exit (-1);
	    /* NOTREACHED */

	default:
	    if (!serial) {
		ct->c_pid = child_id;
		if (xtty)
		    xpid = child_id;
	    } else {
		pidcheck (pidXwait (child_id, NULL));
	    }

	    if (fd != NOTOK)
		(*ct->c_ceclosefnx) (ct);
	    return (alternate ? DONE : OK);
    }
}


/*
 * Store contents of a message or message part to
 * a folder, a file, the standard output, or pass
 * the contents to a command.
 */

static int
store_content (CT ct, char *append)
{
    int	appending;
    long last, pos;
    char *bp, *cp;
    char *file, buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;
    FILE *fp;
    struct msgs *mp = NULL;
    struct stat st;
    char *folder;


    if ((appending = (append && *append))) {
	strcpy (buffer, append);
	goto got_filename;
    }

    /*
     * Get storage formatting string.
     *
     * 1) If we have storeproc defined, then use that
     * 2) Else check for a mhn-store-<type>/<subtype> entry
     * 3) Else check for a mhn-store-<type> entry
     * 4) Else if content is "message", use "+" (current folder)
     * 5) Else use string "%m%P.%s".
     */
    if ((cp = ct->c_storeproc) == NULL || *cp == '\0') {
	sprintf (buffer, "%s-store-%s/%s", invo_name, ci->ci_type, ci->ci_subtype);
	if ((cp = context_find (buffer)) == NULL || *cp == '\0') {
	    sprintf (buffer, "%s-store-%s", invo_name, ci->ci_type);
	    if ((cp = context_find (buffer)) == NULL || *cp == '\0') {
		cp = ct->c_type == CT_MESSAGE ? "+" : "%m%P.%s";
	    }
	}
    }

    /*
     * Check the beginning of storage formatting string
     */
    switch (*cp) {
	case '+':
	case '@':
	    /*
	     * Save content to a folder.
	     */
	    if (cp[1])
		folder = path (cp + 1, *cp == '+' ? TFOLDER : TSUBCWF);
	    else
		folder = getfolder (1);
	    bp = m_mailpath (folder);

	    /* Check if folder exists */
	    if (stat (bp, &st) == NOTOK) {
		int answer;
		char *ep;

		if (errno != ENOENT) {
		    advise (bp, "error on folder");
		    goto losing_folder;
		}

		ep = concat ("Create folder \"", bp, "\"? ", NULL);
		answer = getanswer (ep);
		free (ep);

		if (!answer)
		    goto losing_folder;
		if (!makedir (bp)) {
		    advise (NULL, "unable to create folder %s", bp);
		    goto losing_folder;
		}
	    }

	    /*
	     * Read the folder and get the filename of the
	     * message where we want to store the content
	     */
	    if ((mp = folder_read (folder)))
		sprintf (buffer, "%s/%d", mp->foldpath, mp->hghmsg + 1);
	    else
		advise (NULL, "unable to read folder %s", folder);

losing_folder:
	    if (cp[1])
		free (folder);
	    if (mp)
		folder_free (mp);	/* free folder/message structure */
	    else
		return NOTOK;
	    goto got_filename;

	case '/':
	case '|':
	case '!':
	    /*
	     * If formatting string is a command or pathname
	     * then we need to expand the escapes first.
	     */
	    bp = buffer;
	    buffer[0] = '\0';
	    break;

	case '-':
	    /*
	     * Send content to standard output
	     */
	    if (!cp[1]) {
		sprintf (buffer, "-");
		goto got_filename;
	    }
	    /* else fall ... */

	default:
	    /*
	     * If formatting string is a pathname that doesn't
	     * begin with '/', then check the -auto switch and
	     * preface path with the appropriate directory.
	     */
	    bp = autosw ? cwd : dir;
	    sprintf (buffer, "%s/", bp[1] ? bp : "");
	    bp = buffer + strlen (buffer);
	    break;
    }

    /*
     * Parse the storage formatting string
     */
    for (; *cp; cp++) {

	/* We are processing a storage escape */
	if (*cp == '%') {
	    switch (*++cp) {
		case 'a':
		    /*
		     * Insert parameters from Content-Type.
		     * This is only valid for '|' commands.
		     */
		    if (buffer[0] != '|' && buffer[0] != '!') {
			*bp++ = *--cp;
			*bp = '\0';
			continue;
		    } else {
			char **ap, **ep;
			char *s = "";

			for (ap = ci->ci_attrs, ep = ci->ci_values;
			         *ap;
			         ap++, ep++) {
			    sprintf (bp, "%s%s=\"%s\"", s, *ap, *ep);
			    bp += strlen (bp);
			    s = " ";
			}
		    }
		    break;

		case 'm':
		    /* insert message number */
		    sprintf (bp, "%s", r1bindex (ct->c_file, '/'));
		    break;

		case 'P':
		    /* insert part number with leading dot */
		    if (ct->c_partno)
			sprintf (bp, ".%s", ct->c_partno);
		    break;

		case 'p':
		    /* insert part number withouth leading dot */
		    if (ct->c_partno)
			strcpy (bp, ct->c_partno);
		    break;

		case 't':
		    /* insert content type */
		    strcpy (bp, ci->ci_type);
		    break;

		case 's':
		    /* insert content subtype */
		    strcpy (bp, ci->ci_subtype);
		    break;

		case '%':
		    /* insert the character % */
		    goto raw;

		default:
		    *bp++ = *--cp;
		    *bp = '\0';
		    continue;
	    }
	    bp += strlen (bp);

	} else {
raw:;
	    *bp++ = *cp;
	    *bp = '\0';
	}
    }

    /*
     * Pass content to standard input of a command
     */
    if (buffer[0] == '|' || buffer[0] == '!')
	return show_content_aux (ct, 1, 0, buffer + 1, autosw ? cwd : dir);

got_filename:;
    ct->c_storage = add (buffer, NULL);
    fflush (stdout);

    fprintf (stderr, "storing message %s", ct->c_file);
    if (ct->c_partno)
	fprintf (stderr, " part %s", ct->c_partno);
    if (!strcmp(ct->c_storage, "-"))
	fprintf (stderr, " to stdout\n");
    else
	fprintf (stderr, " as file %s\n",
		strncmp (ct->c_storage, cwd, cwdlen)
		|| ct->c_storage[cwdlen] != '/'
		? ct->c_storage : ct->c_storage + cwdlen + 1);

    if (strchr(ct->c_storage, '/')
	    && make_intermediates (ct->c_storage) == NOTOK)
	return NOTOK;

    if (ct->c_encoding != CE_7BIT) {
	int cc, fd;

	if (!ct->c_ceopenfnx) {
	    advise (NULL, "don't know how to decode part %s of message %s",
		    ct->c_partno, ct->c_file);
	    return NOTOK;
	}

	file = appending || !strcmp (ct->c_storage, "-") ? NULL
							   : ct->c_storage;
	if ((fd = (*ct->c_ceopenfnx) (ct, &file)) == NOTOK)
	    return NOTOK;
	if (!strcmp (file, ct->c_storage)) {
	    (*ct->c_ceclosefnx) (ct);
	    return OK;
	}

	/*
	 * Send to standard output
	 */
	if (!strcmp (ct->c_storage, "-")) {
	    int	gd;

	    if ((gd = dup (fileno (stdout))) == NOTOK) {
		advise ("stdout", "unable to dup");
losing: ;
		(*ct->c_ceclosefnx) (ct);
		return NOTOK;
	    }
	    if ((fp = fdopen (gd, appending ? "a" : "w")) == NULL) {
		advise ("stdout", "unable to fdopen (%d, \"%s\") from", gd,
			appending ? "a" : "w");
		close (gd);
		goto losing;
	    }
	} else {
	    /*
	     * Open output file
	     */
	    if ((fp = fopen (ct->c_storage, appending ? "a" : "w")) == NULL) {
		advise (ct->c_storage, "unable to fopen for %s",
			appending ? "appending" : "writing");
		goto losing;
	    }
	}

	if (append && !*append)
	    copy_some_headers (fp, ct);

	for (;;) {
	    switch (cc = read (fd, buffer, sizeof(buffer))) {
		case NOTOK:
		    advise (file, "error reading content from");
		    break;

		case OK:
		    break;

		default:
		    fwrite (buffer, sizeof(*buffer), cc, fp);
		    continue;
	    }
	    break;
	}

	(*ct->c_ceclosefnx) (ct);

	if (cc != NOTOK && fflush (fp))
	    advise (ct->c_storage, "error writing to");

	fclose (fp);

	return (cc != NOTOK ? OK : NOTOK);
    }

    if (!ct->c_fp && (ct->c_fp = fopen (ct->c_file, "r")) == NULL) {
	advise (ct->c_file, "unable to open for reading");
	return NOTOK;
    }

    fseek (ct->c_fp, pos = ct->c_begin, SEEK_SET);
    last = ct->c_end;

    if (!strcmp (ct->c_storage, "-")) {
	int gd;

	if ((gd = dup (fileno (stdout))) == NOTOK) {
	    advise ("stdout", "unable to dup");
	    return NOTOK;
	}
	if ((fp = fdopen (gd, appending ? "a" : "w")) == NULL) {
	    advise ("stdout", "unable to fdopen (%d, \"%s\") from", gd,
		    appending ? "a" : "w");
	    close (gd);
	    return NOTOK;
	}
    } else {
	if ((fp = fopen (ct->c_storage, appending ? "a" : "w")) == NULL) {
	    advise (ct->c_storage, "unable to fopen for %s",
		    appending ? "appending" : "writing");
	    return NOTOK;
	}
    }

    if (append && !*append) {
	copy_some_headers (fp, ct);
	appending = 1;
    } else {
	appending = 0;
    }

    while (fgets (buffer, sizeof(buffer) - 1, ct->c_fp)) {
	if ((pos += strlen (buffer)) > last) {
	    int diff;

	    diff = strlen (buffer) - (pos - last);
	    if (diff >= 0)
		buffer[diff] = '\0';
	}

	if (appending)
	    switch (buffer[0]) {
		case ' ':
		case '\t':
		    if (appending < 0)
			buffer[0] = 0;
		    break;

		case '\n':
		    appending = 0;
		    break;

		default:
		    if (!uprf (buffer, XXX_FIELD_PRF)
			    && !uprf (buffer, "Encrypted:")
			    && !uprf (buffer, "Message-ID:")) {
			appending = -1;
			buffer[0] = 0;
			break;
		    }
		    appending = 1;
		    break;
	    }

	fputs (buffer, fp);
	if (pos >= last)
	    break;
    }

    if (fflush (fp))
	advise (ct->c_storage, "error writing to");

    fclose (fp);
    fclose (ct->c_fp);
    ct->c_fp = NULL;
    return OK;
}


static int
copy_some_headers (FILE *out, CT ct)
{
    int	state;
    char buf[BUFSIZ], name[NAMESZ];
    FILE *in;

    if ((in = fopen (ct->c_file, "r")) == NULL) {
	advise (ct->c_file, "unable to open for reading");
	return NOTOK;
    }

    for (state = FLD;;) {
	switch (state = m_getfld (state, name, buf, sizeof(buf), in)) {
	    case FLD:
	    case FLDPLUS:
	    case FLDEOF:
		if (uprf (name, XXX_FIELD_PRF)
		        || !strcasecmp (name, "Encrypted")
		        || !strcasecmp (name, "Message-ID")) {
		    while (state == FLDPLUS)
			state = m_getfld (state, name, buf, sizeof(buf), in);
		    continue;
		}

		fprintf (out, "%s:%s", name, buf);
		while (state == FLDPLUS) {
		    state = m_getfld (state, name, buf, sizeof(buf), in);
		    fputs (buf, out);
		}
		if (state != FLDEOF)
		    continue;
		/* else fall... */
	   case BODY:
	   case BODYEOF:
	   case FILEEOF:
		break;

	   case LENERR:
	   case FMTERR:
	   default:
		break;
	}

	break;
    }

    fclose (in);
    return OK;
}


static int
make_intermediates (char *file)
{
    char *cp;

    for (cp = file + 1; cp = strchr(cp, '/'); cp++) {
	struct stat st;

	*cp = '\0';
	if (stat (file, &st) == NOTOK) {
	    int	answer;
	    char *ep;

	    if (errno != ENOENT) {
		advise (file, "error on directory");
losing_directory: ;
		*cp = '/';
		return NOTOK;
	    }

	    ep = concat ("Create directory \"", file, "\"? ", NULL);
	    answer = getanswer (ep);
	    free (ep);

	    if (!answer)
		goto losing_directory;
	    if (!makedir (file)) {
		advise (NULL, "unable to create directory %s", file);
		goto losing_directory;
	    }
	}

	*cp = '/';
    }

    return OK;
}


static void
free_ctinfo (CT ct)
{
    char **ap;
    CI ci;

    ci = &ct->c_ctinfo;
    if (ci->ci_type) {
	free (ci->ci_type);
	ci->ci_type = NULL;
    }
    if (ci->ci_subtype) {
	free (ci->ci_subtype);
	ci->ci_subtype = NULL;
    }
    for (ap = ci->ci_attrs; *ap; ap++) {
	free (*ap);
	*ap = NULL;
    }
    if (ci->ci_comment) {
	free (ci->ci_comment);
	ci->ci_comment = NULL;
    }
    if (ci->ci_magic) {
	free (ci->ci_magic);
	ci->ci_magic = NULL;
    }
}


static void
free_content (CT ct)
{
    if (!ct)
	return;

    if (ct->c_partno)
	free (ct->c_partno);

    if (ct->c_vrsn)
	free (ct->c_vrsn);

    if (ct->c_ctline)
	free (ct->c_ctline);

    free_ctinfo (ct);

    if (ct->c_ctfreefnx)
	(*ct->c_ctfreefnx) (ct);

    if (ct->c_showproc)
	free (ct->c_showproc);
    if (ct->c_termproc)
	free (ct->c_termproc);
    if (ct->c_storeproc)
	free (ct->c_storeproc);

    if (ct->c_celine)
	free (ct->c_celine);
    if (ct->c_cefreefnx)
	(*ct->c_cefreefnx) (ct, 1);

    if (ct->c_id)
	free (ct->c_id);
    if (ct->c_descr)
	free (ct->c_descr);

    if (ct->c_file) {
	if (ct->c_unlink)
	    unlink (ct->c_file);
	free (ct->c_file);
    }
    if (ct->c_fp)
	fclose (ct->c_fp);

    if (ct->c_storage)
	free (ct->c_storage);

    free (ct);
}


static int
part_ok (CT ct, int sP)
{
    char **ap;

    if ((ct->c_type == CT_MULTIPART && (sP || ct->c_subtype))
	    || npart == 0)
	return 1;

    for (ap = parts; *ap; ap++)
	if (strcmp (*ap, ct->c_partno) == 0)
	    return 1;

    return 0;
}


static int
type_ok (CT ct, int sP)
{
    char **ap;
    char buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    if ((ct->c_type == CT_MULTIPART && (sP || ct->c_subtype))
	    || ntype == 0)
	return 1;

    sprintf (buffer, "%s/%s", ci->ci_type, ci->ci_subtype);
    for (ap = types; *ap; ap++)
	if (!strcasecmp (*ap, ci->ci_type) || !strcasecmp (*ap, buffer))
	    return 1;

    return 0;
}


/*
 * CONTENTS
 *
 * Handles content types audio, image, and video
 */

static int
InitGeneric (CT ct)
{
    char **ap, **ep, *cp;
    CI ci = &ct->c_ctinfo;

    /* check if content specifies a filename */
    if (autosw) {
	for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
	    if (!ct->c_storeproc
		&& !strcasecmp (*ap, "name")
		&& *(cp = *ep) != '/'
		&& *cp != '.'
		&& *cp != '|'
		&& *cp != '!'
		&& !strchr(cp, '%'))
		ct->c_storeproc = add (cp, NULL);
	}
    }

    return OK;
}


/*
 * TEXT
 */

static int
InitText (CT ct)
{
    char buffer[BUFSIZ];
    char *chset;
    char **ap, **ep, *cp;
    struct k2v *kv;
    struct text *t;
    CI ci = &ct->c_ctinfo;

    /* check for missing subtype */
    if (!*ci->ci_subtype)
	ci->ci_subtype = add ("plain", ci->ci_subtype);

    /* match subtype */
    for (kv = SubText; kv->kv_key; kv++)
	if (!strcasecmp (ci->ci_subtype, kv->kv_key))
	    break;
    ct->c_subtype = kv->kv_value;

    /* setup methods for this type */
    ct->c_ctshowfnx = show_text;
    ct->c_ctfreefnx = free_text;

    /* allocate text structure */
    if ((t = (struct text *) calloc (1, sizeof(*t))) == NULL)
	adios (NULL, "out of memory");
    ct->c_ctparams = (void *) t;

    /* scan for charset parameter */
    for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++)
	if (!strcasecmp (*ap, "charset"))
	    break;

    if (*ap)
	chset = *ep;
    else
	chset = "US-ASCII";	/* default for text */

    /* match character set, or set to unknown */
    for (kv = Charset; kv->kv_key; kv++)
	if (!strcasecmp (chset, kv->kv_key))
	    break;
    t->tx_charset = kv->kv_value;

    /*
     * If we can not handle character set natively,
     * then check profile for string to modify the
     * terminal or display method.
     */
    if (!check_charset (chset, strlen (chset))) {
	sprintf (buffer, "%s-charset-%s", invo_name, chset);
	if ((cp = context_find (buffer)))
	    ct->c_termproc = getcpy (cp);
    }

    /* check if content specifies a filename */
    if (autosw) {
	for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
	    if (!ct->c_storeproc
		&& !strcasecmp (*ap, "name")
		&& *(cp = *ep) != '/'
		&& *cp != '.'
		&& *cp != '|'
		&& *cp != '!'
		&& !strchr(cp, '%'))
		ct->c_storeproc = add (cp, NULL);
	}
    }

    return OK;
}


/*
 * show content of type "text"
 */

static int
show_text (CT ct, int serial, int alternate)
{
    char *cp, buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    /* Check for mhn-show-type/subtype */
    sprintf (buffer, "%s-show-%s/%s", invo_name, ci->ci_type, ci->ci_subtype);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_content_aux (ct, serial, alternate, cp, NULL);

    /* Check for mhn-show-type */
    sprintf (buffer, "%s-show-%s", invo_name, ci->ci_type);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_content_aux (ct, serial, alternate, cp, NULL);

    /*
     * Use default method if content is text/plain, or if
     * if it is not a text part of a multipart/alternative
     */
    if (!alternate || ct->c_subtype == TEXT_PLAIN) {
	sprintf (buffer, "%%p%s '%%F'", progsw ? progsw :
		moreproc && *moreproc ? moreproc : "more");
	cp = (ct->c_showproc = add (buffer, NULL));
	return show_content_aux (ct, serial, alternate, cp, NULL);
    }

    return NOTOK;
}


static void
free_text (CT ct)
{
    struct text *t;

    if (!(t = (struct text *) ct->c_ctparams))
	return;

    free ((char *) t);
    ct->c_ctparams = NULL;
}


/*
 * MULTIPART
 */

static int
InitMultiPart (CT ct)
{
    int	inout;
    long last, pos;
    char *cp, *dp, **ap, **ep;
    char *bp, buffer[BUFSIZ];
    struct multipart *m;
    struct k2v *kv;
    struct part *part, **next;
    CI ci = &ct->c_ctinfo;
    CT p;
    FILE *fp;

    /*
     * The encoding for multipart messages must be either
     * 7bit, 8bit, or binary (per RFC2045).
     */
    if (ct->c_encoding != CE_7BIT && ct->c_encoding != CE_8BIT
	&& ct->c_encoding != CE_BINARY) {
	admonish (NULL,
		  "\"%s/%s\" type in message %s must be encoded in 7bit, 8bit, or binary",
		  ci->ci_type, ci->ci_subtype, ct->c_file);
	return NOTOK;
    }

    /* match subtype */
    for (kv = SubMultiPart; kv->kv_key; kv++)
	if (!strcasecmp (ci->ci_subtype, kv->kv_key))
	    break;
    ct->c_subtype = kv->kv_value;

    /* setup methods for this type */
    ct->c_ctlistfnx  = list_multi;
    ct->c_ctshowfnx  = show_multi;
    ct->c_ctstorefnx = store_multi;
    ct->c_ctfreefnx  = free_multi;

    /*
     * Check for "boundary" parameter, which is
     * required for multipart messages.
     */
    for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++)
	if (!strcasecmp (*ap, "boundary")) {
	    bp = *ep;
	    break;
	}
    if (!*ap) {
	advise (NULL,
		"a \"boundary\" parameter is mandatory for \"%s/%s\" type in message %s's %s: field",
		ci->ci_type, ci->ci_subtype, ct->c_file, TYPE_FIELD);
	return NOTOK;
    }
    
    if ((m = (struct multipart *) calloc (1, sizeof(*m))) == NULL)
	adios (NULL, "out of memory");
    ct->c_ctparams = (void *) m;

    for (cp = bp; isspace (*cp); cp++)
	continue;
    if (!*cp) {
	advise (NULL, "invalid \"boundary\" parameter for \"%s/%s\" type in message %s's %s: field",
		ci->ci_type, ci->ci_subtype, ct->c_file, TYPE_FIELD);
	return NOTOK;
    }
    for (cp = bp, dp = cp + strlen (cp) - 1; dp > cp; dp--)
	if (!isspace (*dp))
	    break;
    *++dp = '\0';
    m->mp_start = concat (bp, "\n", NULL);
    m->mp_stop = concat (bp, "--\n", NULL);

    if (!ct->c_fp && (ct->c_fp = fopen (ct->c_file, "r")) == NULL) {
	advise (ct->c_file, "unable to open for reading");
	return NOTOK;
    }

    fseek (fp = ct->c_fp, pos = ct->c_begin, SEEK_SET);
    last = ct->c_end;
    next = &m->mp_parts;
    part = NULL;
    inout = 1;

    while (fgets (buffer, sizeof(buffer) - 1, fp)) {
	if (pos > last)
	    break;

	pos += strlen (buffer);
	if (buffer[0] != '-' || buffer[1] != '-')
	    continue;
	if (inout) {
	    if (strcmp (buffer + 2, m->mp_start))
		continue;
next_part:
	    if ((part = (struct part *) calloc (1, sizeof(*part))) == NULL)
		adios (NULL, "out of memory");
	    *next = part;
	    next = &part->mp_next;

	    if (!(p = get_content (fp, ct->c_file,
		rfc934sw && ct->c_subtype == MULTI_DIGEST ? -1 : 0))) {
		fclose (ct->c_fp);
		ct->c_fp = NULL;
		return NOTOK;
	    }
	    p->c_fp = NULL;
	    part->mp_part = p;
	    pos = p->c_begin;
	    fseek (fp, pos, SEEK_SET);
	    inout = 0;
	} else {
	    if (strcmp (buffer + 2, m->mp_start) == 0) {
		inout = 1;
end_part:
		p = part->mp_part;
		p->c_end = ftell(fp) - (strlen(buffer) + 1);
		if (p->c_end < p->c_begin)
		    p->c_begin = p->c_end;
		if (inout)
		    goto next_part;
		goto last_part;
	    } else {
		if (strcmp (buffer + 2, m->mp_stop) == 0)
		    goto end_part;
	    }
	}
    }

    advise (NULL, "bogus multipart content in message %s", ct->c_file);
    if (!inout && part) {
	p = part->mp_part;
	p->c_end = ct->c_end;

	if (p->c_begin >= p->c_end) {
	    for (next = &m->mp_parts;
		     *next != part;
		     next = &((*next)->mp_next))
		continue;
	    *next = NULL;
	    free_content (p);
	    free ((char *) part);
	}
    }

last_part:
    if (ct->c_subtype == MULTI_ALTERNATE && m->mp_parts->mp_next) {
	int i;
	struct part **base, **bmp;

	i = 0;
	for (part = m->mp_parts; part; part = part->mp_next)
	    i++;
	if ((base = (struct part **) calloc ((size_t) (i + 1), sizeof(*base)))
	        == NULL)
	    adios (NULL, "out of memory");
	bmp = base;
	for (part = m->mp_parts; part; part = part->mp_next)
	    *bmp++ = part;
	*bmp = NULL;

	next = &m->mp_parts;
	for (bmp--; bmp >= base; bmp--) {
	    part = *bmp;
	    *next = part, next = &part->mp_next;
	}
	*next = NULL;

	free ((char *) base);
    }

    {
	int partnum;
	char *pp;
	char partnam[BUFSIZ];

	if (ct->c_partno) {
	    sprintf (partnam, "%s.", ct->c_partno);
	    pp = partnam + strlen (partnam);
	} else {
	    pp = partnam;
	}

	for (part = m->mp_parts, partnum = 1; part;
	         part = part->mp_next, partnum++) {
	    p = part->mp_part;

	    sprintf (pp, "%d", partnum);
	    p->c_partno = add (partnam, NULL);

	    if (p->c_ctinitfnx && (*p->c_ctinitfnx) (p) == NOTOK) {
		fclose (ct->c_fp);
		ct->c_fp = NULL;
		return NOTOK;
	    }
	}
    }

    fclose (ct->c_fp);
    ct->c_fp = NULL;
    return OK;
}


/*
 * list content types in multipart message
 */

static int
list_multi (CT ct, int toplevel)
{
    struct multipart *m = (struct multipart *) ct->c_ctparams;
    struct part *part;

    list_content (ct, toplevel);

    for (part = m->mp_parts; part; part = part->mp_next) {
	CT p = part->mp_part;

	if (part_ok (p, 1) && type_ok (p, 1) && p->c_ctlistfnx)
	    (*p->c_ctlistfnx) (p, 0);
    }

    return OK;
}


/*
 * show message body of type multipart
 */

static int
show_multi (CT ct, int serial, int alternate)
{
    char *cp, buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    /* Check for mhn-show-type/subtype */
    sprintf (buffer, "%s-show-%s/%s", invo_name, ci->ci_type, ci->ci_subtype);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_multi_aux (ct, serial, alternate, cp);

    /* Check for mhn-show-type */
    sprintf (buffer, "%s-show-%s", invo_name, ci->ci_type);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_multi_aux (ct, serial, alternate, cp);

    if ((cp = ct->c_showproc))
	return show_multi_aux (ct, serial, alternate, cp);

    /*
     * Use default method to display this multipart content
     * if it is not a (nested) part of a multipart/alternative,
     * or if it is one of the known subtypes of multipart.
     */
    if (!alternate || ct->c_subtype != MULTI_UNKNOWN)
	return show_multi_internal (ct, serial, alternate);

    return NOTOK;
}


/*
 * show message body of subtypes of multipart that
 * we understand directly (mixed, alternate, etc...)
 */

static int
show_multi_internal (CT ct, int serial, int alternate)
{
    int	alternating, nowalternate, nowserial, result;
    struct multipart *m = (struct multipart *) ct->c_ctparams;
    struct part *part;
    CT p;
    sigset_t set, oset;

    alternating = 0;
    nowalternate = alternate;

    if (ct->c_subtype == MULTI_PARALLEL) {
	nowserial = serialsw;
    } else if (ct->c_subtype == MULTI_ALTERNATE) {
	nowalternate = 1;
	alternating  = 1;
	nowserial = serial;
    } else {
	/*
	 * multipart/mixed
	 * mutlipart/digest
	 * unknown subtypes of multipart (treat as mixed per rfc2046)
	 */
	nowserial = serial;
    }

    /* block a few signals */
    if (!nowserial) {
	sigemptyset (&set);
	sigaddset (&set, SIGHUP);
	sigaddset (&set, SIGINT);
	sigaddset (&set, SIGQUIT);
	sigaddset (&set, SIGTERM);
	SIGPROCMASK (SIG_BLOCK, &set, &oset);
    }

/*
 * alternate   -> we are a part inside an multipart/alternative
 * alternating -> we are a multipart/alternative 
 */

    result = alternate ? NOTOK : OK;

    for (part = m->mp_parts; part; part = part->mp_next) {
	p = part->mp_part;

	if (part_ok (p, 0) && type_ok (p, 0) && p->c_ctshowfnx) {
	    int	inneresult;

	    inneresult = (*p->c_ctshowfnx)(p, nowserial, nowalternate);
	    switch (inneresult) {
		case NOTOK:
		    if (alternate && !alternating) {
			result = NOTOK;
			goto out;
		    }
		    continue;

		case OK:
		case DONE:
		    if (alternating) {
			result = DONE;
			break;
		    }
		    if (alternate) {
			alternate = nowalternate = 0;
			if (result == NOTOK)
			    result = inneresult;
		    }
		    continue;
	    }
	    break;
	}
    }

    if (alternating && !part) {
	if (!alternate)
	    content_error (NULL, ct, "don't know how to display any of the contents");
	result = NOTOK;
	goto out;
    }

    if (serial && !nowserial) {
	pid_t pid;
	int kids;
#ifdef WAITINT
	int status;
#else
	union wait status;
#endif

	kids = 0;
	for (part = m->mp_parts; part; part = part->mp_next) {
	    p = part->mp_part;

	    if (p->c_pid > OK)
		if (kill (p->c_pid, 0) == NOTOK)
		    p->c_pid = 0;
		else
		    kids++;
	}

	while (kids > 0 && (pid = wait (&status)) != NOTOK) {
#ifdef WAITINT
	    pidcheck (status);
#else
	    pidcheck (status.w_status);
#endif

	    for (part = m->mp_parts; part; part = part->mp_next) {
		p = part->mp_part;

		if (xpid == pid)
		    xpid = 0;
		if (p->c_pid == pid) {
		    p->c_pid = 0;
		    kids--;
		    break;
		}
	    }
	}
    }

out:
    if (!nowserial) {
	/* reset the signal mask */
	SIGPROCMASK (SIG_SETMASK, &oset, &set);
    }

    return result;
}


/*
 * Parse display string for multipart content
 * and use external program to display it.
 */

static int
show_multi_aux (CT ct, int serial, int alternate, char *cp)
{
    int xlist, xpause, xtty;
    char *bp, *file;
    char buffer[BUFSIZ];
    struct multipart *m = (struct multipart *) ct->c_ctparams;
    struct part *part;
    CI ci = &ct->c_ctinfo;
    CT p;

    for (part = m->mp_parts; part; part = part->mp_next) {
	p = part->mp_part;

	if (!p->c_ceopenfnx) {
	    if (!alternate)
		content_error (NULL, p, "don't know how to decode content");
	    return NOTOK;
	}

	if (p->c_storage == NULL) {
	    file = NULL;
	    if ((*p->c_ceopenfnx) (p, &file) == NOTOK)
		return NOTOK;

	    /* I'm not sure if this is necessary? */
	    p->c_storage = add (file, NULL);

	    if (p->c_showproc && !strcmp (p->c_showproc, "true"))
		return (alternate ? DONE : OK);
	    (*p->c_ceclosefnx) (p);
	}
    }

    xlist     = 0;
    xpause    = 0;
    xtty      = 0;
    buffer[0] = '\0';

    for (bp = buffer; *cp; cp++) {
	if (*cp == '%') {
	    switch (*++cp) {
	    case 'a':
		/* insert parameters from Content-Type field */
	    {
		char **ap, **ep;
		char *s = "";

		for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
		    sprintf (bp, "%s%s=\"%s\"", s, *ap, *ep);
		    bp += strlen (bp);
		    s = " ";
		}
	    }
	    break;

	    case 'd':
		/* insert content description */
		if (ct->c_descr) {
		    char *s;

		    strcpy (bp, s = trimcpy (ct->c_descr));
		    free (s);
		}
		break;

	    case 'e':
		/* exclusive execution */
		xtty = 1;
		break;

	    case 'F':
		/* %e and %f */
		xtty = 1;
		/* and fall... */

	    case 'f':
		/* insert filename(s) containing content */
	    {
		char *s = "";
			
		for (part = m->mp_parts; part; part = part->mp_next) {
		    p = part->mp_part;

		    sprintf (bp, "%s'%s'", s, p->c_storage);
		    bp += strlen (bp);
		    s = " ";
		}
	    }
	    break;

	    case 'p':
		/* %l, and pause prior to displaying content */
		xpause = pausesw;
		/* and fall... */

	    case 'l':
		/* display listing prior to displaying content */
		xlist = !nolist;
		break;

	    case 's':
		/* insert subtype of content */
		strcpy (bp, ci->ci_subtype);
		break;

	    case '%':
		/* insert character % */
		goto raw;

	    default:
		*bp++ = *--cp;
		*bp = '\0';
		continue;
	    }
	    bp += strlen (bp);
	} else {
raw:;
	*bp++ = *cp;
	*bp = '\0';
	}
    }

    /* use charset string to modify display method */
    if (ct->c_termproc) {
	char term[BUFSIZ];

	strcpy (term, buffer);
	sprintf (buffer, ct->c_termproc, term);
    }

    return show_content_aux2 (ct, serial, alternate, NULL, buffer,
			      NOTOK, xlist, xpause, 0, xtty);
}


/*
 * Store the content of a multipart message
 */

static int
store_multi (CT ct, char *unused)
{
    int	result;
    struct multipart *m = (struct multipart *) ct->c_ctparams;
    struct part *part;

    result = NOTOK;
    for (part = m->mp_parts; part; part = part->mp_next) {
	CT  p = part->mp_part;

	if (part_ok (p, 1) && type_ok (p, 1) && p->c_ctstorefnx
	        && (result = (*p->c_ctstorefnx) (p, NULL)) == OK
	        && ct->c_subtype == MULTI_ALTERNATE)
	    break;
    }

    return result;
}


static void
free_multi (CT ct)
{
    struct multipart *m = (struct multipart *) ct->c_ctparams;
    struct part *part, *next;

    if (!m)
	return;

    if (m->mp_start)
	free (m->mp_start);
    if (m->mp_stop)
	free (m->mp_stop);
	
    for (part = m->mp_parts; part; part = next) {
	next = part->mp_next;

	free_content (part->mp_part);

	free ((char *) part);
    }
    m->mp_parts = NULL;

    free ((char *) m);
    ct->c_ctparams = NULL;
}


/*
 * MESSAGE
 */

static int
InitMessage (CT ct)
{
    struct k2v *kv;
    CI ci = &ct->c_ctinfo;

    if (ct->c_encoding != CE_7BIT) {
	admonish (NULL,
		  "\"%s/%s\" type in message %s should be encoded in 7bit",
		  ci->ci_type, ci->ci_subtype, ct->c_file);
	return NOTOK;
    }

    /* check for missing subtype */
    if (!*ci->ci_subtype)
	ci->ci_subtype = add ("rfc822", ci->ci_subtype);

    /* match subtype */
    for (kv = SubMessage; kv->kv_key; kv++)
	if (!strcasecmp (ci->ci_subtype, kv->kv_key))
	    break;
    ct->c_subtype = kv->kv_value;

    switch (ct->c_subtype) {
	case MESSAGE_RFC822:
	    ct->c_ctshowfnx = show_message;
	    break;

	case MESSAGE_PARTIAL:
	    {
		char **ap, **ep;
		struct partial *p;

		ct->c_ctshowfnx = NULL;
		ct->c_ctstorefnx = NULL;

		if ((p = (struct partial *) calloc (1, sizeof(*p))) == NULL)
		    adios (NULL, "out of memory");
		ct->c_ctparams = (void *) p;
		ct->c_ctfreefnx = free_partial;

		for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
		    if (!strcasecmp (*ap, "id")) {
			p->pm_partid = add (*ep, NULL);
			continue;
		    }
		    if (!strcasecmp (*ap, "number")) {
			if (sscanf (*ep, "%d", &p->pm_partno) != 1
			        || p->pm_partno < 1) {
invalid_param: ;
			    advise (NULL,
				    "invalid %s parameter for \"%s/%s\" type in message %s's %s field",
				    *ap, ci->ci_type, ci->ci_subtype,
				    ct->c_file, TYPE_FIELD);
			    return NOTOK;
			}
			continue;
		    }
		    if (!strcasecmp (*ap, "total")) {
			if (sscanf (*ep, "%d", &p->pm_maxno) != 1
			        || p->pm_maxno < 1)
			    goto invalid_param;
			continue;
		    }
		}

		if (!p->pm_partid
		        || !p->pm_partno
		        || (p->pm_maxno && p->pm_partno > p->pm_maxno)) {
		    advise (NULL,
			    "invalid parameters for \"%s/%s\" type in message %s's %s field",
			    ci->ci_type, ci->ci_subtype,
			    ct->c_file, TYPE_FIELD);
		    return NOTOK;
		}

		ct->c_ctlistfnx = list_partial;
		ct->c_ctstorefnx = store_partial;
	    }
	    break;

	case MESSAGE_EXTERNAL:
	    {
		int exresult;
		struct exbody *e;
		CT p;
		FILE *fp;

		ct->c_ctshowfnx = NULL;
		ct->c_ctstorefnx = NULL;

		if ((e = (struct exbody *) calloc (1, sizeof(*e))) == NULL)
		    adios (NULL, "out of memory");
		ct->c_ctparams = (void *) e;
		ct->c_ctfreefnx = free_external;

		if (!ct->c_fp
		        && (ct->c_fp = fopen (ct->c_file, "r")) == NULL) {
		    advise (ct->c_file, "unable to open for reading");
		    return NOTOK;
		}

		fseek (fp = ct->c_fp, ct->c_begin, SEEK_SET);

		if (!(p = get_content (fp, ct->c_file, 0))) {
		    fclose (ct->c_fp);
		    ct->c_fp = NULL;
		    return NOTOK;
		}

		e->eb_parent = ct;
		e->eb_content = p;
		p->c_ctexbody = e;
		if ((exresult = params_external (ct, 0)) != NOTOK
		        && p->c_ceopenfnx == openMail) {
		    int	cc, size;
		    char *bp;
		    
		    if ((size = ct->c_end - p->c_begin) <= 0) {
			if (!e->eb_subject)
			    content_error (NULL, ct,
					   "empty body for access-type=mail-server");
			goto no_body;
		    }
		    
		    if ((e->eb_body = bp = malloc ((unsigned) size)) == NULL)
			adios (NULL, "out of memory");
		    fseek (p->c_fp, p->c_begin, SEEK_SET);
		    while (size > 0)
			switch (cc = fread (bp, sizeof(*bp), size, p->c_fp)) {
			    case NOTOK:
			        adios ("failed", "fread");

			    case OK:
				adios (NULL, "unexpected EOF from fread");

			    default:
				bp += cc, size -= cc;
				break;
			}
		    *bp = 0;
		}
no_body:
		p->c_fp = NULL;
		p->c_end = p->c_begin;

		fclose (ct->c_fp);
		ct->c_fp = NULL;

		ct->c_ctlistfnx = list_external;

		if (exresult == NOTOK)
		    return NOTOK;
		if (e->eb_flags == NOTOK)
		    return OK;

		if (autosw) {
		    char *cp;

		    if ((cp = e->eb_name)
			&& *cp != '/'
			&& *cp != '.'
			&& *cp != '|'
			&& *cp != '!'
			&& !strchr(cp, '%')) {
			if (!ct->c_storeproc)
			    ct->c_storeproc = add (cp, NULL);
			if (!p->c_storeproc)
			    p->c_storeproc = add (cp, NULL);
		    }
		}

		ct->c_ctshowfnx = show_external;
		ct->c_ctstorefnx = store_external;
		switch (p->c_type) {
		    case CT_MULTIPART:
		        break;

		    case CT_MESSAGE:
			if (p->c_subtype != MESSAGE_RFC822)
			    break;
			/* else fall... */
		    default:
			e->eb_partno = ct->c_partno;
			if (p->c_ctinitfnx)
			    (*p->c_ctinitfnx) (p);
			break;
		}
	    }
	    break;

	default:
	    break;
    }

    return OK;
}


static int
params_external (CT ct, int composing)
{
    char **ap, **ep;
    struct exbody *e = (struct exbody *) ct->c_ctparams;
    CI ci = &ct->c_ctinfo;

    for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
	if (!strcasecmp (*ap, "access-type")) {
	    struct str2init *s2i;
	    CT p = e->eb_content;

	    for (s2i = str2methods; s2i->si_key; s2i++)
		if (!strcasecmp (*ep, s2i->si_key))
		    break;
	    if (!s2i->si_key) {
		e->eb_access = *ep;
		e->eb_flags = NOTOK;
		p->c_encoding = CE_EXTERNAL;
		continue;
	    }
	    e->eb_access = s2i->si_key;
	    e->eb_flags = s2i->si_val;
	    p->c_encoding = CE_EXTERNAL;

	    /* Call the Init function for this external type */
	    if ((*s2i->si_init)(p) == NOTOK)
		return NOTOK;
	    continue;
	}
	if (!strcasecmp (*ap, "name")) {
	    e->eb_name = *ep;
	    continue;
	}
	if (!strcasecmp (*ap, "permission")) {
	    e->eb_permission = *ep;
	    continue;
	}
	if (!strcasecmp (*ap, "site")) {
	    e->eb_site = *ep;
	    continue;
	}
	if (!strcasecmp (*ap, "directory")) {
	    e->eb_dir = *ep;
	    continue;
	}
	if (!strcasecmp (*ap, "mode")) {
	    e->eb_mode = *ep;
	    continue;
	}
	if (!strcasecmp (*ap, "size")) {
	    sscanf (*ep, "%lu", &e->eb_size);
	    continue;
	}
	if (!strcasecmp (*ap, "server")) {
	    e->eb_server = *ep;
	    continue;
	}
	if (!strcasecmp (*ap, "subject")) {
	    e->eb_subject = *ep;
	    continue;
	}
	if (composing && !strcasecmp (*ap, "body")) {
	    e->eb_body = getcpy (*ep);
	    continue;
	}
    }

    if (!e->eb_access) {
	advise (NULL,
		"invalid parameters for \"%s/%s\" type in message %s's %s field",
		ci->ci_type, ci->ci_subtype, ct->c_file, TYPE_FIELD);
	return NOTOK;
    }

    return OK;
}


/*
 * show content of type "message"
 *
 * We aren't handling message/partial or message/external
 * here yet.
 */

static int
show_message (CT ct, int serial, int alternate)
{
    char *cp, buffer[BUFSIZ];
    CI ci = &ct->c_ctinfo;

    /* Check for mhn-show-type/subtype */
    sprintf (buffer, "%s-show-%s/%s", invo_name, ci->ci_type, ci->ci_subtype);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_content_aux (ct, serial, alternate, cp, NULL);

    /* Check for mhn-show-type */
    sprintf (buffer, "%s-show-%s", invo_name, ci->ci_type);
    if ((cp = context_find (buffer)) && *cp != '\0')
	return show_content_aux (ct, serial, alternate, cp, NULL);

    if ((cp = ct->c_showproc))
	return show_content_aux (ct, serial, alternate, cp, NULL);

    /* default method for message/rfc822 */
    if (ct->c_subtype == MESSAGE_RFC822) {
	cp = (ct->c_showproc = add ("%pshow -file '%F'", NULL));
	return show_content_aux (ct, serial, alternate, cp, NULL);
    }

    /* complain if we are not a part of a multipart/alternative */
    if (!alternate)
	content_error (NULL, ct, "don't know how to display content");

    return NOTOK;
}


static int
show_external (CT ct, int serial, int alternate)
{
    struct exbody *e = (struct exbody *) ct->c_ctparams;
    CT p = e->eb_content;

    if (!type_ok (p, 0))
	return OK;

    if (p->c_ctshowfnx)
	return (*p->c_ctshowfnx) (p, serial, alternate);

    content_error (NULL, p, "don't know how to display content");
    return NOTOK;
}


static int
list_partial (CT ct, int toplevel)
{
    struct partial *p = (struct partial *) ct->c_ctparams;

    list_content (ct, toplevel);
    if (verbosw) {
	printf ("\t     [message %s, part %d", p->pm_partid, p->pm_partno);
	if (p->pm_maxno)
	    printf (" of %d", p->pm_maxno);
	printf ("]\n");
    }

    return OK;
}


static int
list_external (CT ct, int toplevel)
{
    struct exbody *e = (struct exbody *) ct->c_ctparams;

    list_content (ct, toplevel);
    if (verbosw) {
	if (e->eb_name)
	    printf ("\t     retrieve %s\n", e->eb_name);
	if (e->eb_dir)
	    printf ("\t in directory %s\n", e->eb_dir);
	if (e->eb_site)
	    printf ("\t         from %s\n", e->eb_site);
	if (e->eb_server)
	    printf ("\t from mailbox %s\n", e->eb_server);
	if (e->eb_subject)
	    printf ("\t with subject %s\n", e->eb_subject);
	printf     ("\t        using %s", e->eb_access);
	if (e->eb_mode)
	    printf (" (in %s mode)", e->eb_mode);
	if (e->eb_permission)
	    printf (" (permission %s)", e->eb_permission);
	if (e->eb_flags == NOTOK)
	    printf (" [service unavailable]");
	printf ("\n");
    }
    list_content (e->eb_content, 0);

    return OK;
}


static int
ct_compar (CT *a, CT *b)
{
    struct partial *am = (struct partial *) ((*a)->c_ctparams);
    struct partial *bm = (struct partial *) ((*b)->c_ctparams);

    return (am->pm_marked - bm->pm_marked);
}


static int
store_partial (CT ct, char *unused)
{
    int	cur, hi, i;
    CT p, *ctp, *ctq;
    CT *base;
    struct partial *qm = (struct partial *) ct->c_ctparams;

    if (qm->pm_stored)
	return OK;

    hi = i = 0;
    for (ctp = cts; *ctp; ctp++) {
	p = *ctp;
	if (p->c_type == CT_MESSAGE && p->c_subtype == ct->c_subtype) {
	    struct partial *pm = (struct partial *) p->c_ctparams;

	    if (!pm->pm_stored
	            && strcmp (qm->pm_partid, pm->pm_partid) == 0) {
		pm->pm_marked = pm->pm_partno;
		if (pm->pm_maxno)
		    hi = pm->pm_maxno;
		pm->pm_stored = 1;
		i++;
	    }
	    else
		pm->pm_marked = 0;
	}
    }

    if (hi == 0) {
	advise (NULL, "missing (at least) last part of multipart message");
	return NOTOK;
    }

    if ((base = (CT *) calloc ((size_t) (i + 1), sizeof(*base))) == NULL)
	adios (NULL, "out of memory");

    ctq = base;
    for (ctp = cts; *ctp; ctp++) {
	p = *ctp;
	if (p->c_type == CT_MESSAGE && p->c_subtype == ct->c_subtype) {
	    struct partial *pm = (struct partial *) p->c_ctparams;

	    if (pm->pm_marked)
		*ctq++ = p;
	}
    }
    *ctq = NULL;

    if (i > 1)
	qsort ((char *) base, i, sizeof(*base), (qsort_comp) ct_compar);

    cur = 1;
    for (ctq = base; *ctq; ctq++) {
	struct partial *pm;

	p = *ctq;
	pm = (struct partial *) p->c_ctparams;
	if (pm->pm_marked != cur) {
	    if (pm->pm_marked == cur - 1) {
		admonish (NULL,
			  "duplicate part %d of %d part multipart message",
			  pm->pm_marked, hi);
		continue;
	    }

missing_part: ;
	    advise (NULL,
		    "missing %spart %d of %d part multipart message",
		    cur != hi ? "(at least) " : "", cur, hi);
	    goto losing;
	}
        else
	    cur++;
    }
    if (hi != --cur) {
	cur = hi;
	goto missing_part;
    }

    ctq = base;
    ct = *ctq++;
    if (store_content (ct, "") == NOTOK) {
losing: ;
	free ((char *) base);
	return NOTOK;
    }

    for (; *ctq; ctq++) {
	p = *ctq;
	if (store_content (p, ct->c_storage) == NOTOK)
	    goto losing;
    }

    free ((char *) base);
    return OK;
}


static int
store_external (CT ct, char *unused)
{
    int	result = NOTOK;
    struct exbody *e = (struct exbody *) ct->c_ctparams;
    CT p = e->eb_content;

    if (!type_ok (p, 1))
	return OK;

    p->c_partno = ct->c_partno;
    if (p->c_ctstorefnx)
	result = (*p->c_ctstorefnx) (p, NULL);
    p->c_partno = NULL;

    return result;
}


static void
free_partial (CT ct)
{
    struct partial *p = (struct partial *) ct->c_ctparams;

    if (!p)
	return;

    if (p->pm_partid)
	free (p->pm_partid);

    free ((char *) p);
    ct->c_ctparams = NULL;
}


static void
free_external (CT ct)
{
    struct exbody *e = (struct exbody *) ct->c_ctparams;

    if (!e)
	return;

    free_content (e->eb_content);
    if (e->eb_body)
	free (e->eb_body);

    free ((char *) e);
    ct->c_ctparams = NULL;
}


/*
 * APPLICATION
 */

static int
InitApplication (CT ct)
{
    char **ap, **ep, *cp;
    struct k2v *kv;
    CI ci = &ct->c_ctinfo;

    /* match subtype */
    for (kv = SubApplication; kv->kv_key; kv++)
	if (!strcasecmp (ci->ci_subtype, kv->kv_key))
	    break;
    ct->c_subtype = kv->kv_value;

    /* setup methods for this type */
    ct->c_ctlistfnx = list_application;

    /* check if content specifies a filename */
    if (autosw) {
	for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
	    if (!ct->c_storeproc
		&& !strcasecmp (*ap, "name") 
		&& *(cp = *ep) != '/'
		&& *cp != '.'
		&& *cp != '|'
		&& *cp != '!'
		&& !strchr(cp, '%'))
		ct->c_storeproc = add (cp, NULL);
	}
    }

    if (ct->c_subtype == APPLICATION_OCTETS) {
	int tarP = 0;
	int zP = 0;

	for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
	    /* check for "type=tar" attribute */
	    if (!strcasecmp (*ap, "type")) {
		if (strcasecmp (*ep, "tar"))
		    break;

		tarP = 1;
		continue;
	    }

	    /* check for "conversions=compress" attribute */
	    if ((!strcasecmp (*ap, "conversions") || !strcasecmp (*ap, "x-conversions"))
		&& (!strcasecmp (*ep, "compress") || !strcasecmp (*ep, "x-compress"))) {
		zP = 1;
		continue;
	    }
	}

	if (tarP) {
	    ct->c_showproc = add (zP ? "%euncompress | tar tvf -"
				  : "%etar tvf -", NULL);
	    if (!ct->c_storeproc)
		if (autosw) {
		    ct->c_storeproc = add (zP ? "| uncompress | tar xvpf -"
					   : "| tar xvpf -", NULL);
		    ct->c_umask = 0022;
		} else {
		    ct->c_storeproc = add (zP ? "%m%P.tar.Z" : "%m%P.tar",
					   NULL);
		}
	}
    }

    return OK;
}


static int
list_application (CT ct, int toplevel)
{
    list_content (ct, toplevel);
    if (verbosw) {
	char **ap, **ep;
	CI ci = &ct->c_ctinfo;

	for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++)
	    printf ("\t     %s=\"%s\"\n", *ap, *ep);
    }

    return OK;
}


/*
 * TRANSFER ENCODINGS
 */

static int
init_encoding (CT ct, OpenCEFunc openfnx)
{
    CE ce;

    if ((ce = (CE) calloc (1, sizeof(*ce))) == NULL)
	adios (NULL, "out of memory");

    ct->c_cefile     = ce;
    ct->c_ceopenfnx  = openfnx;
    ct->c_ceclosefnx = close_encoding;
    ct->c_cesizefnx  = size_encoding;
    ct->c_celistfnx  = list_encoding;
    ct->c_cefreefnx  = free_encoding;

    return OK;
}


static int
list_encoding (CT ct)
{
    CE ce;

    if ((ce = ct->c_cefile))
	fprintf (stderr, "  decoded fp 0x%x file \"%s\"\n",
		 (unsigned int) ce->ce_fp, ce->ce_file ? ce->ce_file : "");

    return OK;
}


static void
close_encoding (CT ct)
{
    CE ce;

    if (!(ce = ct->c_cefile))
	return;

    if (ce->ce_fp) {
	fclose (ce->ce_fp);
	ce->ce_fp = NULL;
    }
}


static unsigned long
size_encoding (CT ct)
{
    int	fd;
    unsigned long size;
    char *file;
    CE ce;
    struct stat st;

    if (!(ce = ct->c_cefile))
	return (ct->c_end - ct->c_begin);

    if (ce->ce_fp && fstat (fileno (ce->ce_fp), &st) != NOTOK)
	return (long) st.st_size;

    if (ce->ce_file) {
	if (stat (ce->ce_file, &st) != NOTOK)
	    return (long) st.st_size;
	else
	    return 0L;
    }

    if (ct->c_encoding == CE_EXTERNAL)
	return (ct->c_end - ct->c_begin);	

    file = NULL;
    if ((fd = (*ct->c_ceopenfnx) (ct, &file)) == NOTOK)
	return (ct->c_end - ct->c_begin);

    if (fstat (fd, &st) != NOTOK)
	size = (long) st.st_size;
    else
	size = 0L;

    (*ct->c_ceclosefnx) (ct);
    return size;
}


static void
free_encoding (CT ct, int toplevel)
{
    CE ce;

    if (!(ce = ct->c_cefile))
	return;

    if (ce->ce_fp) {
	fclose (ce->ce_fp);
	ce->ce_fp = NULL;
    }

    if (ce->ce_file) {
	if (ce->ce_unlink)
	    unlink (ce->ce_file);
	free (ce->ce_file);
    }

    if (toplevel) {
	free ((char *) ce);
	ct->c_cefile = NULL;
    } else {
	ct->c_ceopenfnx = NULL;
    }
}


/*
 * BASE64
 */

static unsigned char b642nib[0x80] = {
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0xff, 0xff, 0x3e, 0xff, 0xff, 0xff, 0x3f,
    0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
    0x3c, 0x3d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 
    0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
    0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
    0x17, 0x18, 0x19, 0xff, 0xff, 0xff, 0xff, 0xff,
    0xff, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 
    0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
    0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
    0x31, 0x32, 0x33, 0xff, 0xff, 0xff, 0xff, 0xff
};


static int
InitBase64 (CT ct)
{
    return init_encoding (ct, openBase64);
}


static int
openBase64 (CT ct, char **file)
{
    int	bitno, cc, digested;
    int fd, len, skip;
    unsigned long bits;
    unsigned char value, *b, *b1, *b2, *b3;
    char *cp, *ep, buffer[BUFSIZ];
    CE ce;
    MD5_CTX mdContext;

    b  = (unsigned char *) &bits;
    b1 = &b[endian > 0 ? 1 : 2];
    b2 = &b[endian > 0 ? 2 : 1];
    b3 = &b[endian > 0 ? 3 : 0];

    ce = ct->c_cefile;
    if (ce->ce_fp) {
	fseek (ce->ce_fp, 0L, SEEK_SET);
	goto ready_to_go;
    }

    if (ce->ce_file) {
	if ((ce->ce_fp = fopen (ce->ce_file, "r")) == NULL) {
	    content_error (ce->ce_file, ct, "unable to fopen for reading");
	    return NOTOK;
	}
	goto ready_to_go;
    }

    if (*file == NULL) {
	ce->ce_file = add (m_scratch ("", tmp), NULL);
	ce->ce_unlink = 1;
    } else {
	ce->ce_file = add (*file, NULL);
	ce->ce_unlink = 0;
    }

    if ((ce->ce_fp = fopen (ce->ce_file, "w+")) == NULL) {
	content_error (ce->ce_file, ct, "unable to fopen for reading/writing");
	return NOTOK;
    }

    if ((len = ct->c_end - ct->c_begin) < 0)
	adios (NULL, "internal error(1)");

    if (!ct->c_fp && (ct->c_fp = fopen (ct->c_file, "r")) == NULL) {
	content_error (ct->c_file, ct, "unable to open for reading");
	return NOTOK;
    }
    
    if ((digested = ct->c_digested))
	MD5Init (&mdContext);

    bitno = 18;
    bits = 0L;
    skip = 0;

    lseek (fd = fileno (ct->c_fp), (off_t) ct->c_begin, SEEK_SET);
    while (len > 0) {
	switch (cc = read (fd, buffer, sizeof(buffer) - 1)) {
	case NOTOK:
	    content_error (ct->c_file, ct, "error reading from");
	    goto clean_up;

	case OK:
	    content_error (NULL, ct, "premature eof");
	    goto clean_up;

	default:
	    if (cc > len)
		cc = len;
	    len -= cc;

	    for (ep = (cp = buffer) + cc; cp < ep; cp++) {
		switch (*cp) {
		default:
		    if (isspace (*cp))
			break;
		    if (skip
			|| (*cp & 0x80)
			|| (value = b642nib[*cp & 0x7f]) > 0x3f) {
			if (debugsw)
			    fprintf (stderr, "*cp=0x%x pos=%ld skip=%d\n", *cp,
				     lseek (fd, (off_t)0, SEEK_CUR) - (ep - cp), skip);
			content_error (NULL, ct,
				       "invalid BASE64 encoding -- continuing");
			continue;
		    }

		    bits |= value << bitno;
test_end: ;
		    if ((bitno -= 6) < 0) {
			putc ((char) *b1, ce->ce_fp);
			if (digested)
			    MD5Update (&mdContext, b1, 1);
			if (skip < 2) {
			    putc ((char) *b2, ce->ce_fp);
			    if (digested)
				MD5Update (&mdContext, b2, 1);
			    if (skip < 1) {
				putc ((char) *b3, ce->ce_fp);
				if (digested)
				    MD5Update (&mdContext, b3, 1);
			    }
			}

			if (ferror (ce->ce_fp)) {
			    content_error (ce->ce_file, ct,
					   "error writing to");
			    goto clean_up;
			}
			bitno = 18, bits = 0L, skip = 0;
		    }
		    break;

		case '=':
		    if (++skip > 3)
			goto self_delimiting;
		    goto test_end;
		}
	    }
	}
    }

    if (bitno != 18) {
	if (debugsw)
	    fprintf (stderr, "premature ending (bitno %d)\n", bitno);

	content_error (NULL, ct, "invalid BASE64 encoding");
	goto clean_up;
    }

self_delimiting: ;
    fseek (ct->c_fp, 0L, SEEK_SET);

    if (fflush (ce->ce_fp)) {
	content_error (ce->ce_file, ct, "error writing to");
	goto clean_up;
    }

    if (digested) {
	unsigned char digest[16];

	MD5Final (digest, &mdContext);
	if (memcmp((char *) digest, (char *) ct->c_digest,
		   sizeof(digest) / sizeof(digest[0])))
	    content_error (NULL, ct,
			   "content integrity suspect (digest mismatch) -- continuing");
	else
	    if (debugsw)
		fprintf (stderr, "content integrity confirmed\n");
    }

    fseek (ce->ce_fp, 0L, SEEK_SET);

ready_to_go: ;
    *file = ce->ce_file;
    return fileno (ce->ce_fp);

clean_up: ;
    free_encoding (ct, 0);
    return NOTOK;
}


static void
set_endian (void)
{
    char *cp;
    union {
	long l;
	char c[sizeof(long)];
    } un;

    un.l = 1;
    endian = un.c[0] ? -1 : 1;
    if (debugsw)
	fprintf (stderr, "%s endian architecture\n",
		 endian > 0 ? "big" : "little");

    if ((cp = getenv ("MM_NOASK")) && !strcmp (cp, "1")) {
	nolist  = 1;
	listsw  = 0;
	pausesw = 0;
    }
}


/*
 * QUOTED PRINTABLE
 */

static char hex2nib[0x80] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


static int 
InitQuoted (CT ct)
{
    return init_encoding (ct, openQuoted);
}


static int
openQuoted (CT ct, char **file)
{
    int	cc, digested, len, quoted;
    char *cp, *ep;
    char buffer[BUFSIZ];
    unsigned char mask;
    CE ce;
    MD5_CTX mdContext;

    ce = ct->c_cefile;
    if (ce->ce_fp) {
	fseek (ce->ce_fp, 0L, SEEK_SET);
	goto ready_to_go;
    }

    if (ce->ce_file) {
	if ((ce->ce_fp = fopen (ce->ce_file, "r")) == NULL) {
	    content_error (ce->ce_file, ct, "unable to fopen for reading");
	    return NOTOK;
	}
	goto ready_to_go;
    }

    if (*file == NULL) {
	ce->ce_file = add (m_scratch ("", tmp), NULL);
	ce->ce_unlink = 1;
    } else {
	ce->ce_file = add (*file, NULL);
	ce->ce_unlink = 0;
    }

    if ((ce->ce_fp = fopen (ce->ce_file, "w+")) == NULL) {
	content_error (ce->ce_file, ct, "unable to fopen for reading/writing");
	return NOTOK;
    }

    if ((len = ct->c_end - ct->c_begin) < 0)
	adios (NULL, "internal error(2)");

    if (!ct->c_fp && (ct->c_fp = fopen (ct->c_file, "r")) == NULL) {
	content_error (ct->c_file, ct, "unable to open for reading");
	return NOTOK;
    }

    if ((digested = ct->c_digested))
	MD5Init (&mdContext);

    quoted = 0;
#ifdef lint
    mask = 0;
#endif

    fseek (ct->c_fp, ct->c_begin, SEEK_SET);
    while (len > 0) {
	char *dp;

	if (fgets (buffer, sizeof(buffer) - 1, ct->c_fp) == NULL) {
	    content_error (NULL, ct, "premature eof");
	    goto clean_up;
	}

	if ((cc = strlen (buffer)) > len)
	    cc = len;
	len -= cc;

	for (ep = (cp = buffer) + cc - 1; cp <= ep; ep--)
	    if (!isspace (*ep))
		break;
	*++ep = '\n', ep++;

	for (; cp < ep; cp++) {
	    if (quoted) {
		if (quoted > 1) {
		    if (!isxdigit (*cp)) {
invalid_hex:
			dp = "expecting hexidecimal-digit";
			goto invalid_encoding;
		    }
		    mask <<= 4;
		    mask |= hex2nib[*cp & 0x7f];
		    putc (mask, ce->ce_fp);
		    if (digested)
			MD5Update (&mdContext, &mask, 1);
		} else {
		    switch (*cp) {
		    case ':':
			putc (*cp, ce->ce_fp);
			if (digested)
			    MD5Update (&mdContext, (unsigned char *) ":", 1);
			break;

		    default:
			if (!isxdigit (*cp))
			    goto invalid_hex;
			mask = hex2nib[*cp & 0x7f];
			quoted = 2;
			continue;
		    }
		}

		if (ferror (ce->ce_fp)) {
		    content_error (ce->ce_file, ct, "error writing to");
		    goto clean_up;
		}
		quoted = 0;
		continue;
	    }

	    switch (*cp) {
	    default:
		if (*cp < '!' || *cp > '~') {
		    int	i;
		    dp = "expecting character in range [!..~]";

		invalid_encoding: ;
		    i = strlen (invo_name) + 2;
		    content_error (NULL, ct,
				   "invalid QUOTED-PRINTABLE encoding -- %s,\n%*.*sbut got char 0x%x",
				   dp, i, i, "", *cp);
		    goto clean_up;
		}
		/* and fall...*/
	    case ' ':
	    case '\t':
	    case '\n':
		putc (*cp, ce->ce_fp);
		if (digested) {
		    if (*cp == '\n')
			MD5Update (&mdContext, (unsigned char *) "\r\n",2);
		    else
			MD5Update (&mdContext, (unsigned char *) cp, 1);
		}
		if (ferror (ce->ce_fp)) {
		    content_error (ce->ce_file, ct, "error writing to");
		    goto clean_up;
		}
		break;

	    case '=':
		if (*++cp != '\n') {
		    quoted = 1;
		    cp--;
		}
		break;
	    }
	}
    }
    if (quoted) {
	content_error (NULL, ct,
		       "invalid QUOTED-PRINTABLE encoding -- end-of-content while still quoting");
	goto clean_up;
    }

    fseek (ct->c_fp, 0L, SEEK_SET);

    if (fflush (ce->ce_fp)) {
	content_error (ce->ce_file, ct, "error writing to");
	goto clean_up;
    }

    if (digested) {
	unsigned char digest[16];

	MD5Final (digest, &mdContext);
	if (memcmp((char *) digest, (char *) ct->c_digest,
		   sizeof(digest) / sizeof(digest[0])))
	    content_error (NULL, ct,
			   "content integrity suspect (digest mismatch) -- continuing");
	else
	    if (debugsw)
		fprintf (stderr, "content integrity confirmed\n");
    }

    fseek (ce->ce_fp, 0L, SEEK_SET);

ready_to_go: ;
    *file = ce->ce_file;
    return fileno (ce->ce_fp);

clean_up: ;
    free_encoding (ct, 0);
    return NOTOK;
}


/*
 * 7BIT
 */

static int
Init7Bit (CT ct)
{
    if (init_encoding (ct, open7Bit) == NOTOK)
	return NOTOK;

    ct->c_cesizefnx = NULL;	/* no need to decode for real size */
    return OK;
}


static int
open7Bit (CT ct, char **file)
{
    int	cc, fd, len;
    char buffer[BUFSIZ];
    CE ce;

    ce = ct->c_cefile;
    if (ce->ce_fp) {
	fseek (ce->ce_fp, 0L, SEEK_SET);
	goto ready_to_go;
    }

    if (ce->ce_file) {
	if ((ce->ce_fp = fopen (ce->ce_file, "r")) == NULL) {
	    content_error (ce->ce_file, ct, "unable to fopen for reading");
	    return NOTOK;
	}
	goto ready_to_go;
    }

    if (*file == NULL) {
	ce->ce_file = add (m_scratch ("", tmp), NULL);
	ce->ce_unlink = 1;
    } else {
	ce->ce_file = add (*file, NULL);
	ce->ce_unlink = 0;
    }

    if ((ce->ce_fp = fopen (ce->ce_file, "w+")) == NULL) {
	content_error (ce->ce_file, ct, "unable to fopen for reading/writing");
	return NOTOK;
    }

    if (ct->c_type == CT_MULTIPART) {
	char **ap, **ep;
	CI ci = &ct->c_ctinfo;

	len = 0;
	fprintf (ce->ce_fp, "%s: %s/%s", TYPE_FIELD, ci->ci_type, ci->ci_subtype);
	len += strlen (TYPE_FIELD) + 2 + strlen (ci->ci_type)
	    + 1 + strlen (ci->ci_subtype);
	for (ap = ci->ci_attrs, ep = ci->ci_values; *ap; ap++, ep++) {
	    putc (';', ce->ce_fp);
	    len++;

	    sprintf (buffer, "%s=\"%s\"", *ap, *ep);

	    if (len + 1 + (cc = strlen (buffer)) >= CPERLIN) {
		fputs ("\n\t", ce->ce_fp);
		len = 8;
	    } else {
		putc (' ', ce->ce_fp);
		len++;
	    }
	    fprintf (ce->ce_fp, "%s", buffer);
	    len += cc;
	}

	if (ci->ci_comment) {
	    if (len + 1 + (cc = 2 + strlen (ci->ci_comment)) >= CPERLIN) {
		fputs ("\n\t", ce->ce_fp);
		len = 8;
	    }
	    else {
		putc (' ', ce->ce_fp);
		len++;
	    }
	    fprintf (ce->ce_fp, "(%s)", ci->ci_comment);
	    len += cc;
	}
	fprintf (ce->ce_fp, "\n");
	if (ct->c_id)
	    fprintf (ce->ce_fp, "%s:%s", ID_FIELD, ct->c_id);
	if (ct->c_descr)
	    fprintf (ce->ce_fp, "%s:%s", DESCR_FIELD, ct->c_descr);
	fprintf (ce->ce_fp, "\n");
    }

    if ((len = ct->c_end - ct->c_begin) < 0)
	adios (NULL, "internal error(3)");

    if (!ct->c_fp && (ct->c_fp = fopen (ct->c_file, "r")) == NULL) {
	content_error (ct->c_file, ct, "unable to open for reading");
	return NOTOK;
    }

    lseek (fd = fileno (ct->c_fp), (off_t) ct->c_begin, SEEK_SET);
    while (len > 0)
	switch (cc = read (fd, buffer, sizeof(buffer) - 1)) {
	case NOTOK:
	    content_error (ct->c_file, ct, "error reading from");
	    goto clean_up;

	case OK:
	    content_error (NULL, ct, "premature eof");
	    goto clean_up;

	default:
	    if (cc > len)
		cc = len;
	    len -= cc;

	    fwrite (buffer, sizeof(*buffer), cc, ce->ce_fp);
	    if (ferror (ce->ce_fp)) {
		content_error (ce->ce_file, ct, "error writing to");
		goto clean_up;
	    }
	}

    fseek (ct->c_fp, 0L, SEEK_SET);

    if (fflush (ce->ce_fp)) {
	content_error (ce->ce_file, ct, "error writing to");
	goto clean_up;
    }

    fseek (ce->ce_fp, 0L, SEEK_SET);

ready_to_go: ;
    *file = ce->ce_file;
    return fileno (ce->ce_fp);

clean_up: ;
    free_encoding (ct, 0);
    return NOTOK;
}


/*
 * External
 */

static int
openExternal (CT ct, CT cb, CE ce, char **file, int *fd)
{
    char cachefile[BUFSIZ];

    if (ce->ce_fp) {
	fseek (ce->ce_fp, 0L, SEEK_SET);
	goto ready_already;
    }

    if (ce->ce_file) {
	if ((ce->ce_fp = fopen (ce->ce_file, "r")) == NULL) {
	    content_error (ce->ce_file, ct, "unable to fopen for reading");
	    return NOTOK;
	}
	goto ready_already;
    }

    if (find_cache (ct, rcachesw, (int *) 0, cb->c_id, cachefile) != NOTOK) {
	if ((ce->ce_fp = fopen (cachefile, "r"))) {
	    ce->ce_file = getcpy (cachefile);
	    ce->ce_unlink = 0;
	    goto ready_already;
	} else {
	    admonish (cachefile, "unable to fopen for reading");
	}
    }

    return OK;

ready_already:;
    *file = ce->ce_file;
    *fd = fileno (ce->ce_fp);
    return DONE;
}

/*
 * File
 */

static int
InitFile (CT ct)
{
    return init_encoding (ct, openFile);
}


static int
openFile (CT ct, char **file)
{
    int	fd, cachetype;
    char cachefile[BUFSIZ];
    struct exbody *e = ct->c_ctexbody;
    CE ce = ct->c_cefile;

    switch (openExternal (e->eb_parent, e->eb_content, ce, file, &fd)) {
	case NOTOK:
	    return NOTOK;

	case OK:
	    break;

	case DONE:
	    return fd;
    }

    if (!e->eb_name) {
	content_error (NULL, ct, "missing name parameter");
	return NOTOK;
    }

    ce->ce_file = getcpy (e->eb_name);
    ce->ce_unlink = 0;

    if ((ce->ce_fp = fopen (ce->ce_file, "r")) == NULL) {
	content_error (ce->ce_file, ct, "unable to fopen for reading");
	return NOTOK;
    }

    if ((!e->eb_permission || strcasecmp (e->eb_permission, "read-write"))
	    && find_cache (NULL, wcachesw, &cachetype,
			   e->eb_content->c_id, cachefile) != NOTOK) {
	int mask;
	FILE *fp;

	mask = umask (cachetype ? ~m_gmprot () : 0222);
	if ((fp = fopen (cachefile, "w"))) {
	    int	cc;
	    char buffer[BUFSIZ];
	    FILE *gp = ce->ce_fp;

	    fseek (gp, 0L, SEEK_SET);

	    while ((cc = fread (buffer, sizeof(*buffer), sizeof(buffer), gp))
		       > 0)
		fwrite (buffer, sizeof(*buffer), cc, fp);
	    fflush (fp);

	    if (ferror (gp)) {
		admonish (ce->ce_file, "error reading");
		unlink (cachefile);
	    }
	    else
		if (ferror (fp)) {
		    admonish (cachefile, "error writing");
		    unlink (cachefile);
		}
	    fclose (fp);
	}
	umask (mask);
    }

    fseek (ce->ce_fp, 0L, SEEK_SET);
    *file = ce->ce_file;
    return fileno (ce->ce_fp);
}

/*
 * FTP
 */

static int
InitFTP (CT ct)
{
    return init_encoding (ct, openFTP);
}


static int
openFTP (CT ct, char **file)
{
    int	cachetype, caching, fd;
    char *bp, *ftp, *user, *pass;
    char buffer[BUFSIZ], cachefile[BUFSIZ];
    struct exbody *e;
    CE ce;
    static char *username = NULL;
    static char *password = NULL;

    e  = ct->c_ctexbody;
    ce = ct->c_cefile;

    sprintf (buffer, "%s-access-ftp", invo_name);
    if ((ftp = context_find (buffer)) && !*ftp)
	ftp = NULL;

#ifndef BUILTIN_FTP
    if (!ftp)
	return NOTOK;
#endif

    switch (openExternal (e->eb_parent, e->eb_content, ce, file, &fd)) {
	case NOTOK:
	    return NOTOK;

	case OK:
	    break;

	case DONE:
	    return fd;
    }

    if (!e->eb_name || !e->eb_site) {
	content_error (NULL, ct, "missing %s parameter",
		       e->eb_name ? "site": "name");
	return NOTOK;
    }

    if (xpid) {
	if (xpid < 0)
	    xpid = -xpid;
	pidcheck (pidwait (xpid, NOTOK));
	xpid = 0;
    }

    bp = buffer;
    sprintf (bp, "Retrieve %s", e->eb_name);
    bp += strlen (bp);
    if (e->eb_partno) {
	sprintf (bp, " (content %s)", e->eb_partno);
	bp += strlen (bp);
    }
    sprintf (bp, "\n    using %sFTP from site %s",
		    e->eb_flags ? "anonymous " : "", e->eb_site);
    bp += strlen (bp);
    if (e->eb_size > 0) {
	sprintf (bp, " (%lu octets)", e->eb_size);
	bp += strlen (bp);
    }
    sprintf (bp, "? ");
    if (!getanswer (buffer))
	return NOTOK;

    if (e->eb_flags) {
	user = "anonymous";
	sprintf (pass = buffer, "%s@%s", getusr (), LocalName ());
    } else {
	ruserpass (e->eb_site, &username, &password);
	user = username;
	pass = password;
    }

    ce->ce_unlink = (*file == NULL);
    caching = 0;
    cachefile[0] = '\0';
    if ((!e->eb_permission || strcasecmp (e->eb_permission, "read-write"))
	    && find_cache (NULL, wcachesw, &cachetype,
			   e->eb_content->c_id, cachefile) != NOTOK) {
	if (*file == NULL) {
	    ce->ce_unlink = 0;
	    caching = 1;
	}
    }

    if (*file == NULL)
	ce->ce_file = add (*file, NULL);
    else if (caching)
	ce->ce_file = add (cachefile, NULL);
    else
	ce->ce_file = add (m_scratch ("", tmp), NULL);

    if ((ce->ce_fp = fopen (ce->ce_file, "w+")) == NULL) {
	content_error (ce->ce_file, ct, "unable to fopen for reading/writing");
	return NOTOK;
    }

#ifdef BUILTIN_FTP
    if (ftp)
#endif
    {
	int child_id, i, vecp;
	char *vec[9];

	vecp = 0;
	vec[vecp++] = r1bindex (ftp, '/');
	vec[vecp++] = e->eb_site;
	vec[vecp++] = user;
	vec[vecp++] = pass;
	vec[vecp++] = e->eb_dir;
	vec[vecp++] = e->eb_name;
	vec[vecp++] = ce->ce_file,
	vec[vecp++] = e->eb_mode && !strcasecmp (e->eb_mode, "ascii")
	    		? "ascii" : "binary";
	vec[vecp] = NULL;

	fflush (stdout);

	for (i = 0; (child_id = vfork ()) == NOTOK && i < 5; i++)
	    sleep (5);
	switch (child_id) {
	    case NOTOK:
	        adios ("fork", "unable to");
		/* NOTREACHED */

	    case OK:
		close (fileno (ce->ce_fp));
		execvp (ftp, vec);
		fprintf (stderr, "unable to exec ");
		perror (ftp);
		_exit (-1);
		/* NOTREACHED */

	    default:
		if (pidXwait (child_id, NULL)) {
#ifdef BUILTIN_FTP
losing_ftp: ;
#endif
		    username = password = NULL;
		    ce->ce_unlink = 1;
		    return NOTOK;
		}
		break;
	}
    }
#ifdef BUILTIN_FTP
    else
	if (ftp_get (e->eb_site, user, pass, e->eb_dir, e->eb_name,
		     ce->ce_file,
		     e->eb_mode && !strcasecmp (e->eb_mode, "ascii"), 0)
	        == NOTOK)
	    goto losing_ftp;
#endif

    if (cachefile[0])
	if (caching)
	    chmod (cachefile, cachetype ? m_gmprot () : 0444);
	else {
	    int	mask;
	    FILE *fp;

	    mask = umask (cachetype ? ~m_gmprot () : 0222);
	    if ((fp = fopen (cachefile, "w"))) {
		int cc;
		FILE *gp = ce->ce_fp;

		fseek (gp, 0L, SEEK_SET);

		while ((cc = fread (buffer, sizeof(*buffer), sizeof(buffer), gp))
		           > 0)
		    fwrite (buffer, sizeof(*buffer), cc, fp);
		fflush (fp);

		if (ferror (gp)) {
		    admonish (ce->ce_file, "error reading");
		    unlink (cachefile);
		}
		else
		    if (ferror (fp)) {
			admonish (cachefile, "error writing");
			unlink (cachefile);
		    }
		fclose (fp);
	    }
	    umask (mask);
	}

    fseek (ce->ce_fp, 0L, SEEK_SET);
    *file = ce->ce_file;
    return fileno (ce->ce_fp);
}


/*
 * Mail
 */

static int
InitMail (CT ct)
{
    return init_encoding (ct, openMail);
}


static int
openMail (CT ct, char **file)
{
    int	child_id, fd, i, vecp;
    char *bp, buffer[BUFSIZ], *vec[7];
    struct exbody *e = ct->c_ctexbody;
    CE ce = ct->c_cefile;

    switch (openExternal (e->eb_parent, e->eb_content, ce, file, &fd)) {
	case NOTOK:
	    return NOTOK;

	case OK:
	    break;

	case DONE:
	    return fd;
    }

    if (!e->eb_server) {
	content_error (NULL, ct, "missing server parameter");
	return NOTOK;
    }

    if (xpid) {
	if (xpid < 0)
	    xpid = -xpid;
	pidcheck (pidwait (xpid, NOTOK));
	xpid = 0;
    }

    bp = buffer;
    sprintf (bp, "Retrieve content");
    bp += strlen (bp);
    if (e->eb_partno) {
	sprintf (bp, " %s", e->eb_partno);
	bp += strlen (bp);
    }
    sprintf (bp, " by asking %s\n\n%s\n? ",
		    e->eb_server,
		    e->eb_subject ? e->eb_subject : e->eb_body);
    if (!getanswer (buffer))
	return NOTOK;

    vecp = 0;
    vec[vecp++] = r1bindex (mailproc, '/');
    vec[vecp++] = e->eb_server;
    vec[vecp++] = "-subject";
    vec[vecp++] = e->eb_subject ? e->eb_subject : "mail-server request";
    vec[vecp++] = "-body";
    vec[vecp++] = e->eb_body;
    vec[vecp] = NULL;

    for (i = 0; (child_id = vfork ()) == NOTOK && i < 5; i++)
	sleep (5);
    switch (child_id) {
	case NOTOK:
	    advise ("fork", "unable to");
	    return NOTOK;

	case OK:
	    execvp (mailproc, vec);
	    fprintf (stderr, "unable to exec ");
	    perror (mailproc);
	    _exit (-1);
	    /* NOTREACHED */

	default:
	    if (pidXwait (child_id, NULL) == OK)
		advise (NULL, "request sent");
	    break;
    }

    if (*file == NULL) {
	ce->ce_file = add (m_scratch ("", tmp), NULL);
	ce->ce_unlink = 1;
    } else {
	ce->ce_file = add (*file, NULL);
	ce->ce_unlink = 0;
    }

    if ((ce->ce_fp = fopen (ce->ce_file, "w+")) == NULL) {
	content_error (ce->ce_file, ct, "unable to fopen for reading/writing");
	return NOTOK;
    }

    if (ct->c_showproc)
	free (ct->c_showproc);
    ct->c_showproc = add ("true", NULL);

    fseek (ce->ce_fp, 0L, SEEK_SET);
    *file = ce->ce_file;
    return fileno (ce->ce_fp);
}

/*
 * CACHE
 */

static int
find_cache (CT ct, int policy, int *writing, char *id, char *buffer)
{
    int	status = NOTOK;

    if (id == NULL)
	return NOTOK;
    id = trimcpy (id);

    if (debugsw)
	fprintf (stderr, "find_cache %s(%d) %s %s\n",
		 caches[policy].sw, policy, writing ? "writing" : "reading",
		 id);

    switch (policy) {
	case CACHE_NEVER:
	default:
	    break;

	case CACHE_ASK:
	case CACHE_PUBLIC:
	    if (cache_private
		    && !writing
		    && find_cache_aux (writing ? 2 : 0, cache_private, id,
				       buffer) == OK) {
		if (access (buffer, R_OK) != NOTOK) {
got_private:
		    if (writing)
			*writing = 1;
got_it:
		    status = OK;
		    break;
		}
	    }
	    if (cache_public
		    && find_cache_aux (writing ? 1 : 0, cache_public, id,
				       buffer) == OK) {
		if (writing || access (buffer, R_OK) != NOTOK) {
		    if (writing)
			*writing = 0;
		    goto got_it;
		}
	    }
	    break;

	case CACHE_PRIVATE:
	    if (cache_private
		    && find_cache_aux (writing ? 2 : 0, cache_private, id,
				       buffer) == OK) {
		if (writing || access (buffer, R_OK) != NOTOK)
		    goto got_private;
	    }
	    break;

    }

    if (status == OK && policy == CACHE_ASK) {
	char   *bp,
		query[BUFSIZ];

	if (xpid) {
	    if (xpid < 0)
		xpid = -xpid;
	    pidcheck (pidwait (xpid, NOTOK));
	    xpid = 0;
	}

	bp = query;
	if (writing)
	    sprintf (bp, "Make cached, publically-accessible copy");
	else {
	    struct stat st;

	    sprintf (bp, "Use cached copy");
	    bp += strlen (bp);
	    if (ct->c_partno) {
		sprintf (bp, " of content %s", ct->c_partno);
		bp += strlen (bp);
	    }
	    stat (buffer, &st);
	    sprintf (bp, " (size %lu octets)",
			    (unsigned long) st.st_size);
	}
	bp += strlen (bp);
	sprintf (bp, "\n    in file %s? ", buffer);
	if (!getanswer (query))
	    status = NOTOK;
    }
    if (status == OK && writing) {
	if (*writing && strchr(buffer, '/'))
	    make_intermediates (buffer);
	unlink (buffer);
    }

    free (id);
    return status;
}


static int
find_cache_aux (int writing, char *directory, char *id, char *buffer)
{
    int	mask, usemap;
    char mapfile[BUFSIZ], mapname[BUFSIZ];
    FILE *fp;
    static int partno, pid;
    static time_t clock = 0;

#ifdef BSD42
    usemap = strchr(id, '/') ? 1 : 0;
#else
    usemap = 1;
#endif

    if (debugsw)
	fprintf (stderr, "find_cache_aux %s usemap=%d\n", directory, usemap);

    sprintf (mapfile, "%s/cache.map", directory);
    if (find_cache_aux2 (mapfile, id, mapname) == OK)
	goto done_map;

    if (!writing) {
	if (usemap)
	    return NOTOK;

use_raw: ;
	sprintf (buffer, "%s/%s", directory, id);
	return OK;
    }

    if (!usemap && access (mapfile, W_OK) == NOTOK)
	goto use_raw;

    if (clock != 0) {
	time_t now;
	
	time (&now);
	if (now > clock)
	    clock = 0;
    }
    else
	pid = getpid ();
    if (clock == 0) {
	time (&clock);
	partno = 0;
    }
    else
	if (partno > 0xff) {
	    clock++;
	    partno = 0;
	}

    sprintf (mapname, "%08x%04x%02x",
		(unsigned int) (clock & 0xffffffff),
		(unsigned int) (pid & 0xffff),
		(unsigned int) (partno++ & 0xff));

    if (debugsw)
	fprintf (stderr, "creating mapping %s->%s\n", mapname, id);

    make_intermediates (mapfile);
    mask = umask (writing == 2 ? 0077 : 0);
    if (!(fp = lkfopen (mapfile, "a")) && errno == ENOENT) {
	int fd;

	if ((fd = creat (mapfile, 0666)) != NOTOK) {
	    close (fd);
	    fp = lkfopen (mapfile, "a");
	}
    }
    umask (mask);
    if (!fp)
	return NOTOK;
    fprintf (fp, "%s: %s\n", mapname, id);
    lkfclose (fp, mapfile);

done_map: ;
    if (*mapname == '/')
	strcpy (buffer, mapname);
    else
	sprintf (buffer, "%s/%s", directory, mapname);
    if (debugsw)
	fprintf (stderr, "use %s\n", buffer);

    return OK;
}


static int
find_cache_aux2 (char *mapfile, char *id, char *mapname)
{
    int	state;
    char buf[BUFSIZ], name[NAMESZ];
    FILE *fp;

    if (!(fp = lkfopen (mapfile, "r")))
	return NOTOK;

    for (state = FLD;;) {
	int result;
	char *cp, *dp;

	switch (state = m_getfld (state, name, buf, sizeof(buf), fp)) {
	    case FLD:
	    case FLDPLUS:
	    case FLDEOF:
	        strcpy (mapname, name);
		if (state != FLDPLUS)
		    cp = buf;
		else {
		    cp = add (buf, NULL);
		    while (state == FLDPLUS) {
			state = m_getfld (state, name, buf, sizeof(buf), fp);
			cp = add (buf, cp);
		    }
		}
		dp = trimcpy (cp);
		if (cp != buf)
		    free (cp);
		if (debugsw)
		    fprintf (stderr, "compare %s to %s <- %s\n", id, dp,
			     mapname);
		result = strcmp (id, dp);
		free (dp);
		if (result == 0) {
		    lkfclose (fp, mapfile);
		    return OK;
		}
		if (state != FLDEOF)
		    continue;
		/* else fall... */

	    case BODY:
	    case BODYEOF:
	    case FILEEOF:
	    default:
		break;
	}
	break;
    }

    lkfclose (fp, mapfile);
    return NOTOK;
}


static void
cache_content (CT ct)
{
    int	cachetype;
    char *file, cachefile[BUFSIZ];
    CE ce = ct->c_cefile;

    if (!ct->c_id) {
	advise (NULL, "no %s: field in %s", ID_FIELD, ct->c_file);
	return;
    }

    if (!ce) {
	advise (NULL, "unable to decode %s", ct->c_file);
	return;
    }

    if (ct->c_ceopenfnx == openMail) {
	advise (NULL, "a radish may no know Greek, but I do...");
	return;
    }

    if (find_cache (NULL, wcachesw != CACHE_NEVER ? wcachesw : CACHE_ASK,
		    &cachetype, ct->c_id, cachefile)
	    == NOTOK) {
	advise (NULL, "unable to cache %s's contents", ct->c_file);
	return;
    }
    if (wcachesw != CACHE_NEVER && wcachesw != CACHE_ASK) {
	fflush (stdout);
	fprintf (stderr, "caching message %s as file %s\n", ct->c_file,
		 cachefile);
    }

    if (ce->ce_file) {
	int mask = umask (cachetype ? ~m_gmprot () : 0222);
	FILE *fp;

	if (debugsw)
	    fprintf (stderr, "caching by copying %s...\n", ce->ce_file);

	file = NULL;
	if ((*ct->c_ceopenfnx) (ct, &file) == NOTOK)
	    goto reset_umask;

	if ((fp = fopen (cachefile, "w"))) {
	    int	cc;
	    char buffer[BUFSIZ];
	    FILE *gp = ce->ce_fp;

	    fseek (gp, 0L, SEEK_SET);

	    while ((cc = fread (buffer, sizeof(*buffer), sizeof(buffer), gp))
		       > 0)
		fwrite (buffer, sizeof(*buffer), cc, fp);
	    fflush (fp);

	    if (ferror (gp)) {
		admonish (ce->ce_file, "error reading");
		unlink (cachefile);
	    }
	    else
		if (ferror (fp)) {
		    admonish (cachefile, "error writing");
		    unlink (cachefile);
		}
	    fclose (fp);
	}
	else
	    content_error (cachefile, ct, "unable to fopen for writing");
reset_umask: ;
	umask (mask);
    }
    else {
	if (debugsw)
	    fprintf (stderr, "in place caching...\n");

	file = cachefile;
	if ((*ct->c_ceopenfnx) (ct, &file) != NOTOK)
	    chmod (cachefile, cachetype ? m_gmprot () : 0444);
    }
}


static char nib2b64[0x40+1] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";


static int
writeBase64aux (FILE *in, FILE *out)
{
    int	cc, n;
    char inbuf[3];

    n = BPERLIN;
    while ((cc = fread (inbuf, sizeof(*inbuf), sizeof(inbuf), in)) > 0) {
	unsigned long bits;
	char *bp;
	char outbuf[4];

	if (cc < sizeof(inbuf)) {
	    inbuf[2] = 0;
	    if (cc < sizeof(inbuf) - 1)
		inbuf[1] = 0;
	}
	bits = (inbuf[0] & 0xff) << 16;
	bits |= (inbuf[1] & 0xff) << 8;
	bits |= inbuf[2] & 0xff;

	for (bp = outbuf + sizeof(outbuf); bp > outbuf; bits >>= 6)
	    *--bp = nib2b64[bits & 0x3f];
	if (cc < sizeof(inbuf)) {
	    outbuf[3] = '=';
	    if (cc < sizeof inbuf - 1)
		outbuf[2] = '=';
	}

	fwrite (outbuf, sizeof(*outbuf), sizeof(outbuf), out);

	if (cc < sizeof(inbuf)) {
	    putc ('\n', out);
	    return OK;
	}

	if (--n <= 0) {
	    n = BPERLIN;
	    putc ('\n', out);
	}
    }
    if (n != BPERLIN)
	putc ('\n', out);

    return OK;
}


static int
readDigest (CT ct, char *cp)
{
    int	bitno, skip;
    unsigned long bits;
    char *bp = cp;
    unsigned char *dp, value, *ep;
    unsigned char *b, *b1, *b2, *b3;

    b  = (unsigned char *) &bits,
    b1 = &b[endian > 0 ? 1 : 2],
    b2 = &b[endian > 0 ? 2 : 1],
    b3 = &b[endian > 0 ? 3 : 0];
    bitno = 18;
    bits = 0L;
    skip = 0;

    for (ep = (dp = ct->c_digest)
	         + sizeof(ct->c_digest) / sizeof(ct->c_digest[0]); *cp; cp++)
	switch (*cp) {
	    default:
	        if (skip
		        || (*cp & 0x80)
		        || (value = b642nib[*cp & 0x7f]) > 0x3f) {
		    if (debugsw)
			fprintf (stderr, "invalid BASE64 encoding\n");
		    return NOTOK;
		}

		bits |= value << bitno;
test_end: ;
		if ((bitno -= 6) < 0) {
		    if (dp + (3 - skip) > ep)
			goto invalid_digest;
		    *dp++ = *b1;
		    if (skip < 2) {
			*dp++ = *b2;
			if (skip < 1)
			    *dp++ = *b3;
		    }
		    bitno = 18;
		    bits = 0L;
		    skip = 0;
		}
		break;

	    case '=':
		if (++skip > 3)
		    goto self_delimiting;
		goto test_end;
	}
    if (bitno != 18) {
	if (debugsw)
	    fprintf (stderr, "premature ending (bitno %d)\n", bitno);

	return NOTOK;
    }
self_delimiting: ;
    if (dp != ep) {
invalid_digest: ;
	if (debugsw) {
	    while (*cp)
		cp++;
	    fprintf (stderr, "invalid MD5 digest (got %d octets)\n",
		     cp - bp);
	}

	return NOTOK;
    }

    if (debugsw) {
	fprintf (stderr, "MD5 digest=");
	for (dp = ct->c_digest; dp < ep; dp++)
	    fprintf (stderr, "%02x", *dp & 0xff);
	fprintf (stderr, "\n");
    }

    return OK;
}


/*
 * VIAMAIL
 */

static int
via_mail (char *mailsw, char *subjsw, char *parmsw, char *descsw,
          char *cmntsw, int slowsw, char *fromsw)
{
    int	nlines, nparts, status;
    long pos;
    long offset;
    char tmpfil[BUFSIZ];
    struct stat st;
    FILE *fp;

    umask (~m_gmprot ());

    strcpy (tmpfil, m_tmpfil (invo_name));
    if ((fp = fopen (tmpfil, "w+")) == NULL)
	adios (tmpfil, "unable to open for writing");
    chmod (tmpfil, 0600);

    if (!strchr(mailsw, '@'))
	mailsw = concat (mailsw, "@", LocalName (), NULL);
    fprintf (fp, "To: %s\n", mailsw);
    nlines = 1;
    if (subjsw)
	fprintf (fp, "Subject: %s\n", subjsw), nlines++;
    if (fromsw) {
	if (!strchr(fromsw, '@'))
	    fromsw = concat (fromsw, "@", LocalName (), NULL);
	fprintf (fp, "From: %s\n", fromsw), nlines++;
    }
    fprintf (fp, "%s: %s\n", VRSN_FIELD, VRSN_VALUE), nlines++;
    offset = ftell (fp);
    fprintf (fp, "%s: application/octet-stream", TYPE_FIELD);
    if (parmsw)
	fprintf (fp, "; %s", parmsw);
    if (cmntsw)
	fprintf (fp, "\n\t(%s)", cmntsw), nlines++;
    if (descsw)
	fprintf (fp, "\n%s: %s", DESCR_FIELD, descsw), nlines++;
    fprintf (fp, "\n%s: %s\n\n", ENCODING_FIELD, "base64"), nlines += 2;
    if (fflush (fp))
	adios (tmpfil, "error writing to");

    pos = ftell (fp);
    writeBase64aux (stdin, fp);
    if (fflush (fp))
	adios (tmpfil, "error writing to");

    if (fstat (fileno (fp), &st) == NOTOK)
	adios ("failed", "fstat of %s", tmpfil);
    nlines += (((long) st.st_size - pos) + CPERLIN) / (CPERLIN + 1);
    nparts = (nlines + (LPERMSG - 1)) / LPERMSG;

    if (nparts <= 1)
	status = via_post (tmpfil, 0);
    else {
	int partno;
	time_t clock;
	char buffer[BUFSIZ];
	char msgid[BUFSIZ];

	if (verbosw) {
	    printf ("sending binary image as %d partial messages\n", nparts);
	    fflush (stdout);
	}

	time (&clock);
	sprintf (msgid, "<%d.%ld@%s>",
		(int) getpid(), (long) clock, LocalName());

	fseek (fp, offset, SEEK_SET);
	for (partno = 1; partno <= nparts; partno++) {
	    int	lineno;
	    char tmpdrf[BUFSIZ];
	    FILE *out;

	    strcpy (tmpdrf, m_tmpfil (invo_name));
	    if ((out = fopen (tmpdrf, "w")) == NULL)
		adios (tmpdrf, "unable to open for writing");
	    chmod (tmpdrf, 0600);

	    fprintf (out, "To: %s\n", mailsw);
	    if (subjsw)
		fprintf (out, "Subject: %s\n", subjsw);
	    fprintf (out, "%s: %s\n", VRSN_FIELD, VRSN_VALUE);
	    fprintf (out,
		     "%s: message/partial; id=\"%s\";\n\tnumber=%d; total=%d\n",
		     TYPE_FIELD, msgid, partno, nparts);
	    fprintf (out, "%s: part %d of %d\n\n", DESCR_FIELD, partno,
		     nparts);

	    if (partno == 1)
		fprintf (out, "Message-ID: %s\n", msgid);

	    for (lineno = LPERMSG; lineno > 0; lineno--) {
		if (!fgets (buffer, sizeof buffer, fp)) {
		    if (partno == nparts)
			break;
		    adios (NULL, "premature eof");
		}

		fputs (buffer, out);
	    }
	    offset = ftell (fp);

	    if (fflush (out))
		adios (tmpdrf, "error writing to");

	    fclose (out);

	    status = via_post (tmpdrf, slowsw == 0);
	    unlink (tmpdrf);
	    if (status)
		break;

	    if (slowsw > 0 && partno < nparts) {
		if (verbosw) {
		    printf ("pausing %d seconds before sending part %d...\n",
			    slowsw, partno + 1);
		    fflush (stdout);
		}

		sleep ((unsigned) slowsw);
	    }
	}
    }

    fclose (fp);
    unlink (tmpfil);
    done (status ? 1 : 0);
}


static int
via_post (char *file, int queued)
{
    pid_t child_id;
    int i;

    for (i = 0; (child_id = fork()) == NOTOK && i < 5; i++)
	sleep (5);
    switch (child_id) {
        case NOTOK:
	    adios ("fork", "unable to");
	    /* NOTREACHED */

	case OK:
	    execlp (postproc, r1bindex (postproc, '/'), file,
			   queued ? "-queued" : NULL, NULL);
	    fprintf (stderr, "unable to exec ");
	    perror (postproc);
	    _exit (-1);
	    /* NOTREACHED */

	default:
	    return pidXwait (child_id, postproc);
    }
}


void done (int status)
{
    CT *ctp;

    if ((ctp = cts))
	for (; *ctp; ctp++)
	    free_content (*ctp);

    exit (status);
}


static int
pidcheck (int status)
{
    if ((status & 0xff00) == 0xff00 || (status & 0x007f) != SIGQUIT)
	return status;

    fflush (stdout);
    fflush (stderr);
    done (1);
    /* NOTREACHED */
}
