
/*
 * folder_addmsg.c -- Link message into folder
 *
 * $Id$
 */

#include <h/mh.h>
#include <fcntl.h>
#include <errno.h>

/*
 * Link message into a folder.  Return the new number
 * of the message.  If an error occurs, return -1.
 */

int
folder_addmsg (struct msgs **mpp, char *msgfile, int selected,
               int unseen, int preserve)
{
    int infd, outfd, linkerr, msgnum;
    char *nmsg, newmsg[BUFSIZ];
    struct msgs *mp;
    struct stat st1, st2;

    mp = *mpp;

    /*
     * We might need to make several attempts
     * in order to add the message to the folder.
     */
    for (;;) {

	/* should we preserve the numbering of the message? */
	if (preserve && (msgnum = m_atoi (msgfile)) > 0) {
	    nmsg = msgfile;
	    preserve = 0;	/* only try this once */
	} else {
	    /* else use the next available number */
	    msgnum = mp->hghmsg + 1;
	    nmsg = m_name (msgnum);
	}

	/*
	 * See if we need more space.  If so, then allocate
         * space for an addition 100 messages.
         */
	if (msgnum > mp->hghoff) {
	    if ((mp = folder_realloc (mp, 0, msgnum + 100)))
		*mpp = mp;
	    else {
		advise (NULL, "unable to allocate folder storage");
		return -1;
	    }
	}

	/*
	 * If a message is already in that slot,
	 * then loop to next available slot.
	 */
	if (does_exist (mp, msgnum)) {
	    continue;
	} else {
	    clear_msg_flags (mp, msgnum);
	    set_exists (mp, msgnum);
	    mp->nummsg++;
	}

	/* should we set the UNSEEN bit? */
	if (unseen)
	    set_unseen (mp, msgnum);

	/* should we set the SELECTED bit? */
	if (selected) {
	    set_selected (mp, msgnum);

	    if (mp->lowsel == 0 || msgnum < mp->lowsel)
		mp->lowsel = msgnum;
	    if (msgnum > mp->hghsel)
		mp->hghsel = msgnum;
	}

	if (mp->lowmsg == 0 || msgnum < mp->lowmsg)
	    mp->lowmsg = msgnum;
	if (msgnum > mp->hghmsg)
	    mp->hghmsg = msgnum;

	sprintf (newmsg, "%s/%s", mp->foldpath, nmsg);

	/*
	 * Now try to link message into folder
	 */
	if (link (msgfile, newmsg) != NOTOK) {
	    return msgnum;
	} else {
	    linkerr = errno;

#ifdef EISREMOTE
	    if (linkerr == EISREMOTE)
		linkerr = EXDEV;
#endif /* EISREMOTE */

	    /*
	     * Check if the file in our desired location is the same
	     * as the source file.  If so, then just leave it alone
	     * and return.  Otherwise, we will continue the main loop
	     * and try again at another slot (hghmsg+1).
	     */
	    if (linkerr == EEXIST) {
		if (stat (msgfile, &st2) == OK && stat (newmsg, &st1) == OK
		    && st2.st_ino == st1.st_ino) {
		    return msgnum;
		} else {
		    continue;
		}
	    }

	    /*
	     * If link failed because we are trying to link
	     * across devices, then check if there is a message
	     * already in the desired location.  If so, then return
	     * error, else just copy the message.
	     */
	    if (linkerr == EXDEV) {
		if (stat (newmsg, &st1) == OK) {
		    advise (NULL, "message %s:%s already exists", newmsg);
		    return -1;
		} else {
		    if ((infd = open (msgfile, O_RDONLY)) == NOTOK) {
			advise (msgfile, "unable to open message %s");
			return -1;
		    }
		    fstat (infd, &st1);
		    if ((outfd = creat (newmsg, (int) st1.st_mode & 0777)) == NOTOK) {
			advise (newmsg, "unable to create");
			close (infd);
			return -1;
		    }
		    cpydata (infd, outfd, msgfile, newmsg);
		    close (infd);
		    close (outfd);
		    return msgnum;
		}
	    }

	    /*
	     * Else, some other type of link error,
	     * so just return error.
	     */
	    advise (newmsg, "error linking %s to", msgfile);
	    return -1;
	}
    }
}
