
/*
 * rcvstore.c -- asynchronously add mail to a folder
 *
 * $Id$
 */

#include <h/mh.h>
#include <h/signals.h>
#include <errno.h>
#include <signal.h>

static struct swit switches[] = {
#define CRETSW         0
    { "create",	0 },
#define NCRETSW        1
    { "nocreate", 0 },
#define PUBSW          2
    { "public",	0 },
#define NPUBSW         3
    { "nopublic",  0 },
#define SEQSW          4
    { "sequence name", 0 },
#define ZEROSW         5
    { "zero",	0 },
#define NZEROSW        6
    { "nozero",	0 },
#define VERSIONSW      7
    { "version", 0 },
#define HELPSW         8
    { "help", 4 },
    { NULL, 0 }
};

extern int errno;

static char *tmpfilenam = NULL;

/*
 * prototypes
 */
static int folder_addmsg(struct msgs *, char *);


main (int argc, char **argv)
{
    int publicsw = -1, zerosw = 0, msgnum;
    int create = 1, fd, seqp = 0;
    char *cp, *maildir, *folder = NULL, buf[100];
    char **ap, **argp, *arguments[MAXARGS], *seqs[NUMATTRS+1];
    struct msgs *mp;
    struct stat st;

#ifdef LOCALE
    setlocale(LC_ALL, "");
#endif
    invo_name = r1bindex (argv[0], '/');
    mts_init (invo_name);
    if ((cp = m_find (invo_name))) {
	ap = brkstring (cp = getcpy (cp), " ", "\n");
	ap = copyip (ap, arguments);
    } else {
	ap = arguments;
    }
    copyip (argv + 1, ap);
    argp = arguments;

    while ((cp = *argp++)) {
	if (*cp == '-')
	    switch (smatch (++cp, switches)) {
		case AMBIGSW: 
		    ambigsw (cp, switches);
		    done (1);
		case UNKWNSW: 
		    adios (NULL, "-%s unknown", cp);

		case HELPSW: 
		    sprintf (buf, "%s [+folder] [switches]", invo_name);
		    print_help (buf, switches);
		    done (1);
		case VERSIONSW:
		    print_version(invo_name);
		    done (1);

		case SEQSW: 
		    if (!(cp = *argp++) || *cp == '-')
			adios (NULL, "missing argument name to %s",
				argp[-2]);
		    if (seqp < NUMATTRS)
			seqs[seqp++] = cp;
		    else
			adios (NULL, "only %d sequences allowed!", NUMATTRS);
		    continue;
		case PUBSW: 
		    publicsw = 1;
		    continue;
		case NPUBSW: 
		    publicsw = 0;
		    continue;
		case ZEROSW: 
		    zerosw++;
		    continue;
		case NZEROSW: 
		    zerosw = 0;
		    continue;

		case CRETSW: 
		    create++;
		    continue;
		case NCRETSW: 
		    create = 0;
		    continue;
	    }
	if (*cp == '+' || *cp == '@') {
	    if (folder)
		adios (NULL, "only one folder at a time!");
	    else
		folder = path (cp + 1, *cp == '+' ? TFOLDER : TSUBCWF);
	}
	else
	    adios (NULL, "usage: %s [+folder] [switches]", invo_name);
    }

    seqs[seqp] = NULL;	/* NULL terminate list of sequences */

    if (!m_find ("path"))
	free (path ("./", TFOLDER));
    if (!folder && !(folder = m_find (inbox)))
	folder = defalt;
    maildir = m_maildir (folder);

    if (stat (maildir, &st) == NOTOK) {
	if (errno != ENOENT)
	    adios (maildir, "error on folder");
	if (!create)
	    adios (NULL, "folder %s doesn't exist", maildir);
	if (!makedir (maildir))
	    adios (NULL, "unable to create folder %s", maildir);
    }

    if (chdir (maildir) == NOTOK)
	adios (maildir, "unable to change directory to");

    /* read folder and create message structure */
    if (!(mp = m_readfolder (folder)))
	adios (NULL, "unable to read folder %s", folder);

    SIGNAL (SIGHUP, SIG_IGN);
    SIGNAL (SIGINT, SIG_IGN);
    SIGNAL (SIGQUIT, SIG_IGN);
    SIGNAL (SIGTERM, SIG_IGN);

    tmpfilenam = m_scratch ("", invo_name);
    if ((fd = creat (tmpfilenam, m_gmprot ())) == NOTOK)
	adios (tmpfilenam, "unable to create");
    chmod (tmpfilenam, m_gmprot ());

    /* copy the message from stdin into temp file */
    cpydata (fileno (stdin), fd, "standard input", tmpfilenam);

    if (fstat (fd, &st) == NOTOK) {
	unlink (tmpfilenam);
	adios (tmpfilenam, "unable to fstat");
    }
    if (close (fd) == NOTOK)
	adios (tmpfilenam, "error closing");
    if (st.st_size == 0) {
	unlink (tmpfilenam);
	advise (NULL, "empty file");
	done (0);
    }

    /* Link message into folder */
    if ((msgnum = folder_addmsg(mp, tmpfilenam)) == -1)
	done (1);

    unlink (tmpfilenam);
    tmpfilenam = NULL;

    /* add the message to all the specified sequences */
    for (seqp = 0; seqs[seqp]; seqp++) {
	if (!seq_addmsg (mp, seqs[seqp], msgnum, publicsw, zerosw))
	    done (1);
    }

    seq_setunseen (mp, 0);	/* add message to the Unseen-Sequence */
    seq_save (mp);		/* synchronize message sequences      */
    m_freefolder (mp);		/* free folder/message structure      */
    m_update ();		/* update the context file            */
    done (0);
}


/*
 * Link message into a folder.  Set the EXISTS and
 * UNSEEN bits.  Return the new number of the message.
 * If an error, return -1;
 */
static int
folder_addmsg(struct msgs *mp, char *msgfile)
{
    int msgnum;

    msgnum = mp->hghmsg;
    do {
	msgnum++;
	mp->hghmsg++;
	if (msgnum > mp->hghoff) {
	    if (!(mp = m_remsg (mp, 0, mp->hghoff + MAXFOLDER))) {
		advise (NULL, "unable to allocate folder storage");
		return -1;
	    }
	}

	set_exists (mp, msgnum);
	set_unseen (mp, msgnum);
	errno = 0;
    } while (link (msgfile, m_name (msgnum)) == NOTOK && errno == EEXIST);

    if (errno != 0) {
	advise (NULL, "can't file message %d", msgnum);
	return -1;
    }

    if (mp->lowmsg == 0)
	mp->lowmsg = msgnum;
    mp->msgflags |= SEQMOD;

    return msgnum;
}


/*
 * Clean up and exit
 */
void
done(int status)
{
    if (tmpfilenam && *tmpfilenam)
	unlink (tmpfilenam);
    exit (status);
}
