/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1993 Electrotechnical Laboratry (ETL)

Permission to use, copy, modify, and distribute this material 
for any purpose and without fee is hereby granted, provided 
that the above copyright notice and this permission notice 
appear in all copies, and that the name of ETL not be 
used in advertising or publicity pertaining to this 
material without the specific, prior written permission 
of an authorized representative of ETL.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY 
OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", 
WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	mailcap.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	930721	extracted from mmssynthe.o
///////////////////////////////////////////////////////////////////////*/
#define STATIC static
static struct MailcapEntry *FirstMailcapEntry;

/*
 *	mailcap (copied from mailto.o of metamail-2.6)
 */
/*
Copyright (c) 1992 Bell Communications Research, Inc. (Bellcore)

Permission to use, copy, modify, and distribute this material
for any purpose and without fee is hereby granted, provided
that the above copyright notice and this permission notice
appear in all copies, and that the name of Bellcore not be
used in advertising or publicity pertaining to this
material without the specific, prior written permission
of an authorized representative of Bellcore.  BELLCORE
MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY
OF THIS MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS",
WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
*/
#include <stdio.h>
#include <config.h>
#include <ctype.h>
#include <pwd.h>
char *getenv();

extern char *malloc(), *realloc(), *index(), *getmyname();
#define MAX_LINELENGTH  2000

struct MailcapEntry {
    char *contenttype;
    char *command;
    char *testcommand;
    char *editcommand;
    char *composecommand;
    char *composetypedcommand;
    char *label;
    int needsterminal;
    int copiousoutput;
    struct MailcapEntry *next;
};

/* There are a fair number of core leaks in what follows.  That should matter little -- the mailcap files are only parsed once, and are usually pretty small anyway. */

STATIC char *
GetCommand(s, t)
char *s, **t;
{
    char *s2;
    int quoted = 0;
    s2 = malloc(strlen(s)*2); /* absolute max, if all % signs */
    if (!s2) nomemabort();
    *t = s2;
    while (s && *s) {
	if (quoted) {
            if (*s == '%') *s2++ = '%'; /* Quote through next level, ugh! */

            *s2++ = *s++;
	    quoted = 0;
	} else {
	    if (*s == ';') {
                *s2 = '\0';
		return(++s);
	    }
	    if (*s == '\\') {
		quoted = 1;
		++s;
	    } else {
		*s2++ = *s++;
	    }
	}
    }
    *s2 = '\0';
    return(NULL);
}	

STATIC char *Cleanse(s, dolc) /* no leading or trailing space, all lower case */
char *s;
int dolc;
{
    char *tmp, *news;
    
    /* strip leading white space */
    while (*s && isspace((unsigned char) *s)) ++s;
    news = s;
    /* put in lower case, find end */
    for (tmp=s; *tmp; ++tmp) {
        if (dolc && isupper((unsigned char) *tmp)) *tmp = tolower((unsigned char) *tmp);
    }
    /* strip trailing white space */
    while (--tmp && *tmp && isspace((unsigned char) *tmp)) *tmp = '\0';
    return(news);
}

STATIC char *DeQuote(s)
char *s;
{
    char *retval;
    s = Cleanse(s, 0);
    if (*s != '"') return(s);
    retval = ++s;
    while (s && *s) {
        s = index(s, '\"');
        if (!s) return(retval); /* but it's a bad parse */
        if (*(s-1) != '\\') {
            *s = '\0';
            return(retval);
        }
        ++s;
    }
    return(retval); /* also a bad parse */
}    

STATIC struct MailcapEntry *
GetMailcapEntry(fp)
FILE *fp;
{
    int rawentryalloc = MAX_LINELENGTH, len;
    char *rawentry, *s, *t, *LineBuf;
    struct MailcapEntry *mc;

    LineBuf = malloc(MAX_LINELENGTH);
    if (!LineBuf) nomemabort();
    rawentry = malloc(1 + rawentryalloc);
    mc = (struct MailcapEntry *) malloc(sizeof (struct MailcapEntry));
    if (!rawentry || !mc) nomemabort();
    *rawentry = '\0';
    while (fgets(LineBuf, MAX_LINELENGTH, fp)) {
	if (LineBuf[0] == '#') continue;
	len = strlen(LineBuf);
        if (LineBuf[len-1] == '\n') LineBuf[--len] = '\0';
	if ((len + strlen(rawentry)) > rawentryalloc) {
            rawentryalloc += MAX_LINELENGTH;
	    rawentry = realloc(rawentry, rawentryalloc+1);
	    if (!rawentry) nomemabort();
	}
	if (LineBuf[len-1] == '\\') {
            LineBuf[len-1] = '\0';
	    strcat(rawentry, LineBuf);
	} else {
	    strcat(rawentry, LineBuf);
	    break;
	}
    }
    free(LineBuf);
    for (s=rawentry; *s && isspace((unsigned char) *s); ++s) ;
    if (!*s) {
	/* totally blank entry -- quietly ignore */
	free(rawentry);
	return(NULL);
    }
    s = index(rawentry, ';');
    if (!s) {
	fprintf(stderr, "mailto: Ignoring invalid mailcap entry: %s\n", rawentry);
	free(rawentry);
	return(NULL);
    }
    *s++ = '\0';
    mc->needsterminal = 0;
    mc->copiousoutput = 0;
    mc->testcommand = NULL;
    mc->composecommand = NULL;
    mc->composetypedcommand = NULL;
    mc->editcommand = NULL;
    mc->label = NULL;
    mc->contenttype = malloc(1+strlen(rawentry));
    mc->next = NULL;
    if (!mc->contenttype) nomemabort();
    strcpy(mc->contenttype, rawentry);
    t = GetCommand(s, &mc->command);
    s = t;
    while (s) {
	char *arg, *eq;

        t = GetCommand(s, &arg);
/*        if (t) *t++ = '\0'; */
        eq = index(arg, '=');
        if (eq) *eq++ = '\0';
        arg = Cleanse(arg, 1);
	if (!strcmp(arg, "needsterminal")) {
	    mc->needsterminal = 1;
	} else if (!strcmp(arg, "copiousoutput")) {
	    mc->copiousoutput = 1;
        } else if (eq && !strcmp(arg, "test")) {
            mc->testcommand = DeQuote(eq);
        } else if (eq && !strcmp(arg, "edit")) {
            mc->editcommand = DeQuote(eq);
        } else if (eq && !strcmp(arg, "compose")) {
            mc->composecommand = DeQuote(eq);
        } else if (eq && !strcmp(arg, "composetyped")) {
            mc->composetypedcommand = DeQuote(eq);
        } else if (eq && !strcmp(arg, "description")) {
            mc->label = DeQuote(eq);
        } else if (eq && !strcmp(arg, "label")) { 
            mc->label = DeQuote(eq); /* bogus old name for description */
        } else if (eq && !strcmp(arg, "textualnewlines")) {
            ExceptionalNewline(mc->contenttype, atoi(eq));
	} else if (strcmp(arg, "notes")) { /* IGNORE notes field */
/*	    if (*arg) fprintf(stderr, "mailto: Ignoring invalid mailcap flag: %s\n", arg);  */
	}
	s = t;
    }
    free(rawentry);
    return(mc);
}
STATIC ProcessMailcapFiles() 
{
    char *s, *path = getenv("MAILCAPS"), *origpath;
    static char *stdpath = STDPATH;
    struct MailcapEntry *mc, *CurrentMailcapEntry = NULL;
    FILE *fp;

    if (!path) {
#ifdef AMIGA
        path = malloc(5 + strlen(stdpath));
        if (!path) nomemabort();
        strcpy(path, stdpath);
#else
        int uid = getuid();
        struct passwd *p;
        p = getpwuid(uid);
        if (p) path = malloc(5+strlen(p->pw_dir) + strlen(stdpath));
        if (!p || !path) nomemabort();
        strcpy(path, p->pw_dir);
        strcat(path, stdpath);
#endif
    } else {
        char *pathcopy;
        pathcopy = malloc(1+strlen(path));
        if (!pathcopy) nomemabort();
        strcpy(pathcopy, path);
        path = pathcopy;
    }
    origpath = path;
    while(path) {
        s = index(path, PATH_SEPARATOR);
        if (s) *s++ = '\0';
        fp = fopen(path, "r");
        while (fp && !feof(fp)) {
            mc = GetMailcapEntry(fp);
            if (!mc) continue;
            if ((mc->composecommand || mc->composetypedcommand || mc->editcommand) && index(mc->contenttype, '*')) {
                printf("\nWARNING:  a mailcap file is configured to allow the generation\n");
                printf("of a content-type with a wildcard.  This is probably a mistake.\n");
                printf("The relevant mailcap entry is in the file %s\n", path);
                printf("and is for content-type %s.\n\n", mc->contenttype);
            }
            if (!FirstMailcapEntry) {
                FirstMailcapEntry = mc;
                CurrentMailcapEntry = mc;
            } else {
                CurrentMailcapEntry->next = mc;
                CurrentMailcapEntry = mc;
            }
	}
        if (fp) fclose(fp);
        path = s;
    }
    free(origpath);
    return(-1);
}

/*////////////////////////////////////////////////////////////////////////
 /////////////////////////////////////////////////////////////////////////
 /////////////////////////////////////////////////////////////////////////
 /////////////////////////////////////////////////////////////////////////
 /////////////////////////////////////////////////////////////////////////
 *
 */
static nomemabort(){ MMS_nomemabort(); }

char *MailcapPath =
"%s/.mailcap:%s/mailcap:/etc/mailcap:/usr/etc/mailcap:/usr/local/etc/mailcap";

struct MailcapEntry
*MMS_ProcessMailcap(force)
{	static int initdone;
	char *home,*lib;

	if( force || !initdone ){
		initdone = 1;
		home = getenv("HOME");
		lib = getenv("SYSLIB");
		MMS_putenv("MAILCAPS",MailcapPath,home?home:"",lib?lib:"");
		ProcessMailcapFiles();
		if( FirstMailcapEntry == NULL )
		    MMS_abort(1,"ProcessMailcap: no mailcap found\n%s=%s\n",
			"MAILCAPS",getenv("MAILCAPS"));
	}
	return FirstMailcapEntry;
}
MMS_mailcapAvailableCtype(actype)
	char *actype;
{	struct MailcapEntry *mc;
	char *ctype;

	MMS_ProcessMailcap(0);
	for(mc = FirstMailcapEntry; mc; mc = mc->next ){
		ctype = mc->contenttype;
		if(ctype && index(ctype,'/') && !index(ctype,'*'))
			if( strcmp(ctype,actype) == 0 )
				return 1;
	}
	return 0;
}
MMS_mailcapPrintTypes(out)
	FILE *out;
{	struct MailcapEntry *mc;
	char *ctype;

	MMS_ProcessMailcap(0);
	for(mc = FirstMailcapEntry; mc; mc = mc->next ){
		ctype = mc->contenttype;
		if(ctype && index(ctype,'/') && !index(ctype,'*'))
			fprintf(out,"%s\n",mc->contenttype);
	}
}
char *
MMS_mailcapGetCommand(ctype,entry)
	char *ctype,*entry;
{	char maintype[128],subtype[128],amaintype[128],asubtype[128];
	char *actype;
	struct MailcapEntry *mc;

	MMS_ProcessMailcap(0);

	strcpy(subtype,"*");
	sscanf(ctype,"%[^/]/%s",maintype,subtype);

	for(mc = FirstMailcapEntry; mc; mc = mc->next ){
		if( actype = mc->contenttype ){
			if( strcmp(ctype,actype) == 0 )
				goto found;

			sscanf(actype,"%[^/]/%s",amaintype,asubtype);
			if( strcmp(maintype,amaintype) == 0 ){
				if( strcmp(asubtype,"*") == 0 )
					goto found;
			}
		}
	}
	return 0;

found:
	if( entry == 0 || *entry == 0 )		return mc->command;
	if( strcmp(entry,"label") == 0 )	return mc->label;
	if( strcmp(entry,"test") == 0 )		return mc->testcommand;
	if( strcmp(entry,"edit") == 0 )		return mc->editcommand;
	if( strcmp(entry,"compose") == 0 )	return mc->composecommand;
	return 0;
}
