;;; zenirc-notify.el --- Notifies you when people signon/off

;; Copyright (C) 1995, 1996 Per Persson

;; Author: Per Persson <pp@gnu.ai.mit.edu>
;; Maintainer: pp@gnu.ai.mit.edu
;; Keywords: zenirc, notify, extensions
;; Created: 1995-03-30

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, you can either send email to this
;; program's maintainer or write to: The Free Software Foundation,
;; Inc.; 675 Massachusetts Avenue; Cambridge, MA 02139, USA.

;;; Commentary:

;; TODO: If notifee joins channel, don't show notification.

;;; Code:

(require 'zenirc)

(defvar zenirc-notify-list '()
  "*A list of nicknames that you want to watch for on IRC.")

(defvar zenirc-notify-interval '(0 60)
  "*Time between ISON's sent to server.")

(defvar zenirc-last-notify '(0 0)
  "Time previous ISON sent.")
(make-variable-buffer-local 'zenirc-last-notify)

(zenirc-add-hook 'zenirc-timer-hook 'zenirc-notify-timer)

;; used with notify code to see if we did /ison or not
(defvar zenirc-manual-ison nil)
(make-variable-buffer-local 'zenirc-manual-ison)

;; people you previously recieved notification of
(defvar zenirc-previous-ison nil)
(make-variable-buffer-local 'zenirc-previous-ison)

;; should you get a user@host reply on notify?
(defvar zenirc-userhost-on-notify t)
(make-variable-buffer-local 'zenirc-userhost-on-notify)

(defvar zenirc-userhost-by-notify nil)
(make-variable-buffer-local 'zenirc-userhost-by-notify)

(defvar zenirc-command-notify-hook '(zenirc-command-notify)
  "*Hook to call when a /notify command is issude in ZeniIRC.

The syntax of the command is: /notify victim.
This toggles the presence of `victim' in your notify-list.
If no victim is specified, you will see your notify-list instead.")

; hooks to make zenirc aware of the notify code.
(defvar zenirc-command-ison-hook '(zenirc-command-ison))
(defvar zenirc-command-userhost-hook '(zenirc-command-userhost))

(defun zenirc-notify-install-message-catalogs ()
  (zenirc-lang-define-catalog 'english
   '((notify_list . "[info] Your current notify list: %s\n")
     (notify_on . "[info] detected %s wasting time.\n")
     (notify_off . "[info] detected that %s stopped wasting time.\n")
     (notify_current . "[info] Notificated people wasting time: %s\n")
     )))

(defun zenirc-notify-timer (proc now)
  "Call zenirc-command-notify-hook with arguments that cause it to send an
ISON message to the server. This is used to notice when people in the notify
list have come on or off of IRC."
  (if (zenirc-time< zenirc-notify-interval
                    (zenirc-time-diff now zenirc-last-notify))
      (progn
        (zenirc-run-hook 'zenirc-command-notify-hook proc 
			 '("notify" . "%auto"))
        (setq zenirc-last-notify now))))


;; /notify handler
;;
;; *** NOTE ***
;; this is also called from the zenirc event handling code
;;
(defun zenirc-command-notify (proc parsedcmd)
  (let ((arg (cdr parsedcmd)))
    (if (string-equal "" arg)
	; output list of notificated people (online)
	(progn 
	  (if (not zenirc-previous-ison)
	      (zenirc-message proc 'notify_current "")
	    (zenirc-message proc 'notify_current zenirc-previous-ison))
	  (zenirc-message proc 'notify_list
			  (mapconcat 'identity zenirc-notify-list " ")))
      (if (not (string-match "%" arg))
	  ; add or remove nick from zenirc-notify-list
	  (progn
	    (setq arg (zenirc-parse-words arg))
	    (while arg
	      (if (zenirc-string-match-list (car arg) zenirc-notify-list)
		  (setq zenirc-notify-list (zenirc-delete-case-insensitive 
					    (car arg) zenirc-notify-list))
		(setq zenirc-notify-list (cons (car arg) zenirc-notify-list)))
	      (setq arg (cdr arg)))
	    ; output new list of notificated people
	    (zenirc-message proc 'notify_list
			    (mapconcat 'identity zenirc-notify-list " "))))
      (if zenirc-notify-list
	  ; if automated, check to see if anything has changed
	  (process-send-string
	   proc
	   (concat "ISON " 
		   (mapconcat 'identity zenirc-notify-list " ") "\n"))))))

;; /ison nick1 [nick2 [nick3...]]
(defun zenirc-command-ison (proc parsedcmd)
  (process-send-string proc
                       (concat "ISON " (cdr parsedcmd) "\n"))
  (setq zenirc-manual-ison 1))

;; /userhost nick1 [nick2 [nick3...]]
(defun zenirc-command-userhost (proc parsedcmd)
  (process-send-string proc (concat "USERHOST " (cdr parsedcmd) "\n"))
  (setq zenirc-userhost-by-notify nil))

(defun zenirc-server-303-notify (proc parsedmsg)
  (let* ((ison-list (zenirc-parse-words (aref parsedmsg 3)))
         (temp-list ison-list))
    (if zenirc-manual-ison
        ; we did /ison and don't want to notify code to execute
        (progn
          (zenirc-message proc 's303 (aref parsedmsg 3))
          (setq zenirc-manual-ison nil))
      (if (string= (aref parsedmsg 3) "")
        ; nobody on our list is on
        (setq zenirc-previous-ison nil)
      ; someone on our notify list is on
      (while temp-list
        (if (not (member (car temp-list)
                         zenirc-previous-ison))
            (if zenirc-userhost-on-notify
                (progn
                  (process-send-string 
		   proc 
		   (concat "USERHOST " (car temp-list) "\n"))
                  (setq zenirc-userhost-by-notify t))
              (zenirc-message proc 'notify_on (car temp-list))))
        (setq temp-list (cdr temp-list)))
      (while zenirc-previous-ison
        (if (not (member (car zenirc-previous-ison)
                         ison-list))
            (zenirc-message proc 'notify_off (car zenirc-previous-ison)))
        (setq zenirc-previous-ison (cdr zenirc-previous-ison)))
      (setq zenirc-previous-ison ison-list)))))

(defun zenirc-server-302-notify (proc parsedmsg)
  (if zenirc-userhost-by-notify
      (if (not (string= " " (aref parsedmsg 3)))
	  (zenirc-message proc 'notify_on (aref parsedmsg 3)))
    (zenirc-message proc 's302 (aref parsedmsg 3))))


(provide 'zenirc-notify)

(zenirc-notify-install-message-catalogs)

(zenirc-remove-hook 'zenirc-server-303-hook 'zenirc-server-303)
(zenirc-add-hook 'zenirc-server-303-hook 'zenirc-server-303-notify)
(zenirc-remove-hook 'zenirc-server-302-hook 'zenirc-server-302)
(zenirc-add-hook 'zenirc-server-302-hook 'zenirc-server-302-notify)

;;; zenirc-notify.el ends here
