;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Copyright (c) 1993 by William M. Perry (wmperry@indiana.edu)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Normal Emacs Specific Functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(require 'w3)

(defvar w3-node-style nil)
(defvar w3-header-style nil)
(defvar w3-address-style nil)
(defvar w3-tt-style nil)
(defvar w3-bold-style nil)
(defvar w3-italic-style nil)
(defvar w3-underline-style nil)
(defvar w3-default-style nil)

(defun w3-find-specific-link (link)
  "Find LINK in the current document"
  (let* ((thezones w3-zones-list))
    (while (and thezones
		(not
		 (equal link
			(car-safe (cdr (car (cdr (cdr (car thezones)))))))))
      (setq thezones (cdr thezones)))
    (if thezones (goto-char (car (car thezones)))
      (message "Link %s was not found." link))))

(defun w3-view-this-url (&optional no-show)
  "View the url for the current link."
  (interactive)
  (let ((zn (w3-zone-at (point))))
    (if zn
	(if (not no-show)
	    (message "%s" (car (cdr (cdr (car (cdr (cdr zn)))))))
	  (car (cdr (cdr (car (cdr (cdr zn)))))))
      (error "Not on a link!"))))

(defun w3-add-zone (start end style data)
  "Add a zone (normal emacs)"
  (cond
   ((memq (cdr data) '(style address header)) nil)
   ((or (null w3-zones-list)
	(< start (car (car w3-zones-list))))
    (setq w3-zones-list (cons (list (copy-marker start)
				    (copy-marker end)
				    data) w3-zones-list)))
   (t
    (let ((zones w3-zones-list))
      (while (and (cdr zones)
		  (< (car (car (cdr zones))) start))
	(setq zones (cdr zones)))
      (setcdr zones (cons (list
			   (copy-marker start)
			   (copy-marker end)
			   data)
			  (cdr zones)))))))


(defun w3-zones-list ()
  "Return a list of zones for this buffer"
  w3-zones-list)

(defun w3-zone-at (pt)
  "Return zone (if any) at PT"
  (let ((zones w3-zones-list))
    (while (and zones
		(not (and
		      (>= (car (cdr (car zones))) pt)
		      (<= (car (car zones)) pt))))
      (setq zones (cdr zones)))
    (if zones (car zones)
      nil)))

(defun w3-zone-end (zone)
  (car (cdr zone)))

(defun w3-zone-start (zone)
  (car zone))

(defun w3-next-zone (zone)
  "Return zone (if any) after ZONE"
  (let ((zones w3-zones-list))
    (while (and zones
		(not (and (equal (car (cdr (car zones)))
				 (car (cdr zone)))
			  (equal (car (car zones))
				 (car zone)))))
      (setq zones (cdr zones)))
    (if (cdr zones)
	(car (cdr zones))
      nil)))

(defun w3-previous-zone (zone)
  "Return zone (if any) before ZONE"
  (let ((zones w3-zones-list)
	(last nil))
    (while (not (eql (car zones) zone))
      (setq last (car zones)
	    zones (cdr zones)))
    (if zones
	last
      nil)))

(defun w3-zone-data (zone)
  (car (cdr (cdr zone))))

(defun w3-forward-link ()
  "Go forward 1 link"
  (interactive)
  (let ((zones w3-zones-list))
    (while (and zones
		(<= (car (car zones)) (point)))
      (setq zones (cdr zones)))
    (if zones (goto-char (car (car zones)))
      (error "No more links."))))

(defun w3-back-link ()
  "Go back 1 link"
  (interactive)
  (cond
   ((null w3-zones-list)
    (error "No links in this document."))
   ((> (car (car (cdr w3-zones-list))) (point))
    (error "No previous link"))
   (t
    (let ((zones w3-zones-list))
      (while (and (cdr zones)
		  (< (car (cdr (car (cdr zones)))) (point)))
	(setq zones (cdr zones)))
      (goto-char (car (car zones)))))))

(defun w3-follow-link ()
  "Follow a link"
  (interactive)
  (let ((zn (w3-zone-at (point))))
    (if zn
	(w3-maybe-relative (car (cdr (cdr (w3-zone-data zn)))))
      (message "Not on a link!"))))

(defvar w3-emphasis-style w3-italic-style)
(defvar w3-strong-style w3-bold-style)
(defvar w3-code-style w3-tt-style)
(defvar w3-samp-style w3-tt-style)
(defvar w3-kbd-style w3-tt-style)
(defvar w3-var-style w3-tt-style)
(defvar w3-dfn-style w3-bold-style)
(defvar w3-cite-style w3-underline-style)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; NeXT Related Stuff
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(if (fboundp 'define-mouse)
    (progn
      (defvar w3-NeXT-mousemap (make-mousemap) "Mousemap for use on NeXT")
      (define-mouse w3-NeXT-mousemap '(text right) 'w3-follow-mouse)
      
      (defun w3-follow-mouse (window x y)
	"Follow the link under the mouse cursor"
	(interactive)
	(goto-char (window-start))
	(if (not (eobp)) (next-line y))
	(let ((tmp 0))
	  (while (< tmp x)
	    (if (not (eolp)) (forward-char 1))
	    (setq tmp (1+ tmp))))
	(w3-follow-link))))

(provide 'w3-emacs)

