;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; $Id: vcs.el,v 1.15 19-Sep-1992 22:33:23 EDT don Exp $
;;;
;;; Version control made easy
;;; 
;;; Copyright (C) Donald Beaudry <don@vicorp.com> 1992
;;;
;;; This file is not part of GNU Emacs, but is made available under
;;; the same conditions.
;;;
;;; This program is free software; you can redistribute it and/or
;;; modify it under the terms of the GNU General Public License as
;;; published by the Free Software Foundation; either version 1, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;; INTRODUCTION
;;;
;;; Vcs is an emacs interface to a generic version control system.
;;; Like most other version control interfaces vcs provides its users
;;; with the ability to check in, check out, lock, unlock and get a
;;; delta history for its files.  So you might be wondering why you
;;; need vcs.  Well it's quite possible that you don't.  But, if your 
;;; working environment is anything like mine I think you'll find 
;;; that vcs has some very important advantages over most other 
;;; interfaces.  
;;;
;;; I work in an environment where nearly all source code is kept
;;; under the control of sccs.  I, however, much preferred to use rcs
;;; for my own projects.  What I wanted was a single emacs interface
;;; that would allow me to manipulate either my employer's sccs files
;;; or my personal rcs files.  I was unable to find an interface
;;; package that met my needs.  Nor could I find an existing rcs
;;; interface that was similar enough to an existing sccs interface
;;; that I could feel comfortable learning and using both
;;; simultaneously.  To make matters a bit worse, my employer, like
;;; most that use sccs or rcs to manage a LARGE project, put shell
;;; script wrappers around all of the normal sccs commands.  And if
;;; that isn't enough to make most available interfaces nearly
;;; useless, throw in the fact that the source is scattered about a
;;; directory tree containing more than thirty sub-directories.
;;;
;;; The other interfaces that I looked at were all designed to work in
;;; an environment where the history (s. or ,v) files live in the same
;;; directory as or a sub-directory of the working file's directory.
;;; This is great if you are the only one working on the source tree.
;;; But with many programmers, each possibly wanting to make
;;; conflicting (though usually temporary) changes to the same source
;;; file, this approach leaves much to be desired.
;;;
;;; Vcs was designed to work in an environment where the history files
;;; live in one directory tree and development is done out of a
;;; programmers home directory.  The normal course of events for
;;; fixing a bug or making an enhancement starts with copying the
;;; files you need to somewhere in your home directory tree and
;;; locking those that you know will need to be changed.  These files
;;; are then modified and compiled locally and linked against the most
;;; recent version of the libraries for testing.  When the changes are
;;; complete the unmodified files are either unlocked or removed and
;;; the modified ones are incorporated back into the main source tree.
;;; If your environment is similar to this, then vcs just might be for 
;;; you. 
;;;
;;; Vcs will hunt down the source file you are looking for, without
;;; regard for which version control system manages it or where in the
;;; file system it lives, and allow you to manipulate it from the
;;; privacy of your own directory.  You can even keep copy of public
;;; file in your own version control directory to incrementally log
;;; your changes before putting the file back into its public version
;;; control directory.  I often do this when I am making extensive
;;; changes to a file and don't want to put an inconsistent version
;;; back into the public source tree, but do want to be able to
;;; gracefully recover from a mad editing session (it sure beats
;;; leaving numbered copies of the file lying around).  On top of all
;;; that vcs provides a framework that makes extending its
;;; capabilities to support other version control systems (or
;;; wrappers) quite painless.
;;;
;;;
;;; SET UP
;;;
;;; Vcs comes pre-configured to use both rcs and sccs.  If this is 
;;; satisfactory for your needs, you can simply copy the all of the 
;;; *.el files from the vcs distribution into a directory found on the 
;;; load-path list and add 
;;;
;;;  	     (load "vcs-init")
;;;
;;; to your .emacs file.  It is important that the directory be found
;;; on the load path list because vcs-init loads a few files itself
;;; and will only be able to do so if load is able to find them.  If
;;; this is unacceptable to you then you must change the load commands
;;; in vcs-init to use absolute path names.
;;;
;;;  
;;; FILES
;;;
;;; By itself vcs.el doesn't do much.  Fortunately the vcs
;;; distribution comes with three additional files: vcs-rcs.el,
;;; vcs-sccs.el, and vcs-local.el.  As you might have guessed
;;; vcs-rcs.el is the vcs interface to rcs and vcs-sccs.el is the vcs
;;; interface to sccs.  While vcs.el itself is of modest size,
;;; vcs-rcs.el and vcs.sccs.el are both rather small (around 250
;;; lines).  These files contain all of the system specific code for
;;; their respective version control systems.  Vcs-local.el contains
;;; an example of a vcs-put function that calls a shell script wrapped
;;; around the sccs admin and delta commands.  As it turns out, this
;;; was the only non-standard function I needed to support my
;;; employer's wrapper based interface to sccs.  All of the other
;;; wrappers were used to support an automatic file lookup mechanism
;;; that would find the given file name anywhere in the source tree.
;;; Vcs already knows how to do this so there was no need to support
;;; those wrappers.  The delta wrapper, on the other hand,
;;; interactivly prompted the user for a lot of extra information
;;; which was is used to track the changes to the source.  If you need
;;; to extend the capabilities of vcs you should take a look at
;;; vcs-rcs.el, vcs-sccs.el, and vcs-local.el.
;;;
;;; Vcs-init.el is intended to be modified for use at the local site.  
;;; It sets up the default key binding, file search information, and 
;;; loads the version control system specific modules needed by the 
;;; site. Take a look at it an make any changes that you think are
;;; necessary.
;;;
;;; Also included with the distribution is a package called form.el.
;;; It doesn't have anything to do with version control but it turned
;;; to be a convenient way to get the log message from the user.
;;; Form.el is similar to the popular forms-mode package written by
;;; Johan Vromans <jv@mh.nl> except it does not provide the database
;;; capabilities of his package.  Take a look at form.el for more
;;; information
;;; 
;;;
;;; COMMANDS
;;;
;;; The following commands all act on the file in the current buffer.
;;; By default, if there is more than one history file that could be
;;; associated with the file in the current buffer all of these
;;; commands will prompt you for the history file name with a 
;;; completion list containing all possibilities.  This default 
;;; behavior can be modified by various variables.  See the section on 
;;; variables for for more information. 
;;;
;;; vcs-get-buffer
;;;     Get the most recent version of the file associated with the 
;;;  	current buffer.  A copy of the file is deposited in the buffer's 
;;;     default directory and the contents of the buffer is updated.
;;;  	     
;;; vcs-lock-buffer
;;;     Lock the most recent version of the file associated with the 
;;;     current buffer.  A copy of the file is deposited in the buffer's 
;;;     default directory and the contents of the buffer updated.
;;;
;;; vcs-put-buffer
;;;     Put contents of the current buffer into its associated history 
;;;     file.  Usually the file must have been locked for this 
;;;     operation to succeed.  After successful completion of the put, 
;;;     a get is performed and the buffer contents are updated.
;;;
;;; vcs-unlock-buffer
;;;     Release the version control lock on the history file 
;;;     associated with the current current buffer.
;;;
;;; vcs-info-buffer
;;;     Show the delta information for the history file associated 
;;;     with the current buffer.
;;;
;;;
;;; The following commands, when called interactively, will prompt the
;;; user for a file name.  If a working file name is given the
;;; associated history file will be located and used by the command.
;;; If a history file name is given it will be used but the working
;;; file name will be derived from the history file name and the
;;; current buffer's default directory directory.  By default, if a
;;; working file name is given and there is more than one history file
;;; that could be associated with the that file, all of these commands
;;; will prompt you for the history file name with a completion list 
;;; containing all possibilities.  This default behavior can be 
;;; modified by various variables.  See the section on variables for 
;;; for more information. 
;;; 
;;; 
;;; vcs-insert-file
;;;     From the history file associated with FILE-NAME, extract the 
;;;     most recent version and insert the it into the current buffer 
;;;     at the current point. 
;;;     
;;; vcs-get-file
;;;     From the history file associated with FILE-NAME, extract the 
;;;     most recent version depositing it into the given file name and 
;;;     then visit the file.  If the file is already in a buffer, the 
;;;     buffer is refreshed with the most recent version then selected.
;;;
;;; vcs-lock-file
;;;     From the history file associated with FILE-NAME, extract and
;;;     lock for editing the most recent version depositing it into
;;;     the given file name and then visit the file.  If the file is
;;;     already in a buffer the buffer is refreshed with the most
;;;     recent version then selected.
;;;
;;; vcs-unlock-file
;;;     Release any version control locks on the history file
;;;     associated with FILE-NAME.  A prefix argument says to kill any
;;;     buffer and remove any working file associated with the file.
;;;
;;; vcs-put-file
;;;     Put FILE-NAME under the control of a version control system.
;;;  
;;; vcs-info-file
;;;     Show the delta information for the version control history 
;;;     file associated with FILE-NAME.
;;;
;;;
;;;
;;; VARIABLES
;;;
;;; The following variables modify the behavior of many of the vcs 
;;; funtions.  Each variable is followed by its default value.
;;;
;;; vcs-auto-lookup  t  (must be set prior to loading vcs-init)
;;;     If non-nil, specifies that vcs should automatically try to
;;;     lookup a file when find-file fails.
;;;
;;; vcs-show-lock-status t (must be set prior to loading vcs-init)
;;;     If non-nil, specifies that vcs should display the lock status
;;;     of a file in the buffer's mode line.
;;;
;;; vcs-use-default-bindings t (must be set prior to loading vcs-init)
;;;     If non-nil, specifies that vcs should add default bindings to 
;;;     to the global keymap.
;;;
;;; vcs-file-patterns '((".*" "."))
;;;     Specifies the list of file patterns used to locate vcs history
;;;     files.  This is a list of lists.  The car of each list
;;;     contains a regular expression that is used to match file names.
;;;     The cdr of the list contains directories to be searched to
;;;     find the file.  If nothing else, this list should contain an 
;;;     entry like (".*" ".") which tells vcs to look for any file in 
;;;     the current directory.  If the version control systems being 
;;;     used keep their files in a special sub-directory (like RCS) 
;;;     those sub-directories should not be included in the list.  
;;;     They should be checked by vcs-history-file-name instead.
;;;     See vcs-history-file-name for details.
;;;
;;; vcs-*-lookup-method 'ask-if-ambiguous
;;;     There is a differnt copy of this variable for each of the file 
;;;     level commands; just replace the * with get, put, lock, 
;;;     unlock, insert, or info.
;;;
;;;     These variables specify the method used by the command to 
;;;     disambiguate history file lookup.  They can take one of four 
;;;     possible values:
;;;
;;;  	      'ask-if-ambiguous -- will cause interaction if the list 
;;;  	      	     	           of possible history-files contains 
;;;  	      	     	           more than a single name. 
;;;
;;;  	      'always-ask       -- is just like ask-if-ambiguous but 
;;;  	      	     	           will always cause interaction.
;;;
;;;  	      'first            -- does not cause interaction and will 
;;;          	     	           simply return the first name from 
;;;          	     	           the list of possible history file 
;;;          	     	           names.
;;;
;;;  	      'most-recent      -- does not cause interaction and will 
;;;  	                           return the most recently modified 
;;;  	                           history file from the list of 
;;;  	                           possible history file names.
;;;
;;; vcs-ask-save t
;;;     Set to t if you want to be asked to save a buffer a modified 
;;;     that is about to be reloaded by vcs-lock-file or vcs-get-file. 
;;;     Set to nil if you always want to save the buffer.  Set to 
;;;     'dont-save if you don't want to be ask and you don't want to 
;;;     save the buffer either.
;;;
;;; vcs-ask-backup t
;;;     Set to t if you want to be asked to backup a file that is 
;;;     about to be clobbered by vcs-lock-file or vcs-get-file.  Set 
;;;     to nil if you always want to backup the file.  Set to 
;;;     'dont-backup if you don't want to be asked and you don't want 
;;;     the file backed up.
;;;
;;; vcs-put-only-suggest-subdirs t
;;;     If t, vcs-put-file will only suggest history files that are in 
;;;     special sub-directories.  This only applies to non-existent 
;;;     history files.  Pre-existing history files that are not in 
;;;     special subdirectories will still be considered.  This 
;;;     variable does not restrict where a file can be put, rather, it 
;;;     only effects what completions are possible when put prompts 
;;;     for a history file name.
;;;
;;; vcs-use-other-window t
;;;     If t vcs commands will split the current window when appropiate.
;;;
;;; vcs-get-file-mode nil
;;;     If non nil, a mode string to be passed to chmod(1) after 
;;;     getting a file without locking it.
;;;
;;; vcs-lock-file-mode nil
;;;     If non nil, a mode string to be passed to chmod(1) after 
;;;     getting a file that has been locked.
;;;
;;;     
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 
;;;  
;;; $Log: vcs.el,v $
;;; Revision 1.15  19-Sep-1992 22:33:23 EDT  don
;;; Replaced bogus variable reference
;;;
;;; Revision 1.14  19-Sep-1992 00:44:32 EDT  don
;;; Added copious amounts of documentation
;;;
;;; Revision 1.13  17-Sep-1992 21:58:55 EDT  don
;;; Small changes... not much at all
;;;
;;; Revision 1.12  16-Sep-1992 21:28:11 EDT  don
;;; Added variables for controling the lookup method used for all file level
;;; functions
;;;
;;; Revision 1.11  14-Sep-1992 22:23:52 EDT  don
;;; Cleaned up a bit and changed all file level routines to use
;;; get-file-names to disambiguate history file names
;;;
;;; Revision 1.10  13-Sep-1992 22:25:29 EDT  don
;;; A lot of changes to allow history file prompting on gets an such
;;;
;;; Revision 1.9  12-Sep-1992 19:24:02 EDT  don
;;; Added V.I. stuff back in
;;;
;;; Revision 1.8  12-Sep-1992 18:41:38 EDT  don
;;; Separated all of the sccs, rcs, and V.I. specific code into their own
;;; modules
;;;
;;; Revision 1.7  10-Sep-1992 22:16:57 EDT  don
;;; More changes to ask-history-name
;;;
;;; Revision 1.6   9-Sep-1992 22:22:00 EDT  don
;;; Added hack to vi-put to copy the working file to the vi directory.
;;; This is a workaround for a bug in Delta that nobody has noticed because
;;; it has been compensated for by Put
;;;
;;; Revision 1.5   9-Sep-1992 22:14:53 EDT  don
;;; removed absolute path name from the (require 'form) line
;;;
;;; Revision 1.4   9-Sep-1992 22:13:53 EDT  don
;;; Fixed up ask-history-name to use a completion list generated from the
;;; vcs-file-patterns list
;;;
;;; Revision 1.3   7-Sep-1992 20:36:20 EDT  don
;;; Played with the documentation a bit
;;;
;;; Revision 1.2   7-Sep-1992 20:13:08 EDT  don
;;; Added rcs keywords
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; TO DO
;;;
;;;   * Write vcs-is-locked-by-p to to pass to vcs-get-file-name in
;;;     vcs-unlock-file.  If strict locking can be assumed, it would
;;;     also be useful in vcs-put-file as well.  If not then adding
;;;     another variable to hold the predicate for each high level
;;;     command would be alright (Why not... what's 6 more variables).
;;;
;;;   * Write a new version of backup-buffer that looks in the current 
;;;     directory for a directory called BACKUP and if found will do a 
;;;     vcs-put-file rather than backup the file in a traditional 
;;;     sense.  And I will of course have to write a new version of 
;;;     revert-buffer to go with it. 
;;;
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 

(require 'form)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; User variables -- feel free to change the values of any of these 
;;; variables anytime you like.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar vcs-file-patterns '((".*" "."))
  "*File patterns used to locate vcs history files.
This is a list of lists.  The car of each list contains a regular
expression that is used to match filenames.  The cdr of the list
contains directories to be searched to find the file.  The list is
always scanned from the start to the end and early matched take
precedence over later matches. If nothing else, this list should contain
an entry like (\".*\" \".\") which tells vcs to look for any file in the
current directory.")



(defvar vcs-insert-lookup-method 'ask-if-ambiguous
  "*Method used by vcs-insert-file to disambiguate history file lookup.
See vcs-get-file-names for a list of possible values.")

(defvar vcs-get-lookup-method 'ask-if-ambiguous
  "*Method used by vcs-get-file to disambiguate history file lookup.
See vcs-get-file-names for a list of possible values.")

(defvar vcs-lock-lookup-method 'ask-if-ambiguous
  "*Method used by vcs-lock-file to disambiguate history file lookup.
See vcs-get-file-names for a list of possible values.")

(defvar vcs-unlock-lookup-method 'first
  "*Method used by vcs-unlock-file to disambiguate history file lookup.
See vcs-get-file-names for a list of possible values.")

(defvar vcs-put-lookup-method 'always-ask
  "*Method used by vcs-put-file to disambiguate history file lookup.
See vcs-get-file-names for a list of possible values.")

(defvar vcs-info-lookup-method 'ask-if-ambiguous
  "*Method used by vcs-info-file to disambiguate history file lookup.
See vcs-get-file-names for a list of possible values.")



(defvar vcs-ask-save t
  "*Set to t if you want to be asked to save a buffer a modified that
is about to be reloaded by vcs-lock-file or vcs-get-file.  Set to nil
if you always want to save the buffer.  Set to 'dont-save if you don't
want to be ask and you don't want to save the buffer either.")

(defvar vcs-ask-backup t
  "*Set to t if you want to be asked to backup a file that is about to
be clobbered by vcs-lock-file or vcs-get-file.  Set to nil if you
always want to backup the file.  Set to 'dont-backup if you don't want
to be asked and you don't want the file backed up.") 



(defvar vcs-put-mode-hooks nil
  "*Hooks to run after entering the put buffer.")

(defvar vcs-put-only-suggest-subdirs t
  "*If t, vcs-put-file will only suggest history files that are in
special subdirectories.  This only applies to non-existent history
files.  Pre-existing history files that are not in special
subdirectories will still be considered.  This variable does not
restrict where a file can be put, rather, it only effects what
completions are possible when put prompts for a history file name.")

(defvar vcs-put-buffer "*VCS put*"
  "Temporary buffer used for editing the log message")

(defvar vcs-use-other-window t
  "*If t vcs commands will split the current window when appropiate.")

(defvar vcs-get-file-mode nil
  "*If non nil, a mode string to be passed to chmod(1) after getting a
file without locking it.")

(defvar vcs-lock-file-mode nil
  "*If non nil, a mode string to be passed to chmod(1) after getting a
file that has been locked.")

(defvar vcs-shell-path "/bin/sh"
  "*If non-nil, the pathname for the shell to be uses to spawn all
external commands.  If nil, shell-file-name's value is used instead.")

(defvar vcs-accumulate-shell-output nil
  "*If non-nil, all output from vcs shell commands will accumulate
in the vcs-temp-buffer; otherwise the buffer is erased before a 
shell command is executed.  This is mostly useful for debugging.")

(defvar vcs-temp-buffer "*VCS output*"
  "Temporary buffer to hold the output of shell commands")

(defvar vcs-temp-dir "/tmp"
  "*Directory to use when creating temporary files.")



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Extension hooks -- These hooks are provided to allow for easily 
;;; extending vcs to support most any version control system or to 
;;; support local "wrappers" to version control systems.  Support for 
;;; both SCCS and RCS are provided with vcs.  Take a look at 
;;; vcs-rcs.el and vcs-sccs.el to see how to use these hooks.
;;;
;;; In general, the functions called by these hooks all work at the
;;; file level.  Unless explicitly allowed by the hook, they should 
;;; make no assumptions about the current buffer or environment.
;;;
;;; To safely add a function to one of these hooks, use vcs-add-hook.
;;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst vcs-get-hooks nil
  "Whenever a get operation is requested the functions on this list
are called in order until one of them returns non-nil.  All functions
must accept a HISTORY-FILE, WORKING-FILE, and LOCK-FLAG as parameters.
They should return nil if the HISTORY-FILE is not under the control of
the version control system being supported by the function.
Otherwise, the function should extract the most recent version of the
HISTORY-FILE and deposit it in the WORKING-FILE.  If the LOCK-FLAG is
non nil, it should also attempt to lock the file.")


(defconst vcs-unlock-hooks nil
  "Whenever an unlock operation is requested the functions on this
list are called in order until one of them returns a non-nil value.
All functions must accept a HISTORY-FILE and a WORKING-FILE as
parameters.  They should return nil if the HISTORY-FILE is not under
the control of the version control system being supported by the
function.  Otherwise,the function should unlock some version of the
HISTORY-FILE which is locked by the current user.") 


(defconst vcs-insert-info-hooks nil
  "Whenever an info operation is requested the functions on this list
are called in order until one of them returns a non-nil value.  All
functions must accept a HISTORY-FILE and a WORKING-FILE as parameters.
They should return nil if the HISTORY-FILE is not under the control of
the version control system being supported by the function.
Otherwise, it should insert the history log of HISTORY-FILE into the
current buffer.") 


(defconst vcs-put-hooks nil
  "Whenever a put operation is requested the functions on this list
are called in order until one of them returns a non-nil value.  All
functions must accept a HISTORY-FILE and a WORKING-FILE as parameters.
They should return nil if the HISTORY-FILE is not under the control of
the version control system being supported by the function and its
name does not meet the systems naming convention.  Otherwise, it
should deposit the contents of the WORKING-FILE into the next revision
of the HISTORY-FILE. These function don't really have to do the put
right away.  They can set a data entry form in buffer and finish the
put when the user is done filling out the form.  After the put is
finally finished vcs-cleanup-after-put should be call to do the
standard after-the-put stuff.")  


(defconst vcs-locker-name-hooks nil
  "Whenever the name of a file's locker is needed the functions on this
list are called in order, until one of them returns a non-nil value.
All functions must accept a HISTORY-FILE as its only parameter.  They
should return nil if the HISTORY-FILE is not the under control of the 
version control system being supported by the function.  Otherwise, it
return a string specifing the name or names of the HISTORY-FILE's
lockers.  If the file is not currently locked, the empty string \"\"
should be returned.")


(defconst vcs-working-file-hooks nil
  "Whenever a history file name needs to be converted to a working
file name the functions on this list are called in order until one of
them returns a non-nil value.  All functions must accept a
HISTORY-FILE name as their only parameter.  They should return nil if
the HISTORY-FILE is not the under control of the version control
system being supported by the function.  Otherwise, it should convert
the HISTORY-FILE name to a working file name and return it.")


(defconst vcs-history-file-hooks nil
  "Functions on this list are called in order, until one of them returns
a non-nil value, whenever a working file name needs to be converted to
a history file name.  All functions must accept a WORKING-FILE name as
their only parameter.  They should always return the name that would
be given to WORKING-FILE's history file by the version control system
being supported by the function.  Care must be taken to insure that if
given a valid history file name that same name should be returned.")


(defconst vcs-scan-for-error-hooks nil
  "Just after a shell command performing a version control operation
is executed the functions on this list are called in order until one
of them returns a non-nil value.  No parameters are passed.  Each
function is responsible for scanning ahead to the end of the current
buffer to look for errors which might have been reported by the
version control system being supported.  If an error message is found
it should extracted from the buffer and returned, otherwise nil must be
returned.  These functions can assume the current buffer contains the
the output of a version control operation and that the point is at the
start of that output but not necessarily at the start of the buffer.")



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; System variables
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst vcs-buffer-history-file nil
  "Buffer local variable containing the name of the history
file that is most likely associated with the buffer.  This variable
can get confused if the buffer is written out under a name different
from what was used to load it.")


(defconst vcs-mode-line-format
  '("" mode-line-modified mode-line-buffer-identification " "
    global-mode-string " " vcs-mode-string
    " %[(" mode-name minor-mode-alist "%n"
    mode-line-process ")%]----" (-3 . "%p") "-%-")
  "Mode line format used by buffers which have been modified by
vcs-hack-modeline.") 


(defconst vcs-mode-string ""
  "Displays vcs history file status in the mode line.")



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Buffer related commands
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;
;;;
;;;
(defun vcs-get-buffer ()
  "Get the most recent version of the file associated with the current buffer.
A copy of the file is deposited in the buffer's default directory and
the contents of the buffer is updated."
  (interactive)
  (if (not (buffer-file-name))
      (error "No file associated with current buffer")
    (vcs-get-file buffer-file-name)))


;;;
;;;
;;;
(defun vcs-lock-buffer ()
  "Lock the most recent version of the file associated with the current buffer.
A copy of the file is deposited in the buffer's default directory and
the contents of the buffer updated."
  (interactive)
  (if (not (buffer-file-name))
      (error "No file associated with current buffer")
    (vcs-lock-file buffer-file-name)))


;;;
;;; 
;;;
(defun vcs-put-buffer ()
  "Put contents of the current buffer into its associated history
file.  Usually the file must have been locked for this operation to
succeed.  After successful completion of the put, a get is performed
and the buffer contents are updated." 
  (interactive)
  (if (not (buffer-file-name))
      (error "No file associated with current buffer")
    (vcs-put-file buffer-file-name)))


;;;
;;;
;;;
(defun vcs-unlock-buffer (kill-and-remove)
  "Release the version control lock on the history file associated
with the current current buffer." 
  (interactive "P")
  (if (not (buffer-file-name))
      (error "No file associated with current buffer")
    (vcs-unlock-file buffer-file-name kill-and-remove)))


;;;
;;;
;;;
(defun vcs-info-buffer ()
  "Show the delta information for the version control history file
associated with the current buffer."
  (interactive)
  (if (buffer-file-name)
      (vcs-info-file (buffer-file-name))
    (error "No file associated with the current buffer.")))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; File related commands
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;
;;;
;;;
(defun vcs-insert-file (file-name &optional visit)
  "From the history file associated with FILE-NAME, extract the most
recent version and insert into the current buffer at the current point."
  (interactive "FVCS Insert file: ")
  (if (not file-name)
      (error "Invalid file name"))
  (let* ((names (vcs-get-file-names file-name
				    'vcs-has-history-p
				    vcs-insert-lookup-method
				    "VCS Insert from: "))
	 (history-file (car names))
	 (working-file (cdr names)))
    (if (not history-file)
	(error "No history file found for %s" working-file))
    (message "Checking out %s" history-file)
    (vcs-insert-file-contents history-file visit)
    (message "Checked out %s" history-file)))


;;;
;;;
;;;
(defun vcs-get-file (file-name)
  "From the history file associated with FILE-NAME, extract the most
recent version depositing it into the given file name and then visit
the file.  If the file is already in a buffer, the buffer is refreshed 
with the most recent version then selected."  
  (interactive "FVCS Get file: ")
  (let* ((names (vcs-get-file-names file-name
				    'vcs-has-history-p
				    vcs-get-lookup-method
				    "VCS Get from: "))
	 (history-file (car names))
	 (working-file (cdr names)))
    (if (not history-file)
	(error "No history file for %s" working-file))
    (vcs-save-and-backup working-file vcs-ask-save vcs-ask-backup)
    (message "Checking out %s..." history-file)
    (vcs-get history-file working-file nil)
    (switch-to-buffer (vcs-kill-and-reload history-file working-file))
    (message "Checked out %s" history-file)
    t))


;;;
;;;
;;;
(defun vcs-lock-file (file-name)
  "From the history file associated with FILE-NAME, extract and lock
for editing the most recent version, depositing it into the given file
name and then visit the file.  If the file is already in a buffer the
buffer is refreshed with the most recent version then selected."
  (interactive "FVCS Lock file: ")
  (let* ((names (vcs-get-file-names file-name
				    'vcs-has-history-p
				    vcs-lock-lookup-method
				    "VCS Lock from: "))
	 (history-file (car names))
	 (working-file (cdr names)))
    (if (not history-file)
	(error "No history file for %s" working-file))
    (vcs-save-and-backup working-file vcs-ask-save vcs-ask-backup)
    (message "Locking %s..." history-file)
    (vcs-get history-file working-file t)
    (switch-to-buffer (vcs-kill-and-reload history-file working-file))
    (message "Locked %s" history-file)
    t))


;;;
;;; It would be way cool (and maybe a little slow) if we had a
;;; vcs-is-locked-by-me-p to pass to get-file-names as the predicate
;;;
(defun vcs-unlock-file (file-name &optional kill-and-remove)
  "Release any version control locks on FILE.
A prefix argument says to kill any buffer and remove any working file
associated with the file."
  (interactive "FVCS Unlock file: \nP")
  (let* ((names (vcs-get-file-names file-name
				    'vcs-has-history-p
				    vcs-unlock-lookup-method
				    "VCS Unlock from: "))
	 (history-file (car names))
	 (working-file (cdr names)))
    (if (not history-file)
	(error "No history file found for %s" working-file))
    (message "Unlocking %s..." history-file)
    (vcs-unlock history-file working-file kill-and-remove)
    (if (get-file-buffer working-file)
	(vcs-update-modeline  (get-file-buffer working-file)))
    (message "Unlocked %s" history-file))
  t)


;;;
;;;
;;;
(defun vcs-put-file (file-name)
  "Put FILE-NAME under the control of a version control system."
  (interactive "FVCS Put file: ")
  (let* ((names (vcs-get-file-names file-name
				    'vcs-filter-put-suggestions
				    vcs-put-lookup-method
				    "VCS Put into: "))
	 (history-file (car names))
	 (working-file (cdr names)))
    (if (not history-file)
	(error "No history file for %s" file-name))
    (vcs-put history-file working-file)))


;;;
;;;
;;;
(defun vcs-filter-put-suggestions (history-file)
  (or (and vcs-put-only-suggest-subdirs
	   (vcs-uses-special-subdirectory-p history-file))
      (file-exists-p history-file)))


;;;
;;;
;;;
(defun vcs-info-file (file-name)
  "Show the delta information for the version control history file
associated with FILE-NAME."
  (interactive "FVCS Info file: ")
  (let* ((names (vcs-get-file-names file-name
				    'vcs-has-history-p
				    vcs-info-lookup-method
				    "VCS Info from: "))
	 (history-file (car names))
	 (working-file (cdr names)))
    (if (not history-file)
	(error "No history file for %s" working-file))
    (let ((buf (get-buffer-create "*VCS info*")))
      (save-excursion
	(set-buffer buf)
	(setq mode-line-buffer-identification
	      (concat "VCS info: " (file-name-nondirectory working-file)))
	(erase-buffer)
	(message "Getting info for %s..." history-file)
	(vcs-insert-info history-file working-file))
      (if vcs-use-other-window
	  (switch-to-buffer-other-window buf)
	(switch-to-buffer buf))
      (message "Info for %s" history-file))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Interaction support routines
;;;
;;; These routines help the user decide which history file goes with 
;;; which working file.  They were a lot of work so I hope someone 
;;; appreciates them.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;; Given just a file name (either history or working) this routine 
;;; goes through a lot of trouble to return a cons cell, the car of 
;;; which contains the history file name and cdr contains the working 
;;; file.  
;;;
(defun vcs-get-file-names (file-name &optional predicate method prompt)
  "Given a FILE-NAME return the associated history file name and
working file name.  If the optional PREDICATE, if given, must evaluate
to a non-nil value for a history file to be considered.  LOOKUP-METHOD
refers to the method to use for disambiguating history file names.
PROMPT is the string to prompt with if interaction is necessary.
LOOKUP-METHOD can take one of four possible values:
  'ask-if-ambiguous -- (default) will cause interaction if the list
                       of possible history-files contains more than a 
                       single name.

  'always-ask       -- is just like ask-if-ambiguous but will always
                       cause interaction.

  'first            -- does not cause interaction and will simply
                       return the first name from the list of possible
                       history file names.

  'most-recent     -- does not cause interaction and will return the
                      most recently modified history file from the
                      list of possible history file names.

See vcs-lookup-history-path for a description of how the list of
possible names is generated."
  
  (setq method (or method 'ask-if-ambiguous))
  (let ((history-file) (working-file) (buffer))
    (if (vcs-is-history-file-p file-name)
	;; we already have a history name make sure that PREDICATE 
	;; says is alright
	(progn
	  (setq working-file (vcs-working-file-name file-name))
	  (setq history-file
		(if predicate
		    (and (funcall predicate (expand-file-name file-name))
			 (expand-file-name file-name))
		  (expand-file-name file-name))))
      ;; we were given a working file name, so check if there is a 
      ;; buffer visiting the file and use the vcs-buffer-history-file 
      ;; if it has one and PREDICATE says it's alright
      (setq working-file (expand-file-name file-name))
      (setq buffer (get-file-buffer working-file))
      (setq history-file
	    (and buffer 
		 (save-excursion
		   (set-buffer buffer)
		   (if (and predicate vcs-buffer-history-file)
		       (and (funcall predicate
				     vcs-buffer-history-file)
			    vcs-buffer-history-file)
		     vcs-buffer-history-file)))))
    ;; at this point history-file was either passed in or was found in
    ;; a buffer visiting the working-file
    (cond
     ((eq method 'most-recent)
      ;; find the most recent file of them all
      (let ((history-name-list
	     (vcs-lookup-history-path working-file t predicate)))
	(if history-file
	    (setq history-name-list (cons history-file history-name-list)))
	(setq history-name-list (sort history-name-list
				      'file-newer-than-file-p))
	(setq history-file (car history-name-list))))
     ((eq method 'first)
      ;; if we don't already have it, just return the first one
      (if (not history-file)
	  (setq history-file (car (vcs-lookup-history-path
				   working-file t predicate)))))
     ((eq method 'always-ask)
      (setq history-file
	    (vcs-ask-history-name file-name
				  predicate nil prompt history-file)))
     ((eq method 'ask-if-ambiguous)
      (setq history-file
	    (vcs-ask-history-name file-name
				  predicate t prompt history-file))))
    (cons history-file working-file)))


;;;
;;; This function does some weird stuff... It will prompt the user a
;;; second time if an answer is not given the first time.  The first
;;; time they are prompted it is with completing-read with the list of
;;; possible history names.  At this point they can type an arbitrary
;;; file name but they dont get any help from the normal file name
;;; completer.  So... if they want its help they must enter a blank
;;; line and the normal file name completer will take over...  too
;;; much trouble??? you decide.
;;;
(defun vcs-ask-history-name
  (file-name predicate only-if-ambiguous prompt default)
  "Ask the user for a path to the version control history file for
FILE-NAME.  If PREDICATE is non-nil, only those name that cause it to
evaluate to a non-nil value will be considered (names entered during
interaction are not subject to this test).  If ONLY-IF-AMBIGUOUS is
non-nil, interaction will only take place if the list of possible
history file names contain more than a single name.  The PROMPT string
will be used to prompt the user and DEFAULT will be the initial
choice.  Note that DEFAULT counts as a possible history file name."  
  (let* ((history-name-list (vcs-lookup-history-path file-name t predicate))
	 (history-name))
    (if default
	(setq history-name-list (cons default history-name-list))
      (setq default (car history-name-list)))
    ;; sort the list and remove duplicates
    (setq history-name-list (sort history-name-list 'string-lessp))
    (let ((name history-name-list))
      (while (cdr name)
	(if (string= (car name) (car (cdr name)))
	    (setcdr name (cdr (cdr name)))
	  (setq name (cdr name)))))
    ;; a zero length list counts as being ambiguous maybe it should be 
    ;; an error... let the user decide
    (if (and only-if-ambiguous
	     (= (length history-name-list) 1))
	(car history-name-list)
      ;; turn it into an alist for the completer
      (setq history-name-list
	    (mapcar '(lambda (name) (list name))
		    history-name-list))
      (setq history-name
	    (completing-read (or prompt "History file: ")
			     history-name-list
			     nil nil default))
      (while (string= "" history-name)
	(setq history-name
	      (read-file-name (or prompt "History file: ")
			      (file-name-directory default)
			      default)))
      ;; we shouldn't really trust the user so make sure it is a history 
      ;; file name  
      (vcs-history-file-name history-name))))


;;;
;;; This function is useful when you know a vcs command might clobber 
;;; an existing working file.
;;;
(defun vcs-save-and-backup (filename &optional ask-save ask-backup)
  "Save the FILE and copy it to FILE.BAK  The optional parameter
ASK-SAVE specifies whether the user should be asked before saving a
modified buffer which is visiting the file.  If t they will be asked,
if nil they won't be asked but the buffer will be saved and if
'dont-save specifies not to ask and don't save the buffer.  ASK-BACKUP
specifies whether the user should be asked about backing the file if a
writeable copy of the file exists.  If t they will be asked, if nil
they they won't be asked but the file will be backed up, if
'dont-backup they won't be asked and the file will not be backed up"
  (setq file-name (expand-file-name file-name))
  (let ((buf (get-file-buffer file-name)))
    (if (and (not (eq ask-save 'dont-save))
	     buf
	     (buffer-modified-p buf))
	(save-excursion
	  (set-buffer buf)
	  (if (or (not ask-save)
		  (y-or-n-p (concat "Save " buffer-file-name "? ")))
	      (save-buffer))))
    (if (and (not (eq ask-backup 'dont-backup))
	     (file-exists-p file-name)
	     (file-writable-p file-name)
	     (or (not ask-backup)
		 (y-or-n-p (format "Backup %s? "
				   file-name
				   (file-name-nondirectory file-name)))))
	(vcs-backup-file file-name))))


;;;
;;; This function was mostly stolen from files.el backup-buffer It
;;; looks at the same variables for determining how to do the backup.
;;; The only real difference is that it takes a file name argument
;;; rather than assuming the current buffer.  And, it is used to
;;; backup a file after a save has been made, rather than before it.  
;;; So the return value is not needed and has been eliminated.
;;;
(defun vcs-backup-file (file-name)
  "Make a backup of the FILE-NAME."
  (and (file-exists-p buffer-file-name)
       (memq (aref (elt (file-attributes file-name) 8) 0)
	     '(?- ?l))
       (or (< (length file-name) 5)
	   (not (string-equal "/tmp/" (substring file-name 0 5))))
       (condition-case ()
	   (let* ((backup-info (find-backup-file-name file-name))
		  (backupname (car backup-info))
		  (targets (cdr backup-info)))
	     (condition-case ()
		 (if (or file-precious-flag
			 (file-symlink-p file-name)
			 backup-by-copying
			 (and backup-by-copying-when-linked
			      (> (file-nlinks file-name) 1))
			 (and backup-by-copying-when-mismatch
			      (let ((attr (file-attributes file-name)))
				(or (nth 9 attr)
				    (/= (nth 2 attr) (user-uid))))))
		     (copy-file file-name backupname t t)
		   (condition-case ()
		       (delete-file backupname)
		     (file-error nil))
		   (rename-file file-name backupname t))
	       (file-error
		;; If trouble writing the backup, write it in ~.
		(setq backupname (expand-file-name "~/%backup%~"))
		(message
		 "Cannot write backup file; backing up in ~/%%backup%%~")
		(sleep-for 1)
		(copy-file file-name backupname t t)))
	     (if (and targets
		      (or trim-versions-without-asking
			  (y-or-n-p
			   (format "Delete excess backup versions of %s? "
				   buffer-file-name))))
		 (while targets
		   (condition-case ()
		       (delete-file (car targets))
		     (file-error nil))
		   (setq targets (cdr targets)))))
	 (file-error nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Generic support routines
;;;
;;; These routine don't know anything about any particular version
;;; control system they are based only on other vcs-routines
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;;
;;;
(defun vcs-insert-file-contents (history-file  &optional visit)
  "Insert the contents of a vcs HISTORY-FILE into the current buffer."
  (let ((tmp-dir (concat vcs-temp-dir "/" (make-temp-name "vcs"))))
    (call-process "rm" nil nil nil "-rf" tmp-dir)
    (call-process "mkdir" nil nil nil tmp-dir)
    (if (not (file-directory-p tmp-dir))
	(error "Unable to create temporary directory %s" tmp-dir))
    (let ((working-file
	   (concat tmp-dir "/"
		   (file-name-nondirectory (vcs-working-file-name
					    history-file)))))
      (unwind-protect
	  (vcs-get history-file working-file nil)
	(if (file-exists-p working-file)
	    (insert-file-contents working-file visit))
	(call-process "rm" nil nil nil "-rf" tmp-dir)))))


;;;
;;;
;;;
(defun vcs-lookup-path-name (file-name &optional find-all predicate)
  "Return the path to FILE-NAME by looking down the vcs-file-patterns list.
The directory (if any) given in FILE-NAME is ignored.  If FIND-ALL is
true the a list of all possible path names is returned.  If PREDICATE
is given each possible path name is passed to it; only those that
generate a non-nil return value are added to the list.  If FIND-ALL is
nil only the first name found is returned."
  (let* ((pattern-list vcs-file-patterns)
	 (file-name (file-name-sans-versions (expand-file-name file-name)))
	 (name (file-name-nondirectory (file-name-sans-versions file-name)))
	 (default-directory (file-name-directory file-name))
	 (path-name-list nil)
	 (full-name nil)
	 (pattern) (dir-list))
    (while (and pattern-list
		(or find-all (not path-name-list)))
      (setq pattern (car (car pattern-list)))
      (setq dir-list (cdr (car pattern-list)))
      (if (string-match pattern name)
	  (while (and dir-list
		      (or find-all (not path-name-list)))
	    (condition-case nil
		(let ((full-name (expand-file-name
				  (substitute-in-file-name
				   (concat (car dir-list) "/" name)))))
		  (if (or (not predicate)
			  (funcall predicate full-name))
		      (setq path-name-list
			    (cons full-name path-name-list))))
	      (error nil))
	    (setq dir-list (cdr dir-list))))
      (setq pattern-list (cdr pattern-list)))
    (setq path-name-list (nreverse path-name-list))
    (if (not find-all)
	(car path-name-list)
      path-name-list)))



;;;
;;;
;;;
(defun vcs-lookup-history-path (file-name &optional find-all predicate)
  "Return the path to the version control history file for FILE-NAME.
Or nil if one does not exist."
  (setq file-name (expand-file-name file-name))
  (let* ((path-name-list (vcs-lookup-path-name file-name t nil))
	 (history-name-list
	  (apply 'append
		 (mapcar '(lambda (name)
			    (setq name
				  (vcs-history-file-name name find-all
							 predicate))
			    (if (listp name)
				name
			      (list name)))
			    path-name-list))))
    (if (not find-all)
	(car history-name-list)
      history-name-list)))


;;;
;;;
;;;
(defun vcs-shorten-file-name (file-name)
  ;; Get rid of the prefixes added by the automounter.
  (setq file-name (expand-file-name file-name)) 
  (if (and (string-match automount-dir-prefix file-name)
	   (file-exists-p (file-name-directory
			   (substring file-name (1- (match-end 0))))))
      (setq file-name (substring file-name (1- (match-end 0)))))
  (if (string= default-directory "/")
      file-name
    ;; Replace the default directory with .
    (if (string-match (concat "^" (regexp-quote default-directory))
		      file-name)
	(setq file-name (concat "." (substring file-name (1- (match-end 0))))))
    ;; Replace the parent directory with ..
    (if (string-match
	 (concat "^" (regexp-quote (expand-file-name
				    (concat default-directory "../"))))
	 file-name)
	(setq file-name (concat ".."
				(substring file-name (1- (match-end 0)))))))
  ;; Replace the HOME directory with ~
  (if (string-match (concat "^" (regexp-quote (getenv "HOME"))) file-name)
      (setq file-name (concat "~" (substring file-name (match-end 0)))))
  file-name)
  
  

;;;
;;;
;;;
(defun vcs-file-exists-p (file-name)
  "Return t if FILE-NAME exists or has a history file."
  (or (file-exists-p file-name)
      (vcs-has-history-p file-name)))


;;;
;;;
;;;
(defun vcs-has-history-p (file-name)
  "Return the name of the version control history file iff FILE has a one."
  (if file-name
      (vcs-history-file-name file-name nil 'file-exists-p)))


;;;
;;;
;;;
(defun vcs-uses-special-subdirectory-p (name)
  (not (string= (file-name-directory (vcs-working-file-name name))
		(file-name-directory name))))


;;;
;;;
;;;
(defun vcs-is-history-file-p (history-name)
  "Returns t if FILE-NAME the name of a history file for any of
version control system being used."
  (not (string= (expand-file-name history-name)
		(vcs-working-file-name history-name))))


;;;
;;;
;;;
(defun vcs-kill-and-reload (history-file working-file)
  "Extract the most recent version of HISTORY-FILE and reload the
buffer visiting WORKING-FILE with its contents and return the buffer
used.  If there is no buffer visiting WORKING-FILE one will be
created.  If the buffer has been modified the file will NOT be saved
before reloading.  Makes a lame attempt to keep the mark and point in
the same location." 
  (let ((buf (get-file-buffer working-file)))
    (if buf
	(let ((curp (point))
	      (curm (mark)))
	  (set-buffer buf)
	  (if (file-exists-p working-file)
	      (revert-buffer t t)
	    (and (not (verify-visited-file-modtime (current-buffer)))
		 (setq buffer-backed-up nil))
	    ;; Discard all the undo information.
	    (or (eq buffer-undo-list t)
		(setq buffer-undo-list nil))
	    (let ((buffer-read-only nil)
		  ;; Don't record undo info for the revert itself.
		  ;; Doing so chews up too much storage.
		  (buffer-undo-list t))
	      ;; Bind buffer-file-name to nil
	      ;; so that we don't try to lock the file.
	      (let ((buffer-file-name nil))
		(unlock-buffer)
		(erase-buffer))
	      (vcs-insert-file-contents history-file nil))
	    (set-buffer-modified-p nil)
	    (after-find-file nil nil))
	  (set-mark curm)
	  (goto-char curp)
	  buf)
      (let ((find-file-not-found-hooks (list 'vcs-try-file)))
	(setq buf (find-file-noselect working-file t))))
    (if buf
	(save-excursion
	  (set-buffer buf)
	  (set (make-local-variable 'vcs-buffer-history-file) history-file)
	  buf))))
      	  
  

;;;
;;;
;;;
(defun vcs-update-all-modelines nil
  "Run through the list of buffer and update the modelines of all with
vcs files."
  (interactive)
  (let ((list (buffer-list)))
    (while list
      (vcs-update-modeline (car list))
      (setq list (cdr list)))))


;;;
;;; 
;;;
(defun vcs-update-modeline (&optional buffer)
  (interactive)
  (setq buffer (or buffer (current-buffer)))
  (if buffer
      (save-excursion
	(set-buffer buffer)
	(if (and (eq mode-line-format vcs-mode-line-format)
		 vcs-buffer-history-file)
	    (let ((locker))
	      (setq locker
		    (condition-case nil
			(vcs-locker-name vcs-buffer-history-file)
		      (error "<error>")))
	      (setq vcs-mode-string
		    (if (not locker)
			"[Not locked]"
		      (concat "[Locker: " locker "]")))
	      (set-buffer-modified-p (buffer-modified-p)))))))
		    
		      


;;;
;;; If we already know the current buffer to be a vcs buffer than we 
;;; just hack the mode line using vcs-buffer-history-file.  Otherwise
;;; we try to look up the history file and if we find it mark the 
;;; buffer as a vcs buffer and set its mode line format.
;;;
(defun vcs-hack-modeline ()
  "Modify the modeline for vcs files."
  (let* ((names (vcs-get-file-names buffer-file-name
				    'vcs-has-history-p 'most-recent)) 
	 (history-file (car names))
	 (working-file (cdr names)))
    (setq history-file (or vcs-buffer-history-file history-file))
    (if history-file
	(progn
	  (set (make-local-variable 'vcs-buffer-history-file) history-file)
	  (make-local-variable 'vcs-mode-string)
	  (setq mode-line-format vcs-mode-line-format)
	  (vcs-update-modeline)))))

;;;
;;;
;;;
(defun vcs-try-file (&optional file-name terse)
  "Check the vcs-file-patterns list for a file when find-file fails."
  (if (not terse)
      (message "Looking for %s..." (file-name-nondirectory buffer-file-name)))
  (kill-local-variable 'vcs-buffer-history-file)
  (let ((loaded-file (vcs-load buffer-file-name)))
    (if loaded-file
	(progn
	  (if (not terse)
	      (message "Found in %s" loaded-file))
	  (set-buffer-modified-p nil)
	  ;; keep after-find-file quiet
	  (setq error nil)
	  t))))


(defun vcs-mark-as-read-only nil
  "Mark a buffer as read only if it is a vcs file and does not exist"
  (if (and (not (file-exists-p (buffer-file-name)))
	   vcs-buffer-history-file
	   (file-exists-p vcs-buffer-history-file))
      (setq buffer-read-only t)))

;;;
;;;
;;;
(defun vcs-load (file-name)
  (let* ((default-directory (file-name-directory file-name))
	 (file-name (vcs-lookup-path-name file-name nil 'vcs-file-exists-p)))
    (if file-name
	(let* ((names (vcs-get-file-names file-name 'vcs-has-history-p
					 'most-recent))
	       (history-file (car names))
	       (working-file (cdr names)))
	  (cond
	   ((not working-file)
	    nil)
	   ((and history-file
		 (file-exists-p working-file)
		 (file-newer-than-file-p working-file history-file))
	    (insert-file-contents working-file nil)
	    (set (make-local-variable 'vcs-buffer-history-file) history-file)
	    working-file)
	   (history-file
	    (vcs-insert-file-contents history-file nil)
	    (set (make-local-variable 'vcs-buffer-history-file) history-file)
	    history-file)
	   (t ;file-name must exist
	    (insert-file-contents file-name nil)
	    file-name))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; These routines know that there is more to version control than
;;; just vcs.  They generally decide which system specific routine to
;;; use by calling a list of functions until one of them returns a 
;;; non-nil value.  They then know that the operation has taken place.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; 

;;;
;;; Calls vcs-get-hooks...
;;;
(defun vcs-get (history-file working-file &optional lock)
  (let ((status)
	(hooks vcs-get-hooks))
    (while (and hooks (not status))
      (setq status (funcall (car hooks) history-file working-file lock))
      (setq hooks (cdr hooks)))
    (if (not status)
	(error "Invalid history file name: %s" history-file))
    (if lock
	(if vcs-lock-file-mode 
	    (call-process "chmod" nil nil nil
			  vcs-lock-file-mode working-file))
      (if vcs-get-file-mode
	  (call-process "chmod" nil nil nil
			vcs-get-file-mode working-file)))))
      

;;;
;;; Calls vcs-unlock-hooks...
;;;
(defun vcs-unlock (history-file working-file &optional remove)
  (let ((status)
	(hooks vcs-unlock-hooks))
    (while (and hooks (not status))
      (setq status (funcall (car hooks) history-file working-file))
      (setq hooks (cdr hooks)))
    (if (not status)
	(error "Invalid history file name: %s" history-file)))
  (if remove
      (let ((buf (get-file-buffer working-file)))
	(call-process "rm" nil nil nil "-f" working-file)
	(if buf
	    (save-excursion
	      (set-buffer buf)
	      (set-buffer-modified-p nil)
	      (kill-buffer buf))))))


;;;
;;; Calls vcs-insert-info-hooks...
;;;
(defun vcs-insert-info (history-file working-file)
  (let ((status)
	(hooks vcs-insert-info-hooks))
    (while (and hooks (not status))
      (setq status (funcall (car hooks) history-file working-file))
      (setq hooks (cdr hooks)))
    (if (not status)
	(error "Invalid history file name: %s" history-file))))



;;;
;;; Calls vcs-put-hooks...
;;;
(defun vcs-put (history-file working-file)
  (let ((status)
	(hooks vcs-put-hooks))
    (while (and hooks (not status))
      (setq status (funcall (car hooks) history-file working-file))
      (setq hooks (cdr hooks)))
    (if (not status)
	(error "Invalid history file name: %s" history-file))))


;;;
;;; This function has been seperated from vcs-put to allow the
;;; vcs-put-hooks functions to do their thing in two steps.  The first
;;; step usually involves setting up a form buffer and letting the
;;; user edit the form.  When the user signals that they are done
;;; editing the put operation can actually take place.  Afterwards,
;;; vcs-clean-up-after-put should be called to take care of any
;;; buffers that are displaying the just put file.
;;;
(defun vcs-cleanup-after-put (history-file working-file)
  "Reload a buffer after the file has been put."
  (let ((buf (get-file-buffer working-file)))
    (if (not buf)
	(vcs-get history-file working-file nil)
      (if (not (buffer-modified-p buf))
	  (save-window-excursion
	    (vcs-get history-file working-file nil)
	    (vcs-kill-and-reload history-file working-file))))))


;;;
;;; Calls vcs-locker-name-hooks...
;;;
(defun vcs-locker-name (history-file)
  "Return the name of the person who has FILE-NAME locked"
  (if (not (file-exists-p history-file))
      (error "No history file found for %s" file-name))
  (let ((locker-name)
	(hooks vcs-locker-name-hooks))
    (while (and hooks (not locker-name))
      (setq locker-name (funcall (car hooks) history-file))
      (setq hooks (cdr hooks)))
    (if (string= "" locker-name)
	nil
      locker-name)))


;;;
;;; Calls vcs-working-file-hooks
;;;
(defun vcs-working-file-name (history-file)
  "Return the working file name for a HISTORY-FILE name.
If the given name is not a valid history file name it is returned as given."
  (if history-file
      (let ((working-file)
	    (history-file (expand-file-name history-file))
	    (hooks vcs-working-file-hooks))
	(while (and hooks (not working-file))
	  (setq working-file (funcall (car hooks) history-file))
	  (setq hooks (cdr hooks)))
	(if working-file
	    working-file
	  history-file))))
	

;;;
;;; Calls vcs-history-file-hooks...
;;;
(defun vcs-history-file-name (file-name &optional all predicate)
  "Return the name or names of the version control history file for
FILE-NAME.  If the optional parameter PREDICATE is given any returned
file name will cause predicate to evaluate to true.  If the optional
parameter ALL is not nil, then all possible names are returned on a list."
  (if (not file-name)
      nil
    (if (and (not all) (vcs-is-history-file-p file-name))
	(if predicate
	    (and (funcall predicate file-name)
		 (expand-file-name file-name))
	  (expand-file-name file-name))
      (let* ((file-name (vcs-working-file-name file-name))
	     (directory (file-name-directory file-name))
	     (history-file-list)
	     (history-file)
	     (hooks vcs-history-file-hooks))
	(while hooks
	  (setq history-file (funcall (car hooks) file-name))
	  (if predicate
	      (if (funcall predicate history-file)
		  (setq history-file-list
			(cons history-file history-file-list)))
	    (setq history-file-list
		  (cons history-file history-file-list)))
	  (setq hooks (cdr hooks)))
	(setq history-file-list (nreverse history-file-list))
	(if all
	    history-file-list
	  (setq history-file nil)
	  (setq list history-file-list)
	  (while (and list (not history-file))
	    (if (and (not (string= directory (file-name-directory (car list))))
		     (string= directory (expand-file-name
					 (concat (file-name-directory
						  (car list)) 
						 "../"))))
		(setq history-file (car list)))
	    (setq list (cdr list)))
	  (or history-file
	      (car history-file-list)))))))


;;;
;;; Calls vcs-scan-for-error-hooks...
;;; 
(defun vcs-execute-command (directory command &rest arguments)
  "In DIRECTORY, execute COMMAND with ARGUMENTS."
  (setq vcs-last-error-message nil)
  (save-excursion
    (let ((case-fold-search nil)
	  (buftmp (get-buffer-create vcs-temp-buffer))
	  (curpoint)
	  (command (concat (if directory
			       (concat "cd "
				       (expand-file-name directory) "\n"))
			   command " "
			   (mapconcat (function identity)
				      arguments " "))))
      (set-buffer buftmp)
      (if vcs-accumulate-shell-output
	  (goto-char (point-max))
	(erase-buffer))
      (insert "* " (current-time-string) " *" "\n" command "\n")
      (setq curpoint (point))
      (call-process vcs-shell-path nil t nil "-c" command)
      (insert "\n")
      (goto-char curpoint)
      (let ((hooks vcs-scan-for-error-hooks))
	(while (and hooks (not vcs-last-error-message))
	  (setq vcs-last-error-message
		(save-excursion (funcall (car hooks))))
	  (setq hooks (cdr hooks))))
      (if vcs-last-error-message
	  (error vcs-last-error-message)
	t))))

;;;
;;;
;;;
(defun vcs-add-hook (hook function)
  (if (not (symbolp hook))
      (error "hook must be a symbol"))
  (if (not (symbolp function))
      (error "function must be a symbol"))
  (if (not (memq function (eval hook)))
      (set hook (cons function (eval hook)))))

