/*
 * (C)opyright 1995 by Darren Reed.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and due credit is given
 * to the original author and the contributors.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include "y.tab.h"
#include "var.h"
#include "misc.h"

#ifndef	lint
static	char	sccsid[] = "@(#)var.c	1.5 12/17/95 (C) 1995 Darren Reed";
#endif

var_t	*varlist = NULL;

static	int	nextv;

#define	ALREADY(x, d, y)	synerr("%s%s already specified: %s\n", \
					(((d) == 1) ? "from " : \
					 ((d) == 2) ? "to " : ""), (x), (y))

/*
 * add a new value to the variable specified by token, at the correct lvl.
 * If there is no variable structure for this level, create one.  "dir" is
 * for host/port rules which have "to" and "from" fields.  str is the value
 * to be given to the variable.
 *
 * Don't replace a variable if it has been previously set but complain.
 */
void	add_var(lvl, dir, token, str)
int	lvl, dir, token;
char	*str;
{
	var_t	*v, **vp;
	char	**d = NULL, *s, **d2 = NULL, *str2 = NULL;
	int	idx = dir - 1;

	/*
	 * Linked list of variable levels, where the set of variables closest
	 * to the top has the highest level number.
	 */
	for (vp = &varlist; (v = *vp); vp = &v->va_next)
		if ((v->va_lvl < lvl) || (v->va_lvl == lvl))
			break;;
	if (!v || (v->va_lvl < lvl)) {
		v = (var_t *)malloc(sizeof(*v));
#if defined(__SVR4) || defined(__sysv__) || defined(__svr4__)
		memset((char *)v, 0, sizeof(*v));
#else
		bzero((char *)v, sizeof(*v));
#endif
		v->va_lvl = lvl;
		v->va_next = *vp;
		*vp = v;
	}

	v->va_policy |= nextv;
	nextv = 0;

#ifdef	DEBUG
	printf("add_var(%d,%d,%d,%s) v %x\n", lvl, dir, token, str, v);
#endif

	switch (token)
	{
	case FP_IN :
	case FP_OUT :
		if (v->va_inout)
			ALREADY("in/out", 0, v->va_inout);
		else
			d = &v->va_inout;
		break;
	case FP_INBOUND :
	case FP_OUTBOUND :
		d = &v->va_inout;
		if (*str == 'i' || *str == 'I')
			str = "in";
		else
			str = "out";
		break;
	case FP_MASK :
		if ((dir < 1) || (dir > 2))
			break;
		if (v->va_mask[idx])
			ALREADY("mask", dir, v->va_mask[idx]);
		else
			d = &v->va_mask[idx];
		break;
	case FP_NETMASK :
		if (v->va_mask[0] || v->va_mask[1])
			ALREADY("netmask", dir, "(set netmask)");
		else {
			v->va_mask[0] = strdup(str);
			v->va_mask[1] = strdup(str);
		}
		break;
	/*
	 * The "net" and "host" keywords compete for the same
	 * variable space.
	 */
	case FP_NET :
		if (v->va_net[idx])
			ALREADY("net", dir, v->va_net[idx]);
		else if (v->va_host[idx])
			ALREADY("host", dir, v->va_host[idx]);
		else {
			if ((s = strchr(str, '/')))
				if (v->va_mask[idx])
					ALREADY("netmask", dir, s);
				else {
					*s++ = '\0';
					str2 = s;
					d2 = &v->va_mask[idx];
				}
			d = &v->va_net[idx];
		}
		break;
	case FP_HOST :
		if ((dir < 1) || (dir > 2))
			break;
		if (v->va_host[idx])
			ALREADY("host", dir, v->va_host[idx]);
		else if (v->va_net[idx])
			ALREADY("net", dir, v->va_net[idx]);
		else
			d = &v->va_host[idx];
		break;
	case FP_PORT :
		if ((dir < 1) || (dir > 2))
			break;
		if (v->va_port1[idx])
			ALREADY("port", dir, v->va_port1[idx]);
		else
			d = &v->va_port1[idx];
		break;
	case FP_PORTCMP :
		if ((dir < 1) || (dir > 2))
			break;
		if (v->va_portcmp[idx])
			ALREADY("port comparison", dir, v->va_portcmp[idx]);
		else
			d = &v->va_portcmp[idx];
		break;
	case FP_INSIDE :
	case FP_OUTSIDE :
		if ((dir < 1) || (dir > 2))
			break;
		if (v->va_port2[idx])
			ALREADY("port", dir, v->va_port2[idx]);
		else
			d = &v->va_port2[idx];
		break;
	case FP_PROTO :
		if (v->va_proto)
			ALREADY("protocol", 0, v->va_proto);
		else
			d = &v->va_proto;
		break;
	case FP_ON :
	case FP_INTERFACE :
		if (v->va_iface)
			ALREADY("interface", 0, v->va_iface);
		else
			d = &v->va_iface;
		break;
	case FP_ACCESSLIST :
		if (v->va_aclnum)
			ALREADY("access-list", 0, v->va_aclnum);
		else
			d = &v->va_aclnum;
		break;
	case FP_FLAGS :
		if (v->va_flags)
			ALREADY("tcp flags", 0, v->va_flags);
		else
			d = &v->va_flags;
		break;
	case FP_ESTABLISHED :
		if (v->va_flags)
			ALREADY("tcp flags", 0, v->va_flags);
		else
			d = &v->va_flags;
		break;
	case FP_OPENING :
		if (v->va_flags)
			ALREADY("tcp flags", 0, v->va_flags);
		else
			d = &v->va_flags;
		break;
	/*
	 * Options and security options.  Almost none of the other freely
	 * available packages support even close to half of this.  sigh.
	 */
	case FP_SECCLASS :
		if ((s = v->va_opts)) {
			if (strstr(s, str)) {
				ALREADY("IP security option", 0, str);
				break;
			} else if (!strcasecmp(s, "ipopts")) {
				ALREADY("IP options", 0, str);
				break;
			}
		}
		if (s)
			s = (char *)realloc(s, strlen(str) + strlen(s) + 12);
		else
			s = (char *)malloc(strlen(str) + 12);
		strcat(s, " sec-class=");
		strcat(s, str);
		v->va_opts = s;
		break;
	case FP_OPT :
		if ((s = v->va_opts)) {
			if (strstr(s, str)) {
				ALREADY("IP option", 0, str);
				break;
			} else if (!strcasecmp(s, "ipopts")) {
				ALREADY("IP options", 0, str);
				break;
			}
		}
		if (s)
			s = (char *)realloc(s, strlen(str) + strlen(s) + 6);
		else
			s = (char *)malloc(strlen(str) + 6);
		strcat(s, " opt ");
		strcat(s, str);
		v->va_opts = s;
		break;
	case FP_FRAG :
	case FP_SHORT :
	case FP_IPOPTS :
		if ((s = v->va_opts)) {
			if (strstr(s, str)) {
				ALREADY("IP option", 0, str);
				break;
			} else if (!strcasecmp(s, "ipopts")) {
				ALREADY("IP options", 0, str);
				break;
			}
		}
		if (s) {
			s = (char *)realloc(s, strlen(str) + strlen(s) + 2);
			strcat(s, " ");
			strcat(s, str);
		} else
			s = strdup(str);
		v->va_opts = s;
		break;
	case FP_LOG :
	case FP_PASS :
	case FP_BODY :
	case FP_BLOCK :
		if ((s = v->va_action)) {
			if (strstr(s, str)) {
				ALREADY("action", 0, str);
				break;
			if ((strstr(s, "pass") && !strcasecmp(str, "block")) ||
			    (strstr(s, "pass") && !strcasecmp(str, "block")))
				synerr("conflicting pass/block\n");
				break;
			}
		} else if (token == FP_BODY) {
			synerr("body specified without log option\n");
			break;
		}
		if (s) {
			s = (char *)realloc(s, strlen(str) +
					    strlen(v->va_action) + 2);
			strcat(s, " ");
			strcat(s, str);
		} else
			s = strdup(str);
		v->va_action = s;
		return;
	}
	if (d)
		*d = strdup(str);
	if (d2)
		*d2 = strdup(str2);
	return;
}


void	del_lvl(lvl)
int	lvl;
{
	var_t	**vp, *v;
	int	i;

	for (vp = &varlist; (v = *vp); vp = &v->va_next)
		if (v->va_lvl == lvl)
			break;
#ifdef	DEBUG
	printf("del_lvl(%d) = %x[%x] %x\n", lvl, vp, *vp, v);
#endif
	if (!v)
		return;
	*vp = v->va_next;

	for (i = 0; i < 2; i++) {
		if (v->va_host[i])
			free(v->va_host[i]);
		if (v->va_port1[i])
			free(v->va_port1[i]);
		if (v->va_port2[i])
			free(v->va_port2[i]);
		if (v->va_net[i])
			free(v->va_net[i]);
		if (v->va_portcmp[i])
			free(v->va_portcmp[i]);
		if (v->va_port1[i])
			free(v->va_port1[i]);
		if (v->va_port2[i])
			free(v->va_port2[i]);
		if (v->va_net[i])
			free(v->va_net[i]);
	}

	if (v->va_action)
		free(v->va_action);
	if (v->va_inout)
		free(v->va_inout);
	if (v->va_proto)
		free(v->va_proto);
	if (v->va_iface)
		free(v->va_iface);
	if (v->va_ttl)
		free(v->va_ttl);
	if (v->va_tos)
		free(v->va_tos);
	free(v);
}


void	clear_var(which)
int	which;
{
	var_t	*v;

	/*
	 * find the bottom most...
	 */
	if (which == FP_POLICY)
		for (v = varlist; v && v->va_next; v = v->va_next)
			;
	else
		v = varlist;
	if (!v)
		return;

	switch (which)
	{
	case FP_POLICY :
		if (v->va_action) {
			free(v->va_action);
			v->va_action = NULL;
		}
		nextv = VP_ACTION;
		break;
	case FP_INTERFACE :
		if (v->va_iface) {
			free(v->va_iface);
			v->va_iface = NULL;
		}
		nextv = VP_INTERFACE;
		break;
	case FP_ACCESSLIST :
		if (v->va_aclnum) {
			free(v->va_aclnum);
			v->va_aclnum = NULL;
		}
		nextv = VP_ACCESSLIST;
		break;
	case FP_PROTOCOL :
		if (v->va_proto) {
			free(v->va_proto);
			v->va_proto = NULL;
		}
		nextv = VP_PROTOCOL;
		break;
	case FP_IN :
	case FP_OUT :
	case FP_INBOUND :
	case FP_OUTBOUND :
		if (v->va_inout) {
			free(v->va_inout);
			v->va_inout = NULL;
		}
		nextv = VP_INOUT;
		break;
	}
}


int	setup_policy(state, which)
int	state, which;
{
	switch (which)
	{
	case FP_LOG :
	case FP_PASS :
	case FP_BLOCK :
	case FP_POLICY :
		varlist->va_policy |= VP_ACTION;
		break;
	case FP_INTERFACE :
		varlist->va_policy |= VP_INTERFACE;
		break;
	case FP_ACCESSLIST :
		varlist->va_policy |= VP_ACCESSLIST;
		break;
	case FP_PROTOCOL :
		varlist->va_policy |= VP_PROTOCOL;
		break;
	case FP_IN :
	case FP_OUT :
	case FP_INBOUND :
	case FP_OUTBOUND :
		varlist->va_policy |= VP_INOUT;
		break;
	}
	return state;
}


void	call_emitpolicy(emitfunc)
void	(*emitfunc)();
{
	var_t	*v;

	if (!(v = varlist) || !v->va_policy)
		return;
	if ((v->va_policy & (VP_ACTION|VP_INOUT)) != (VP_ACTION|VP_INOUT)) {
		synerr("policy inadequately defined\n");
		fprintf(stderr, "\tmissing: %s\n",
			(v->va_policy & VP_INOUT) ? "<action>" : "<in-out>");
		return;
	}
	(*emitfunc)();
}


void	clear_policy(v)
var_t	*v;
{
	if (v->va_policy & VP_ACTION) {
		free(v->va_action);
		v->va_action = NULL;
	}

	if (v->va_policy & VP_INTERFACE) {
		free(v->va_iface);
		v->va_iface = NULL;
	}

	if (v->va_policy & VP_ACCESSLIST) {
		free(v->va_aclnum);
		v->va_aclnum = NULL;
	}
	if (v->va_policy & VP_PROTOCOL) {
		free(v->va_proto);
		v->va_proto = NULL;
	}

	if (v->va_policy & VP_INOUT) {
		free(v->va_inout);
		v->va_inout = NULL;
	}
}
