/* pop-gw.c */

/* Copyright 1997 by Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
   Donated to the public domain.  No warranty.

   Version 0.1	1997-04-09 Initial version */

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/file.h>
#include "firewall.h"
#include "../libem/libemfw.h"
#include "../libem/libemtn.h"

static Cfg *confp;
static const char *server;
static char user[32];
static char pass[32];
static char riaddr[512];
static char rladdr[512];
static char input[512];
static tnconn *tc, *ts;


static void ok (void)
{
  tn_puts (tc, "+OK\r\n");
}


static void command (const char *cmd, const char *arg)
{
  tn_printf (ts, "%s %s\r\n", cmd, arg);
  tn_gets (ts, input, sizeof (input), PROXY_TIMEOUT, 0);
  if (input[0] != '+')
    {
      tn_puts (tc, "-ERR authorization failed\r\n");
      exit (1);
    }
}


#define CHANNEL_BUFSIZ  4096

struct channel
{
  int fdr, fdw;
  int count, eof;
  char buf[CHANNEL_BUFSIZ];
};

static void channel_init (struct channel *c, int fdr, int fdw)
{
  int fl;

  c->fdr = fdr; c->fdw = fdw;
  c->count = 0; c->eof = 0;
  fl = fcntl (fdw, F_GETFL, 0);
  if (fl != -1)
    fl = fcntl (fdw, F_SETFL, fl | O_NDELAY);
  if (fl == -1)
    {
      syslog (LLEV, "fcntl");
      exit (1);
    }
}

static void channel_fdset (const struct channel *c, fd_set *rfds, fd_set *wfds)
{
  if (c->count < CHANNEL_BUFSIZ)
    FD_SET (c->fdr, rfds);
  if (c->count > 0)
    FD_SET (c->fdw, wfds);
}

static void channel_read (struct channel *c, fd_set *rfds)
{
  int n;

  if (FD_ISSET (c->fdr, rfds))
    {
      n = read (c->fdr, c->buf + c->count, CHANNEL_BUFSIZ - c->count);
      if (n == -1)
        {
          syslog (LLEV, "channel_read: %m");
          exit (1);
        }
      if (n == 0)
        c->eof = 1;
      c->count += n;
    }
}

static void channel_write (struct channel *c, fd_set *wfds)
{
  int n;

  if (FD_ISSET (c->fdw, wfds))
    {
      n = write (c->fdw, c->buf, c->count);
      if (n == -1)
        {
          syslog (LLEV, "channel_write: %m");
          exit (1);
        }
      if (n < c->count)
        memmove (c->buf, c->buf + n, c->count - n);
      c->count -= n;
    }
}


static void forward (int fd1, int fd2)
{
  struct channel c1, c2;
  fd_set rfds, wfds;
  struct timeval tv;
  int n, maxfd;

  maxfd = fd1 > fd2 ? fd1 : fd2;
  if (maxfd >= FD_SETSIZE)
    {
      syslog (LLEV, "FD_SETSIZE exceeded");
      exit (1);
    }
  channel_init (&c1, fd1, fd2);
  channel_init (&c2, fd2, fd1);
  while (!c1.eof && !c2.eof)
    {
      FD_ZERO (&rfds); FD_ZERO (&wfds);
      channel_fdset (&c1, &rfds, &wfds);
      channel_fdset (&c2, &rfds, &wfds);
      tv.tv_sec = PROXY_TIMEOUT;
      tv.tv_usec = 0;
      n = select (maxfd + 1, &rfds, &wfds, (fd_set *)0, &tv);
      if (n == 0)
        {
          syslog (LLEV, "timeout");
          exit (1);
        }
      if (n < 0)
        {
          syslog (LLEV, "select: %m");
          exit (1);
        }
      channel_read (&c1, &rfds);
      channel_read (&c2, &rfds);
      channel_write (&c1, &wfds);
      channel_write (&c2, &wfds);
    }
}


static void user_ok (void)
{
  int sfd, port = 110;

  if ((sfd = conn_server (server, port, 0, (char *)0)) < 0)
    {
      syslog (LLEV, "cannot connect to server %.512s/%d: %m", server, port);
      exit (1);
    }
  syslog (LLEV, "connect host=%.512s/%.20s destination=%.512s/%d",
          rladdr, riaddr, server, port);
  ts = tn_init (sfd, 0);
  if (ts == NULL)
    exit (1);

  tn_gets (ts, input, sizeof (input), PROXY_TIMEOUT, 0);
  if (input[0] != '+')
    {
      tn_printf (tc, "%.500s\r\n", input);
      exit (1);
    }

  command ("USER", user);
  command ("PASS", pass);
  ok ();
  forward (0, sfd);
}


static void do_quit (void)
{
  ok ();
  exit (0);
}


static void do_user (const char *s)
{
  while (*s == ' ') ++s;
  if (*s == 0 || strlen (s) >= sizeof (user))
    {
      tn_puts (tc, "-ERR syntax error\r\n");
      return;
    }
  strcpy (user, s);
  ok ();
}


static void do_pass (const char *s)
{
  Cfg *cf;
  int i;

  if (user[0] == 0)
    {
      tn_puts (tc, "-ERR need USER\r\n");
      return;
    }
  while (*s == ' ') ++s;
  if (*s == 0 || strlen (s) >= sizeof (pass))
    {
      tn_puts (tc, "-ERR syntax error\r\n");
      return;
    }
  strcpy (pass, s);

  if ((cf = cfg_get ("user", confp)) != (Cfg *)0)
    {
      for (i = 0; i < cf->argc; ++i)
        if (strcmp (user, cf->argv[i]) == 0)
          {
            user_ok ();
            exit (0);
          }
    }
  tn_puts (tc, "-ERR authorization failed\r\n");
  exit (1);
}


static void config_pop_gw (Cfg *confp)
{
  Cfg *cf;

  if ((cf = cfg_get ("server", confp)) == (Cfg *)0)
    {
      syslog (LLEV, "fwtkcfgerr: `server' required");
      exit (1);
    }
  if (cf->argc != 1)
    {
      syslog (LLEV, "fwtkcfgerr: server must have one parameter, line %d",
              cf->ln);
      exit (1);
    }
  server = cf->argv[0];
}


int main (int argc, char *argv[])
{
  openlog ("pop-gw", LOG_PID|LOG_NDELAY, LFAC);

  if (argc >= 3 && strcmp (argv[1], "-daemon") == 0)
    {
      int port = str_to_port (argv[2]);
      argc -= 2; argv += 2;
      syslog (LLEV, "Starting daemon mode on port %d", port);
      do_daemon (port);
    }

  if ((confp = cfg_read ("pop-gw")) == (Cfg *)-1)
    exit (1);
  config_groupid (confp, 1);
  config_userid (confp, 1);
  config_pop_gw (confp);

  if (peername (0, rladdr, riaddr, sizeof (riaddr)) != 0)
    {
      syslog (LLEV, "fwtksyserr: cannot get peer name: %m");
      exit (1);
    }

  if (config_hosts (confp, rladdr, riaddr) == (Cfg *)0)
    exit (1);

  tc = tn_init (0, 0);
  ok ();

  user[0] = 0;
  for (;;)
    {
      tn_gets (tc, input, sizeof (input), PROXY_TIMEOUT, 0);
      if (strncasecmp (input, "quit", 4) == 0 && input[4] == 0)
        do_quit ();
      else if (strncasecmp (input, "user", 4) == 0 && input[4] == ' ')
        do_user (input+4);
      else if (strncasecmp (input, "pass", 4) == 0 && input[4] == ' ')
        do_pass (input+4);
      else
        tn_puts (tc, "-ERR authorization first\r\n");
    }
}
