/*
 * Copyright (c) 1993,1994
 *	Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Texas A&M University
 *	and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *             David K. Hess, Douglas Lee Schales, David R. Safford
 */
#include "db.h"

int startDebug = 0;
int stopDebug = 0;
unsigned char debugStrings[MAX_NUM_DEBUG_STRINGS][DEBUG_STRING_LENGTH];

// The two card handles that are passed back.
static CardHandle insideHandle;
static CardHandle outsideHandle;

unsigned short frameSize = 0;
unsigned short headerSize = 0;
int mediaType = 0;

// Sequence number counter.
unsigned long packetSeq = 0;

// The IP packet queue.
Queue protocolQueue;

CardHandle *cardHandleLookup[MAX_NUM_CARDS];

ProtocolManager protocolManager;
unsigned short protocolManagerDS;

CommonCharacteristics common;
ProtocolCharacteristics protChars;
ProtocolLowerDispatch lowerTable;

FailingModules failingModules;

BindingsList bindings;
	
static unsigned char *freeStacks[STACK_POOL_SIZE];
static int freeStackPtr = STACK_POOL_SIZE - 1;

// This routine is called from both threads.
void freeStack(unsigned char *aStack)
{
	GUARD

	if (freeStackPtr == STACK_POOL_SIZE - 1) {
		PERROR("tried to free too many stacks")
	}

	freeStacks[++freeStackPtr] = aStack;

	if (freeStackPtr == 0) 
		sprintf(GET_DEBUG_STRING,"freeStackPtr went positive\n",freeStackPtr);

	UNGUARD
}

// This routine is called from both threads.
unsigned char *allocStack(void)
{
	unsigned char *aStack;

	GUARD

	if (freeStackPtr < 0) {
		// Ran out of stack buffers. Return NULL which will start dropping packets.
		sprintf(GET_DEBUG_STRING,"freeStackPtr went negative\n",freeStackPtr);
		aStack = 0;
	}
	else
		aStack = freeStacks[freeStackPtr--];

	UNGUARD

	//sprintf(GET_DEBUG_STRING,"######### freePktBufPtr = %d\n",freePktBufPtr);

	return aStack;
}

unsigned char *hardware_ntoa(unsigned char *buffer,HardwareAddress *addr)
{
	sprintf(buffer,"%02X:%02X:%02X:%02X:%02X:%02X",
		addr->bytes[0],
		addr->bytes[1],
		addr->bytes[2],
		addr->bytes[3],
		addr->bytes[4],
		addr->bytes[5]);

	return buffer;
}

HardwareAddress *hardware_aton(unsigned char *buffer,HardwareAddress *addr)
{
	int t0;
	int t1;
	int t2;
	int t3;
	int t4;
	int t5;

	sscanf(buffer,"%x:%x:%x:%x:%x:%x",&t0,&t1,&t2,&t3,&t4,&t5);

	addr->bytes[0] = t0;
	addr->bytes[1] = t1;
	addr->bytes[2] = t2;
	addr->bytes[3] = t3;
	addr->bytes[4] = t4;
	addr->bytes[5] = t5;

	return addr;
}

unsigned short __pascal systemRequest(unsigned long param1,unsigned long param2,unsigned short param3,
						unsigned short opcode,unsigned short targetDS)
{
	static int bindEntry = 0;
	CommonCharacteristics *macCommon;
	volatile unsigned short result;

	// Handle the request.
	switch (opcode) {
		case NDIS_SYS_REQ_INITIATE_BIND:
			macCommon = (CommonCharacteristics *) param2;

			// For some reason the Protocol Manager is telling us we can't bind.
			if (macCommon == NULL) {
				fprintf(stderr,"There is an NDIS misconfiguration.\n");
				result = NDIS_ERROR_GENERAL_FAILURE;
				break;
			}

			//fprintf(stderr,"module name %s\n",macCommon->moduleName);
			//fprintf(stderr,"module type %s\n",((MacCharacteristics *) macCommon->serviceCharacteristics)->macName);

			// Binding to the MAC.
			result = macCommon->systemRequest((unsigned long) &common,(unsigned long) &macCommon,
				0,NDIS_SYS_REQ_BIND,macCommon->moduleDS);

			//macCommon = ((MacUpperDispatch *) macCommon->upperDispatchTable)->backPointer;
			//fprintf(stderr,"module name %s\n",macCommon->moduleName);

			if (strcmp(bindings.moduleName[bindEntry],insideHandle.moduleName) == 0) 
				insideHandle.common = macCommon;
			else if (strcmp(bindings.moduleName[bindEntry],outsideHandle.moduleName) == 0) 
				outsideHandle.common = macCommon;
			else
				PERROR("unknown module")

			++bindEntry;

			break;
		case NDIS_SYS_REQ_INITIATE_UNBIND:

			// XXXXX

			//fprintf(stderr,"UNBIND request\n");

			macCommon = (CommonCharacteristics *) param2;

			result = macCommon->systemRequest((unsigned long) &common,0,
				0,NDIS_SYS_REQ_UNBIND,macCommon->moduleDS);

			break;
		default:
			result = NDIS_ERROR_GENERAL_FAILURE;
			break;
	}

	return result;
}

unsigned short __pascal requestConfirm(unsigned short protId,unsigned short macId,unsigned short reqHandle,
						unsigned short status,unsigned short request,unsigned short protDS)
{
	return NDIS_ERROR_SUCCESS;
}

unsigned short __pascal transmitConfirm(unsigned short protId,unsigned short macId,
						unsigned short reqHandle,unsigned short status,unsigned short protDS)
{
	PktBuf *pktBuf;

	//cputs("transmitConfirm called\n\r");

	//sprintf(GET_DEBUG_STRING,"packet send completed\n");

	GUARD

	// Decrement the send count for the indicated card.
	--cardHandleLookup[macId]->sendsPending;

	UNGUARD

	pktBuf = pktBufFromHandle(reqHandle);

	// Add passed ECB back to the free list.
	freePktBuf(pktBuf);

	//cputs("returning from transmitConfirm\n\r");

	return NDIS_ERROR_SUCCESS;
}


unsigned short __pascal receiveLookahead(unsigned short macId,unsigned short frameSize,
						unsigned short bytesAvail,unsigned char *buffer,
						unsigned char *indicate,unsigned short protDS)
{
	int result;
	PktBuf *localPktBuf;
	CardHandle *card;
	unsigned short bytesCopied;
	TDBufDescr tDBufDescr;

	//cputs("receiveLookahead called\n\r");

	//sprintf(GET_DEBUG_STRING,"lookahead length = %d\n",bytesAvail);
	//sprintf(GET_DEBUG_STRING,"ecb = %08lX\n",*ecb);
	//sprintf(GET_DEBUG_STRING,"count = %08lX\n",count);
	//sprintf(GET_DEBUG_STRING,"offset = %08lX\n",offset);
	//sprintf(GET_DEBUG_STRING,"timesAllowed = %d\n",timesAllowed);

	//sprintf(GET_DEBUG_STRING,"looking at a packet size = %d protocolId = %02X:%02X:%02X:%02X:%02X:%02X\n",
	//	look->dataLookAheadLen,
	//	look->protocolId[0],
	//	look->protocolId[1],
	//	look->protocolId[2],
	//	look->protocolId[3],
	//	look->protocolId[4],
	//	look->protocolId[5]);

	// Check the packet against the bridge table to check and see if we even need to look at it.
	//   Note that if the protocol stack wants it (ecb != NULL) then we still have put the source
	//   address into the bridge table. This will enable us to reply to whatever request this is
	//   even if we don't actually bridge the packet.
	result = bridge(macId,buffer);

	//sprintf(GET_DEBUG_STRING,"bridging said %d\n",result);

	if (result == NO) {

		// Otherwise tell the driver I don't want the packet.
		return NDIS_ERROR_FRAME_REJECTED;
	}

	// If bridging it then allocate an packet buffer and drop a sequence number in it.
	localPktBuf = allocPktBuf();

	//fprintf(stderr,"got a packet buffer\n");

	// If we ran out of ECB's then drop the packet.
	if (localPktBuf == NULL) {

		++theStats.droppedPackets;

		//sprintf(GET_DEBUG_STRING,"Dropping a packet\n");
		return NDIS_ERROR_FRAME_REJECTED;
	}

	card = cardHandleLookup[macId];

	// Now kludge things. Note we will have to undo this later. This will make the packet
	//    contiguous after the MLID has done the requested copy.

	// Stamp it with a sequence number. Note that this sequence number may rollover. Note that
	//   if it rolls we really ought to renumber the current packets in the queue. 
	GUARD

	if (packetSeq == 0xFFFFFFFFUL) {
		// Whoops. We rolled. Go through and renumber all of the packets in the
		//   queues.
		renumberPacketQueues();
	}

	localPktBuf->sequence = ++packetSeq;

	UNGUARD

	tDBufDescr.tDDataCount = 1;
	tDBufDescr.tDBufDescrRec[0].tDPtrType = NDIS_PTR_PHYSICAL;
	tDBufDescr.tDBufDescrRec[0].tDDataPtr = localPktBuf->buffer;
	tDBufDescr.tDBufDescrRec[0].tDDataLen = localPktBuf->length;
	tDBufDescr.tDBufDescrRec[0].dummy = 0;

	result = MAC_DISPATCH(card)->transferData(&bytesCopied,0,&tDBufDescr,card->common->moduleDS);

	localPktBuf->packetLength = bytesCopied;

	if (result == NDIS_ERROR_SUCCESS) {
		enqueuePktBuf(&card->queue,localPktBuf);
	}
	else {
		// The receive failed for some reason. Free up the packet buffer.
		freePktBuf(localPktBuf);
	}

	//sprintf(GET_DEBUG_STRING,"returning pktBuf = %08lX\n",pktBuf);

	return NDIS_ERROR_SUCCESS;
}

unsigned short __pascal indicationComplete(unsigned short macId,unsigned short protDS)
{
	// We don't give a hoot about these. Just return.
	return NDIS_ERROR_SUCCESS;
}

// Ok, this is the OTHER way we may receive packets.
unsigned short __pascal receiveChain(unsigned short macId,unsigned short frameSize,unsigned short reqHandle,
						RxBufDescr *rxBufDescr,unsigned char *indicate,unsigned short protDS)
{
	int i;
	int result;
	CardHandle *card;
	RxBufDescrRec *currRxBufDescrRec;
	PktBuf *localPktBuf;

	// For now we copy the entire packet over to a PktBuf structure. This may be a performance hit
	//   but this routine probably isn't called very much, and it is a lot of work to do it otherwise.
	//   Also if it is a filter protocol packet we could end up sucking up MAC buffes.

	// Check the packet against the bridge table to check and see if we even need to look at it.
	//   Note that if the protocol stack wants it (ecb != NULL) then we still have put the source
	//   address into the bridge table. This will enable us to reply to whatever request this is
	//   even if we don't actually bridge the packet.
	result = bridge(macId,rxBufDescr->rxBufDescrRec[0].rxDataPtr);

	//sprintf(GET_DEBUG_STRING,"receive chain!\n");

	if (result == NO) {

		// Otherwise tell the driver I don't want the packet.
		return NDIS_ERROR_FRAME_REJECTED;
	}

	// If bridging it then allocate an packet buffer and drop a sequence number in it.
	localPktBuf = allocPktBuf();

	//fprintf(stderr,"got a packet buffer\n");

	// If we ran out of ECB's then drop the packet.
	if (localPktBuf == NULL) {
		++theStats.droppedPackets;

		//sprintf(GET_DEBUG_STRING,"Dropping a packet\n");
		return NDIS_ERROR_FRAME_REJECTED;
	}

	card = cardHandleLookup[macId];

	// Stamp it with a sequence number. Note that this sequence number may rollover. Note that
	//   if it rolls we really ought to renumber the current packets in the queue. Need to test
	//   and see if this is really needed.

	GUARD

	if (packetSeq == 0xFFFFFFFFUL) {
		// Whoops. We rolled. Go through and renumber all of the packets in the
		//   queues.
		renumberPacketQueues();
	}

	localPktBuf->sequence = ++packetSeq;

	UNGUARD

	localPktBuf->packetLength = 0;

	// Ok, copy the packet to the buffer.
	for (i = 0;i < rxBufDescr->rxDataCount;++i) {
		currRxBufDescrRec = &rxBufDescr->rxBufDescrRec[i];
		fastCopy(localPktBuf->buffer + localPktBuf->packetLength,
			currRxBufDescrRec->rxDataPtr,
			currRxBufDescrRec->rxDataLen);
		localPktBuf->packetLength += currRxBufDescrRec->rxDataLen;
	}

	enqueuePktBuf(&card->queue,localPktBuf);

	//sprintf(GET_DEBUG_STRING,"returning pktBuf = %08lX\n",pktBuf);

	// This frees up the buffer for the MAC to use.
	return NDIS_ERROR_SUCCESS;
}

unsigned short __pascal status(unsigned short macId,unsigned short param1,unsigned char *indicate,
						unsigned short opcode,unsigned short protDS)
{
	switch (opcode) {
		case NDIS_STATUS_RING_STATUS:
			break;
		case NDIS_STATUS_ADAPTER_CHECK:
			break;
		case NDIS_STATUS_START_RESET:
			break;
		case NDIS_STATUS_INTERRUPT:
			break;
		case NDIS_STATUS_END_RESET:
			break;
		default:
			break;
	}

	// We don't need to do anything about this stuff yet.
	return NDIS_ERROR_SUCCESS;
}

void scanQueue(Queue *queue,unsigned long *lowestSeq,unsigned long *highestSeq,unsigned long adjust)
{
	PktBuf *currPktBuf;

	// Scan for the lowest and highest sequence number in the queued packets.
	currPktBuf = queue->head;
	while (currPktBuf) {

		if (lowestSeq && currPktBuf->sequence < *lowestSeq)
				*lowestSeq = currPktBuf->sequence;

		if (highestSeq && currPktBuf->sequence > *highestSeq)
				*highestSeq = currPktBuf->sequence;

		if (adjust)
			currPktBuf->sequence -= adjust;

		currPktBuf = currPktBuf->nextLink;
	}
}

// This routine will get called once in a very blue moon.
void renumberPacketQueues(void)
{
	unsigned long lowestSeq;
	unsigned long highestSeq;

	//sprintf(GET_DEBUG_STRING,"renumbering the packet queues\n");

	// Handle the easy case.
	if (insideHandle.queue.head == NULL &&
		outsideHandle.queue.head == NULL) {
		packetSeq = 0;
		return;
	}

	lowestSeq = 0xFFFFFFFFUL;
	highestSeq = 0;

	// Scan for the lowest and highest sequence number in the queued packets.
	//  Note that theoretically, we don't need to do this since the packets
	//  *should* be in sequence order on the queues, however we don't trust
	//  the order that the drivers may do things.
	scanQueue(&insideHandle.queue,&lowestSeq,&highestSeq,0);
	scanQueue(&outsideHandle.queue,&lowestSeq,&highestSeq,0);

	//sprintf(GET_DEBUG_STRING,"highest = %lu lowest = %lu\n",highestSeq,lowestSeq);

	// Now go through and decrement them all by lowest.
	scanQueue(&insideHandle.queue,NULL,NULL,lowestSeq);
	scanQueue(&outsideHandle.queue,NULL,NULL,lowestSeq);

	// Set the sequence number to be the highest - the lowest + 1.
	packetSeq = highestSeq - lowestSeq + 1;

	//sprintf(GET_DEBUG_STRING,"new packet sequence = %lu\n",packetSeq);
}

// Tell the NDIS driver to start the delivery of the packet.
void sendPacket(PktBuf *pktBuf,int macId)
{
	int i;
	int resultCode;
	CardHandle *card;
	TxBufDescr txBufDescr;

	card = cardHandleLookup[macId];

	GUARD

	// Increment the send count for the indicated card.
	++card->sendsPending;

	UNGUARD

	//fprintf(stderr,"calling sendPacket\n");
	//fprintf(stderr,"sending to board number %d\n",macId);

	txBufDescr.txImmedLen = 0;
	txBufDescr.txImmedPtr = NULL;
	txBufDescr.txDataCount = 1;
	txBufDescr.txBufDescrRec[0].txPtrType = NDIS_PTR_PHYSICAL;
	txBufDescr.txBufDescrRec[0].dummy     = 0;
	txBufDescr.txBufDescrRec[0].txDataLen = pktBuf->packetLength;
	txBufDescr.txBufDescrRec[0].txDataPtr = pktBuf->buffer;

	// Evenize the packet. This throws on an extra byte of junk. I just love PCs.
	//if (pktBuf->packetLength & 0x01)
	//	++txBufDescr.txBufDescrRec[0].txDataLen;

	resultCode = MAC_DISPATCH(card)->transmitChain(common.moduleId,pktBuf->handle,&txBufDescr,card->common->moduleDS);

	switch (resultCode) {
		case NDIS_ERROR_OUT_OF_RESOURCE: // Note that this should not happen but if it does there is not
						 //   much we can do about it.
			fprintf(stderr,"ERROR: transmit queue overflowed\n");
		case NDIS_ERROR_SUCCESS:
			//fprintf(stderr,"synchronous send\n");
			// Everything was hunky dory and synchronous. Free up the packet buffer.
			GUARD

			--card->sendsPending;

			UNGUARD

			freePktBuf(pktBuf);
			break;
		case NDIS_ERROR_REQUEST_QUEUED:
			// Everything was hunky dory and asynchronous. Do nothing.
			//fprintf(stderr,"asynchronous send\n");
			break;
		default:
			fprintf(stderr,"The transmission of the packet failed. (code = %04X)\n",resultCode);
			//fprintf(stderr,"packet length = %d\n",pktBuf->packetLength);
			//fprintf(stderr,"first 20 bytes of packet:\n");
			//for (i = 0;i < 20;++i) {
			//	fprintf(stderr," %02X",pktBuf->buffer[i]);
			//}
			//fprintf(stderr,"\n");
			exit(1);
	}

	//fprintf(stderr,"returning from sendPacket\n");
}


void printNdisError(unsigned short errorCode)
{
	switch (errorCode) {
		case NDIS_ERROR_SUCCESS:
			fprintf(stderr,"The function completed successfully.\n");
			break;
		case NDIS_ERROR_WAIT_FOR_RELEASE:
			fprintf(stderr,"The ReceiveChain completed successfully but the protocol has\n"
				       "retained control of the buffer.\n");
			break;
		case NDIS_ERROR_REQUEST_QUEUED:
			fprintf(stderr,"The current request has been queued.\n");
			break;
		case NDIS_ERROR_FRAME_NOT_RECOGNIZED:
			fprintf(stderr,"Frame not recognized.\n");
			break;
		case NDIS_ERROR_FRAME_REJECTED:
			fprintf(stderr,"Frame was discarded.\n");
			break;
		case NDIS_ERROR_FORWARD_FRAME:
			fprintf(stderr,"Protocol wishes to forward frame to another protocol.\n");
			break;
		case NDIS_ERROR_OUT_OF_RESOURCE:
			fprintf(stderr,"Out of resource.\n");
			break;
		case NDIS_ERROR_INVALID_PARAMETER:
			fprintf(stderr,"Invalid parameter.\n");
			break;
		case NDIS_ERROR_INVALID_FUNCTION:
			fprintf(stderr,"Invalid function.\n");
			break;
		case NDIS_ERROR_NOT_SUPPORTED:
			fprintf(stderr,"Not supported.\n");
			break;
		case NDIS_ERROR_HARDWARE_ERROR:
			fprintf(stderr,"Hardware error.\n");
			break;
		case NDIS_ERROR_TRANSMIT_ERROR:
			fprintf(stderr,"The packet was not transmitted due to an error.\n");
			break;
		case NDIS_ERROR_NO_SUCH_DESTINATION:
			fprintf(stderr,"Token ring packet was not recognized when transmitted.\n");
			break;
		case NDIS_ERROR_BUFFER_TOO_SMALL:
			fprintf(stderr,"Provided buffer was too small.\n");
			break;
		case NDIS_ERROR_ALREADY_STARTED:
			fprintf(stderr,"Network drivers already started.\n");
			break;
		case NDIS_ERROR_INCOMPLETE_BINDING:
			fprintf(stderr,"Protocol driver could not complete its bindings.\n");
			break;
		case NDIS_ERROR_DRIVER_NOT_INITIALIZED:
			fprintf(stderr,"MAC did not initialize properly.\n");
			break;
		case NDIS_ERROR_HARDWARE_NOT_FOUND:
			fprintf(stderr,"Hardware not found.\n");
			break;
		case NDIS_ERROR_HARDWARE_FAILURE:
			fprintf(stderr,"Hardware failure.\n");
			break;
		case NDIS_ERROR_CONFIGURATION_FAILURE:
			fprintf(stderr,"Configuration failure.\n");
			break;
		case NDIS_ERROR_INTERRUPT_CONFLICT:
			fprintf(stderr,"Interrupt conflict.\n");
			break;
		case NDIS_ERROR_INCOMPATIBLE_MAC:
			fprintf(stderr,"The MAC is not compatible with the protocol.\n");
			break;
		case NDIS_ERROR_INITIALIZATION_FAILED:
			fprintf(stderr,"Initialization failed.\n");
			break;
		case NDIS_ERROR_NO_BINDING:
			fprintf(stderr,"Binding did not occur.\n");
			break;
		case NDIS_ERROR_NETWORK_MAY_NOT_BE_CONNECTED:
			fprintf(stderr,"The network may not be connected to the adapter.\n");
			break;
		case NDIS_ERROR_INCOMPATIBLE_OS_VERSION:
			fprintf(stderr,"The version of the operating system is incompatible with the protocol.\n");
			break;
		case NDIS_ERROR_ALREADY_REGISTERED:
			fprintf(stderr,"The protocol is already registered.\n");
			break;
		case NDIS_ERROR_PATH_NOT_FOUND:
			fprintf(stderr,"PROTMAN.EXE could not be found.\n");
			break;
		case NDIS_ERROR_INSUFFICIENT_MEMORY:
			fprintf(stderr,"Insufficient memory.\n");
			break;
		case NDIS_ERROR_INFO_NOT_FOUND:
			fprintf(stderr,"Protocol Mananger info structure is lost or corrupted.\n");
			break;
		case NDIS_ERROR_GENERAL_FAILURE:
			fprintf(stderr,"General failure.\n");
			break;
		default:
			fprintf(stderr,"Unknown NDIS error.\n");
			break;
	}
}

void openNdis(void)
{
	ReqBlock reqBlock;
	int ndisFd;
	int result;

	ndisFd = open(NDIS_PATH,O_RDONLY);

	if (ndisFd == -1) {
		fprintf(stderr,"Could not open NDIS Protocol Manager device.\n");
		fprintf(stderr,"Is PROTMAN.DOS in your CONFIG.SYS?\n");
		exit(1);
	}

	//fprintf(stderr,"handle = %d\n",ndisFd);

	memset(&reqBlock,0x00,sizeof(ReqBlock));

	reqBlock.opcode = NDIS_PROT_MAN_GET_PROTOCOL_MANAGER_LINKAGE;

	//fprintf(stderr,"size = %d\n",sizeof(ReqBlock));

	result = ioctl(ndisFd,2,&reqBlock,sizeof(ReqBlock));

	//fprintf(stderr,"result = %d\n",result);

	if (result == -1) {
		fprintf(stderr,"Could not pass IOCTL request to Protocol Manager.\n");
		exit(1);
	}

	(void) close(ndisFd);

	//fprintf(stderr,"Entry Point = %08lX\n",reqBlock.pointer1);
	//fprintf(stderr,"Protocol DS = %04X\n",reqBlock.word1);
	
	protocolManager = (ProtocolManager) reqBlock.pointer1;
	protocolManagerDS = reqBlock.word1;
}

// manager <ip address>[,<ip address>]
// ipaddress <ip address>
// subnetmask <ip address>
// gateway <ip address>
// port <udp port number>
// listen (inside|outside|both)
void parseNdisConfig(void)
{
	ReqBlock reqBlock;
	int result;
	ModuleConfig *currModule;
	KeyWord *currKeyWord;
	Param *param;
	char tempBuffer[32];
	int i;

	insideHandle.moduleName[0] = '\0';
	outsideHandle.moduleName[0] = '\0';

	// Get a pointer to the configuration image.
	memset(&reqBlock,0,sizeof(ReqBlock));
	reqBlock.opcode = NDIS_PROT_MAN_GET_PROTOCOL_MANAGER_INFO;
	result = protocolManager(&reqBlock,protocolManagerDS);

	if (result != 0) {
		fprintf(stderr,"Could not get PROTOCOL.INI image from Protocol Manager.\n");
		printNdisError(result);
		exit(1);
	}

	//fprintf(stderr,"result = %d\n",result);
	//fprintf(stderr,"status = %d\n",reqBlock.status);

	currModule = (ModuleConfig *) reqBlock.pointer1;

	//fprintf(stderr,"opening file\n");

	// Set up default values.
	filterConfig.internalMacSet          = NO;
	filterConfig.numBuffers              = BUFFER_POOL_SIZE;
	filterConfig.discardOther            = NO;
	filterConfig.discardSuspectOffset    = NO;
	filterConfig.discardOtherIp          = NO;
	filterConfig.logMask                 = DEFAULT_SYSL_MASK;

	while (currModule) {

		if (stricmp(currModule->moduleName,"drawbridge") == 0) {

			//fprintf(stderr,"found drawbridge line\n");

			currKeyWord = &currModule->keyWord;

			// Hunt for the configuration lines.
			while (currKeyWord) {

				if (stricmp("outside",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Outside = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING || param->paramLen > 16) {
						fprintf(stderr,"Invalid Outside = syntax.\n");
						exit(1);
					}

					// Paranoia
					strncpy(outsideHandle.moduleName,param->value.string,16);
					outsideHandle.moduleName[15] = '\0';
				}
				else if (stricmp("inside",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Inside = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING || param->paramLen > 16) {
						fprintf(stderr,"Invalid Inside = syntax.\n");
						exit(1);
					}

					// Paranoia
					strncpy(insideHandle.moduleName,param->value.string,16);
					insideHandle.moduleName[15] = '\0';
				}
				else if (stricmp("manager",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams == 0) {
						fprintf(stderr,"Invalid Manager = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					for (i = 0;i < currKeyWord->numParams;++i) {
						if (param->paramType != NDIS_PARAM_STRING) {
							fprintf(stderr,"Invalid Manager = syntax.\n");
							exit(1);
						}

						if (filterConfig.numManagers == MAX_NUM_MANAGERS) {
							fprintf(stderr,"Only %d managers are allowed.\n",MAX_NUM_MANAGERS);
							exit(1);
						}

						(void) inet_aton(param->value.string,
							&filterConfig.managers[filterConfig.numManagers]);
						(void) inet_ntoa(tempBuffer,&filterConfig.managers[filterConfig.numManagers]);

						if (!IN_CLASSA(filterConfig.managers[filterConfig.numManagers].S_addr) &&
						    !IN_CLASSB(filterConfig.managers[filterConfig.numManagers].S_addr) &&
						    !IN_CLASSC(filterConfig.managers[filterConfig.numManagers].S_addr)) {
							fprintf(stderr,"Invalid manager IP address.\n");
							exit(1);
						}

						fprintf(stdout,"Added Manager: %s\n",tempBuffer);

						++filterConfig.numManagers;
						
						// This is the grossest thing I have ever seen.
						param = (Param *) &param->value.string[param->paramLen];
					}
				}
				else if (stricmp("ipaddress",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Ipaddress = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid Ipaddress = syntax.\n");
						exit(1);
					}

					(void) inet_aton(param->value.string,&filterConfig.myIpAddr);
					(void) inet_ntoa(tempBuffer,&filterConfig.myIpAddr);

					if (!IN_CLASSA(filterConfig.myIpAddr.S_addr) &&
					    !IN_CLASSB(filterConfig.myIpAddr.S_addr) &&
					    !IN_CLASSC(filterConfig.myIpAddr.S_addr)) {
						fprintf(stderr,"Invalid IP address for Filter.\n");
						exit(1);
					}

					fprintf(stdout,"My IP address: %s\n",tempBuffer);
				}
				else if (stricmp("listen",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Listen = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid Listen = syntax.\n");
						exit(1);
					}

					if (stricmp("inside",param->value.string) == 0) {
						filterConfig.listenMode = INSIDE_MASK;
						fprintf(stdout,"Listening on inside only\n");
					}
					else if (stricmp("outside",param->value.string) == 0) {
						filterConfig.listenMode = OUTSIDE_MASK;
						fprintf(stdout,"Listening on outside only\n");
					}
					else if (stricmp("both",param->value.string) == 0) {
						filterConfig.listenMode = INSIDE_MASK | OUTSIDE_MASK;
						fprintf(stdout,"Listening on both sides\n");
					}
					else {
						fprintf(stderr,"Unknown listen mode (%s).\n",param->value.string);
					}
				}
				else if (stricmp("logmask",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid LogMask = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_INTEGER) {
						fprintf(stderr,"Invalid LogMask = syntax.\n");
						exit(1);
					}

					filterConfig.logMask = param->value.numeric;
				}
				else if (stricmp("loghost",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid LogHost = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid LogHost = syntax.\n");
						exit(1);
					}

					(void) inet_aton(param->value.string,&filterConfig.logHost);

					(void) inet_ntoa(tempBuffer,&filterConfig.logHost);
					fprintf(stdout,"Syslog host: %s\n",tempBuffer);
				}
				else if (stricmp("logfacility",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid LogFacility = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid LogFacility = syntax.\n");
						exit(1);
					}

					if (sscanf(param->value.string,"local%d",&filterConfig.logFacility) != 1) {
						fprintf(stderr,"Invalid LogFacility = syntax.\n");
						exit(1);
					}

					if (filterConfig.logFacility < 0 || filterConfig.logFacility > 7) {
						fprintf(stderr,"Invalid value for LogFacility (must be local0 thru local7).\n");
						exit(1);
					}
				}
				else if (stricmp("gateway",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Gateway = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid Gateway = syntax.\n");
						exit(1);
					}

					(void) inet_aton(param->value.string,&filterConfig.myGateway);

					(void) inet_ntoa(tempBuffer,&filterConfig.myGateway);
					fprintf(stdout,"Default IP gateway: %s\n",tempBuffer);
				}
				else if (stricmp("subnetmask",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Subnetmask = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid Subnetmask = syntax.\n");
						exit(1);
					}

					(void) inet_aton(param->value.string,&filterConfig.mySubnetMask);

					(void) inet_ntoa(tempBuffer,&filterConfig.mySubnetMask);
					fprintf(stdout,"Subnet mask: %s\n",tempBuffer);
				}
				else if (stricmp("port",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Port = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_INTEGER) {
						fprintf(stderr,"Invalid Port = syntax.\n");
						exit(1);
					}

					filterConfig.listenPort = param->value.numeric;

					fprintf(stdout,"Management on UDP port: %d\n",filterConfig.listenPort);
				}
				else if (stricmp("internalmac",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid InternalMac = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid InternalMac = syntax.\n");
						exit(1);
					}

					(void) hardware_aton(param->value.string,&filterConfig.internalMac);

					(void) hardware_ntoa(tempBuffer,&filterConfig.internalMac);

					fprintf(stdout,"Internal MAC address: %s\n",tempBuffer);
					filterConfig.internalMacSet = YES;
				}
				else if (stricmp("buffers",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid Buffers = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_INTEGER) {
						fprintf(stderr,"Invalid Buffers = syntax.\n");
						exit(1);
					}

					filterConfig.numBuffers = param->value.numeric;

					fprintf(stdout,"Number of packet buffers: %d\n",filterConfig.numBuffers);
				}
				else if (stricmp("suspectoffset",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid SuspectOffset = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid SuspectOffset = syntax.\n");
						exit(1);
					}

					if (stricmp(param->value.string,"PASS") == 0)
						filterConfig.discardSuspectOffset = NO;
					else if (stricmp(param->value.string,"DISCARD") == 0)
						filterConfig.discardSuspectOffset = YES;
					else {
						fprintf(stderr,"Invalid SuspectOffset = syntax.\n");
						exit(1);
					}
				}
				else if (stricmp("otherip",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid OtherIp = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid OtherIp = syntax.\n");
						exit(1);
					}

					if (stricmp(param->value.string,"PASS") == 0)
						filterConfig.discardOtherIp = NO;
					else if (stricmp(param->value.string,"DISCARD") == 0)
						filterConfig.discardOtherIp = YES;
					else {
						fprintf(stderr,"Invalid OtherIp = syntax.\n");
						exit(1);
					}
				}
				else if (stricmp("nonip",currKeyWord->keyWord) == 0) {
					if (currKeyWord->numParams != 1) {
						fprintf(stderr,"Invalid NonIp = syntax.\n");
						exit(1);
					}

					param = &currKeyWord->param;

					if (param->paramType != NDIS_PARAM_STRING) {
						fprintf(stderr,"Invalid NonIp = syntax.\n");
						exit(1);
					}

					if (stricmp(param->value.string,"PASS") == 0)
						filterConfig.discardOther = NO;
					else if (stricmp(param->value.string,"DISCARD") == 0)
						filterConfig.discardOther = YES;
					else {
						fprintf(stderr,"Invalid NonIp = syntax.\n");
						exit(1);
					}
				}
				else if (stricmp("drivername",currKeyWord->keyWord) == 0) {
					// Skip the drivername parameter.
				}
				else {
					fprintf(stderr,"Unrecognized keyword in protocol.ini file (%s).\n",currKeyWord->keyWord);
					exit(1);
				}

				currKeyWord = currKeyWord->nextKeyword;
			}

			if (insideHandle.moduleName[0] == '\0' || outsideHandle.moduleName[0] == '\0') {
				fprintf(stderr,"Must bind cards to the inside and the outside.\n");
				exit(1);
			}

			//fprintf(stderr,"out of lines or broke\n");

			// Nothing else to read once we have finished with the drawbridge clause.
			break;
		}

		currModule = currModule->nextModule;
	}

	if (insideHandle.moduleName[0] == '\0' || outsideHandle.moduleName[0] == '\0') {
		fprintf(stderr,"Must bind cards to the inside and the outside.\n");
		exit(1);

	}

	if (filterConfig.logHost.S_addr != 0UL) {
		fprintf(stdout,"Syslog Facility: local%d\nSyslog Mask: 0x%08lX\n",
			filterConfig.logFacility,
			filterConfig.logMask);
	}


	if (filterConfig.discardOther == NO) 
		fprintf(stdout,"Will pass MAC layer protocols other than IP/ARP/RARP\n");
	else
		fprintf(stdout,"Will discard MAC layer protocols other than IP/ARP/RARP\n");

#ifdef DENY_MULTICAST
	fprintf(stdout,"Will discard IP multicast\n");
#else
	fprintf(stdout,"Will pass IP multicast\n");
#endif

	if (filterConfig.discardOtherIp == NO)
		fprintf(stdout,"Will pass IP protocols other than TCP/UDP/ICMP\n");
	else
		fprintf(stdout,"Will discard IP protocols other than TCP/UDP/ICMP\n");

	if (filterConfig.discardSuspectOffset == NO)
		fprintf(stdout,"Will pass TCP packets with suspect IP fragment offset\n");
	else
		fprintf(stdout,"Will discard TCP packets with suspect IP fragment offset\n");

	if (filterConfig.listenMode) {
		if (filterConfig.mySubnetMask.S_addr == 0 || 
			filterConfig.myIpAddr.S_addr == 0) {
			fprintf(stderr,"Management is enabled and the IP address and/or subnet mask is not set.\n");
			exit(1);
		}

		if (filterConfig.myGateway.S_addr == 0) {
			fprintf(stdout,"WARNING: gateway IP address not set.\n");
		}
	}
}

void registerNdisAndBind(void)
{
	unsigned short result;
	ReqBlock reqBlock;

	memset(&common,0,sizeof(CommonCharacteristics));

	common.tableSize = sizeof(CommonCharacteristics);

	//fprintf(stderr,"common size = %d\n",common.tableSize);

	common.majorNdisVersion = 2;
	common.minorNdisVersion = 0;

	common.majorModuleVersion = 2;
	common.minorModuleVersion = 0;

	// Indicates binding from below and dynamically loaded.
	common.moduleFlags = 0x00000006L;

	strcpy(common.moduleName,"DRAWBRIDGE");

	common.protocolLevelUpper = 0xFF;
	common.protocolLevelLower = 1;
	common.interfaceLower = 1;

	common.moduleDS = _DS;
	common.systemRequest = (SystemRequest) systemRequestGlue;

	common.serviceCharacteristics = (unsigned char *) &protChars;
	common.serviceStatus = NULL;

	common.upperDispatchTable = NULL;
	common.lowerDispatchTable = (unsigned char *) &lowerTable;

	protChars.length = sizeof(ProtocolCharacteristics);
	protChars.name[0] = 0;
	protChars.type = 0;

	lowerTable.backPointer = &common;

	// Give me ALL the packets.
	lowerTable.flags = 0x00000007;

	lowerTable.requestConfirm     = requestConfirmGlue;
	lowerTable.transmitConfirm    = transmitConfirmGlue;
	lowerTable.receiveLookahead   = receiveLookaheadGlue;
	lowerTable.indicationComplete = indicationCompleteGlue;
	lowerTable.receiveChain       = receiveChainGlue;
	lowerTable.status             = statusGlue;

	/*
	printf("enter the module to bind to\n");

	scanf("%s",bindings.moduleName);

	bindings.numBindings = 1;

	printf("trying module %s\n",bindings.moduleName[0]);
	*/

	bindings.numBindings = 2;
	strcpy(bindings.moduleName[0],insideHandle.moduleName);
	strcpy(bindings.moduleName[1],outsideHandle.moduleName);

	// Register ourselves with NDIS.
	reqBlock.opcode   = NDIS_PROT_MAN_REGISTER_MODULE;
	reqBlock.pointer1 = (unsigned char *) &common;
	reqBlock.pointer2 = (unsigned char *) &bindings;

	result = protocolManager(&reqBlock,protocolManagerDS);

	if (result != 0) {
		fprintf(stderr,"Could not register with Protocol Manager\n");
		printNdisError(result);
		exit(1);
	}

	// Start the binding process.
	reqBlock.opcode   = NDIS_PROT_MAN_BIND_AND_START;
	reqBlock.pointer1 = (unsigned char *) &failingModules;

	result = protocolManager(&reqBlock,protocolManagerDS);

	if (result != 0) {
		fprintf(stderr,"Could not start binding process with Protocol Manager\n");
		printNdisError(result);
		exit(1);
	}

	//exit(1);

	//fprintf(stderr,"binding succeeded\n");
}

void checkMacFeatures(CardHandle *card)
{
	unsigned long serviceFlags;
	unsigned char *mediaString;
	unsigned short macFrameSize;
	HardwareAddress *hardwareAddress;

	//fprintf(stderr,"checking card features\n");

	//fprintf(stderr,"common table address = %08lX\n",card->common);

	//fprintf(stderr,"macId = %d\n",card->common->moduleId);

	serviceFlags = MAC_CHAR(card)->serviceFlags;

	if (!(serviceFlags & NDIS_MAC_SERVICE_FLAGS_PROMISCUOUS)) {
		fprintf(stderr,"The MAC %s does not support promiscuous mode.\n",card->moduleName);
		exit(1);
	}

	mediaString = MAC_CHAR(card)->macName;

	//fprintf(stderr,"media type = %s\n",mediaString);

	// Get the media type. And set the header size.
	if (strcmp(mediaString,"802.3") == 0 ||
	    strcmp(mediaString,"DIX") == 0 ||
	    strcmp(mediaString,"DIX+802.3") == 0) {
	    	if (mediaType == 0)
			mediaType = MEDIA_ETHERNET;
		else if (mediaType != MEDIA_ETHERNET) {
			fprintf(stderr,"Media types for the two cards don't match.\n");
			exit(1);
		}
		headerSize = sizeof(EthernetIIHeader);
	}
	else if (strcmp(mediaString,"802.5") == 0) {
		headerSize = sizeof(Ieee802Dot5Header) + sizeof(Ieee802Dot2SnapHeader);
		fprintf(stderr,"sorry, token ring is not yet supported.\n");
		exit(1);

	    	if (mediaType == 0)
			mediaType = MEDIA_TOKEN;
		else if (mediaType != MEDIA_TOKEN) {
			fprintf(stderr,"Media types for the two cards don't match.\n");
			exit(1);
		}
	}
	else if (strcmp(mediaString,"FDDI") == 0) {
	    	if (mediaType == 0)
			mediaType = MEDIA_FDDI;
		else if (mediaType != MEDIA_FDDI) {
			fprintf(stderr,"Media types for the two cards don't match.\n");
			exit(1);
		}
		headerSize = sizeof(FddiHeader) + sizeof(Ieee802Dot2SnapHeader);
	}
	else {
		fprintf(stderr,"Unsupported MAC type.\n");
		fprintf(stderr,"Supported MAC are Ethernet and FDDI.\n");
		exit(1);
	}

	macFrameSize = MAC_CHAR(card)->maxFrameSize;

	// Get the frame size. It had better be the same for both cards.
	if (frameSize == 0)
		frameSize = macFrameSize;
	else if (frameSize != macFrameSize) {
		fprintf(stderr,"Frame sizes for the two cards don't match.\n");
		exit(1);
	}

	hardwareAddress = (HardwareAddress *) MAC_CHAR(card)->currentAddress;

	fprintf(stdout,"Hardware address: %02X:%02X:%02X:%02X:%02X:%02X\n",
		hardwareAddress->bytes[0],
		hardwareAddress->bytes[1],
		hardwareAddress->bytes[2],
		hardwareAddress->bytes[3],
		hardwareAddress->bytes[4],
		hardwareAddress->bytes[5]);

	// Set up the max pending sends. We reserve 3 queue positions (if we can) for management
	//   packets.
	card->maxSends = MAC_CHAR(card)->txQueueDepth;

	//fprintf(stderr,"max transmits = %d\n",card->maxSends);
}

void startMac(CardHandle *card)
{
	unsigned short result;

	//fprintf(stderr,"Starting card %s\n",card->moduleName);

	// Set the lookahead length.
	result = MAC_DISPATCH(card)->request(common.moduleId,
		0,
		headerSize,
		0,
		NDIS_GENERAL_REQUEST_SET_LOOKAHEAD,
		card->common->moduleDS);

	// Well we assume that if we got INVALID PARAMETER then either this is not 
	//   supported or will work anyway. NE2000 does this.
	if (result != NDIS_ERROR_SUCCESS && result != NDIS_ERROR_INVALID_PARAMETER) {
		fprintf(stderr,"Set lookahead failed.\n");
		printNdisError(result);
		exit(1);
	}

	// Set the packet filter. Note that for some medias and drivers we *must* specify
	//   all three flags or the card(s) will not operate correctly.
	result = MAC_DISPATCH(card)->request(common.moduleId,
		0,
		NDIS_PACKET_FILTER_PROMISCUOUS |
			NDIS_PACKET_FILTER_DIRECTED |
			NDIS_PACKET_FILTER_BROADCAST,
		0,
		NDIS_GENERAL_REQUEST_SET_PACKET_FILTER,
		card->common->moduleDS);

	if (result != NDIS_ERROR_SUCCESS) {
		fprintf(stderr,"Set packet filter failed.\n");
		printNdisError(result);
		exit(1);
	}

	// If OPEN/CLOSE supported then open the adapter.
	if (MAC_CHAR(card)->serviceFlags & NDIS_MAC_SERVICE_FLAGS_OPEN_CLOSE) {
		//fprintf(stderr,"card supports open\n");

		result = MAC_DISPATCH(card)->request(common.moduleId,
			0,
			0,
			NULL,
			NDIS_GENERAL_REQUEST_OPEN_ADAPTER,
			card->common->moduleDS);

		if (result != NDIS_ERROR_SUCCESS) {
			fprintf(stderr,"Opening the MAC failed.\n");
			printNdisError(result);
			exit(1);
		}
	}
}

void shutdownNdis(void)
{
	ReqBlock reqBlock;
	int resultCode;
	int i;

	fprintf(stdout,"\nShutting down...");

	// Release the extended memory allocated for the networks.
	for (i = 0; i < MAX_NUM_NETWORKS; ++i)
		if (addrTable[i].network.S_addr)
			xmsFreeMem(addrTable[i].hostTable);

	// If the adapters support open and are open then close them.
	if (MAC_CHAR(&insideHandle)->serviceFlags & NDIS_MAC_SERVICE_FLAGS_OPEN_CLOSE &&
		MAC_STATUS(&insideHandle)->macStatus & NDIS_MAC_STATUS_MAC_OPEN) {

		// XXXXX
		//fprintf(stderr,"closing the inside MAC\n");

		resultCode = MAC_DISPATCH(&insideHandle)->request(common.moduleId,
			0,
			0,
			0,
			NDIS_GENERAL_REQUEST_CLOSE_ADAPTER,
			insideHandle.common->moduleDS);

		if (resultCode != NDIS_ERROR_SUCCESS) {
			fprintf(stderr,"\nClosing the MAC failed.\n");
			printNdisError(resultCode);
			exit(1);
		}
	}

	// If the adapters support open and are open then close them.
	if (MAC_CHAR(&outsideHandle)->serviceFlags & NDIS_MAC_SERVICE_FLAGS_OPEN_CLOSE &&
		MAC_STATUS(&outsideHandle)->macStatus & NDIS_MAC_STATUS_MAC_OPEN) {

		// XXXXX
		//fprintf(stderr,"closing the outside MAC\n");

		resultCode = MAC_DISPATCH(&outsideHandle)->request(common.moduleId,
			0,
			0,
			0,
			NDIS_GENERAL_REQUEST_CLOSE_ADAPTER,
			outsideHandle.common->moduleDS);

		if (resultCode != NDIS_ERROR_SUCCESS) {
			fprintf(stderr,"\nClosing the MAC failed.\n");
			printNdisError(resultCode);
			exit(1);
		}
	}

	// XXXXX
	//fprintf(stderr,"requesting UNBIND\n");

	// Tell the Protocol Manager to unbind and stop.
	reqBlock.opcode   = NDIS_PROT_MAN_UNBIND_AND_STOP;
	reqBlock.pointer1 = (unsigned char *) &failingModules;
	//reqBlock.pointer2 = "DRAWBRIDGE";
	reqBlock.pointer2 = NULL;

	resultCode = protocolManager(&reqBlock,protocolManagerDS);

	if (resultCode != 0) {
		fprintf(stderr,"Could not unbind with Protocol Manager.\n");
		printNdisError(resultCode);
		exit(1);
	}

	fprintf(stdout," done\n");
}

void initNdis(CardHandle **campus, CardHandle **internet)
{
	int resultCode;
	int i;

	//putch('*');

	// Allocate the real mode stacks.
	for (i = 0;i < STACK_POOL_SIZE;++i) {
		// Allocate the buffer itself.
		freeStacks[i] = farmalloc(STACK_SIZE);

		if (!freeStacks[i]) {
			fprintf(stderr,"Could not allocate stack.\n");
			exit(1);
		}

		freeStacks[i] += STACK_SIZE;
	}

	//fprintf(stderr,"Initializing NDIS\n");

	openNdis();

	// Print out the results.
	//fprintf(stderr,"Network Protocol entry point: %08lX\n",protocolManager);

	//fprintf(stderr,"parsing NDIS config info\n");

	parseNdisConfig();

	//fprintf(stderr,"parsed ok\n");

	// Now we initialize the management stuff since we now have all of our IP addresses
	//   and what not.
	initManage();

	// We can also now init the Syslog stuff.
	initSyslog();

	registerNdisAndBind();

	//fprintf(stderr,"my module id = %d\n",common.moduleId);
	//fprintf(stderr,"inside id = %d\n",insideHandle.common->moduleId);
	//fprintf(stderr,"outside module id = %d\n",outsideHandle.common->moduleId);

	// Register the exit routine.
	atexit(shutdownNdis);

	// Check out each card and make sure it is reasonable.
	fprintf(stdout,"Inside card: %-16s - ",insideHandle.moduleName);
	checkMacFeatures(&insideHandle);
	fprintf(stdout,"Outside card: %-15s - ",outsideHandle.moduleName);
	checkMacFeatures(&outsideHandle);

	switch (mediaType) {
		case MEDIA_FDDI:
			fprintf(stdout,"Media type: FDDI");
			break;
		case MEDIA_ETHERNET:
			fprintf(stdout,"Media type: ETHERNET");
			break;
		default:
			fprintf(stdout,"Unsupported media.\n");
			exit(1);
	}

	fprintf(stdout," - Frame size: %d\n",frameSize);

	// Now that we know the frame size we can init the queue stuff.
	//fprintf(stderr,"initing the queue\n");
	initQueue();

	// Mark sends pending as 0.
	insideHandle.sendsPending  = 0;
	outsideHandle.sendsPending = 0;

	// Mark the queues as empty.
	insideHandle.queue.tail  = insideHandle.queue.head  = NULL;
	outsideHandle.queue.tail = outsideHandle.queue.head = NULL;

	insideHandle.mgmtQueue.tail  = insideHandle.mgmtQueue.head  = NULL;
	outsideHandle.mgmtQueue.tail = outsideHandle.mgmtQueue.head = NULL;

	// Set up the inverse table lookup.
	cardHandleLookup[insideHandle.common->moduleId]  = &insideHandle;
	cardHandleLookup[outsideHandle.common->moduleId] = &outsideHandle;

	// Return the handles.
	*campus   = &insideHandle;
	*internet = &outsideHandle;

	fprintf(stderr,"Starting cards\n");

	// Start the cards.
	startMac(&insideHandle);
	startMac(&outsideHandle);
}
