/*
 * Copyright (c) 1993,1995
 *	Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Texas A&M University
 *	and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *             David K. Hess, Douglas Lee Schales, David R. Safford
 */
/*
 *  Communications interface file.
 */
#include "fm.h"

struct in_addr currTarget;

/* 
 * Buffer for an acknowledgement.
 */
static unsigned char syncAcknowledgement[BUFFER_SIZE];
static unsigned char syncMessage[BUFFER_SIZE];
static unsigned char acknowledgement[BUFFER_SIZE];
static unsigned char message[BUFFER_SIZE];
int sockFd;
int synced = NO;

Key sessionKey;

unsigned short chkSum(unsigned char *buf,unsigned short length,unsigned short *prevSum)
{
	int i;
	unsigned short *curr;
	unsigned long sum;

	/* If a previous sum was specified then use it to keep chaining. */
	if (prevSum == NULL)
		sum = 0;
	else
		sum = ~*prevSum;

	curr = (unsigned short *) buf;

	for (i = 0;i < length / 2;++i)
		sum += *curr++;

	if (length % 2 == 1) 
		sum += ((unsigned short) *(unsigned char *) curr) << 8;

	sum = (sum & 0xFFFF) + ((sum & 0xFFFF0000) >> 16);
	sum = (sum & 0xFFFF) + ((sum & 0xFFFF0000) >> 16);

	return (unsigned short) ~sum;
}

/*
 * This function initializes the communications interface.
 */
void initCommunications()
{
    /*
     * Get a UDP socket.
     */
    sockFd = socket(PF_INET,SOCK_DGRAM,0);

    if (sockFd == -1) {
	fprintf(messages,"socket: %s\n",sys_errlist[errno]);
	exit(1);
    }
}

int doSync(void)
{
    FiltHeader *packetHeader;
    FiltHeader *replyPacketHeader;
    SyncPacket tempSync;
    ErrorPacket *errorPacket;
    struct timeval waitTime;
    fd_set fdSet;
    int result;
    int retries;
    int dtablesize;
    Key myKey;
    Key theirKey;

    /* fprintf(stderr,"starting sync\n"); */

#ifdef SYSV
    dtablesize = sysconf(_SC_OPEN_MAX);
#else
    dtablesize = getdtablesize();
#endif

    packetHeader = (FiltHeader *) syncMessage;

    /*
     * Send a SYNC and wait for a SYNCACK. Note we shouldn't even be in this routine if we are not
     *   in encryption mode.
     */
    packetHeader->type  = FM_M_SYNC;
    packetHeader->flags = FM_F_CRYPTED;
    packetHeader->randomInject = rand();
    packetHeader->chkSum = 0;

    clientInit(password,&myKey,&tempSync);
    memcpy(syncMessage + sizeof(FiltHeader),&tempSync,sizeof(SyncPacket));

    /* fprintf(stderr,"called potp clientinit\n"); */

    /*
     * Send the packet.
     */
    result = send(sockFd,syncMessage,sizeof(FiltHeader) + sizeof(SyncPacket),0);

    /* fprintf(stderr,"sent packet; waiting on reply\n"); */

    if (result < 0) {
	fprintf(messages,"send: %s\n",sys_errlist[errno]);
	return -1;
    }

    /* 
     * Wait for the response.
     */
    FD_ZERO(&fdSet);
    FD_SET(sockFd,&fdSet);

    waitTime.tv_sec = sendTimeout;
    waitTime.tv_usec = 0;

    retries = 0;

    while (retries < sendRetries) {
	result = select(dtablesize,
			&fdSet,
			(fd_set *) NULL,
			(fd_set *) NULL,
			&waitTime);
	
	if (result == 0) {
		/* fprintf(stderr,"timeout on recv\n");*/

	    result = send(sockFd,syncMessage,sizeof(FiltHeader) + sizeof(SyncPacket),0);

	    if (result < 0) {
		fprintf(messages,"send: %s\n",sys_errlist[errno]);
		return -1;
	    }

	    ++retries;
	}
	else {
	    /*
	     * Packet must have arrived.
	     */
	    result = recv(sockFd,syncAcknowledgement,sizeof(syncAcknowledgement),0);

	    /* fprintf(stderr,"received packet\n");*/

	    if (result < 0) {
		fprintf(messages,"recv: %s\n",sys_errlist[errno]);
		return -1;
	    }
	    else {
		/*
		 * Check the packet.
		 */
		replyPacketHeader = (FiltHeader *) syncAcknowledgement;
		memcpy(&tempSync,syncAcknowledgement + sizeof(FiltHeader),sizeof(SyncPacket));

		/*
		 * Make sure the packet is coming from the filter and that it is the correct
		 *  protocol. Silently drop any that are not.
		 */
		if (replyPacketHeader->type == FM_M_SYNCACK) {
		    /*
		     * SYNC succeeded. Pull out the key and build a session key.
		     */
		    serverInit(password,&theirKey,&tempSync);
		    buildNewSessionKey(&myKey,&theirKey,&sessionKey);
		    break;
		}
		else if (replyPacketHeader->type == FM_M_ERROR) {
		    /*
		     * We had some kind of error. Print it and return.
		     */
		    errorPacket = (ErrorPacket *) (syncAcknowledgement + sizeof(FiltHeader));
		    printError(errorPacket->errorCode,output);
		    return -1;
		}
	    }
	}

	/* 
	 * Set up for the next select call.
	 */
	FD_ZERO(&fdSet);
	FD_SET(sockFd,&fdSet);
	
	waitTime.tv_sec = sendTimeout;
	waitTime.tv_usec = 0;
    }

    if (retries == sendRetries) {
	fprintf(messages,"target is not responding\n");
	return -1;
    }

    /* fprintf(stderr,"sync suceeded\n");*/

    return 0;
}

/*
 *  This function sends a message to the current target and returns the
 *    acknowledgement in a static buffer (which will be overwritten by
 *    the next call).
 */
unsigned char *sendMessage(unsigned char type,void *data,unsigned long size)
{
    fd_set fdSet;
    FiltHeader *filtHeader;
    FiltHeader *replyFiltHeader;
    ErrorPacket *errorPacket;
    struct timeval waitTime;
    int result;
    int retries;
    int dtablesize;
    int localSyncRetries;

    /* fprintf(stderr,"size = %d\n",size); */

    if (targetSet != YES) {
	fprintf(messages,"must set the target first\n");
	return NULL;
    }

    localSyncRetries = 0;

retrySync:

    if (synced == NO && passwordSet == YES) {
	if (doSync() != 0) {
	    fprintf(messages,"could not synchronize encryption with filter\n");
	    return NULL;
	}
	synced = YES;
    }

#ifdef SYSV
    dtablesize = sysconf(_SC_OPEN_MAX);
#else
    dtablesize = getdtablesize();
#endif

    filtHeader = (FiltHeader *) message;

    filtHeader->type  = type;
    filtHeader->flags = passwordSet;
    filtHeader->randomInject = rand();
    filtHeader->chkSum = 0;

    /*
     * Copy the data portion of the packet over.
     */
    if (size)
	memcpy(message + sizeof(FiltHeader),data,size);

    /*
     * Set the checksum.
     */
    filtHeader->chkSum = chkSum((unsigned char *) &filtHeader->chkSum,
				sizeof(filtHeader->chkSum) + 
					sizeof(filtHeader->randomInject) + 
					size,
				NULL);

    /* fprintf(stderr,"checksum = %04X\n",filtHeader->chkSum);*/

	/*
    {
    int k;
    for (k = 0;k < 20;++k) {
	fprintf(stderr,"%02X ",((unsigned char *) &filtHeader->chkSum)[k]);
    }
    fprintf(stderr,"\n");
    }
	*/

    /*
     * Encrypt the packet if necessary.
     */
    if (passwordSet == YES) {
	/* fprintf(stderr,"encrypting size = %d\n",
		sizeof(filtHeader->chkSum) + sizeof(filtHeader->randomInject) + size); */
	potpEncrypt((unsigned char *) &filtHeader->chkSum,
		(unsigned char *) &filtHeader->chkSum,
		&sessionKey,
		sizeof(filtHeader->chkSum) + sizeof(filtHeader->randomInject) + size);
    }

	/*
    {
    int k;
    for (k = 0;k < 20;++k) {
	fprintf(stderr,"%02X ",((unsigned char *) &filtHeader->chkSum)[k]);
    }
    fprintf(stderr,"\n");
    }
	*/

    result = send(sockFd,(char *) message,sizeof(FiltHeader) + size,0);

    if (result == -1) {
	fprintf(messages,"send: %s\n",sys_errlist[errno]);
	return (char *) NULL;
    }

    /* 
     * Wait for the response.
     */
    retries = 0;

    while (retries < sendRetries) {
	FD_ZERO(&fdSet);
	FD_SET(sockFd,&fdSet);

	waitTime.tv_sec = sendTimeout;
	waitTime.tv_usec = 0;

	result = select(dtablesize,
			&fdSet,
			(fd_set *) NULL,
			(fd_set *) NULL,
			&waitTime);
	
	if (result == 0) {
	    /*
             * Timed out so send the request again.
             */
	    if (send(sockFd,(char *) message,sizeof(FiltHeader) + size,0) == -1) {
		fprintf(messages,"send: %s\n",sys_errlist[errno]);
		return (char *) NULL;
	    }

	    ++retries;
	}
	else {
	    /*
	     * Packet must have arrived.
	     */
	    result = recv(sockFd,acknowledgement,sizeof(acknowledgement),0);

	    if (result <= 0) {
		fprintf(messages,"recv: %s\n",sys_errlist[errno]);
		return (char *) NULL;
	    }
	    else {
		/*
		 * Check the packet. Silently drop any packets that do not have the
		 *   correct target address or the wrong protocol.
		 */
		replyFiltHeader = (FiltHeader *) acknowledgement;

		/*
		fprintf(stderr,"result = %d size = %d\n",result,
			sizeof(filtHeader->chkSum) + sizeof(filtHeader->randomInject) + 
				result - sizeof(FiltHeader));
		*/
		/*
		 * Decrypt the packet.
		 */
	    	if (passwordSet == YES && replyFiltHeader->type != FM_M_ERROR) {
		    potpDecrypt((unsigned char *) &replyFiltHeader->chkSum,
			(unsigned char *) &replyFiltHeader->chkSum,
			&sessionKey,
			sizeof(filtHeader->chkSum) + sizeof(filtHeader->randomInject) + 
				result - sizeof(FiltHeader));
		}

		if (replyFiltHeader->type == FM_M_ERROR) {
		    /*
		     * We had some kind of error.
		     */
		    errorPacket = (ErrorPacket *) (acknowledgement + sizeof(FiltHeader));

			/*
		    fprintf(stderr,"syncRetries = %d localSyncRetries = %d\n",syncRetries,localSyncRetries);
			*/

		    if (errorPacket->errorCode == FM_ERROR_LOSTSYNC && localSyncRetries < syncRetries) {
			fprintf(messages,"[synchronizing encryption]\n");
			synced = NO;
			++localSyncRetries;
			goto retrySync;
		    }

		    printError(errorPacket->errorCode,output);

		    return (char *) NULL;
		}

		break;
	    }
	}

	/* 
	 * Set up for the next select call.
	 */
	 /*
	FD_ZERO(&fdSet);
	FD_SET(sockFd,&fdSet);
	
	waitTime.tv_sec = sendTimeout;
	waitTime.tv_usec = 0;
	*/
    }

    if (retries == sendRetries) {
	fprintf(messages,"target is not responding\n");
	return (char *) NULL;
    }

    /* We perform this move to make sure longs are aligned when structures are mapped to this buffer. */
    /*memmove(acknowledgement,acknowledgement + sizeof(FiltHeader),result - sizeof(FiltHeader));
    */

    return acknowledgement;
}
