/*
 Copyright (c) 2002 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

------------------------------------------------------------------------------
 Module Header
 Filename          : log.c
 Purpose           : Hook for logging messages
 Author            : Can Bican <can@ripe.net>
 Description       : All glib log functions go to logHandler
 Language Version  : C
 $Id: log.c,v 1.5 2002/06/07 11:09:33 can Exp $
------------------------------------------------------------------------------
*/

#include "supconf.h"

void logHandler(const gchar * log_domain, GLogLevelFlags log_level,
		const gchar * message, gpointer user_data)
{
  struct flock fl;
  time_t t;
  int f;
  gchar *ct, logLine[MAXDATALEN], *logLevelStr;

  fl.l_type = F_WRLCK;
  fl.l_whence = SEEK_SET;
  fl.l_start = 0;
  fl.l_len = 0;
  fl.l_pid = getpid();
  f = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, S_IREAD | S_IWRITE);
  if (f < 0) {
    fprintf(stderr, "Error opening %s: %s\n", LOGFILE, strerror(errno));
    exit(1);
  }
  if (fcntl(f, F_SETLKW, &fl) < 0) {
    fprintf(stderr, "Error setting lock for %s: %s\n", LOGFILE,
	    strerror(errno));
    exit(1);
  }
  if (time(&t) == ((time_t) - 1)) {
    fprintf(stderr, "Error getting time: %s\n", strerror(errno));
    exit(1);
  }
  ct=ctime(&t);
  g_strchomp(ct);
  if (log_level == G_LOG_LEVEL_ERROR) {
    logLevelStr = g_strdup("error");
  } else if (log_level == G_LOG_LEVEL_CRITICAL) {
    logLevelStr = g_strdup("critical");
  } else if (log_level == G_LOG_LEVEL_WARNING) {
    logLevelStr = g_strdup("warning");
  } else if (log_level == G_LOG_LEVEL_MESSAGE) {
    logLevelStr = g_strdup("message");
  } else if (log_level == G_LOG_LEVEL_INFO) {
    logLevelStr = g_strdup("info");
  } else if (log_level == G_LOG_LEVEL_DEBUG) {
    logLevelStr = g_strdup("debug");
  } else {
    logLevelStr = g_strdup_printf("%d", log_level);
  }
  g_snprintf(logLine, MAXDATALEN, "%s %s %s %s\n", ct, log_domain,
	     logLevelStr, message);
  g_free(logLevelStr);
  write(f, logLine, strlen(logLine)), strlen(logLine);
  fl.l_type = F_UNLCK;
  if (fcntl(f, F_SETLK, &fl) == -1) {
    fprintf(stderr, "Error unsetting lock for %s: %s\n", LOGFILE,
	    strerror(errno));
    exit(1);
  }
  close(f);
}
