/**************************************************
 * File: annotate.c
 * Purpose: obtain annotation and descriptors,
 *          return length of annotation
 * Invokes: hb.h findDD() 
 * Contents:
 *  annotate: get annotation and data descriptor
 *            it belongs to; get temporary file
 *            to store annotation in, before passing
 *            it to CBReadFile()
 *  findDD: find data descriptor that annotation 
 *          begongs to 
 * Returns: 
 *  annotate: length of annotation
 *  findDD: length of annotation
 * Remarks: annotation file has a bug, or length is wrong
 **************************************************/

#include "hbox.h" 

uint16  descTag, descRef;     /* tag and ref of descriptor to which DIA or DIL belongs */

int annotate(c, ann, ref)
     ctxt_ptr c;
     int   ann, ref;         /* ann: annotation is either a DIL or a DIA 
		  		   ref: reference number of ann */
{
  int   len;                  /* length of annotation */
  char *buf;                  /* annotation buffer */

#if DEBUG
  printf("in annotate \n");
#endif

  /* 
   * get the annotation from the HDF file and store it in a buffer;
   * get the data descriptor that it belongs to; get its length
   */
  
  len = findDD(c, &buf, ann, ref);
  
  if (len == FAIL)
    return len;
  
  /*
   * read annotation into a motif scrolled text widget
   */
  annRead(c, ann, (len-1), buf, descTag, descRef);

#if DEBUG
  printf(" annotate: done\n");
#endif

  return len;
}


/***************************************
 * AnnRead: read label or annotation
 ***************************************/
int annRead(c,tag, strlen, data, dtag, dref)
     ctxt_ptr c;
     int tag;                /* DIA or DIL */
     int dtag, dref;         /* tag and ref of desc that annotation is for */
     int strlen;                /* length of annotation */
     char *data;                /* pointer to annotation or label */
{
  Cardinal     n;             /* number of label resource values to set */
  Arg      wargs[20];           /* number of motif widget arguments */
  XmString label;               /* button label string */
  char     s[100], *title;

#if DEBUG
  printf("in AnnRead\n");
#endif

  /* set global tag and ref of desc which the annotation is for */
  c->dTag = dtag; 
  c->dRef = dref;
  c->editType = tag;
  
  /* set local edit window shell IDs to global edit window shell IDs */
  c->editShellMapped = 1;
  XtPopup(c->editShell, XtGrabNone);

  /*
   * read annotation into the scrolled text window
   * save a copy in c->backup_fileptr
   */

  if(strlen) {
    c->backup_text = (char *) XtMalloc( strlen + 1 );
    data[strlen] = '\0';
    strcpy (c->backup_text, data);
  } else {
    c->backup_text = NULL;
  }

#if DEBUG
  if(data)
    printf("contents: %s    strlen %d\n", data, strlen);
#endif		

  /*
   * set edit cursor to beginning of annotation text 
   */
  c->firstEdit = TRUE;
  XmTextSetTopCharacter(c->textedit, 0);
  XmTextSetString(c->textedit, data);

  /*
   * Display TAG and REF which the annotation was for
   */
  if (tag == DFTAG_DIL) {
    sprintf(s, "Label for:  %s  (Tag %d  Ref %d)", getName(dtag), dtag, dref);
    title = "Label Editor";
  } else {
    sprintf(s, "Annotation for:  %s  (Tag %d  Ref %d)", getName(dtag), dtag, dref);
    title = "Annotation Editor";
  }

  n = 0;
  XtSetArg(wargs[n], XmNtitle, title); n++;
  XtSetValues(c->editShell, wargs, n);

  n = 0;
  label = XmStringCreate(s, XmSTRING_DEFAULT_CHARSET);
  XtSetArg(wargs[n], XmNlabelString, label); n++; 
  XtSetArg(wargs[n], XmNx, EDGE_OFFSET); n++;
  XtSetArg(wargs[n], XmNy, EDGE_OFFSET); n++;
  c->editLabel = XmCreateLabelGadget(c->editForm, "tagLabel", wargs, n);
  XtManageChild(c->editLabel);

#if DEBUG
  printf("AnnRead: done\n");
#endif

  if(data) free(data);
} /* annRead */


/***************************************************
 * CBSave: save label or annotation back to HDF file
 ***************************************************/
void CBSave(w, client_data, call_data)
     Widget    w;
     void      *client_data;
     void      *call_data;
{
  char  *updateStr;
  XmTextPosition length;           /* length of current text */
  XmTextPosition current_pos;      /* current position of insert cursor */
  int    n, ret;

  Arg   wargs[3];
  ctxt_ptr c;

  n = 0;
  XtSetArg(wargs[n], XmNuserData, &c); n++;
  XtGetValues(w, wargs, n);

  /*
   * Update current text string buffer 
   */
  length = XmTextGetLastPosition(c->textedit);
  updateStr = XmTextGetString(c->textedit);

#if DEBUG
  printf("saving text to %d %d\n", c->dTag, c->dRef);
#endif

  /*
   * Update hdf file
   */
  if (c->editType == DFTAG_DIL){
    ret = DFANputlabel(c->hb_file, c->dTag, c->dRef, updateStr);
  } else {
    ret = DFANputdesc(c->hb_file, c->dTag, c->dRef, updateStr, length);
  }
  if(ret == FAIL) {
    hdfErr(c);
    return;
  }

  current_pos = XmTextGetInsertionPosition(c->textedit);
  XmTextSetString(c->textedit, updateStr);
  XmTextSetInsertionPosition(c->textedit, current_pos);

  XtFree(c->backup_text);
  c->backup_text = (char *) XtMalloc( length );
  strcpy(c->backup_text, updateStr);
  
  XtFree(updateStr);
  return;

} /* CBSave */


/**************************************************
 * CBCancel: cancel editing of label or annotation
 **************************************************/
void CBCancel(w, client_data, call_data)
     Widget   w;
     void     *client_data;
     void     *call_data;
{
  int n;
  Arg   wargs[3];
  ctxt_ptr c;

  n = 0;
  XtSetArg(wargs[n], XmNuserData, &c); n++;
  XtGetValues(w, wargs, n);

  if (c->editShellMapped) {
    c->editShellMapped = 0;
    XtPopdown(c->editShell);
    XtDestroyWidget(c->editLabel);
  }
  
} /* CBCancel */


/****************************************
 *  findDD: find data descriptor that the
 *          annotation was written for 
 *  Returns: length of annotation 
 ****************************************/
int  findDD(c, buf, anntag, annref)
     ctxt_ptr c;
     char **buf;             /* pointer to annotation buffer */
     int  anntag, annref;  /* tag and ref of annotation */
{
  char   *dBuf;              /* annotation for current descriptor */
  int32   dlen;              /* length of annotation of descriptor */
  uint16  lastref;           /* ref of last annotation read */
  int     i;              /* nth descriptor in the hdf file */

  dBuf = NULL;

#if DEBUG 
  printf("in findDD, looking for %d %d\n", anntag, annref);
#endif

  for(i = 0; i < c->hb_numDesc; i++) {

    descTag = c->hb_desc[i].tag;   
    descRef = c->hb_desc[i].ref;
    
    /*
     * get length of annotation
     */
    
    /* 
     * search for descriptor reference number matching the
     * annotation reference number --- the annotation belongs
     * to this data descriptor;
     * if data descriptor found, return length of its annotation
     */
    
    if (anntag == DFTAG_DIA) {
      dlen =  DFANgetdesclen(c->hb_file, descTag, descRef);
      
      if (dlen > 0) {
	dBuf = (char *) malloc (dlen + 1);
	if (DFANgetdesc(c->hb_file, descTag, descRef, dBuf, (dlen+1)) == -1) {
	  printf ("findDD: DFANgetdesc failed \n");
	  return FAIL;
	}
      }
    }
    if (anntag == DFTAG_DIL) {
      dlen =  DFANgetlablen(c->hb_file, descTag, descRef);
      
      if (dlen > 0) {
	dBuf = (char *) malloc (dlen + 1);
	if (DFANgetlabel(c->hb_file, descTag, descRef, dBuf, (dlen+1)) == -1) {
	  printf("findDD: DFANgetlabel failed \n");
	  return FAIL;
	}
      }
    }
    
    if (dlen >= 0) {
      lastref = DFANlastref();
#if DEBUG
      printf ("findDD: dtag %ld lastref %ld \n", descTag, lastref);
      printf ("findDD: annref %ld  \n", annref);
#endif	  
      if (lastref == annref) {
	/* found matching data descriptor for annotation */
	*buf = (char *) XtMalloc(dlen+1);
	strcpy(*buf, dBuf);
	
#if DEBUG
	printf ("findDD: dannotation:  %s \n", dBuf);
#endif
	return (dlen+1);
      }
    }
  }
  
#if DEBUG
  printf ("findDD: leaving, didn't find descTag and descRef \n");
#endif
  
  if(dBuf) free(dBuf);

  return (-1);
}

