/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMOID.C - OID functions
 *
 * $Revision:   1.2  $ $Date:   27 Jul 1992 19:57:02  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMOID.C_V  $
 * 
 *    Rev 1.2   27 Jul 1992 19:57:02   gfoster
 * Changed the variable name fNoCheckIndxSeq
 * to fCheckIndxSeq to accommodate the -7
 * option providing stricter index item checking
 * instead of looser checking.
 * 
 *    Rev 1.1   19 Jun 1992 16:13:26   gfoster
 * Copyright text was reformated.
 * 
 * Fixed bug so that situation is correctly reported when
 * duplicate OID objects are defined.
 * 
 *    Rev 1.0   27 May 1992 16:01:18   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <string.h>
#include <ctype.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/** checkOIDcomp - check if OID component is valid
*
* NOTE: no forward references are allowed
*
* call with:
*   pPar - ptr to parent (or NULL)
*   pNa - component name (or NULL)
*   ulVal - component value (or -1 if not present)
*   pMod - module containing component use
*
* returns:
*   ptr to component or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
checkOIDcomp(MIBSYM *pPar, STRTAB *pNa, ULONG ulVal, MIBSYM *pMod)
#else
checkOIDcomp(pPar, pNa, ulVal, pMod)
    MIBSYM *pPar;
    STRTAB *pNa;
    ULONG ulVal;
    MIBSYM *pMod;
#endif
{
    MIBSYM *pComp;
    MIBSYM *pT;


    /* check if name specified */
    if (pNa != NULL) {
        /* name present, check if item defined */
        pComp = pNa->pSym;
        if (pComp == NULL) {
            /* item not defined */
            yyerror("component \"%s\" is not defined",
                    pComp->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }
        if (pComp->pMod != pMod) {
            /* item not defined */
            yyerror("component \"%s\" is not defined in the current module",
                    pComp->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }

        /* check if import */
        if (pComp->usType == MIBSYMimp) {
            pComp->ut.imp.cUse++;
            pComp = pComp->ut.imp.pImpSym;
        }

        /* check if OID type */
        if ((pComp->usType & MIBSYMmask) != MIBSYMoid) {
            yyerror("component \"%s\" must be an item having an OID value",
                    pComp->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }

        /* check for valid data structures */
        if (pComp->ut.oid.pPar == NULL) {
#ifdef OLD
            yyerror("checkOIDcomp: internal error - bad data structure for \"%s\"",
                    pComp->pszName);
            yyterm();
#endif
            return(NULL);
        }

        /* check for matching parents */
        if (pPar == NULL) {
            /* no parent given */
            /* get parent from component */
            pPar = pComp->ut.oid.pPar;
        } else {
            /* parent specified */
            /* check if same parent */
            if (pComp->ut.oid.pPar != pPar) {
                if ((pComp->ut.oid.pPar)->pMod != pMod)
                    yyerror("\"%s\" already defined with parent \"%s\" from module \"%s\" and not \"%s\"",
                            pComp->pszName, (pComp->ut.oid.pPar)->pszName,
                            ((pComp->ut.oid.pPar)->pMod)->pszName,
                            pPar->pszName);
                else
                    yyerror("\"%s\" already defined with parent \"%s\" and not \"%s\"",
                            pComp->pszName, (pComp->ut.oid.pPar)->pszName,
                            pPar->pszName);
#ifdef OLD
                yyterm();
#endif
                return(NULL);
            }
        }

        /* check if number present */
        if (ulVal != -1L) {
            /* check if component has same value */
            if (pComp->ut.oid.ulVal != ulVal) {
                /* mismatch of values */
                yyerror("component \"%s\" already assigned value %lu, and can not be changed to %lu",
                        pComp->pszName, pComp->ut.oid.ulVal, ulVal);
#ifdef OLD
                yyterm();
#endif
                return(NULL);
            }
        }
        /* all done */
        return(pComp);
    }
    
    
    /* just a number specified */
    /* check if parent has value */
    if (pPar == NULL)
        return(NULL);
    for (pT = pPar->ut.oid.pChild; pT != NULL; pT = pT->ut.oid.pSib) {
        if (ulVal == pT->ut.oid.ulVal) {
            /* found match, all done */
            return(pT);
        }
        if (ulVal < pT->ut.oid.ulVal) {
            /* found place it should be */
            break;
        }
    }

    /* value not defined */
    yyerror("No item found for \"%s\" with OID value %lu",
            pPar->pszName, ulVal);
#ifdef OLD
    yyterm();
#endif
    return(NULL);


} /* checkOIDcomp */


/** addOIDname - add pure OID
*
* call with:
*   pNa - new OID
*   pNaPar - name of parent (if NULL, then defining top items)
*   ulVal - component value
*   pMod - containing module
*
* returns:
*   ptr to OID or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
addOIDname(STRTAB *pNa, STRTAB *pNaPar, ULONG ulVal, MIBSYM *pMod)
#else
addOIDname(pNa, pNaPar, ulVal, pMod)
    STRTAB *pNa;
    STRTAB *pNaPar;
    ULONG ulVal;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pOid;
    MIBSYM *pPar;
    MIBSYM *pSymPar;
    MIBSYM *pT;
    MIBSYM *pOT;
    BOOL fCheckCirc = FALSE;


    /* check is self defining */
    if (pNa == pNaPar) {
        yyerror("OID \"%s\" can not be used to define itself\n",
                pNa->pszVal);
#ifdef OLD
        yyterm();
#endif
        return(NULL);
    }

    /* check if item already defined */
    pSym = pNa->pSym;
    if ((pSym != NULL) && (pSym->pMod == pMod)) {
        /* item already defined */
        /* check if forward reference */
        pOid = pSym;
        if (pSym->usType != (MIBSYMFR|MIBSYMoid)) {
            yyerror("\"%s\" already defined in current module",
                    pSym->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }
    } else {
        /* item not defined */
        pOid = NULL;
    }

    /* if parent is specified */
    if (pNaPar != NULL) {
        /* check if parent already defined */
        pSymPar = pNaPar->pSym;
        if ((pSymPar != NULL) && (pSymPar->pMod == pMod)) {
            /* parent defined */
            /* check if import */
            if (pSymPar->usType == MIBSYMimp) {
                pSymPar->ut.imp.cUse++;
                pSymPar = pSymPar->ut.imp.pImpSym;
            } else
                fCheckCirc = TRUE;

            /* check if parent is an OID type */
            if ((pSymPar->usType & MIBSYMmask) != MIBSYMoid) {
                yyerror("parent \"%s\" of \"%s\" must be OID object",
                    pSymPar->pszName, pNa->pszVal);
#ifdef OLD
                yyterm();
#endif
                return(NULL);
            }
            pPar = (MIBSYM *)pSymPar;
        } else {
            /* parent not defined, allocate one */
            pPar = newOID(pNaPar, pMod);
        }
    } else {
        pPar = &OidRoot;
    }

    /* if not forward reference, then allocate */
    if (pOid == NULL) {
        /* new symbol */
        pOid = newOID(pNa, pMod);
    } else {
        /* forward reference */
        if (fCheckCirc) {
            /* parent already defined and not import */
            /* check for circular reference */
            for (pT = pPar; pT != NULL; pT = pT->ut.oid.pPar) {
                if (pT == pOid) {
                    yyerror("\"%s\" is circular defined", pOid->pszName);
#ifdef OLD
                    yyterm();
#endif
                    return(NULL);
                }
            }
        }
    }

    /* store values */
    pOid->usType = MIBSYMoid;
    pOid->ut.oid.ulVal = ulVal;
    pOid->ut.oid.usOType = MIBOToid;
    pOid->ut.oid.pPar = pPar;
    /* insert in children list for parent */
    for (pOT = NULL, pT = pPar->ut.oid.pChild; pT != NULL;
            pOT = pT, pT = pT->ut.oid.pSib) {
        if (ulVal < pT->ut.oid.ulVal) {
            /* found place to insert */
            break;
        }
        if (ulVal == pT->ut.oid.ulVal) {
            /* duplicate values */
            if (pT->pMod == pMod) {
                if (strcmp(pOid->pszName, pT->pszName) == 0)
                    yyerror("\"%s\" is already defined",
                            pOid->pszName);
                else
                    yyerror("\"%s\" and \"%s\" have same registration value",
                            pOid->pszName, pT->pszName);
            } else {
                if (strcmp(pOid->pszName, pT->pszName) == 0)
                    yyerror("\"%s\" is already defined in module \"%s\"",
                            pT->pszName, (pT->pMod)->pszName);
                else
                    yyerror(
                "\"%s\" and \"%s\" from \"%s\" have same registration value",
                        pOid->pszName, pT->pszName,
                        (pT->pMod)->pszName);
            }
#ifdef OLD
            yyterm();
            return(NULL);
#endif
            break;              /* insert */
        }
    }
    if (pOT == NULL) {
        /* insert as first child */
        pPar->ut.oid.pChild = pOid;
    } else {
        /* insert in list */
        pOT->ut.oid.pSib = pOid;
    }
    pOid->ut.oid.pSib = pT;

    return(pOid);

} /* addOIDname */


/** hexVal - convert hex char to value
*
* call with:
*   ch - char
*
* returns:
*   value
*/
    USHORT
#ifdef __STDC__
hexVal(CHAR ch)
#else
hexVal(ch)
    CHAR ch;
#endif
{
    register USHORT us;


    if ((ch >= '0') && (ch <= '9')) {
        us = ch - '0';
    } else if ((ch >= 'A') && (ch <= 'F')) {
        us = ch - 'A' + 10;
    } else {
        us = ch - 'a' + 10;
    }
    return(us);

} /* hexVal */


#define CTABLE 5
#define PSZTABLE "Table"
#define CENTRY 5
#define PSZENTRY "Entry"

/** checkOIDs - that that all OIDs for a module are defined
*               (ie no forward references)
* call with:
*   pMod - module to check
*/
    VOID
#ifdef __STDC__
checkOIDs(MIBSYM *pMod)
#else
checkOIDs(pMod)
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pSym;
    USHORT usSyntax;
    MIBSYM *pO;
    MIBINDX *pIndx;
    MIBSEQI *pSi;
    MIBSYM *pSeq;
    MIBSYM *pSeqPar;
    USHORT cNa;
    USHORT cNaPre;
    MIBENUM *pE;
    USHORT usSizeRange;
    BOOL fNoSR;
    STRTAB *pNa;
    USHORT i;
    PSZ psz;


    for (pT = pOidGHL; pT != NULL; pT = pT->pNext) {
        if (pT->pMod != pMod)
            continue;

        if (!islower(*(pT->pszName))) {
            yywarning("Name of OID object \"%s\" must start with lowercase letter",
                    pT->pszName);
        }
        
        if (pT->usType & MIBSYMFR) {
            yyerror("\"%s\" referenced but not defined",
                    pT->pszName);
        }


        pO = pT->ut.oid.pPar;   /* get parent */

        /* check that there is no sibling with the same name,
        *  but ignore duplicate definitions */
        if (pO != NULL) {
            for (pSym = pT->pSym; pSym != NULL; pSym = pSym->pSym) {
                if ((pSym->usType & MIBSYMmask) != MIBSYMoid)
                    continue;       /* skip non-OID items */
                if ((pSym->ut.oid.ulVal != pT->ut.oid.ulVal) &&
                        (pSym->ut.oid.pPar == pO)) {
                    yyerror("Item \"%s\" has sibling in OID tree from module \"%s\" with the same name",
                            pT->pszName, (pSym->pMod)->pszName);
                    break;
                }
            }
        }

        /* check semantics based on type */
        switch(pT->ut.oid.usOType) {
        case MIBOToid:
            /* pure OID */
            /* check that parent is OID */
            if ((pO != NULL) && (pO->ut.oid.usOType != MIBOToid)) {
                yyerror("Object identifier \"%s\" should be registered under an object identifier",
                        pT->pszName);
            }
            break;

        case MIBOTtab:
            /* SNMP table */
            /* check that parent is an OID */
            if ((pO != NULL) && (pO->ut.oid.usOType != MIBOToid)) {
                yyerror("Table \"%s\" should be registered under an object identifier",
                        pT->pszName);
            }
            /* check that suffix of name is "Table" */
            cNa = strlen(pT->pszName);
            if ((cNa < CTABLE) ||
                    (strcmp((pT->pszName)+(cNa-CTABLE), PSZTABLE) != 0)) {
                if (!fNoCheckTab)
                    yywarning("Name of table \"%s\" should end with Table",
                            pT->pszName);
            }
            /* syntax is seq of */
            if (pT->ut.oid.syn.usSyntax != MIBSYNseqOf) {
                yyerror("\"%s\" has invalid syntax for a table",
                        pT->pszName);
            }
            /* must have row defined */
            pO = pT->ut.oid.pChild;
            if ((pO == NULL) || (pO->ut.oid.usOType != MIBOTrow)) {
                yyerror("Table \"%s\" does not have a row defined",
                        pT->pszName);
            }
            /* access is not-accessible */
            if (!fAllowAccess && (pT->ut.oid.usAccess != MIBACCna) &&
                (pT->ut.oid.usAccess != MIBACCbad)) {
                yyerror("Table \"%s\" must have access of \"not-accessible\"",
                        pT->pszName);
            }
            /* status is mandatory, deprecated, or obsolete */
            if ((pT->usStatus != MIBSTAma)  &&
                    (pT->usStatus != MIBSTAde) &&
                    (pT->usStatus != MIBSTAob) &&
                    (pT->usStatus != MIBSTAbad) &&
                    !fAllowOpt) {
                yyerror("Table \"%s\" must have status of \"mandatory\", \"deprecated\", or \"obsolete\"",
                        pT->pszName);
            }
            /* no index */
            if (pT->ut.oid.cIndx != 0) {
                yyerror("Table \"%s\" can have not INDEX items specified",
                        pT->pszName);
            }
            /* no defval */
            if (pT->ut.oid.usDefVal != MIBDFVno) {
                yyerror("Table \"%s\" can have not a DEFAULT value specified",
                        pT->pszName);
            }
            break;

        case MIBOTrow:
            /* SNMP row */
            /* check that suffix of name is Entry */
            cNa = strlen(pT->pszName);
            cNaPre = cNa - CENTRY;
            if ((cNa < CENTRY) ||
                    (strcmp((pT->pszName)+(cNa-CENTRY), PSZENTRY) != 0)) {
                if (!fNoCheckTab)
                    yywarning("Name of row \"%s\" should end with Entry",
                                pT->pszName);
                cNaPre = 0;
            }
            /* syntax is seq */
            if (pT->ut.oid.syn.usSyntax != MIBSYNseq) {
                yyerror("\"%s\" has invalid syntax for a row",
                        pT->pszName);
                pSeq = NULL;
            } else {
                /* get sequence */
                pSeq = pT->ut.oid.syn.usi.pSeq;
            }
            /* must have table defined */
            if ((pO == NULL) || (pO->ut.oid.usOType != MIBOTtab)) {
                yyerror("Row \"%s\" does not have a containing table defined",
                        pT->pszName);
            } else {
                /* parent exists and is a table */
                if (cNaPre != 0) {
                    /* check that parent has same prefix */
                    cNa = strlen(pO->pszName);
                    if ((cNa > CTABLE) &&
                            (strncmp(pT->pszName, pO->pszName,
                               cNaPre) != 0)) {
                        if (!fNoCheckTab)
                            yywarning("Table \"%s\" and Row \"%s\" should have same name prefix",
                                    pO->pszName, pT->pszName);
                    }
                    /* check that seq name matches except for first char */
                    if ((pSeq != NULL) &&
                            (((INT)(*(pSeq->pszName)) !=
                                    toupper(*(pT->pszName))) ||
                            (strcmp(pSeq->pszName+1, pT->pszName+1) != 0))) {
                        if (!fNoCheckTab)
                            yywarning("Sequence \"%s\" and Row \"%s\" should have related names",
                                    pSeq->pszName, pT->pszName);
                    }
                }
                /* check that parent has same sequence */
                if (pO->ut.oid.syn.usSyntax != MIBSYNseqOf) {
                    pSeqPar = NULL;
                } else {
                    /* get parents's sequence */
                    pSeqPar = pO->ut.oid.syn.usi.pSeq;
                }
                if ((pSeq != NULL) && (pSeqPar != NULL) &&
                        (pSeq != pSeqPar)) {
                    yyerror("Row \"%s\" and Table \"%s\" must use the same sequence",
                            pT->pszName, pO->pszName);
                }
            }

            /* access is not-accessible */
            if (!fAllowAccess && (pT->ut.oid.usAccess != MIBACCna) &&
                    (pT->ut.oid.usAccess != MIBACCbad)) {
                yyerror("Row \"%s\" must have access of \"not-accessible\"",
                        pT->pszName);
            }

            /* status is mandatory, deprecated, or obsolete */
            if ((pT->usStatus != MIBSTAma)  &&
                    (pT->usStatus != MIBSTAde) &&
                    (pT->usStatus != MIBSTAob) &&
                    (pT->usStatus != MIBSTAbad) &&
                    !fAllowOpt) {
                yyerror("Row \"%s\" must have status of \"mandatory\", \"deprecated\", or \"obsolete\"",
                        pT->pszName);
            }

            /* must have index */
            if (pT->ut.oid.cIndx == 0) {
                yyerror("Row \"%s\" must have INDEX items specified",
                        pT->pszName);
            }
            if (pSeq != NULL) {
                /* check for these normal case items, but with
                    flags to turn off error messages */
                /* indices must be objects (or limited syntax items) */
                /* indices must be in sequence */
                /* indices must be read-only */
                /* indices must have no defval */
                /* indices that are octet string must size specified */
                /* indices that are integer must have non-negative range */
                /*?? syntax type check for indices */
                for (pIndx = pT->ut.oid.pIndxL; pIndx != NULL;
                        pIndx = pIndx->pNext) {
                    if (pIndx->usItype == MIBITnloct) {
                        /* index is no length OCTET STRING */
                        /* check that no items follow */
                        if (pIndx->pNext != NULL) {
                            yyerror("Row \"%s\" has index item NOLENGTH OCTET STRING which is not the last item",
                                    pT->pszName);
                            continue;
                        }
                    } else if (pIndx->usItype == MIBITnloid) {
                        /* index is no length OBJECT IDENTIFIER */
                        /* check that no items follow */
                        if (pIndx->pNext != NULL) {
                            yyerror("Row \"%s\" has index item NOLENGTH OBJECT IDENTIFIER which is not the last item",
                                    pT->pszName);
                            continue;
                        }
                    } else if ((pIndx->usItype == MIBITobj) ||
                               (pIndx->usItype == MIBITnlobj)) {
                        pO = pIndx->pOid;
                        if (pO->ut.oid.usOType != MIBOTobj) {
                            yyerror("Row \"%s\" has index item \"%s\" not defined as an object",
                                    pT->pszName, pO->pszName);
                            continue;
                        }
                        for (pSi = pSeq->ut.seq.pSeqIL; pSi != NULL;
                                pSi = pSi->pNext) {
                            /* check for match in sequence with index */
                            if (pSi->pOid == pO)
                                break;
                        }
                        if (fCheckIndxSeq && (pSi == NULL)) {
                            /* index item not in sequence */
                            yyerror("Row \"%s\" has index item \"%s\" which is not defined in sequence",
                                    pT->pszName, pO->pszName);
                        }
                        if (fCheckRoIndx && (pO->ut.oid.usAccess != MIBACCro)) {
                            yyerror("Row \"%s\" has index item \"%s\" whose access must be \"read-only\"",
                                    pT->pszName, pO->pszName);
                        }
                        if (pO->ut.oid.usDefVal != MIBDFVno) {
                            yyerror("Row \"%s\" has index item \"%s\" which must not have a DEFAULT value",
                                    pT->pszName, pO->pszName);
                        }
                        usSyntax = pO->ut.oid.rsyn.usSyntax;
                        usSizeRange = pO->ut.oid.rsyn.usSizeRange;
                        /* check integers to make sure that range is non-negative*/
                        if (usSyntax == MIBSYNint) {
                            if (fCheckISR && (usSizeRange != MIBSRpp)) {
                                yyerror("Row \"%s\" has index item \"%s\" which must have a non-negative range specified",
                                        pT->pszName, pO->pszName);
                            }
                        }
                        /* check octet strings to make sure that size specified */
                        else if (usSyntax == MIBSYNoctstr) {
                            if (fCheckISR && (usSizeRange == MIBSRno)) {
                                yyerror("Row \"%s\" has index item \"%s\" which must have a size specified",
                                        pT->pszName, pO->pszName);
                            }
                        }
                        /* checks for NOLENGTH */
                        if (pIndx->usItype == MIBITnlobj) {
                            /* check for syntax */
                            if ((usSyntax != MIBSYNoctstr) &&
                                    (usSyntax != MIBSYNoid)) {
                                yyerror("Row \"%s\" has index item \"%s\" with NOLENGTH specified, but is not OCTET STRING or OBJECT IDENTIFIER",
                                        pT->pszName, pO->pszName);
                            }
                            /* check for last item */
                            if (pIndx->pNext != NULL) {
                                yyerror("Row \"%s\" has index item \"%s\" which NOLENGTH specifed which is not last item",
                                        pT->pszName, pO->pszName);
                            }
                        }
                    }
                }
            }
            /* no defval */
            if (pT->ut.oid.usDefVal != MIBDFVno) {
                yyerror("Row \"%s\" can have not a DEFAULT value specified",
                        pT->pszName);
            }
            break;

        case MIBOTobj:
            /* SNMP object */
            /* syntax is not "seq of" or "seq" */
            if ((pT->ut.oid.syn.usSyntax == MIBSYNseq) ||
                    (pT->ut.oid.syn.usSyntax == MIBSYNseqOf)) {
                yyerror("Object \"%s\" has invalid syntax",
                        pT->pszName);
            }
            /* check if in table (ie parent is a row) */
            pSeq = NULL;
            if ((pO != NULL) && (pO->ut.oid.usOType == MIBOTrow)) {
                /* get sequence containing item */
                pSeq = pO->ut.oid.syn.usi.pSeq;
            } else if ((pO != NULL) && (pO->ut.oid.usOType != MIBOToid)) {
                /* parent not row or OID */
                yyerror("Object \"%s\" must be registered under a row or an object identifier",
                        pT->pszName);
            }
            /* check to see if in sequence */
            if (pSeq != NULL) {
                /* in a row, check if in sequence for row */
                for (pSi = pSeq->ut.seq.pSeqIL; pSi != NULL;
                        pSi = pSi->pNext) {
                    /* check for match in sequence with object */
                    if (pSi->pOid == pT)
                        break;
                }
                if (pSi == NULL) {
                    /* object not in sequence */
                    yyerror("Object \"%s\" is not in sequence for Row \"%s\"",
                            pT->pszName, pO->pszName);
                } else
                    pT->ut.oid.pDefSeq = pSeq;
            }

            usSyntax = pT->ut.oid.rsyn.usSyntax;
            fNoSR = ((usSizeRange = pT->ut.oid.rsyn.usSizeRange) == MIBSRno);

            /* check that size/range specified */
            if (fCheckSR && fNoSR) {
                if (usSyntax == MIBSYNoctstr) {
                    /* octet string - need size */
                    yywarning("\"%s\" should have SIZE specified",
                            pT->pszName);
                } else if (usSyntax == MIBSYNint) {
                    /* integer - need range */
                    yywarning("\"%s\" should have range specified",
                            pT->pszName);
                }
            }
            /* access is read-only or read-write */
            if ((pT->ut.oid.usAccess != MIBACCro) &&
                    (pT->ut.oid.usAccess != MIBACCrw) &&
                    (pT->ut.oid.usAccess != MIBACCbad) &&
                    !fAllowAccess) {
                yyerror("\"%s\" must have access of \"read-only\" or \"read-write\"",
                        pT->pszName);
            }
            /* status is mandatory, deprecated, or obsolete */
            if ((pT->usStatus != MIBSTAma)  &&
                    (pT->usStatus != MIBSTAde) &&
                    (pT->usStatus != MIBSTAob) &&
                    (pT->usStatus != MIBSTAbad) &&
                    !fAllowOpt) {
                yyerror("\"%s\" must have status of \"mandatory\", \"deprecated\", or \"obsolete\"",
                        pT->pszName);
            }
            /* check for index */
            if (pT->ut.oid.cIndx != 0) {
                /* has INDEX clause */
                if (!fAllowIndx)
                    yyerror("\"%s\" can not have INDEX items specified",
                            pT->pszName);
                else if (pSeq != NULL)
                    yyerror("\"%s\" can not have INDEX items specified and be in a sequence",
                            pT->pszName);
                else {
                    /* item has INDEX clause */
                    /* check that all items are syntax types
                        and NOLENGTH clause used correctly */
                    for (pIndx = pT->ut.oid.pIndxL; pIndx != NULL;
                            pIndx = pIndx->pNext) {
                        if ((pIndx->usItype == MIBITobj) ||
                               (pIndx->usItype == MIBITnlobj)) {
                            yyerror("\"%s\" has index item \"%s\" which is an object and not a syntax type",
                                    pT->pszName, (pIndx->pOid)->pszName);
                        }
                        else if (pIndx->usItype == MIBITnloct) {
                            /* index is no length OCTET STRING */
                            /* check that no items follow */
                            if (pIndx->pNext != NULL) {
                                yyerror("\"%s\" has index item NOLENGTH OCTET STRING which is not the last item",
                                        pT->pszName);
                                continue;
                            }
                        } else if (pIndx->usItype == MIBITnloid) {
                            /* index is no length OBJECT IDENTIFIER */
                            /* check that no items follow */
                            if (pIndx->pNext != NULL) {
                                yyerror("\"%s\" has index item NOLENGTH OBJECT IDENTIFIER which is not the last item",
                                        pT->pszName);
                                continue;
                            }
                        }
                    }
                }
            }
            /* defval must be compatible with type */
            if (pT->ut.oid.usDefVal != MIBDFVno) {
                /* default value specified */
                switch (usSyntax) {
                case MIBSYNbad:
                    /* bad syntax - ignore */
                    pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                    break;

                case MIBSYNint:
                    /* integer */
                    if ((pT->ut.oid.usDefVal != MIBDFVbstr) &&
                            (pT->ut.oid.usDefVal != MIBDFVhstr) &&
                            (pT->ut.oid.usDefVal != MIBDFVint) &&
                            (pT->ut.oid.usDefVal != MIBDFVneg)) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                        break;
                    }
                    /*?? put in checks for range */
                    break;

                case MIBSYNoctstr:
                    /* octet string */
                    if ((pT->ut.oid.usDefVal != MIBDFVstr) &&
                            (pT->ut.oid.usDefVal != MIBDFVbstr) &&
                            (pT->ut.oid.usDefVal != MIBDFVhstr)) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                        break;
                    }
                    /*?? put in checks for size */
                    break;

                case MIBSYNoid:
                    /* object identifier */
                    if (pT->ut.oid.usDefVal == MIBDFVoid) {
                        /* all done */
                        break;
                    }
                    if (pT->ut.oid.usDefVal != MIBDFVna) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                        break;
                    }
                    /* lookup name and see if item is an OID type */
                    pNa = StrTabSearch(pT->ut.oid.udv.pszDefVal);
                    pO = NULL;
                    if ((pNa != NULL) && (pNa->pSym != NULL) &&
                            ((pNa->pSym)->pMod == pMod)) {
                        pO = pNa->pSym;
                        if (pO->usType == MIBSYMimp) {
                            /* item is import */
                            pO->ut.imp.cUse++;
                            pO = pO->ut.imp.pImpSym;
                        }
                    }
                    if ((pO == NULL) || (pO->usType != MIBSYMoid)) {
                        yyerror("\"%s\" doesn't name OID object in current module",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                        break;
                    }
                    /* change defval type to OID */
                    pT->ut.oid.usDefVal = MIBDFVoid;
                    pT->ut.oid.udv.pOid = pO;
                    break;

                case MIBSYNipaddr:
                case MIBSYNnaddr:
                    /* ip or network address */
                    if (pT->ut.oid.usDefVal == MIBDFVint) {
                        if (pT->ut.oid.udv.ulDefVal != 0L) {
                            yyerror("\"%s\" has invalid default value",
                                    pT->pszName);
                            pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                            break;
                        }
                        /* change type to IPaddress */
                        pT->ut.oid.usDefVal = MIBDFVip;

                    } else if (pT->ut.oid.usDefVal == MIBDFVhstr) {
                        if (strlen(pT->ut.oid.udv.pszDefVal) != 8) {
                            yyerror("\"%s\" has invalid default value",
                                    pT->pszName);
                            pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                            break;
                        }
                        /* convert to IP address */
                        for (psz = pT->ut.oid.udv.pszDefVal, i = 0;
                                i < 4; i++) {
                            pT->ut.oid.udv.ipDefVal[i] = (BYTE)((hexVal(psz[2*i]) << 4) |
                                                         hexVal(psz[2*i+1]));
                        }
                        pT->ut.oid.usDefVal = MIBDFVip;

                    } else if (pT->ut.oid.usDefVal != MIBDFVip) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                            pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                            break;
                    } else {
                        /*?? put in checks for value */
                    }
                    break;

                case MIBSYNenum:
                    /* enumerated */
                    if (pT->ut.oid.usDefVal != MIBDFVna) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                        break;
                    }
                    /* check that name is one of the enumerated values */
                    if (pT->ut.oid.syn.usSyntax == MIBSYNtc)
                        pE = (pT->ut.oid.syn.usi.pTC)->ut.tc.syn.usi.pEnumL;
                    else
                        pE = pT->ut.oid.syn.usi.pEnumL;
                    for (;pE != NULL; pE = pE->pNext) {
                        if (strcmp(pT->ut.oid.udv.pszDefVal,
                                pE->pszName) == 0)
                            break;
                    }
                    if (pE == NULL) {
                        yyerror("\"%s\" has invalid default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                        break;
                    }
                    break;
                    
                case MIBSYNcntr:
                case MIBSYNgauge:
                    /* counter or gauge */
                    if (!fAllowDV) {
                        if (((pT->ut.oid.usDefVal == MIBDFVbstr) ||
                             (pT->ut.oid.usDefVal == MIBDFVhstr) ||
                             (pT->ut.oid.usDefVal == MIBDFVint)  ||
                             (pT->ut.oid.usDefVal == MIBDFVneg)) &&
                                (pT->ut.oid.udv.ulDefVal == 0L)) {
                            yywarning("\"%s\" has gratuitous default value",
                                    pT->pszName);
                        } else {
                            yyerror("\"%s\" should not have a default value specified",
                                    pT->pszName);
                            pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                        }
                    }
                    else if ((pT->ut.oid.usDefVal != MIBDFVbstr)  &&
                             (pT->ut.oid.usDefVal != MIBDFVhstr) &&
                             (pT->ut.oid.usDefVal != MIBDFVint)) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;/* set to no value */
                    }
                    break;

                case MIBSYNticks:
                    /* time ticks */
                    if ((pT->ut.oid.usDefVal != MIBDFVbstr) &&
                            (pT->ut.oid.usDefVal != MIBDFVhstr) &&
                            (pT->ut.oid.usDefVal != MIBDFVint)) {
                        yyerror("\"%s\" has invalid type for default value",
                                pT->pszName);
                        pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                        break;
                    }
                    /*?? put in checks for range */
                    break;

                default:
                    /* all other types */
                    yyerror("\"%s\" must not have default value specified",
                            pT->pszName);
                    pT->ut.oid.usDefVal = MIBDFVno;  /* set to no value */
                    break;
                }
            }
            break;

        case MIBOTunk:   /* unknown or bad */
        default:
            break;
        }
    }

} /* checkOIDs */


/* end of file: SMOID.C */
