/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMDUMP.C - functions to dump the various lists in the compiler
 *                  and their necessary support routines.
 *
 * $Revision:   1.4  $ $Date:   08 Jul 1992 17:29:08  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMDUMP.C_V  $
 * 
 *    Rev 1.4   08 Jul 1992 17:29:08   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.3   29 Jun 1992 20:53:50   gfoster
 * Removed the unreferenced local variable pD from
 * the printNameOID function.
 * 
 *    Rev 1.2   29 Jun 1992 19:34:26   gfoster
 * Changed dump of names to always print "**dup**"
 * if fDup is set (this means that the back-end
 * should be changed to clear and set fDup based
 * on include/exclude flag).
 * 
 * Changed dump of names so that if parent is not
 * defined, then "**not defined**" is printed
 * as the OID value instead of 4gig.
 * 
 *    Rev 1.1   19 Jun 1992 15:59:44   gfoster
 * Copyright text was reformated.
 * 
 * The misspelling of "not-accessible" was corrected.
 * 
 * Dump of Opaque will print size, if given in the MIB.
 * 
 * Dumps work with excluded objects.  The OID list,
 * name list, and MOSY lists don't print the excluded
 * objects, dumps do.
 * 
 * Corrects print of OID value if not defined.
 * 
 * The OID list also prints the traps in the MIB.
 * 
 *    Rev 1.0   27 May 1992 16:16:34   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <string.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/** oidType - return values of OID type
*
* call with:
*   usOType
*
* returns:
*   ptr to name of type
*/
    PSZ
#ifdef __STDC__
oidType(USHORT usOType)
#else
oidType(usOType)
    USHORT usOType;
#endif /* __STDC__ */
{
    switch(usOType) {
    case MIBOToid:
        return("oid");
    case MIBOTtab:
        return("table");
    case MIBOTrow:
        return("row");
    case MIBOTobj:
        return("object");
    case MIBOTunk:
        return("**unknown or bad**");
    default:
        sprintf(szBuf, "**unknown type(%d)**", usOType);
        return(&szBuf[0]);
    }
} /* oidType */


/** accessType - return values of access
*
* call with:
*   usAccess - access type
*
* returns:
*   ptr to name of type
*/
    PSZ
#ifdef __STDC__
accessType(USHORT usAccess)
#else
accessType(usAccess)
    USHORT usAccess;
#endif /* __STDC__ */
{
    switch(usAccess) {
    case MIBACCnu:
        return("not used");
    case MIBACCbad:
        return("**bad**");
    case MIBACCro:
        return("read-only");
    case MIBACCrw:
        return("read-write");
    case MIBACCwo:
        return("write-only");
    case MIBACCna:
        return("not-accessible");
    default:
        sprintf(szBuf, "**unknown access(%d)**", usAccess);
        return(&szBuf[0]);
    }
} /* accessType */


/** fdumpSYN - dump syntax info to fp
*
* call with:
*   fp - file pointer
*   pSyn - syntax info
*/

    VOID
#ifdef __STDC__
fdumpSYN(FILE* fp, MIBSYN *pSyn)
#else
fdumpSYN(fp, pSyn)
    FILE* fp;
    MIBSYN *pSyn;
#endif /* __STDC__ */
{

    MIBENUM *pEnum;


    switch(pSyn->usSyntax) {
    case MIBSYNnu:
        fprintf(fp, "  **syntax not used");
        break;
    case MIBSYNbad:
        fprintf(fp, "  **bad syntax");
        break;
    case MIBSYNint:
        fprintf(fp, "  INTEGER%s", dumpSR(pSyn));
        break;
    case MIBSYNoctstr:
        fprintf(fp, "  OCTET STRING%s", dumpSR(pSyn));
        break;
    case MIBSYNoid:
        fprintf(fp, "  OBJECT IDENTIFIER");
        break;
    case MIBSYNnull:
        fprintf(fp, "  NULL");
        break;
    case MIBSYNnaddr:
        fprintf(fp, "  NetworkAddress");
        break;
    case MIBSYNipaddr:
        fprintf(fp, "  IpAddress");
        break;
    case MIBSYNcntr:
        fprintf(fp, "  Counter");
        break;
    case MIBSYNgauge:
        fprintf(fp, "  Gauge");
        break;
    case MIBSYNticks:
        fprintf(fp, "  TimeTicks");
        break;
    case MIBSYNopaque:
        fprintf(fp, "  Opaque%s", dumpSR(pSyn));
        break;
    case MIBSYNenum:
        fprintf(fp, "  enum(%u):\n", pSyn->cEnum);
        for (pEnum = pSyn->usi.pEnumL; pEnum != NULL; pEnum = pEnum->pNext) {
            fprintf(fp, "    %s(%lu)\n", pEnum->pszName, pEnum->ulVal);
        }
        break;
    case MIBSYNseqOf:
        fprintf(fp, "  SEQUENCE OF \"%s\"", (pSyn->usi.pSeq)->pszName);
        break;
    case MIBSYNseq:
        fprintf(fp, "  SEQUENCE \"%s\"", (pSyn->usi.pSeq)->pszName);
        break;
    case MIBSYNtc:
        fprintf(fp, "  [%s%s]->",
                (pSyn->usi.pTC)->pszName, dumpSR(pSyn));
        fdumpSYN(fp, &(pSyn->usi.pTC)->ut.tc.syn);
        break;
    default:
        fprintf(fp, "**unknown value(%d)", pSyn->usSyntax);
        break;
    }
} /* fdumpSYN */


/** dumpSYN - dump syntax info
*
* call with:
*   pSyn - syntax info
*/
    VOID
#ifdef __STDC__
dumpSYN(MIBSYN *pSyn)
#else
dumpSYN(pSyn)
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
  fdumpSYN(fhOut, pSyn);
  fprintf(fhOut, "\n");
}


/** prInst - print encoding for instance
*
* call with:
*  pIndx - list of index items
*
*/
    VOID
#ifdef __STDC__
prInst(MIBINDX *pIndx)
#else
prInst(pIndx)
    MIBINDX *pIndx;
#endif /* __STDC__ */
{
    USHORT i;
    USHORT n;

    
    fprintf(fhOut, "  instance encoding: ");
    while(pIndx != NULL) {
        switch (pIndx->usItype) {
        case MIBITbad:  /* bad index item */
            fprintf(fhOut, ".**BAD**");
            break;

        case MIBITobj:  /* index item is an object */
        case MIBITnlobj:/* index item is no length object */
            if ((pIndx->pOid)->ut.oid.usOType != MIBOTobj) {
                fprintf(fhOut, ".**bad**");
                break;
            }
            switch ((pIndx->pOid)->ut.oid.rsyn.usSyntax) {
            case MIBSYNnu:      /* not used */
            case MIBSYNbad:     /* invalid syntax specified */
            case MIBSYNnull:    /* NULL */
            case MIBSYNopaque:  /* opaque */
            case MIBSYNseqOf:   /* sequence of (table) */
            case MIBSYNseq:     /* sequence (row) */
            case MIBSYNtc:      /* textual convention */
            default:
                fprintf(fhOut, ".**BAD**");
                break;

            case MIBSYNint:     /* integer */
            case MIBSYNcntr:    /* counter */
            case MIBSYNgauge:   /* gauge */
            case MIBSYNticks:   /* timer tics */
            case MIBSYNenum:    /* enumerated */
                fprintf(fhOut, ".<int>");
                break;

            case MIBSYNoctstr:  /* octet string */
                if ((pIndx->pOid)->ut.oid.rsyn.usSizeRange == MIBSRfs) {
                    /* fixed size string */
                    for (i = 0, n = (pIndx->pOid)->ut.oid.rsyn.usr.usSize[0]; i < n; i++) {
                        fprintf(fhOut, ".<oct%d>", i+1);
                    }
                } else if (((pIndx->pOid)->ut.oid.rsyn.usSizeRange == MIBSRvs) ||
                            ((pIndx->pOid)->ut.oid.rsyn.usSizeRange == MIBSRno)) {
                    /* variable size */
                    if (pIndx->usItype == MIBITobj)
                        fprintf(fhOut, ".<len><com_per_octet>");
                    else
                        fprintf(fhOut, "<com_per_octet>");
                } else {
                    /* a problem */
                    fprintf(fhOut, ".***BAD***");
                }
                break;

            case MIBSYNoid:     /* object identifier */
                if (pIndx->usItype == MIBITobj)
                    fprintf(fhOut, ".<len>.<com_per_oid_com>");
                else
                    fprintf(fhOut, ".<com_per_oid_com>");
                break;

            case MIBSYNnaddr:   /* network address */
                fprintf(fhOut, ".<1>.<ip1>.<ip2>.<ip3>.<ip4>");
                break;

            case MIBSYNipaddr:  /* IP address */
                fprintf(fhOut, ".<ip1>.<ip2>.<ip3>.<ip4>");
                break;
            }
            break;

        case MIBITint:  /* index item is INTEGER */
            fprintf(fhOut, ".<int>");
            break;

        case MIBIToct:  /* index item is OCTET STRING */
            fprintf(fhOut, ".<len>.<com_per_octet>");
            break;

        case MIBITnloct:/* index item is no length OCTET STRING */
            fprintf(fhOut, ".<com_per_oct>");
            break;

        case MIBITfloct:/* index item is fixed length OCTET STRING */
            for (i = 0, n = (USHORT)(pIndx->ulLow); i < n; i++) {
                fprintf(fhOut, ".<oct%d>", i+1);
            }
            break;

        case MIBIToid:  /* index item is OBJECT IDENTIFIER */
            fprintf(fhOut, ".<len>.<com_per_oid_com>");
            break;

        case MIBITnloid:/* index item is no length OID */
            fprintf(fhOut, "<com_per_oid_com>");
            break;

        case MIBITnaddr:/* index item is NetworkAddress */
            fprintf(fhOut, ".<1>.<ip1>.<ip2>.<ip3>.<ip4>");
            break;

        case MIBITipaddr:/* index item is IpAddress */
            fprintf(fhOut, ".<ip1>.<ip2>.<ip3>.<ip4>");
            break;

        default:
            fprintf(fhOut, "**unknown type %d**",
                    pIndx->usItype);
            break;
        }

        pIndx = pIndx->pNext;
    }

    fprintf(fhOut, "\n");

} /* prInst */


/** prIndx - print index list
*
* call with:
*  cIndx - number of items in list
*  pIndx - list of index items
*
*/
    VOID
#ifdef __STDC__
prIndx(USHORT cIndx, MIBINDX *pIndx)
#else
prIndx(cIndx, pIndx)
    USHORT cIndx;
    MIBINDX *pIndx;
#endif /* __STDC__ */
{
    fprintf(fhOut, "  indices(%d):\n", cIndx);
    while (pIndx != NULL) {
        switch (pIndx->usItype) {
        case MIBITbad:  /* bad index item */
            fprintf(fhOut, "    **BAD**\n");
            break;

        case MIBITobj:  /* index item is an object */
            fprintf(fhOut, "    %s\n", (pIndx->pOid)->pszName);
            break;

        case MIBITnlobj:/* index item is no length object */
            fprintf(fhOut, "    NOLENGTH %s\n", (pIndx->pOid)->pszName);
            break;

        case MIBITint:  /* index item is INTEGER */
            if ((pIndx->ulLow == 0L) &&
                    (pIndx->ulHigh == 0xFFFFFFFFL))
                fprintf(fhOut, "    INTEGER\n");
            else
                fprintf(fhOut, "    INTEGER (%lu..%lu)\n",
                        pIndx->ulLow, pIndx->ulHigh);
            break;

        case MIBIToct:  /* index item is OCTET STRING */
            if ((pIndx->ulLow == 0L) &&
                    (pIndx->ulHigh == 0L))
                fprintf(fhOut, "    OCTET STRING\n");
            else
                fprintf(fhOut, "    OCTET STRING (SIZE(%lu..%lu))\n",
                        pIndx->ulLow, pIndx->ulHigh);
            break;

        case MIBITnloct:/* index item is no length OCTET STRING */
            if ((pIndx->ulLow == 0L) &&
                    (pIndx->ulHigh == 0L))
                fprintf(fhOut, "    NOLENGTH OCTET STRING\n");
            else
                fprintf(fhOut, "    NOLENGTH OCTET STRING (SIZE(%lu..%lu))\n",
                        pIndx->ulLow, pIndx->ulHigh);
            break;

        case MIBITfloct:/* index item is fixed length OCTET STRING */
            fprintf(fhOut, "    OCTET STRING (SIZE(%lu))\n",
                    pIndx->ulLow);
            break;

        case MIBIToid:  /* index item is OBJECT IDENTIFIER */
            fprintf(fhOut, "    OBJECT IDENTIFIER\n");
            break;

        case MIBITnloid:/* index item is no length OID */
            fprintf(fhOut, "    NOLENGTH OBJECT IDENTIFIER\n");
            break;

        case MIBITnaddr:/* index item is NetworkAddress */
            fprintf(fhOut, "    NetworkAddress\n");
            break;

        case MIBITipaddr:/* index item is IpAddress */
            fprintf(fhOut, "    IpAddress\n");
            break;

        default:
            fprintf(fhOut, "    **unknown type %d**\n",
                    pIndx->usItype);
            break;
        }
        pIndx = pIndx->pNext;
    }

} /* prIndx */


/** dumpOID - dump OID info
*
*/
    VOID
#ifdef __STDC__
dumpOID(VOID)
#else
dumpOID()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;


    fprintf(fhOut, "\nOID object info\n");

    for (pT = pOidAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d oid objects in avail list\n", i, MIBOIDACT);

    for (pT = pOidGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d oid objects specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d oid objects excluded\n", j);


    for (pT = pOidGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
        if (pT->ut.oid.fDup)
            fprintf(fhOut, "  **has dup**\n");
        fprintf(fhOut, "  type: %s\n",
                oidType(pT->ut.oid.usOType));
        if ((pT->ut.oid.pPar == NULL) &&
                (pT->ut.oid.ulVal == 0xFFFFFFFFL))
            fprintf(fhOut, "  val: --not defined--\n");
        else
            fprintf(fhOut, "  val: %s.%lu\n", 
                    (pT->ut.oid.pPar == NULL) ?
                        "--none--" : (pT->ut.oid.pPar)->pszName,
                                    pT->ut.oid.ulVal);
        switch(pT->ut.oid.usOType) {
        case MIBOToid:
            break;

        case MIBOTtab:
            fprintf(fhOut, "  syntax:");
            dumpSYN(&(pT->ut.oid.syn));
            break;

        case MIBOTrow:
            fprintf(fhOut, "  syntax:");
            dumpSYN(&(pT->ut.oid.syn));
            prIndx(pT->ut.oid.cIndx, pT->ut.oid.pIndxL);
            if (pT->ut.oid.pIndxL != NULL)
                prInst(pT->ut.oid.pIndxL);
            break;

        case MIBOTobj:
            fprintf(fhOut, "  syntax:");
            dumpSYN(&(pT->ut.oid.syn));
            if (pT->ut.oid.syn.usSyntax != pT->ut.oid.rsyn.usSyntax) {
                fprintf(fhOut, "  resynt:");
                dumpSYN(&(pT->ut.oid.rsyn));
            }
            fprintf(fhOut, "  access: %s\n", accessType(pT->ut.oid.usAccess));
            if ((pT->ut.oid.pDefSeq == NULL) &&
                    (pT->ut.oid.cIndx == 0)) {
                /* no instance */
                fprintf(fhOut, "  instance: 0\n");
            } else if (pT->ut.oid.cIndx != 0) {
                /* instance determined by index clause */
                prIndx(pT->ut.oid.cIndx, pT->ut.oid.pIndxL);

                fprintf(fhOut, "  instance: from index clause\n");
                prInst(pT->ut.oid.pIndxL);
            } else {
                /* item in table */
                if ((pT->ut.oid.pDefSeq)->ut.seq.pRow != NULL)
                    fprintf(fhOut, "  instance: from row \"%s\"\n",
                            ((pT->ut.oid.pDefSeq)->ut.seq.pRow)->pszName);
                else
                    fprintf(fhOut, "  instance: from sequence \"%s\" (no row defined)\n",
                        (pT->ut.oid.pDefSeq)->pszName);
            }
            switch(pT->ut.oid.usDefVal) {
            case MIBDFVno:
                /* no default value */
                break;
            case MIBDFVbad:
                /* bad default value */
                fprintf(fhOut, "  df: **bad\n");
                break;
            case MIBDFVstr:
                /* string */
                fprintf(fhOut, "  df: \"%s\"\n", pT->ut.oid.udv.pszDefVal);
                break;
            case MIBDFVbstr:
                /* binary string */
                fprintf(fhOut, "  df: \'%s\'b\n", pT->ut.oid.udv.pszDefVal);
                break;
            case MIBDFVhstr:
                /* hex string */
                fprintf(fhOut, "  df: \'%s\'h\n", pT->ut.oid.udv.pszDefVal);
                break;
            case MIBDFVint:
                /* integer */
                fprintf(fhOut, "  df: %lu\n", pT->ut.oid.udv.ulDefVal);
                break;
            case MIBDFVneg:
                /* negative number */
                fprintf(fhOut, "  df: -%lu\n", pT->ut.oid.udv.ulDefVal);
                break;
            case MIBDFVna:
                /* name (enumerated value) */
                fprintf(fhOut, "  df: %s\n", pT->ut.oid.udv.pszDefVal);
                break;
            case MIBDFVip:
                /* ip address */
                fprintf(fhOut, "  df: %u.%u.%u.%u\n",
                        pT->ut.oid.udv.ipDefVal[0],
                        pT->ut.oid.udv.ipDefVal[1],
                        pT->ut.oid.udv.ipDefVal[2],
                        pT->ut.oid.udv.ipDefVal[3]);
                break;
            case MIBDFVoid:
                /* object identifier */
                fprintf(fhOut, "  df: %s\n",
                        (pT->ut.oid.udv.pOid)->pszName);
                break;
            default:
                /* bad value */
                fprintf(fhOut, "  df: **unknown type(%d)**\n", pT->ut.oid.usDefVal);
                break;
            }
            break;

        case MIBOTunk:      /* unknown or bad */
        default:
            break;
        }
    }
    fprintf(fhOut, "**end of oid objects**\n");

} /* dumpOID */

/** printAbsoluteOID - print expanded OID
*
* call with:
*   pO - item
*/
    VOID
#ifdef __STDC__
printAbsoluteOID(MIBSYM *pO)
#else
printAbsoluteOID(pO)
    MIBSYM *pO;
#endif /* __STDC__ */
{
  USHORT usLevel=0;
  int i=0;
  ULONG  aulOid[MXOIDL];
  MIBSYM *pT=pO;

  while (pT != &OidRoot) {
    aulOid[usLevel++] = pT->ut.oid.ulVal;
    pT = pT->ut.oid.pPar;
    if (usLevel == MXOIDL) {
      fprintf(fhOut, "***OIDs nested too deep***\n");
      return;
    }
  }

  for (i=usLevel-1; i>=0; i--) {
    fprintf(fhOut,"%lu ", aulOid[i]);
  }
}

/** printOIDitem - print item for dump of OID tree
*
* call with:
*   pO - item
*   aulOid - vector of ancester OIDs
*   usLevel - number of parents
*/
    VOID
#ifdef __STDC__
printOIDitem(MIBSYM *pO, ULONG aulOid[], USHORT usLevel)
#else
printOIDitem(pO, aulOid, usLevel)
    MIBSYM *pO;
    ULONG aulOid[];
    USHORT usLevel;
#endif /* __STDC__ */
{
    USHORT j;

    /* check if excluded */
    if (pO->usNeed == MIBIEex)
        return;

    /* print OID of ancestors */
    for (j = 0; j < usLevel; j++) {
        fprintf(fhOut, "%lu.", aulOid[j]);
    }

    /* print object oid and type */
    if (pO->ut.oid.usOType == MIBOToid)
        fprintf(fhOut, "%lu  %s: reg point\n",
                pO->ut.oid.ulVal, pO->pszName);
    else if (pO->ut.oid.usOType == MIBOTunk)
        fprintf(fhOut, "%lu  %s: **unknown or bad**\n",
                pO->ut.oid.ulVal, pO->pszName);
    else {
        fprintf(fhOut, "%lu  %s: ", pO->ut.oid.ulVal, pO->pszName);
        dumpSYNshort(&(pO->ut.oid.syn));
    }

} /* printOIDitem */


/** dumpOIDtree - dump OID info in tree form
*
*/
    VOID
#ifdef __STDC__
dumpOIDtree(VOID)
#else
dumpOIDtree()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    SHORT k;
    MIBSYM *pT;
    MIBENT *pE;
    MIBSYM *pPar;
    ULONG aulOid[MXOIDL];


    /* print OIDs */
    fprintf(fhOut, "\nOID tree\n");
    walkOIDtree(printOIDitem);
    fprintf(fhOut, "**end of oid tree**\n");

    /* print traps */
    fprintf(fhOut, "\nTraps\n");
    for (pE = pEntGHL, i = 0; pE != NULL; pE = pE->pNext) {
        /* print traps having the same enterprise value */
        for (pT = pE->pTr, j = 0; pT != NULL; pT = pT->ut.tr.pEntTr) {
            if (pT->usNeed == MIBIEex)
                continue;
            if (j++ == 0) {
                aulOid[0] = (pE->pOid)->ut.oid.ulVal;
                for (pPar = (pE->pOid)->ut.oid.pPar, k = 1; (pPar != NULL) &&
                        (pPar != &OidRoot); pPar = pPar->ut.oid.pPar, k++) {
                    if (k == MXOIDL) {
                        fprintf(fhOut, "***OIDs nested too deep***\n");
                        return;
                    }
                    aulOid[k] = pPar->ut.oid.ulVal;
                }
                fprintf(fhOut, "  enterprise: %s  %lu",
                          (pE->pOid)->pszName, aulOid[--k]);
                for (k--; k != -1; k--) {
                    fprintf(fhOut, ".%lu", aulOid[k]);
                }
                fprintf(fhOut,"\n");
            }
            fprintf(fhOut, "    trap: %s, value %lu\n",
                    pT->pszName, pT->ut.tr.ulVal);
        }
        i += j;
    }
    if (i == 0)
        fprintf(fhOut, "  --None Defined\n");
    fprintf(fhOut, "**end of TRAPs**\n");

} /* dumpOIDtree */


/** mosySYN - return syntax info in mosy form
*
* call with:
*   pSyn - syntax info
*/
    PSZ
#ifdef __STDC__
mosySYN(MIBSYN *pSyn)
#else
mosySYN(pSyn)
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
    switch(pSyn->usSyntax) {
    case MIBSYNnu:
        return("**syntax not used**");
        break;

    case MIBSYNbad:
        return("**bad syntax");
        break;

    case MIBSYNint:
        return("INTEGER");
        break;

    case MIBSYNoctstr:
        return("OctetString");
        break;

    case MIBSYNoid:
        return("ObjectID");
        break;

    case MIBSYNnull:
        return("NULL");
        break;

    case MIBSYNnaddr:
        return("NetworkAddress");
        break;

    case MIBSYNipaddr:
        return("IpAddress");
        break;

    case MIBSYNcntr:
        return("Counter");
        break;

    case MIBSYNgauge:
        return("Gauge");
        break;

    case MIBSYNticks:
        return("TimeTicks");
        break;

    case MIBSYNopaque:
        return("Opaque");
        break;

    case MIBSYNenum:
        return("INTEGER");
        break;

    case MIBSYNseqOf:
        return("Aggregate");
        break;

    case MIBSYNseq:
        return("Aggregate");
        break;

    case MIBSYNtc:
        return((pSyn->usi.pTC)->pszName);
        break;

    default:
        return("**unknown syntax**");
        break;
    }
} /* mosySYN */


CHAR szMosy[80];
/** extMosySYN - return syntax info in extended mosy form
*
* call with:
*   pSyn - syntax info
*/
    PSZ
#ifdef __STDC__
extMosySYN(MIBSYN *pSyn)
#else
extMosySYN(pSyn)
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
    switch(pSyn->usSyntax) {
    case MIBSYNnu:
        return("**syntax not used**");
        break;

    case MIBSYNbad:
        return("**syntax bad");
        break;

    case MIBSYNint:
        if (pSyn->usSizeRange == MIBSRpp)
            sprintf(szMosy, "INTEGER (%lu..%lu)",
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
        else if (pSyn->usSizeRange == MIBSRnp)
            sprintf(szMosy, "INTEGER (-%lu..%lu)",
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
        else if (pSyn->usSizeRange == MIBSRnn)
            sprintf(szMosy, "INTEGER (-%lu..-%lu)",
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
        else
            strcpy(szMosy, "INTEGER");
        return(szMosy);
        break;

    case MIBSYNoctstr:
        if (pSyn->usSizeRange == MIBSRfs)
            sprintf(szMosy, "OctetString (SIZE (%u))",
                    pSyn->usr.usSize[0]);
        else if (pSyn->usSizeRange == MIBSRvs)
            sprintf(szMosy, "OctetString (SIZE (%u..%u))",
                    pSyn->usr.usSize[0], pSyn->usr.usSize[1]);
        else
            strcpy(szMosy, "OctetString");
        return(szMosy);
        break;

    case MIBSYNoid:
        return("ObjectID");
        break;

    case MIBSYNnull:
        return("NULL");
        break;

    case MIBSYNnaddr:
        return("NetworkAddress");
        break;

    case MIBSYNipaddr:
        return("IpAddress");
        break;

    case MIBSYNcntr:
        return("Counter");
        break;

    case MIBSYNgauge:
        return("Gauge");
        break;

    case MIBSYNticks:
        return("TimeTicks");
        break;

    case MIBSYNopaque:
        return("Opaque");
        break;

    case MIBSYNenum:
        return("Enum");
        break;

    case MIBSYNseqOf:
        return("Table");
        break;

    case MIBSYNseq:
        return("Row");
        break;

    case MIBSYNtc:
        switch (pSyn->usSizeRange) {
        case MIBSRpp:
            sprintf(szMosy, "%s (%lu..%lu)", (pSyn->usi.pTC)->pszName,
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
            break;

        case MIBSRnp:
            sprintf(szMosy, "%s (-%lu..%lu)", (pSyn->usi.pTC)->pszName,
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
            break;

        case MIBSRnn:
            sprintf(szMosy, "%s (-%lu..-%lu)", (pSyn->usi.pTC)->pszName,
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
            break;

        case MIBSRfs:
            sprintf(szMosy, "%s (SIZE (%u))", (pSyn->usi.pTC)->pszName,
                    pSyn->usr.usSize[0]);
            break;

        case MIBSRvs:
            sprintf(szMosy, "%s (SIZE (%u..%u))", (pSyn->usi.pTC)->pszName,
                    pSyn->usr.usSize[0], pSyn->usr.usSize[1]);
            break;

        case MIBSRno:
            sprintf(szMosy, "%s", (pSyn->usi.pTC)->pszName);
        }
        return(szMosy);
        break;

    default:
        return("**unknown syntax**");
        break;
    }
} /* extMosySYN */


MIBSYM *pMosyMod;
/** printMosyOidItem - print an OID object in MOSY format
*
* call with:
*   pO - item
*   aulOid - vector of OIDs
*   usLevel - number of parents
*/
    VOID
#ifdef __STDC__
printMosyOidItem(MIBSYM *pO, ULONG aulOid[], USHORT usLevel)
#else
printMosyOidItem(pO, aulOid, usLevel)
    MIBSYM *pO;
    ULONG aulOid[];
    USHORT usLevel;
#endif /* __STDC__ */
{
    if ((pO->pMod != pMosyMod) || (pO->ut.oid.usOType != MIBOToid) ||
            (pO->usNeed == MIBIEex))
        return;


    fprintf(fhOut, "%-20s ", pO->pszName);
    if (pO->ut.oid.pPar == &OidRoot)
        sprintf(szBuf, "%lu", pO->ut.oid.ulVal);
    else
        sprintf(szBuf, "%s.%lu",
                (pO->ut.oid.pPar)->pszName,
                pO->ut.oid.ulVal);

    fprintf(fhOut, "%s\n", szBuf);
    
} /* printMosyOidItem */


/** printMosyOtherItem - print an "Other" object in MOSY format
*
* call with:
*   pO - item
*   aulOid - vector of OIDs
*   usLevel - number of parents
*/
    VOID
#ifdef __STDC__
printMosyOtherItem(MIBSYM *pO, ULONG aulOid[], USHORT usLevel)
#else
printMosyOtherItem(pO, aulOid, usLevel)
    MIBSYM *pO;
    ULONG aulOid[];
    USHORT usLevel;
#endif /* __STDC__ */
{
    if ((pO->pMod != pMosyMod) || (pO->ut.oid.usOType == MIBOToid) ||
            (pO->usNeed == MIBIEex))
        return;


    fprintf(fhOut, "%-20s ", pO->pszName);
    if (pO->ut.oid.pPar == &OidRoot)
        sprintf(szBuf, "%lu", pO->ut.oid.ulVal);
    else
        sprintf(szBuf, "%s.%lu",
                (pO->ut.oid.pPar)->pszName,
                pO->ut.oid.ulVal);

    fprintf(fhOut, "%-16s %-15s %-15s %s\n", szBuf,
                mosySYN(&(pO->ut.oid.syn)),
                accessType(pO->ut.oid.usAccess),
                symStatus(pO->usStatus));

} /* printMosyOtherItem */


/** printExtMosyItem - print any object in Extended MOSY format
*
* call with:
*   pO - item
*   aulOid - vector of OIDs
*   usLevel - number of parents
*/
    VOID
#ifdef __STDC__
printExtMosyItem(MIBSYM *pO, ULONG aulOid[], USHORT usLevel)
#else
printExtMosyItem(pO, aulOid, usLevel)
    MIBSYM *pO;
    ULONG aulOid[];
    USHORT usLevel;
#endif /* __STDC__ */
{
    MIBINDX *pI;


    if (pO->usNeed == MIBIEex)
        return;


    fprintf(fhOut, "%-20s ", pO->pszName);
    if (pO->ut.oid.pPar == &OidRoot)
        sprintf(szBuf, "%lu", pO->ut.oid.ulVal);
    else
        sprintf(szBuf, "%s.%lu",
                (pO->ut.oid.pPar)->pszName,
                pO->ut.oid.ulVal);

    switch(pO->ut.oid.usOType) {

    case MIBOToid:
        /* Object ID */
        fprintf(fhOut, "%-16s regPt\n", szBuf);
        break;

    case MIBOTrow:
        /* a row in a table */
        fprintf(fhOut, "%-16s seq %s %u (", szBuf,
                (pO->ut.oid.syn.usi.pSeq)->pszName,
                pO->ut.oid.cIndx);
        for (pI = pO->ut.oid.pIndxL; pI != NULL; pI = pI->pNext) {
            switch (pI->usItype) {
            case MIBITbad:  /* bad index item */
                fprintf(fhOut, " **BAD**");
                break;

            case MIBITobj:  /* index item is an object */
                fprintf(fhOut, " %s", (pI->pOid)->pszName);
                break;

            case MIBITnlobj:/* index item is no length object */
                fprintf(fhOut, " NOLENGTH %s", (pI->pOid)->pszName);
                break;

            case MIBITint:  /* index item is INTEGER */
                if ((pI->ulLow == 0L) &&
                        (pI->ulHigh == 0xFFFFFFFFL))
                    fprintf(fhOut, " INTEGER");
                else
                    fprintf(fhOut, " INTEGER (%lu..%lu)",
                            pI->ulLow, pI->ulHigh);
                break;

            case MIBIToct:  /* index item is OCTET STRING */
                if ((pI->ulLow == 0L) &&
                        (pI->ulHigh == 0L))
                    fprintf(fhOut, " OCTET STRING");
                else
                    fprintf(fhOut, " OCTET STRING (SIZE(%lu..%lu))",
                                pI->ulLow, pI->ulHigh);
                break;

            case MIBITnloct:/* index item is no length OCTET STRING */
                if ((pI->ulLow == 0L) &&
                        (pI->ulHigh == 0L))
                    fprintf(fhOut, " NOLENGTH OCTET STRING");
                else
                    fprintf(fhOut, " NOLENGTH OCTET STRING (SIZE(%lu..%lu))",
                                pI->ulLow, pI->ulHigh);
                break;

            case MIBITfloct:/* index item is no length OCTET STRING */
                fprintf(fhOut, " OCTET STRING (SIZE (%lu))",
                        pI->ulLow);
                break;

            case MIBIToid:  /* index item is OBJECT IDENTIFIER */
                fprintf(fhOut, " OBJECT IDENTIFIER");
                break;

            case MIBITnloid:/* index item is no length OID */
                fprintf(fhOut, " NOLENGTH OBJECT IDENTIFIER");
                break;

            case MIBITnaddr:/* index item is NetworkAddress */
                fprintf(fhOut, " NetworkAddress");
                break;

            case MIBITipaddr:/* index item is IpAddress */
                fprintf(fhOut, " IpAddress");
                break;

            default:
                fprintf(fhOut, " **unknown type %d**",
                        pI->usItype);
                break;
            }
        }
        fprintf(fhOut, " ) %s %s\n", 
                accessType(pO->ut.oid.usAccess),
                symStatus(pO->usStatus));
        break;

    case MIBOTtab:
    case MIBOTobj:
        /* table or object */
        fprintf(fhOut, "%-16s %-15s %-15s %s\n", szBuf,
                extMosySYN(&(pO->ut.oid.syn)),
                accessType(pO->ut.oid.usAccess),
                symStatus(pO->usStatus));
        break;

    case MIBOTunk:
    default:
        /* unknown or bad or data structure error */
        fprintf(fhOut, "%-16s ** unknown or bad type\n", szBuf);
        break;
    }

} /* printExtMosyItem */


/** dumpMosy - dump Objects in "mosy" format
*
*/
    VOID
#ifdef __STDC__
dumpMosy(VOID)
#else
dumpMosy()
#endif /* __STDC__ */
{
    MIBSYM *pM;


    fprintf(fhOut, "\n-- Mosy format from\n");
    fprintf(fhOut, "--   %s\n\n", getVer());

    fprintf(fhOut, "\n-- From module%s:\n", (cModUsed > 1) ? "s" : "");

    for (pM = pModGHL; pM != NULL; pM = pM->pNext) {
        fprintf(fhOut, "--   %s\n", pM->pszName);
        pMosyMod = pM;
        walkOIDtree(printMosyOidItem);
        walkOIDtree(printMosyOtherItem);
    }

    fprintf(fhOut, "-- **end of Mosy oid tree**\n");


} /* dumpMosy */


/** dumpExtMosy - dump Objects in extended "mosy" format
*
*/
    VOID
#ifdef __STDC__
dumpExtMosy(VOID)
#else
dumpExtMosy()
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pO;
    MIBSYM *pS;
    MIBENUM *pE;
    MIBSEQI *pSi;
    USHORT i;

    fprintf(fhOut, "\n-- Extended Mosy format from\n");
    fprintf(fhOut, "--   %s\n\n", getVer());

    if (cEnumUsed > 0) {
        i = 0;
        /* check textual conventions */
        for (pT = pTcGHL; pT != NULL; pT = pT->pNext) {
            if (pT->usNeed == MIBIEex)
                continue;
            if (pT->ut.tc.syn.usSyntax == MIBSYNenum) {
                if (i++ == 0)
                    fprintf(fhOut, "\n-- Enumerations\n");
                fprintf(fhOut, "*%s %u (\n", pT->pszName, pT->ut.tc.syn.cEnum);
                for (pE = pT->ut.tc.syn.usi.pEnumL; pE != NULL;
                        pE = pE->pNext) {
                    fprintf(fhOut, "    %-20s %lu\n", pE->pszName, pE->ulVal);
                }
                fprintf(fhOut, ")\n");
            }
        }
        /* check objects */
        for (pO = pOidGHL; pO != NULL; pO = pO->pNext) {
            if (pO->usNeed == MIBIEex)
                continue;
            if (pO->ut.oid.syn.usSyntax == MIBSYNenum) {
                if (i++ == 0)
                    fprintf(fhOut, "\n-- Enumerations\n");
                fprintf(fhOut, "*%s %u (\n", pO->pszName, pO->ut.oid.syn.cEnum);
                for (pE = pO->ut.oid.syn.usi.pEnumL; pE != NULL;
                        pE = pE->pNext) {
                    fprintf(fhOut, "    %-20s %lu\n", pE->pszName, pE->ulVal);
                }
                fprintf(fhOut, ")\n");
            }
        }
        if (i != 0)
            fprintf(fhOut, "\n");
    }

    if (cTcUsed > 0) {
        for (pT = pTcGHL, i = 0; pT != NULL; pT = pT->pNext) {
            if (pT->usNeed == MIBIEex)
                continue;
            if (i++ == 0)
                fprintf(fhOut, "\n-- Textual Conventions\n");
            fprintf(fhOut, "+%-20s %s\n", pT->pszName,
                    extMosySYN(&(pT->ut.tc.syn)));
        }
        if (i != 0)
            fprintf(fhOut, "\n");
    }

    if (cSeqUsed > 0) {
        for (pS = pSeqGHL; pS != NULL; pS = pS->pNext) {
            if (pS->usNeed == MIBIEex)
                continue;
            if (i++ == 0)
                fprintf(fhOut, "-- Sequences\n");
            fprintf(fhOut, "$%s %u {\n", pS->pszName, pS->ut.seq.cSeqi);
            for (pSi = pS->ut.seq.pSeqIL; pSi != NULL; pSi = pSi->pNext) {
                fprintf(fhOut, "    %s\n", (pSi->pOid)->pszName);
            }
            fprintf(fhOut, "}\n");
        }
    }

    fprintf(fhOut, "\n-- Extended Mosy OID tree\n");
    walkOIDtree(printExtMosyItem);

    fprintf(fhOut, "\n-- **end of Extended Mosy OID tree**\n");

} /* dumpExtMosy */


/** printTNSDR - print type, name, status, description, and reference
*                in form for intermediate file
*
* call with:
*   pS - symbol
*/
    VOID
#ifdef __STDC__
printTNSDR(MIBSYM *pS)
#else
printTNSDR(pS)
    MIBSYM *pS;
#endif /* __STDC__ */
{
    CHAR chT;
    CHAR chS;

    switch(pS->usType) {
    case MIBSYMmod:
        /* module */
        chT = 'm';
        break;

    case MIBSYMoid:
        /* pure OID, Table, Row, or object */
        if (pS->ut.oid.usOType == MIBOToid)
            chT = 'o';
        else if (pS->ut.oid.usOType == MIBOTtab)
            chT = 't';
        else if (pS->ut.oid.usOType == MIBOTrow)
            chT = 'r';
        else if (pS->ut.oid.usOType == MIBOTobj)
            chT = 'b';
        else
            chT = '?';
        break;

    case MIBSYMseq:
        /* sequence */
        chT = 's';
        break;

    case MIBSYMtc:
        /* textual convention */
        chT = 'x';
        break;

    case MIBSYMtr:
        /* trap */
        chT = 'p';
        break;

    case MIBSYMimp:
        /* import */
        chT = 'i';
        break;

    case MIBSYMsmi:
        /* smi */
        chT = 'y';
        break;

    default:
        chT = '?';
    }

    switch(pS->usStatus) {
    case MIBSTAma:
        /* mandatory */
        chS = 'm';
        break;

    case MIBSTAop:
        /* optional */
        chS = 'o';
        break;

    case MIBSTAob:
        /* obsolete */
        chS = 'b';
        break;

    case MIBSTAde:
        /* deprecated */
        chS = 'd';
        break;
        
    case MIBSTAnu:
        /* not used */
        chS = 'n';
        break;

    default:
        chS = '?';
    }
        
    if ((pS->pszDesc == NULL) && (pS->pszRefer == NULL))
        fprintf(fhOut, "%c %s %c n",
                chT, pS->pszName, chS);
    else if ((pS->pszDesc == NULL) && (pS->pszRefer != NULL))
        fprintf(fhOut, "%c %s %c r \"%s\"",
                chT, pS->pszName, chS, pS->pszRefer);
    else if ((pS->pszDesc != NULL) && (pS->pszRefer == NULL))
        fprintf(fhOut, "%c %s %c d \"%s\"",
                chT, pS->pszName, chS, pS->pszDesc);
    else
        fprintf(fhOut, "%c %s %c b \"%s\" \"%s\"",
                chT, pS->pszName, chS, pS->pszDesc, pS->pszRefer);

} /* printTNSDR */


/** printSyntaxInfo - print syntax info in form for Intermediate file
*
* call with:
*   pSyn - syntax info
*/
    VOID
#ifdef __STDC__
printSyntaxInfo(MIBSYN *pSyn)
#else
printSyntaxInfo(pSyn)
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
    MIBENUM *pE;
    USHORT i;


    switch(pSyn->usSyntax) {
    case MIBSYNnu:
    case MIBSYNbad:
        fprintf(fhOut, " ?");
        break;

    case MIBSYNint:
        if (pSyn->usSizeRange == MIBSRpp)
            fprintf(fhOut, " i p %lu %lu",
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
        else if (pSyn->usSizeRange == MIBSRnp)
            fprintf(fhOut, " i z %lu %lu",
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
        else if (pSyn->usSizeRange == MIBSRnn)
            fprintf(fhOut, " i g %lu %lu",
                    pSyn->usr.ulRange[0], pSyn->usr.ulRange[1]);
        else
            fprintf(fhOut, " i n");
        break;

    case MIBSYNoctstr:
        if (pSyn->usSizeRange == MIBSRfs)
            fprintf(fhOut, " o f %u",
                    pSyn->usr.usSize[0]);
        else if (pSyn->usSizeRange == MIBSRvs)
            fprintf(fhOut, " o v %u %u",
                    pSyn->usr.usSize[0], pSyn->usr.usSize[1]);
        else
            fprintf(fhOut, " o n");
        break;

    case MIBSYNoid:
        fprintf(fhOut, " d n");
        break;

    case MIBSYNnull:
        fprintf(fhOut, " u n");
        break;

    case MIBSYNnaddr:
        fprintf(fhOut, " n n");
        break;

    case MIBSYNipaddr:
        fprintf(fhOut, " p n");
        break;

    case MIBSYNcntr:
        fprintf(fhOut, " c n");
        break;

    case MIBSYNgauge:
        fprintf(fhOut, " g n");
        break;

    case MIBSYNticks:
        fprintf(fhOut, " t n");
        break;

    case MIBSYNopaque:
        if (pSyn->usSizeRange == MIBSRfs)
            fprintf(fhOut, " q f %u",
                    pSyn->usr.usSize[0]);
        else if (pSyn->usSizeRange == MIBSRvs)
            fprintf(fhOut, " q v %u %u",
                    pSyn->usr.usSize[0], pSyn->usr.usSize[1]);
        else
            fprintf(fhOut, " q n");
        break;

    case MIBSYNenum:
        fprintf(fhOut, " e %u", pSyn->cEnum);
        for (i = 2, pE = pSyn->usi.pEnumL; pE != NULL; i++, pE = pE->pNext) {
            if (i%4 == 0)
                fprintf(fhOut, "\n   ");
            fprintf(fhOut, " %s %lu", pE->pszName, pE->ulVal);
        }
        break;

    case MIBSYNseqOf:
        fprintf(fhOut, " ?");
        break;

    case MIBSYNseq:
        fprintf(fhOut, " ?");
        break;

    case MIBSYNtc:
        switch (pSyn->usSizeRange) {
        case MIBSRpp:
            fprintf(fhOut, " x %s p %lu %lu",
                    (pSyn->usi.pTC)->pszName,
                    pSyn->usr.ulRange[0],
                    pSyn->usr.ulRange[1]);
            break;

        case MIBSRnp:
            fprintf(fhOut, " x %s z %lu %lu",
                    (pSyn->usi.pTC)->pszName,
                    pSyn->usr.ulRange[0],
                    pSyn->usr.ulRange[1]);
            break;

        case MIBSRnn:
            fprintf(fhOut, " x %s g %lu %lu",
                    (pSyn->usi.pTC)->pszName,
                    pSyn->usr.ulRange[0],
                    pSyn->usr.ulRange[1]);
            break;

        case MIBSRfs:
            fprintf(fhOut, " x %s f %u",
                    (pSyn->usi.pTC)->pszName,
                    pSyn->usr.usSize[0]);
            break;

        case MIBSRvs:
            fprintf(fhOut, " x %s v %u %u",
                    (pSyn->usi.pTC)->pszName,
                    pSyn->usr.usSize[0],
                    pSyn->usr.usSize[1]);
            break;

        case MIBSRno:
            fprintf(fhOut, " x %s n",
                    (pSyn->usi.pTC)->pszName);
        }
        break;

    default:
        fprintf(fhOut, " ?");
        break;
    }
} /* printSyntaxInfo */


/** printIntermInst - print instance for intermediate file
*
* call with:
*   pI - ptr to list of index items
*
*/
    VOID
#ifdef __STDC__
printIntermInst(MIBINDX *pI)
#else
printIntermInst(pI)
    MIBINDX *pI;
#endif /* __STDC__ */
{
    USHORT i;

    /* print names of items in index */
    for (i = 2; pI != NULL; i++, pI = pI->pNext) {
        if (i%4 == 0)
            fprintf(fhOut, "\n   ");
        switch (pI->usItype) {
        case MIBITbad:  /* bad index item */
            fprintf(fhOut, " 1");
            break;

        case MIBITobj:  /* index item is an object */
            fprintf(fhOut, " 2 %s", (pI->pOid)->pszName);
            break;

        case MIBITnlobj:/* index item is no length object */
            fprintf(fhOut, " 3 %s", (pI->pOid)->pszName);
            break;

        case MIBITint:  /* index item is INTEGER */
            fprintf(fhOut, " 4 %lu %lu",
                    pI->ulLow, pI->ulHigh);
            break;

        case MIBIToct:  /* index item is OCTET STRING */
            fprintf(fhOut, " 5 %lu %lu",
                    pI->ulLow, pI->ulHigh);
            break;

        case MIBITnloct:/* index item is no length OCTET STRING */
            fprintf(fhOut, " 6 %lu %lu",
                    pI->ulLow, pI->ulHigh);
            break;

        case MIBITfloct:/* index item is fixed size OCTET STRING */
            fprintf(fhOut, " 7 %lu", pI->ulLow);
            break;

        case MIBIToid:  /* index item is OBJECT IDENTIFIER */
            fprintf(fhOut, " 8");
            break;

        case MIBITnloid:/* index item is no length OID */
            fprintf(fhOut, " 9");
            break;

        case MIBITnaddr:/* index item is NetworkAddress */
            fprintf(fhOut, " A");
            break;

        case MIBITipaddr:/* index item is IpAddress */
            fprintf(fhOut, " B");
            break;

        default:
            fprintf(fhOut, " ?");
            break;
        }
    }

} /* printIntermInst */



MIBSYM *pModCur;

/** printIntermOID1 - print intermediate OID item in first form
*
* call with:
*   pT - item to print
*/
    VOID
#ifdef __STDC__
printIntermOID1(MIBSYM *pT, ULONG aulOid[], USHORT usLevel)
#else
printIntermOID1(pT, aulOid, usLevel)
    MIBSYM *pT;
    ULONG aulOid[];
    USHORT usLevel;
#endif /* __STDC__ */
{
#ifdef NEW
    MIBSYM *pS;
    MIBSEQI *pSi;
#endif

    if (pT->pMod != pModCur)
        return;

    printTNSDR(pT);

    if (pT->ut.oid.usOType == MIBOTtab) {
        /* table */
        /* print sequence name */
        fprintf(fhOut, " f %s", (pT->ut.oid.syn.usi.pSeq)->pszName);

    } else if (pT->ut.oid.usOType == MIBOTrow) {
        /* row */
        /* print sequence name and number of items in index */
        fprintf(fhOut, " s %s %u",
                (pT->ut.oid.syn.usi.pSeq)->pszName,
                pT->ut.oid.cIndx);
        /* print names of items in index */
        printIntermInst(pT->ut.oid.pIndxL);

#ifdef NEW
/**** start test ****/
/*?? NOTE: The backend parser needs to be changed for
           this to work! */
        /* print info for sequence */
        if ((pS = pT->ut.oid.syn.usi.pSeq) != NULL) {

            fprintf(fhOut, "\n");
            printTNSDR(pS);
            fprintf(fhOut, " %u", pS->ut.seq.cSeqi);
            for (i = 1, pSi = pS->ut.seq.pSeqIL; pSi != NULL;
                    i++, pSi = pSi->pNext) {
                if (i%2 == 0)
                    fprintf(fhOut, "\n   ");
                fprintf(fhOut, " %s", (pSi->pOid)->pszName);
                printSyntaxInfo(&(pSi->syn));
            }
        }
/**** end test ****/
#endif
    } else if (pT->ut.oid.usOType == MIBOTobj) {
        /* object */
        /* print access */
        switch(pT->ut.oid.usAccess) {
        case MIBACCnu:
            /* not used */
            fprintf(fhOut, " ?");
            break;

        case MIBACCro:
            /* read-only */
            fprintf(fhOut, " r");
            break;

        case MIBACCrw:
            /* read-write */
            fprintf(fhOut, " b");
            break;

        case MIBACCwo:
            /* write-only */
            fprintf(fhOut, " w");
            break;

        case MIBACCna:
            /* not-accessible */
            fprintf(fhOut, " n");
            break;

        default:
            fprintf(fhOut, " ?");
        }

        /* print syntax */
        printSyntaxInfo(&(pT->ut.oid.syn));
        /* printf defval */
        switch(pT->ut.oid.usDefVal) {
        case MIBDFVno:
            /* no default value */
            fprintf(fhOut, " u");
            break;

        case MIBDFVstr:
            /* string */
            fprintf(fhOut, " s \"%s\"",
                    pT->ut.oid.udv.pszDefVal);
            break;

        case MIBDFVbstr:
            /* binary string */
            fprintf(fhOut, " b \'%s\'B",
                    pT->ut.oid.udv.pszDefVal);
            break;

        case MIBDFVhstr:
            /* hex string */
            fprintf(fhOut, " h \'%s\'H",
                    pT->ut.oid.udv.pszDefVal);
            break;

        case MIBDFVint:
            /* integer */
            fprintf(fhOut, " i %lu",
                    pT->ut.oid.udv.ulDefVal);
            break;

        case MIBDFVneg:
            /* negative number */
            fprintf(fhOut, " g %lu",
                    pT->ut.oid.udv.ulDefVal);
            break;

        case MIBDFVna:
            /* name (enumerated value) */
            fprintf(fhOut, " n %s",
                    pT->ut.oid.udv.pszDefVal);
            break;

        case MIBDFVip:
            /* ip address */
            fprintf(fhOut, " p %u.%u.%u.%u",
                    pT->ut.oid.udv.ipDefVal[0],
                    pT->ut.oid.udv.ipDefVal[1],
                    pT->ut.oid.udv.ipDefVal[2],
                    pT->ut.oid.udv.ipDefVal[3]);
            break;

        case MIBDFVoid:
            /* oid */
            fprintf(fhOut, " o %s %s",
                    ((pT->ut.oid.udv.pOid)->pMod)->pszName,
                    (pT->ut.oid.udv.pOid)->pszName);
            break;

        default:
            /* bad value */
            fprintf(fhOut, " ?");
        break;
        }

        /* print INDEX clause */
        fprintf(fhOut, "  %u",
                pT->ut.oid.cIndx);
        if (pT->ut.oid.cIndx != 0)
            printIntermInst(pT->ut.oid.pIndxL);

    } else {
        /* oid (or unknown or bad) - nothing to do */
    }
    fprintf(fhOut, "\n");

}  /* printIntermOID1 */


USHORT usOldLevel;

/** printIntermOID2 - print intermediate OID item in second form
*
* call with:
*   pO - item
*   aulOid - vector of ancester OIDs
*   usLevel - number of parents
*/
    VOID
#ifdef __STDC__
printIntermOID2(MIBSYM *pO, ULONG aulOid[], USHORT usLevel)
#else
printIntermOID2(pO, aulOid, usLevel)
    MIBSYM *pO;
    ULONG aulOid[];
    USHORT usLevel;
#endif /* __STDC__ */
{
#ifdef OLD
    CHAR chL;


    if (usLevel > usOldLevel)
        chL = '>';
    else if (usLevel == usOldLevel)
        chL = '=';
    else {
        while (--usOldLevel > usLevel) {
            fprintf(fhOut, "<");
        }
        chL = '<';
    }
    usOldLevel = usLevel;

    fprintf(fhOut, "%c %lu %s:%s\n", chL,
            pO->ut.oid.ulVal, pO->pszName, (pO->pMod)->pszName);
#else
    fprintf(fhOut, "%s %s %lu",
            pO->pszName,
            (pO->pMod)->pszName,
            pO->ut.oid.ulVal);

    if (pO->ut.oid.pPar == &OidRoot)
        fprintf(fhOut, " _ _\n");
    else
        fprintf(fhOut, " %s %s\n",
                (pO->ut.oid.pPar)->pszName,
                ((pO->ut.oid.pPar)->pMod)->pszName);
#endif

} /* printIntermOID2 */



/*
 * include functions to generate GDMO according to the NMF IIMC specs
 *    James.Reilly@tel.vtt.fi
 */
#include "smdump_gdmo.ic"


/** dumpInterm - dump Objects in Intermediate format
*
*/
    VOID
#ifdef __STDC__
dumpInterm(VOID)
#else
dumpInterm()
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pM;
    MIBMODREF *pR;
    MIBIMPI *pI;
    USHORT i;
    MIBSEQI *pSi;
    MIBVAR *pV;

    fprintf(fhOut, "\n-- Intermediate format from\n");
    fprintf(fhOut, "--   %s\n\n", getVer());

    for (pM = pModGHL; pM != NULL; pM = pM->pNext) {
        printTNSDR(pM);
        fprintf(fhOut,"\n");

        /* Imports */
        for (pR = pM->ut.mod.pImodHL; pR != NULL; pR = pR->pNext) {
            fprintf(fhOut, "i %s %u", (pR->pMod)->pszName, pR->cImpi);
            for (i = 2, pI = pR->pImpIL; pI != NULL; i++, pI = pI->pNext) {
                if (i%4 == 0)
                    fprintf(fhOut, "\n   ");
                fprintf(fhOut, " %s", (pI->pNa)->pszVal);
            }
            fprintf(fhOut, "\n");
        }

        /* Smi items */
        for (pT = pSmiGHL; pT != NULL; pT = pT->pNext) {
            if (pT->pMod != pM)
                continue;

            printTNSDR(pT);
            fprintf(fhOut, "\n");
        }

        /* Textual Conventions */
        for (pT = pTcGHL; pT != NULL; pT = pT->pNext) {
            if (pT->pMod != pM)
                continue;

            printTNSDR(pT);

            printSyntaxInfo(&(pT->ut.tc.syn));
            fprintf(fhOut, "\n");
        }

        /* Sequences */
        for (pT = pSeqGHL; pT != NULL; pT = pT->pNext) {
            if (pT->pMod != pM)
                continue;

            printTNSDR(pT);

            fprintf(fhOut, " %u", pT->ut.seq.cSeqi);
            for (i = 1, pSi = pT->ut.seq.pSeqIL; pSi != NULL;
                    i++, pSi = pSi->pNext) {
                if (i%2 == 0)
                    fprintf(fhOut, "\n   ");
                fprintf(fhOut, " %s", (pSi->pOid)->pszName);
                printSyntaxInfo(&(pSi->syn));
            }
            fprintf(fhOut, "\n");
        }

        /* items identified with OIDs */
        pModCur = pM;
        walkOIDtree(printIntermOID1);

        /* traps */
        for (pT = pTrGHL; pT != NULL; pT = pT->pNext) {
            if (pT->pMod != pM)
                continue;

            printTNSDR(pT);

            fprintf(fhOut, " %s %lu %u",
                    (pT->ut.tr.pOid)->pszName,
                    pT->ut.tr.ulVal, pT->ut.tr.cVar);
            for (i = 2, pV = pT->ut.tr.pVarL; pV != NULL;
                    i++, pV = pV->pNext) {
                if (i%4 == 0)
                    fprintf(fhOut, "\n   ");
                fprintf(fhOut, " %s", (pV->pOid)->pszName);
            }
            fprintf(fhOut, "\n");
        }

/*?? this may need to be changed so that when processing
     each symbol type, the aliases for the symbol are
     printed immediately after the symbol */

        /* print aliases */
        for (pT = pAlGHL; pT != NULL; pT = pT->pNext) {
            if (pT->pMod == NULL) {
                if (pT->ut.al.pAlSym != pM)
                    continue;
                fprintf(fhOut, "1 %s %s\n",
                        pT->pszName, (pT->ut.al.pAlSym)->pszName);
            } else {
                if (pT->pMod != pM)
                    continue;
                fprintf(fhOut, "2 %s %s %s\n",
                        pT->pszName, (pT->pMod)->pszName,
                        (pT->ut.al.pAlSym)->pszName);
            }
        }
    }



    /* print separator */
    fprintf(fhOut, "z %u\n", cOidUsed);

    /* print OID tree */
    walkOIDtree(printIntermOID2);


} /* dumpInterm */



/** symFR - return forward reference flag
*
* call with:
*   usType - symbol type
*
* returns:
*   ptr to name of type
*/
    PSZ
#ifdef __STDC__
symFR(USHORT usType)
#else
symFR(usType)
    USHORT usType;
#endif /* __STDC__ */
{
    if (usType & MIBSYMFR)
        return("FR:");
    else
        return("");

} /* symFR */


/** symType - return value of type
*
* call with:
*   usType - symbol type
*
* returns:
*   ptr to name of type
*/
    PSZ
#ifdef __STDC__
symType(USHORT usType)
#else
symType(usType)
    USHORT usType;
#endif /* __STDC__ */
{
    switch(usType & MIBSYMmask) {
    case MIBSYMmod:
        return("module");
    case MIBSYMoid:
        return("OID object");
    case MIBSYMseq:
        return("sequence");
    case MIBSYMtc:
        return("textual convention");
    case MIBSYMtr:
        return("trap");
    case MIBSYMimp:
        return("import");
    case MIBSYMsmi:
        return("SMI");
    case MIBSYMalias:
        return("alias");
    default:
        return("**unknown**");
    }
} /* symType */


/** symStatus - return value of status
*
* call with:
*   usStatus
*
* returns:
*   ptr to name of type
*/
    PSZ
#ifdef __STDC__
symStatus(USHORT usStatus)
#else
symStatus(usStatus)
    USHORT usStatus;
#endif /* __STDC__ */
{
    switch(usStatus) {
    case MIBSTAnu:
        return("not used");
    case MIBSTAbad:
        return("**bad**");
    case MIBSTAma:
        return("mandatory");
    case MIBSTAop:
        return("optional");
    case MIBSTAob:
        return("obsolete");
    case MIBSTAde:
        return("deprecated");
    default:
        return("**unknown**");
    }
} /* symStatus */


/** printText - print formatted text. If text ptr is NULL then
*               don't print anything.
*
* ??NOTE: for now it just dumps the text. Fix so that the
*         text is wrapped
*
* call with:
*   pszLeader - leading text
*   pszText - bulk text
*
*/
    VOID
#ifdef __STDC__
printText(PSZ pszLeader, PSZ pszText)
#else
printText(pszLeader, pszText)
    PSZ pszLeader;
    PSZ pszText;
#endif /* __STDC__ */
{
    if (pszText == NULL)
        return;

    fprintf(fhOut, "%s%s\n", pszLeader, pszText);

} /* printText */


/** printNameOID - print name and OID of an item
*
* call with:
*   pSym - mib symbol
*/
    VOID
#ifdef __STDC__
printNameOID(MIBSYM *pSym)
#else
printNameOID(pSym)
    MIBSYM *pSym;
#endif /* __STDC__ */
{
    ULONG aulOid[MXOIDL];
    SHORT i;
    MIBSYM *pPar;

    while (pSym != NULL) {
        if ((pSym->usType == MIBSYMoid) && (pSym->usNeed != MIBIEex)) {
            aulOid[0] = pSym->ut.oid.ulVal;
            for (pPar = pSym->ut.oid.pPar, i = 1; (pPar != NULL) &&
                    (pPar != &OidRoot); pPar = pPar->ut.oid.pPar, i++) {
                if (i == MXOIDL) {
                    fprintf(fhOut, "***OIDs nested too deep***\n");
                    return;
                }
                aulOid[i] = pPar->ut.oid.ulVal;
            }
            i--;
            if (aulOid[i] == 0xFFFFFFFFL)
                fprintf(fhOut, "%-24s **not defined**",
                        pSym->pszName);
            else
                fprintf(fhOut, "%-24s %lu",
                        pSym->pszName, aulOid[i]);
            for (i--; i != -1; i--) {
                fprintf(fhOut, ".%lu", aulOid[i]);
            }
            if (pSym->ut.oid.fDup)
                fprintf(fhOut, "  **dup**");
            fprintf(fhOut, "\n");
        }
        pSym = pSym->pSym;
    }
} /* printNameOID */


/** OIDnameInOrder - print out names (and OID values) of OID items
*
* call with:
*   pStrTab - subtree of string table
*/
    VOID
#ifdef __STDC__
OIDnameInOrder(STRTAB *pCur)
#else
OIDnameInOrder(pCur)
    STRTAB *pCur;
#endif /* __STDC__ */
{
    /* check if anything to do */
    if (pCur == NULL)
        return;

    /* do left subtree */
    OIDnameInOrder(pCur->pLeft);

    /* do item */
    printNameOID(pCur->pSym);

    /* do right subtree */
    OIDnameInOrder(pCur->pRight);

} /* OIDnameInOrder */


/** symIE - return value for INCLUDE/EXCLUDE field
*
* call with:
*   usNeed - value
*
* returns:
*   ptr to value expressed as a string
*/
    PSZ
#ifdef __STDC__
symIE(USHORT usNeed)
#else
symIE(usNeed)
    USHORT usNeed;
#endif /* __STDC__ */
{
    switch(usNeed) {
    case MIBIEns:
        return("not specified");
    case MIBIEbad:
        return("**bad**");
    case MIBIEin:
        return("included");
    case MIBIEex:
        return("excluded");
    default:
        return("**unknown**");
    }
} /* symIE */



/** dumpOIDname - print out names (and OID values) of OID items
*
*/
    VOID
#ifdef __STDC__
dumpOIDname(VOID)
#else
dumpOIDname()
#endif /* __STDC__ */
{

    OIDnameInOrder(pStrTabRoot);

} /* dumpOIDname */


/** dumpSYM - dump symbol info
*
* call with:
*   pSym - symbol
*/
    VOID
#ifdef __STDC__
dumpSYM(MIBSYM *pSym)
#else
dumpSYM(pSym)
    MIBSYM *pSym;
#endif /* __STDC__ */
{
    MIBSYM *pAl;


    fprintf(fhOut, "%s: \"%s\"", symType(pSym->usType), pSym->pszName);
    if (pSym->pMod != NULL)
        fprintf(fhOut, " from \"%s\"\n", (pSym->pMod)->pszName);
    else
        fprintf(fhOut, "\n");
    if (pSym->pAlist != NULL) {
        for (pAl = pSym->pAlist; pAl != NULL; pAl = pAl->ut.al.pNext) {
            fprintf(fhOut, "  alias: %s\n", pAl->pszName);
        }
    }
    if (pSym->usNeed != MIBIEns)
        fprintf(fhOut, "  I/E: %s\n",
                    symIE(pSym->usNeed));
    if (pSym->usStatus != MIBSTAnu)
        fprintf(fhOut, "  status: %s%s\n",
                    symStatus(pSym->usStatus), symFR(pSym->usStatus));
    if ((pSym->pszDesc != NULL) && (*(pSym->pszDesc) != 0))
        printText("  descr: ", pSym->pszDesc);
    if ((pSym->pszRefer != NULL) && (*(pSym->pszRefer) != 0))
        printText("  refer: ", pSym->pszRefer);

} /* dumpSYM */


/** dumpMOD - dump module info
*
*/
    VOID
#ifdef __STDC__
dumpMOD(VOID)
#else
dumpMOD()
#endif /* __STDC__ */
{
    USHORT i, j, k;
    MIBSYM *pT;
    MIBMODREF *pR;
    MIBIMPI *pI;


    fprintf(fhOut, "\nDumping Module info\n");

    for (pR = pModRefAvail, i = 0; pR != NULL; pR = pR->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d module refs in avail list\n", i, MIBMODREFACT);

    for (pI = pImpiAvail, i = 0; pI != NULL; pI = pI->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d import items in avail list\n", i, MIBIMPIACT);

    for (pT = pModAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d modules in avail list\n", i, MIBMODACT);

    for (pT = pModGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d modules specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d modules excluded\n", j);


    for (pT = pModGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
        for (pR = pT->ut.mod.pImodHL, j = 0; pR != NULL; pR = pR->pNext) {
            j++;
            fprintf(fhOut, "    mod: \"%s\": %d items\n",
                    (pR->pMod)->pszName, pR->cImpi);
            for (pI = pR->pImpIL, k = 0; pI != 0; pI = pI->pNext) {
                k++;
                fprintf(fhOut, "        \"%s\"\n", (pI->pNa)->pszVal);
            }
        }
    }
    fprintf(fhOut, "**end of modules**\n");

} /* dumpMOD */


/** dumpAL - dump alias info
*
*/
    VOID
#ifdef __STDC__
dumpAL(VOID)
#else
dumpAL()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;


    fprintf(fhOut, "\nDumping Alias info\n");

    for (pT = pAlAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d aliases in avail list\n", i, MIBALACT);

    for (pT = pAlGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d aliases specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d aliases excluded\n", j);

    for (pT = pAlGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
        fprintf(fhOut, "    for: %s\n", (pT->ut.al.pAlSym)->pszName);
        fprintf(fhOut, "    use: %d\n", pT->ut.al.cUse);
    }
    fprintf(fhOut, "**end of aliases**\n");

} /* dumpAL */


/** dumpSMI - dump SMI info
*
*/
    VOID
#ifdef __STDC__
dumpSMI(VOID)
#else
dumpSMI()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;


    fprintf(fhOut, "\nDumping SMI item info\n");

    for (pT = pSmiAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d SMI items in avail list\n", i, MIBSMIACT);

    for (pT = pSmiGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d SMI items specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d SMI items excluded\n", j);


    for (pT = pSmiGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
    }
    fprintf(fhOut, "**end of smi items**\n");

} /* dumpSMI */


/** dumpIMP - dump import info
*
*/
    VOID
#ifdef __STDC__
dumpIMP(VOID)
#else
dumpIMP()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;


    fprintf(fhOut, "\nDumping IMPORT info\n");

    for (pT = pImpAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d IMPORT items in avail list\n", i, MIBIMPACT);

    for (pT = pImpGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed != MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d IMPORT items specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d IMPORT items excluded\n", j);


    for (pT = pImpGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
        fprintf(fhOut, "  defined in module \"%s\"\n",
            ((pT->ut.imp.pImpSym)->pMod)->pszName);
        fprintf(fhOut, "  use count: %d\n", pT->ut.imp.cUse);
    }
    fprintf(fhOut, "**end of IMPORT items**\n");

} /* dumpIMP */


/** dumpTRAP - dump traps
*
*/
    VOID
#ifdef __STDC__
dumpTRAP(VOID)
#else
dumpTRAP()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;
    MIBVAR *pV;
    MIBENT *pE;


    fprintf(fhOut, "\nDumping TRAP info\n");

    for (pT = pTrAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d TRAPs in avail list\n", i, MIBTRACT);

    for (pT = pTrGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d Traps specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d Traps excluded\n", j);



    for (pV = pVarAvail, i = 0; pV != NULL; pV = pV->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d TRAP vars in avail list\n", i, MIBVARACT);

    fprintf(fhOut, "%d Traps vars specified\n", cVarUsed);


    for (pE = pEntAvail, i = 0; pE != NULL; pE = pE->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d Enterprises in avail list\n", i, MIBENTACT);

    for (pE = pEntGHL, i = 0; pE != NULL; pE = pE->pNext) {
        i++;
    }
    fprintf(fhOut, "%d Trap Enterprises specified\n", i);

    if (cTrUsed != 0)
        fprintf(fhOut, "There are %d trap%s defined:\n",
                cTrUsed, (cTrUsed == 1) ? "" : "s");
    for (pT = pTrGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
        if (pT->ut.tr.pOid != NULL) {
            fprintf(fhOut, "  enterprise: \"%s\"",
                    (pT->ut.tr.pOid)->pszName);
            if ((pT->ut.tr.pOid)->pMod != NULL)
                fprintf(fhOut, " from \"%s\"\n",
                        ((pT->ut.tr.pOid)->pMod)->pszName);
            else
                fprintf(fhOut, "\n");
        }
        fprintf(fhOut, "  value: %lu\n", pT->ut.tr.ulVal);
        if (pT->ut.tr.cVar == 0)
            fprintf(fhOut, "  no vars\n");
        else {
            fprintf(fhOut, "  vars(%d):\n", pT->ut.tr.cVar);
            for (pV = pT->ut.tr.pVarL; pV != NULL; pV = pV->pNext) {
                fprintf(fhOut, "    %s\n", (pV->pOid)->pszName);
            }
        }
    }

    if (cEntUsed != 0)
        fprintf(fhOut, "There are %d trap enterprise%s used:\n",
                cEntUsed, (cEntUsed == 1) ? "" : "s");

    for (pE = pEntGHL, i = 0; pE != NULL; pE = pE->pNext) {
        i++;
        fprintf(fhOut, "  enterprise: \"%s\"\n", (pE->pOid)->pszName);
        for (pT = pE->pTr; pT != NULL; pT = pT->ut.tr.pEntTr) {
            if ((pT->ut.tr.pOid)->pMod != NULL)
                fprintf(fhOut, "  trap: \"%s\" from \"%s\", value %lu\n",
                    pT->pszName, (pT->pMod)->pszName,
                    pT->ut.tr.ulVal);
            else
                fprintf(fhOut, "  trap: \"%s\", value %lu\n",
                    pT->pszName, pT->ut.tr.ulVal);
        }
    }
    fprintf(fhOut, "**end of TRAPs*\n");


} /* dumpTRAP */


/** dumpSEQ - dump sequences
*
*/
    VOID
#ifdef __STDC__
dumpSEQ(VOID)
#else
dumpSEQ()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;
    MIBSEQI *pI;


    fprintf(fhOut, "\nDumping SEQUENCE info\n");

    for (pT = pSeqAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d SEQUENCEs in avail list\n", i, MIBSEQACT);

    for (pT = pSeqGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d Sequences specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d Sequences excluded\n", j);


    for (pI = pSeqiAvail, i = 0; pI != NULL; pI = pI->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d SEQUENCE items in avail list\n", i, MIBSEQIACT);

    fprintf(fhOut, "%d Sequences items specified\n", cSeqiUsed);


    for (pT = pSeqGHL, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
        dumpSYM((MIBSYM *)pT);
        fprintf(fhOut, "  use: %d\n", pT->ut.seq.cUse);
        if (pT->ut.seq.pRow != NULL)
            fprintf(fhOut, "  row: %s\n", (pT->ut.seq.pRow)->pszName);
        if (pT->ut.seq.cSeqi == 0)
            fprintf(fhOut, "  no items in sequence\n");
        else {
            fprintf(fhOut, "  items(%d):\n", pT->ut.seq.cSeqi);
            for (pI = pT->ut.seq.pSeqIL; pI != NULL; pI = pI->pNext) {
                fprintf(fhOut, "    %s ", (pI->pOid)->pszName);
                dumpSYNshort(&(pI->syn));
            }
        }
    }
    fprintf(fhOut, "**end of SEQUENCEs**\n");


} /* dumpSEQ */


/** dumpTC - dump textual conventions
*
*/
    VOID
#ifdef __STDC__
dumpTC(VOID)
#else
dumpTC()
#endif /* __STDC__ */
{
    USHORT i;
    USHORT j;
    MIBSYM *pT;


    fprintf(fhOut, "\nDumping Textual Convention info\n");

    for (pT = pTcAvail, i = 0; pT != NULL; pT = pT->pNext) {
        i++;
    }
    fprintf(fhOut, "%d of %d TCs in avail list\n", i, MIBTCACT);


    for (pT = pTcGHL, i = 0, j = 0; pT != NULL; pT = pT->pNext) {
        i++;
        if (pT->usNeed == MIBIEex)
            j++;
    }
    fprintf(fhOut, "%d TCs specified\n", i);
    if (j != 0)
        fprintf(fhOut, "%d TCs excluded\n", j);


    for (pT = pTcGHL; pT != NULL; pT = pT->pNext) {
        dumpSYM((MIBSYM *)pT);
        fprintf(fhOut, "  use: %d\n", pT->ut.tc.cUse);
        fprintf(fhOut, "  value: ");
        dumpSYNshort(&(pT->ut.tc.syn));
    }
    fprintf(fhOut, "**end of Textual Conventions**\n");


} /* dumpTC */


/* end of file: SMDUMP.C */

