/*
 * Copyright 1992 SynOptics Communications, Inc.  All Rights Reserved.
 * SynOptics grants a non-exclusive license to use, copy, modify, and
 * distribute this software for any purpose and without fee, provided
 * that this copyright notice and license appear on all copies and
 * supporting documentation.
 * SynOptics makes no representations about the suitability of this
 * software for any particular purpose.  The software is supplied
 * "AS IS", and SynOptics makes no warranty, either express or implied,
 * as to the use, operation, condition, or performance of the software.
 * SynOptics retains all title and ownership in the software.
 *
 * file: SMPARCOM.C - parsing support routines common to the compiler
 *                    and all backends.
 *
 * $Revision:   1.3  $ $Date:   08 Jul 1992 17:34:34  $
 * $Log:   R:/MIBTOOLS/V1.0/SMIC/SRC/SMPARCOM.C_V  $
 * 
 *    Rev 1.3   08 Jul 1992 17:34:34   gfoster
 * Removed unnecessary revision comment lines added by
 * PVCS to make revision history easier to read.
 * 
 *    Rev 1.2   29 Jun 1992 19:57:00   gfoster
 * Casted the value "-1L" to "(ULONG)-1L" to prevent
 * signed/unsigned mismatch when assigning "-1" to
 * the ulVal member of the MIBOID data structure.
 * 
 * Changed the incorrect escape sequences in the fprintf
 * statements from "\%s\ ..." to "\"%s\" ...".
 * 
 *    Rev 1.1   19 Jun 1992 16:17:32   gfoster
 * Copyright text was reformated.
 * 
 *    Rev 1.0   27 May 1992 16:12:24   gfoster
 * Initial revision.
 *
*/

#include <stdio.h>

#ifdef MS_DOS
#include <stdlib.h>
#endif /* MS_DOS */

#include <string.h>
#include <ctype.h>

#include "tds.h"
#include "smscdefs.h"
#include "smstdefs.h"
#include "smsydefs.h"
#include "smic.h"


/** initOIDroot - initialize the Root for the OID tree
*
*/
    VOID
#ifdef __STDC__
initOIDroot(VOID)
#else
initOIDroot()
#endif /* __STDC__ */
{
    OidRoot.usType = MIBSYMoid;
    OidRoot.pSym = NULL;
    OidRoot.pMod = NULL;
    OidRoot.pAlist = NULL;
    OidRoot.pszName = "OID_tree_Root";
    OidRoot.usNeed = MIBIEex;
    OidRoot.pBKEdata = NULL;
    OidRoot.usStatus = MIBSTAnu;
    OidRoot.pszDesc = NULL;
    OidRoot.pszRefer = NULL;
    OidRoot.pNext = NULL;

    OidRoot.ut.oid.fDup = FALSE;
    OidRoot.ut.oid.ulVal = 0L;
    OidRoot.ut.oid.pPar = NULL;
    OidRoot.ut.oid.pChild = NULL;
    OidRoot.ut.oid.pSib = NULL;
    OidRoot.ut.oid.usOType = MIBOToid;
    OidRoot.ut.oid.syn.usSyntax = MIBSYNnu;
    OidRoot.ut.oid.syn.usi.pSI = NULL;
    OidRoot.ut.oid.syn.usSizeRange = MIBSRno;
    OidRoot.ut.oid.usAccess = MIBACCnu;
    OidRoot.ut.oid.pIndxL = NULL;
    OidRoot.ut.oid.cIndx = 0;
    OidRoot.ut.oid.pDefSeq = NULL;
    OidRoot.ut.oid.rsyn.usSyntax = MIBSYNnu;
    OidRoot.ut.oid.rsyn.usi.pSI = NULL;
    OidRoot.ut.oid.rsyn.usSizeRange = MIBSRno;
    OidRoot.ut.oid.usDefVal = MIBDFVno;

} /* initOIDroot */


/** newOID - allocate a new oid object
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pNa - name of oid object
*   pMod - module containing object
*
* returns:
*   ptr to oid object or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
newOID(STRTAB *pNa, MIBSYM *pMod)
#else
newOID(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pOid;
    MIBSYM *pS;
    USHORT i;


    /* check if any available */
    if (pOidAvail == NULL) {
        /* none avail - so allocate some */
        pOidAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBOIDACT);
        if (pOidAvail == NULL) {
            yyerror("newOID: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cOidAlloc += MIBOIDACT;
        /* put oid objects in list */
        for (pT = pOidAvail, i = 1; i < MIBOIDACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get oid object from list */
    pOid = pOidAvail;
    pOidAvail = pOidAvail->pNext;
    cOidUsed++;

    /* init fields */
    pOid->usType = MIBSYMoid | MIBSYMFR; /* make forward reference */
    pOid->pSym = pNa->pSym;
    pNa->pSym = pOid;
    pOid->pMod = pMod;
    pOid->pAlist = NULL;
    pOid->pszName = pNa->pszVal;
    pOid->usStatus = MIBSTAnu;
    pOid->usNeed = MIBIEns;
    pOid->pBKEdata = NULL;
    pOid->pszDesc = NULL;
    pOid->pszRefer = NULL;
    pOid->pNext = NULL;

    pOid->ut.oid.fDup = FALSE;
    for (pS = pOid->pSym; pS != NULL; pS = pS->pSym) {
        if ((pS->usType & MIBSYMmask) == MIBSYMoid) {
            pOid->ut.oid.fDup = TRUE;
            pS->ut.oid.fDup = TRUE;
            break;
        }
    }
    pOid->ut.oid.ulVal = (ULONG)-1L;
    pOid->ut.oid.pPar = NULL;
    pOid->ut.oid.pChild = NULL;
    pOid->ut.oid.pSib = NULL;
    pOid->ut.oid.usOType = MIBOToid;
    pOid->ut.oid.syn.usSyntax = MIBSYNnu;
    pOid->ut.oid.syn.usi.pSI = NULL;
    pOid->ut.oid.syn.usSizeRange = MIBSRno;
    pOid->ut.oid.syn.usr.ulRange[0] = 0L;
    pOid->ut.oid.syn.usr.ulRange[1] = 0L;
    pOid->ut.oid.usAccess = MIBACCnu;
    pOid->ut.oid.pIndxL = NULL;
    pOid->ut.oid.cIndx = 0;
    pOid->ut.oid.pDefSeq = NULL;
    pOid->ut.oid.rsyn.usSyntax = MIBSYNnu;
    pOid->ut.oid.rsyn.usi.pSI = NULL;
    pOid->ut.oid.rsyn.usSizeRange = MIBSRno;
    pOid->ut.oid.rsyn.usr.ulRange[0] = 0L;
    pOid->ut.oid.rsyn.usr.ulRange[1] = 0L;
    pOid->ut.oid.usDefVal = MIBDFVno;
    pOid->ut.oid.udv.ulDefVal = 0L;

    /* link to end of global list of oid objects */
    if (pOidGHL == NULL) {
        /* list empty */
        pOidGHL = pOid;
    } else {
        /* add to end */
        pOidGTL->pNext = pOid;
    }
    pOidGTL = pOid;

    return(pOid);

} /* newOID */


/** newMOD - allocate a module
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pNa - name of module
*
* returns:
*   ptr to module name or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
newMOD(STRTAB *pNa)
#else
newMOD(pNa)
    STRTAB *pNa;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pMod;
    USHORT i;

    /* check if any available */
    if (pModAvail == NULL) {
        /* none avail - so allocate some */
        pModAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBMODACT);
        if (pModAvail == NULL) {
            yyerror("newMOD: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cModAlloc += MIBMODACT;
        /* put modules in list */
        for (pT = pModAvail, i = 1; i < MIBMODACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get module from list */
    pMod = pModAvail;
    pModAvail = pModAvail->pNext;
    cModUsed++;

    /* init fields */
    pMod->usType = MIBSYMmod;
    pMod->pSym = pNa->pSym;
    pNa->pSym = pMod;
    pMod->pMod = NULL;
    pMod->pAlist = NULL;
    pMod->pszName = pNa->pszVal;
    pMod->usStatus = MIBSTAnu;
    pMod->usNeed = MIBIEns;
    pMod->pBKEdata = NULL;
    pMod->pszDesc = NULL;
    pMod->pszRefer = NULL;
    pMod->pNext = NULL;

    pMod->ut.mod.pImodHL = NULL;
    pMod->ut.mod.pImodTL = NULL;

    /* link to end of global list of modules */
    if (pModGHL == NULL) {
        /* list empty */
        pModGHL = pMod;
    } else {
        /* add to end */
        pModGTL->pNext = pMod;
    }
    pModGTL = pMod;

    return(pMod);

} /* newMOD */


/** newMODREF - allocate a new module reference for an import
*
* call with:
*   pMod - module containing the import
*
* returns:
*   ptr to new module reference or NULL if an error
*/
    MIBMODREF *
#ifdef __STDC__
newMODREF(MIBSYM *pMod)
#else
newMODREF(pMod)
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBMODREF *pT;
    MIBMODREF *pModRef;
    USHORT i;

    /* check if any available */
    if (pModRefAvail == NULL) {
        /* none avail - so allocate some */
        pModRefAvail = (MIBMODREF *)malloc(sizeof(MIBMODREF)*MIBMODREFACT);
        if (pModRefAvail == NULL) {
            yyerror("newMODREF: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cModRefAlloc += MIBMODREFACT;
        /* put modules in list */
        for (pT = pModRefAvail, i = 1; i < MIBMODREFACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get module from list */
    pModRef = pModRefAvail;
    pModRefAvail = pModRefAvail->pNext;
    cModRefUsed++;

    /* init fields */
    pModRef->pMod = NULL;
    pModRef->pImpIL = NULL;
    pModRef->cImpi = 0;
    pModRef->pNext = NULL;

    /* add to tail of list for module */
    if (pMod->ut.mod.pImodHL == NULL) {
        /* list empty */
        pMod->ut.mod.pImodHL = pModRef;
    } else {
        /* add to end */
        (pMod->ut.mod.pImodTL)->pNext = pModRef;
    }
    pMod->ut.mod.pImodTL = pModRef;

    return(pModRef);

} /* newMODREF */


/** newIMPI - allocate an IMPORT item
*
* call with:
*   pModRef - import module
*   pNa - name of item
*
* returns:
*   ptr to new item or NULL if error
*/
    MIBIMPI *
#ifdef __STDC__
newIMPI(MIBMODREF *pModRef, STRTAB *pNa)
#else
newIMPI(pModRef, pNa)
    MIBMODREF *pModRef;
    STRTAB *pNa;
#endif /* __STDC__ */
{
    MIBIMPI *pT;
    MIBIMPI *pOT;
    MIBIMPI *pImpi;
    USHORT i;
    SHORT rt;


    /* check if any available */
    if (pImpiAvail == NULL) {
        /* none avail - so allocate some */
        pImpiAvail = (MIBIMPI *)malloc(sizeof(MIBIMPI)*MIBIMPIACT);
        if (pImpiAvail == NULL) {
            yyerror("newIMPI: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cImpiAlloc += MIBIMPIACT;
        /* put items in list */
        for (pT = pImpiAvail, i = 1; i < MIBIMPIACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* insert at end of list after checking for dups */
    for (pOT = NULL, pT = pModRef->pImpIL; pT != NULL;
            pOT = pT, pT = pT->pNext) {
        if ((rt = strcmp(pNa->pszVal, (pT->pNa)->pszVal)) == 0) {
            /* duplicate name */
            yyerror("duplicate name \"%s\" in IMPORTS",
                    pNa->pszVal);
#ifdef OLD
            yyterm();
#endif
            /*?? just ignore for now */
            return(NULL);
        }
    }

    /* get import item from list */
    pImpi = pImpiAvail;
    pImpiAvail = pImpiAvail->pNext;
    cImpiUsed++;
    ++(pModRef->cImpi);

    /* init fields */
    pImpi->pNa = pNa;

    if (pOT == NULL) {
        /* insert at beginning of list */
        pModRef->pImpIL = pImpi;
    } else {
        /* insert in list */
        pOT->pNext = pImpi;
    }
    pImpi->pNext = pT;

    return(pImpi);

} /* newIMPI */


/** newIMP - allocate new imported object
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pMod - module into which symbol is imported
*   pSym - originally defined symbol
*   pNa - place to insert in string table
*
* returns:
*   ptr to imported object or NULL if an error
*/
    MIBSYM *
#ifdef __STDC__
newIMP(MIBSYM *pMod, MIBSYM *pSym, STRTAB *pNa)
#else
newIMP(pMod, pSym, pNa)
    MIBSYM *pMod;
    MIBSYM *pSym;
    STRTAB *pNa;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pImp;
    USHORT i;


    /* check if any available */
    if (pImpAvail == NULL) {
        /* none avail - so allocate some */
        pImpAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBIMPACT);
        if (pImpAvail == NULL) {
            yyerror("newIMP: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cImpAlloc += MIBIMPACT;
        /* put items in list */
        for (pT = pImpAvail, i = 1; i < MIBIMPACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get import item from list */
    pImp = pImpAvail;
    pImpAvail = pImpAvail->pNext;
    cImpUsed++;

    /* init fields */
    pImp->usType = MIBSYMimp;
    pImp->pSym = pNa->pSym;
    pNa->pSym = pImp;
    pImp->pMod = pMod;
    pImp->pAlist = NULL;
    pImp->pszName = pNa->pszVal;
    pImp->usStatus = MIBSTAnu;
    pImp->usNeed = MIBIEns;
    pImp->pBKEdata = NULL;
    pImp->pszDesc = NULL;
    pImp->pszRefer = NULL;
    pImp->pNext = NULL;

    pImp->ut.imp.pImpSym = pSym;
    pImp->ut.imp.cUse = 0;

    /* link to end of global list of imports */
    if (pImpGHL == NULL) {
        /* list empty */
        pImpGHL = pImp;
    } else {
        /* add to end */
        pImpGTL->pNext = pImp;
    }
    pImpGTL = pImp;

    return(pImp);

} /* newIMP */


/** newAL - allocate new alias
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pSym - symbol being aliased
*   pAlNa - name of alias
*
* returns:
*   ptr to alias or NULL if an error
*/
    MIBSYM *
#ifdef __STDC__
newAL(MIBSYM *pSym, STRTAB *pAlNa)
#else
newAL(pSym, pAlNa)
    MIBSYM *pSym;
    STRTAB *pAlNa;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pAl;
    USHORT i;


    /* check if any available */
    if (pAlAvail == NULL) {
        /* none avail - so allocate some */
        pAlAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBALACT);
        if (pAlAvail == NULL) {
            yyerror("newAL: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cAlAlloc += MIBALACT;
        /* put items in list */
        for (pT = pAlAvail, i = 1; i < MIBALACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get alias item from list */
    pAl = pAlAvail;
    pAlAvail = pAlAvail->pNext;
    cAlUsed++;

    /* init fields */
    pAl->usType = MIBSYMalias;
    pAl->pSym = pAlNa->pSym;
    pAlNa->pSym = pAl;
    pAl->pMod = pSym->pMod;
    pAl->pAlist = NULL;
    pAl->pszName = pAlNa->pszVal;
    pAl->usStatus = MIBSTAnu;
    pAl->usNeed = MIBIEns;
    pAl->pBKEdata = NULL;
    pAl->pszDesc = NULL;
    pAl->pszRefer = NULL;
    pAl->pNext = NULL;

    pAl->ut.al.pAlSym = pSym;
    pAl->ut.al.cUse = 0;
    pAl->ut.al.pNext = pSym->pAlist;
    pSym->pAlist = pAl;

    /* link to end of global list of imports */
    if (pAlGHL == NULL) {
        /* list empty */
        pAlGHL = pAl;
    } else {
        /* add to end */
        pAlGTL->pNext = pAl;
    }
    pAlGTL = pAl;

    return(pAl);

} /* newAL */


/** addMODalias - add alias for module
*
* call with:
*   pNa - module name
*   pAlNa - alias for module
*
* returns:
*   TRUE - alias added
*   FALSE - an error adding alias
*/
    BOOL
#ifdef __STDC__
addMODalias(STRTAB *pNa, STRTAB *pAlNa)
#else
addMODalias(pNa, pAlNa)
    STRTAB *pNa;
    STRTAB *pAlNa;
#endif /* __STDC__ */
{
    MIBSYM *pMod;
    MIBSYM *pSym;
    MIBSYM *pAl;
    USHORT usType;


    /* check if module defined */
    for (pMod = pNa->pSym; pMod != NULL; pMod = pMod->pSym) {
        if ((usType = (pMod->usType & MIBSYMmask)) == MIBSYMmod)
            break;
        if ((usType == MIBSYMalias) &&
                (((pMod->ut.al.pAlSym)->usType & MIBSYMmask) == MIBSYMmod)) {
            /* alias to module */
            pMod->ut.al.cUse++;
            pMod = pMod->ut.al.pAlSym;
            break;
        }
    }
    if (pMod == NULL) {
        yyerror("module \"%s\" not defined - ignoring module alias",
                pNa->pszVal);
        return(FALSE);
    }

    /* check if alias already used */
    for (pSym = pAlNa->pSym; pSym != NULL; pSym = pSym->pSym) {
        if (((usType = (pSym->usType & MIBSYMmask)) == MIBSYMmod) ||
                ((usType == MIBSYMalias) &&
                (((pSym->ut.al.pAlSym)->usType & MIBSYMmask) == MIBSYMmod))) {
            yyerror("alias \"%s\" for module \"%s\" is already used - ignoring module alias",
                    pAlNa->pszVal, pNa->pszVal);
            return(FALSE);
        }
    }

    /* add new alias */
    pAl = newAL(pMod, pAlNa);

    return(TRUE);

} /* addMODalias */


/** addSYMalias - add alias for a symbol
*
* call with:
*   pModNa - module containing the symbol
*   pNa - name of symbol
*   pAlNa - alias for the symbol
*
* returns:
*   TRUE - alias added
*   FALSE - an error adding alias
*/
    BOOL
#ifdef __STDC__
addSYMalias(STRTAB *pModNa, STRTAB *pNa, STRTAB *pAlNa)
#else
addSYMalias(pModNa, pNa, pAlNa)
    STRTAB *pModNa;
    STRTAB *pNa;
    STRTAB *pAlNa;
#endif /* __STDC__ */
{
    MIBSYM *pMod;
    MIBSYM *pSym;
    MIBSYM *pAl;
    USHORT usType;


    /* check if module defined */
    for (pMod = pModNa->pSym; pMod != NULL; pMod = pMod->pSym) {
        if ((usType = (pMod->usType & MIBSYMmask)) == MIBSYMmod)
            break;
        if ((usType == MIBSYMalias) &&
                (((pMod->ut.al.pAlSym)->usType & MIBSYMmask) == MIBSYMmod)) {
            /* alias to module */
            pMod->ut.al.cUse++;
            pMod = pMod->ut.al.pAlSym;
            break;
        }
    }
    if (pMod == NULL) {
        yyerror("module \"%s\" not defined - ignoring symbol alias",
                pModNa->pszVal);
        return(FALSE);
    }


    /* check if alias already used */
    for (pSym = pAlNa->pSym; pSym != NULL; pSym = pSym->pSym) {
        if (((usType = (pSym->usType & MIBSYMmask)) == MIBSYMmod) ||
                ((usType == MIBSYMalias) &&
                (((pSym->ut.al.pAlSym)->usType & MIBSYMmask) == MIBSYMmod))) {
            yyerror("alias \"%s\" for module \"%s\" is already used - ignoring module alias",
                    pAlNa->pszVal, pNa->pszVal);
            return(FALSE);
        }
    }

    /* check if item defined in module and can be aliased */
    /* invalid symbols are imports and sequences */
    for (pSym = pNa->pSym; pSym != NULL; pSym = pSym->pSym) {
        if (pSym->pMod == pMod)
                break;
    }
    if (pSym == NULL) {
        yyerror("Item \"%s\" not defined in module \"%s\" - ignoring symbol alias",
                pNa->pszVal, pModNa->pszVal);
        return(FALSE);
    }

    /* check if item is an alias */
    if ((usType = (pSym->usType & MIBSYMmask)) == MIBSYMalias) {
        /* get real symbol and type */
        pSym->ut.al.cUse++;
        pSym = pSym->ut.al.pAlSym;
        usType = pSym->usType & MIBSYMmask;
    }

    /* check if item imported in original module */
    if (usType == MIBSYMimp) {
        yyerror("Item \"%s\" from module \"%s\" is imported - ignoring symbol alias",
                pNa->pszVal, pModNa->pszVal);
        return(FALSE);
    }

    /* check for sequence item */
    if (usType == MIBSYMseq) {
        yyerror("Item \"%s\" from module \"%s\" is a sequence - ignoring symbol alias",
                pNa->pszVal, pModNa->pszVal);
        return(FALSE);
    }

    /* check if SMI item */
    if (usType == MIBSYMsmi) {
        yyerror("Item \"%s\" from module \"%s\" is an SMI item - ignoring symbol alias",
                pNa->pszVal, pModNa->pszVal);
        return(FALSE);
    }

    /* add new alias */
    pAl = newAL(pSym, pAlNa);

    return(TRUE);


} /* addSYMalias */


/** newSMI - allocate a SMI name
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pNa - name to add
*   pMod - module containing name
*
* returns:
*   ptr to symbol or NULL for error
*/
    MIBSYM *
#ifdef __STDC__
newSMI(STRTAB *pNa, MIBSYM *pMod)
#else
newSMI(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    USHORT i;
    MIBSYM *pSmi;


    /* check if any available */
    if (pSmiAvail == NULL) {
        /* none avail - so allocate some */
        pSmiAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBSMIACT);
        if (pSmiAvail == NULL) {
            yyerror("newSMI: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cSmiAlloc += MIBSMIACT;
        /* put items in list */
        for (pT = pSmiAvail, i = 1; i < MIBSMIACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get SMI item from list */
    pSmi = pSmiAvail;
    pSmiAvail = pSmiAvail->pNext;
    cSmiUsed++;

    /* init fields */
    pSmi->usType = MIBSYMsmi;
    pSmi->pSym = pNa->pSym;
    pNa->pSym = pSmi;
    pSmi->pMod = pMod;
    pSmi->pAlist = NULL;
    pSmi->pszName = pNa->pszVal;
    pSmi->usStatus = MIBSTAnu;
    pSmi->usNeed = MIBIEns;
    pSmi->pBKEdata = NULL;
    pSmi->pszDesc = NULL;
    pSmi->pszRefer = NULL;
    pSmi->pNext = NULL;

    pSmi->ut.smi.iKW = 0;

    /* link to end of global list of smi objects */
    if (pSmiGHL == NULL) {
        /* list empty */
        pSmiGHL = pSmi;
    } else {
        /* add to end */
        pSmiGTL->pNext = pSmi;
    }
    pSmiGTL = pSmi;

    return(pSmi);

} /* newSMI */


/** newTR - allocate a trap
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pNa - name of trap
*   pMod - module containing trap
*
* returns:
*   ptr to trap or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
newTR(STRTAB *pNa, MIBSYM *pMod)
#else
newTR(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pTr;
    USHORT i;


    /* check if any available */
    if (pTrAvail == NULL) {
        /* none avail - so allocate some */
        pTrAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBTRACT);
        if (pTrAvail == NULL) {
            yyerror("newTR: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cTrAlloc += MIBTRACT;
        /* put traps in list */
        for (pT = pTrAvail, i = 1; i < MIBTRACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get trap from list */
    pTr = pTrAvail;
    pTrAvail = pTrAvail->pNext;
    cTrUsed++;

    /* init fields */
    pTr->usType = MIBSYMtr | MIBSYMFR; /* make forward reference */
    pTr->pSym = pNa->pSym;
    pNa->pSym = pTr;
    pTr->pMod = pMod;
    pTr->pAlist = NULL;
    pTr->pszName = pNa->pszVal;
    pTr->usStatus = MIBSTAnu;
    pTr->usNeed = MIBIEns;
    pTr->pBKEdata = NULL;
    pTr->pszDesc = NULL;
    pTr->pszRefer = NULL;
    pTr->pNext = NULL;

    pTr->ut.tr.pOid = NULL;
    pTr->ut.tr.pEntTr = NULL;
    pTr->ut.tr.pVarL = NULL;
    pTr->ut.tr.cVar = 0;
    pTr->ut.tr.ulVal = 0L;

    /* link to end of global list of traps */
    if (pTrGHL == NULL) {
        /* list empty */
        pTrGHL = pTr;
    } else {
        /* add to end */
        pTrGTL->pNext = pTr;
    }
    pTrGTL = pTr;

    return(pTr);

} /* newTR */


/** newENT - allocate an enterprise list item
*
* call with:
*   pTr - first trap to add
*
* returns:
*   ptr to Enterprise list item or NULL if error
*/
    MIBENT *
#ifdef __STDC__
newENT(MIBSYM *pFrTr)
#else
newENT(pFrTr)
    MIBSYM *pFrTr;
#endif /* __STDC__ */
{
    MIBENT *pT;
    MIBENT *pEnt;
    USHORT i;


    /* check if any available */
    if (pEntAvail == NULL) {
        /* none avail - so allocate some */
        pEntAvail = (MIBENT *)malloc(sizeof(MIBENT)*MIBENTACT);
        if (pEntAvail == NULL) {
            yyerror("newENT: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cEntAlloc += MIBENTACT;
        /* put enterprise list items in list */
        for (pT = pEntAvail, i = 1; i < MIBENTACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get enterprise item from list */
    pEnt = pEntAvail;
    pEntAvail = pEntAvail->pNext;
    cEntUsed++;

    /* init fields */
    pEnt->pOid = pFrTr->ut.tr.pOid;
    pEnt->pTr = pFrTr;
    pEnt->pNext = NULL;


    /* link to end of global list of enterprise items */
    if (pEntGHL == NULL) {
        /* list empty */
        pEntGHL = pEnt;
    } else {
        /* add to end */
        pEntGTL->pNext = pEnt;
    }
    pEntGTL = pEnt;

    return(pEnt);

} /* newENT */


/** newVAR - allocate a trap var
*
* call with:
*   pTr - trap
*   pOid - variable
*
* returns:
*   ptr to new item or NULL if error
*/
    MIBVAR *
#ifdef __STDC__
newVAR(MIBSYM *pTr, MIBSYM *pOid)
#else
newVAR(pTr, pOid)
    MIBSYM *pTr;
    MIBSYM *pOid;
#endif /* __STDC__ */
{
    MIBVAR *pT;
    MIBVAR *pOT;
    MIBVAR *pVar;
    USHORT i;
    SHORT rt;


    /* check if any available */
    if (pVarAvail == NULL) {
        /* none avail - so allocate some */
        pVarAvail = (MIBVAR *)malloc(sizeof(MIBVAR)*MIBVARACT);
        if (pVarAvail == NULL) {
            yyerror("newVAR: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cVarAlloc += MIBVARACT;
        /* put items in list */
        for (pT = pVarAvail, i = 1; i < MIBVARACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get trap var from list */
    pVar = pVarAvail;
    pVarAvail = pVarAvail->pNext;
    cVarUsed++;
    ++(pTr->ut.tr.cVar);

    /* init fields */
    pVar->pOid = pOid;
    /* insert at end of list after checking for dups */
    for (pOT = NULL, pT = pTr->ut.tr.pVarL; pT != NULL;
            pOT = pT, pT = pT->pNext) {
        if ((rt = strcmp(pOid->pszName, (pT->pOid)->pszName)) == 0) {
            /* duplicate name */
            yyerror("duplicate name \"%s\" in VARIABLES list",
                    pOid->pszName);
            yyterm();
            return(NULL);
        }
    }
    if (pOT == NULL) {
        /* insert at beginning of list */
        pTr->ut.tr.pVarL = pVar;
    } else {
        /* insert in list */
        pOT->pNext = pVar;
    }
    pVar->pNext = pT;

    return(pVar);

} /* newVAR */


/** addVAR - add variable to trap
*
* NOTE: the checking of the variable's type must be done
*       later after all objects are defined
*
* call with:
*   pTr - trap
*   pNa - variable
*   pMod - current module
*
* returns:
*   TRUE - no errors
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
addVAR(MIBSYM *pTr, STRTAB *pNa, MIBSYM *pMod)
#else
addVAR(pTr, pNa, pMod)
    MIBSYM *pTr;
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pOid;


    /* check if trap variable defined */
    pSym = pNa->pSym;
    if ((pSym != NULL) && (pSym->pMod == pMod)) {
        /* item defined */
        /* check if import */
        if (pSym->usType == MIBSYMimp) {
            pSym->ut.imp.cUse++;
            pSym = pSym->ut.imp.pImpSym;
        }
        /* check if OID type (ignore real checks) */
        if ((pSym->usType & MIBSYMmask) != MIBSYMoid) {
            yyerror("\"%s\" is not valid type for trap variable",
                    pSym->pszName);
#ifdef OLD
            yyterm();
#endif
            return(FALSE);
        }
        pOid = pSym;
    } else {
        /* item not defined */
        /* allocate new symbol */
        pOid = newOID(pNa, pMod);
    }

    /* allocate new trap variable */
    newVAR(pTr, pOid);

    return(TRUE);

} /* addVAR */


/** newTC - allocate a new textual convention object
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pNa - name of tc object
*   pMod - module containing object
*
* returns:
*   ptr to tc object or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
newTC(STRTAB *pNa, MIBSYM *pMod)
#else
newTC(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pTc;
    USHORT i;


    /* check if any available */
    if (pTcAvail == NULL) {
        /* none avail - so allocate some */
        pTcAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBTCACT);
        if (pTcAvail == NULL) {
            yyerror("newTC: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cTcAlloc += MIBTCACT;
        /* put tc objects in list */
        for (pT = pTcAvail, i = 1; i < MIBTCACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get tc object from list */
    pTc = pTcAvail;
    pTcAvail = pTcAvail->pNext;
    cTcUsed++;

    /* init fields */
    pTc->usType = MIBSYMtc;
    pTc->pSym = pNa->pSym;
    pNa->pSym = pTc;
    pTc->pMod = pMod;
    pTc->pAlist = NULL;
    pTc->pszName = pNa->pszVal;
    pTc->usStatus = MIBSTAnu;
    pTc->usNeed = MIBIEns;
    pTc->pBKEdata = NULL;
    pTc->pszDesc = NULL;
    pTc->pszRefer = NULL;
    pTc->pNext = NULL;

    memset(&(pTc->ut.tc.syn), 0, sizeof(MIBSYN));
    memset(&(pTc->ut.tc.rsyn), 0, sizeof(MIBSYN));
    pTc->ut.tc.cUse = 0;

    /* link to end of global list of tc objects */
    if (pTcGHL == NULL) {
        /* list empty */
        pTcGHL = pTc;
    } else {
        /* add to end */
        pTcGTL->pNext = pTc;
    }
    pTcGTL = pTc;

    return(pTc);

} /* newTC */


/** finishTC - finish definition of a textual convention
*
* call with:
*   pTc - tc object
*   pSyn - syntax info
*
* returns:
*   ptr to tc or NULL for error
*/
    MIBSYM *
#ifdef __STDC__
finishTC(MIBSYM *pTc, MIBSYN *pSyn)
#else
finishTC(pTc, pSyn)
    MIBSYM *pTc;
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
    /* store values */
    pTc->usType = MIBSYMtc;
    pTc->ut.tc.syn = *pSyn;     /* copy syntax info */

    /* get resolved syntax */
    if (pSyn->usSyntax == MIBSYNtc) {
        /* a TC, so get resolved syntax from TC's resolved syntax */
        pTc->ut.tc.rsyn = (pSyn->usi.pTC)->ut.tc.rsyn;
        /* add size/range if present */
        if ((pSyn->usSizeRange != MIBSRno) &&
                (pSyn->usSizeRange != MIBSRbad)) {
            pTc->ut.tc.rsyn.usSizeRange = pSyn->usSizeRange;
            pTc->ut.tc.rsyn.usr = pSyn->usr;
        }
    } else
        pTc->ut.tc.rsyn = *pSyn; /* syntax and resolved syntax are the same */


    return(pTc);

} /* finishTC */


/** newSEQ - allocate a new seq object
*
* ??NOTE: change to allocate only space needed
*
* call with:
*   pNa - name of seq object
*   pMod - module containing sequence
*
* returns:
*   ptr to seq object or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
newSEQ(STRTAB *pNa, MIBSYM *pMod)
#else
newSEQ(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pT;
    MIBSYM *pSeq;
    USHORT i;


    /* check if any available */
    if (pSeqAvail == NULL) {
        /* none avail - so allocate some */
        pSeqAvail = (MIBSYM *)malloc(sizeof(MIBSYM)*MIBSEQACT);
        if (pSeqAvail == NULL) {
            yyerror("newSEQ: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cSeqAlloc += MIBSEQACT;
        /* put seq objects in list */
        for (pT = pSeqAvail, i = 1; i < MIBSEQACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get seq object from list */
    pSeq = pSeqAvail;
    pSeqAvail = pSeqAvail->pNext;
    cSeqUsed++;

    /* init fields */
    pSeq->usType = MIBSYMseq | MIBSYMFR; /* make forward reference */
    pSeq->pSym = pNa->pSym;
    pNa->pSym = pSeq;
    pSeq->pMod = pMod;
    pSeq->pAlist = NULL;
    pSeq->pszName = pNa->pszVal;
    pSeq->usStatus = MIBSTAnu;
    pSeq->usNeed = MIBIEns;
    pSeq->pBKEdata = NULL;
    pSeq->pszDesc = NULL;
    pSeq->pszRefer = NULL;
    pSeq->pNext = NULL;

    pSeq->ut.seq.pSeqIL = NULL;
    pSeq->ut.seq.cSeqi = 0;
    pSeq->ut.seq.cUse = 0;
    pSeq->ut.seq.pRow = NULL;

    /* link to end of global list of sequences */
    if (pSeqGHL == NULL) {
        /* list empty */
        pSeqGHL = pSeq;
    } else {
        /* add to end */
        pSeqGTL->pNext = pSeq;
    }
    pSeqGTL = pSeq;

    return(pSeq);

} /* newSEQ */


/** newSEQI - allocate a Sequence item
*
* call with:
*   pSeq - sequence containing item
*   pOid - sequence item
*   pSyn - syntax
*
* returns:
*   ptr to new item or NULL if error
*/
    MIBSEQI *
#ifdef __STDC__
newSEQI(MIBSYM *pSeq, MIBSYM *pOid, MIBSYN *pSyn)
#else
newSEQI(pSeq, pOid, pSyn)
    MIBSYM *pSeq;
    MIBSYM *pOid;
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
    MIBSEQI *pT;
    MIBSEQI *pOT;
    MIBSEQI *pSeqi;
    USHORT i;
    SHORT rt;


    /* check if any available */
    if (pSeqiAvail == NULL) {
        /* none avail - so allocate some */
        pSeqiAvail = (MIBSEQI *)malloc(sizeof(MIBSEQI)*MIBSEQIACT);
        if (pSeqiAvail == NULL) {
            yyerror("newSEQI: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cSeqiAlloc += MIBSEQIACT;
        /* put items in list */
        for (pT = pSeqiAvail, i = 1; i < MIBSEQIACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get sequence item from list */
    pSeqi = pSeqiAvail;
    pSeqiAvail = pSeqiAvail->pNext;
    cSeqiUsed++;
    ++(pSeq->ut.seq.cSeqi);

    /* init fields */
    pSeqi->pOid = pOid;
    pSeqi->syn = *pSyn;
    /* insert at end of list after checking for dups */
    for (pOT = NULL, pT = pSeq->ut.seq.pSeqIL; pT != NULL;
            pOT = pT, pT = pT->pNext) {
        if ((rt = strcmp((pSeqi->pOid)->pszName,
                (pT->pOid)->pszName)) == 0) {
            /* duplicate name */
            yyerror("duplicate name \"%s\" in SEQUENCE",
                    (pSeqi->pOid)->pszName);
            yyterm();
            return(NULL);
        }
    }
    if (pOT == NULL) {
        /* insert at beginning of list */
        pSeq->ut.seq.pSeqIL = pSeqi;
    } else {
        /* insert at end of list */
        pOT->pNext = pSeqi;
    }
    pSeqi->pNext = pT;

    return(pSeqi);

} /* newSEQI */


/** addSEQitem - add item to sequence
*
* NOTE: can not determine if item is the proper type
*       at this point in time.
*       MUST check later!!
*
* call with:
*   pSeq - sequence
*   pNa - item
*   pSyn - syntax of item
*   pMod - current module
*
* returns:
*   TRUE - no errors
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
addSEQitem(MIBSYM *pSeq, STRTAB *pNa, MIBSYN *pSyn, MIBSYM *pMod)
#else
addSEQitem(pSeq, pNa, pSyn, pMod)
    MIBSYM *pSeq;
    STRTAB *pNa;
    MIBSYN *pSyn;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pOid;
    MIBSEQI *pSeqi;


    /* check if item already defined */
    pSym = pNa->pSym;
    if ((pSym != NULL) && (pSym->pMod == pMod)) {
        /* item already defined */
        /* check if import */
        if (pSym->usType == MIBSYMimp) {
            pSym->ut.imp.cUse++;
            pSym = pSym->ut.imp.pImpSym;
        }
        /* check if OID type (ignore real checks) */
        if ((pSym->usType & MIBSYMmask) != MIBSYMoid) {
            yyerror("\"%s\" is not valid type for sequence member",
                    pSym->pszName);
            yyterm();
            return(FALSE);
        }
        pOid = pSym;
    } else {
        /* item not defined */
        /* allocate new symbol */
        pOid = newOID(pNa, pMod);
    }


    /* allocate new sequence item */
    pSeqi = newSEQI(pSeq, pOid, pSyn);

    return(TRUE);
    
} /* addSEQitem */


/** addOTname - add snmp object (start definition)
*
* call with:
*   pNa - object name
*   pMod - containing module
*
* returns:
*   ptr to object or NULL if error
*/
    MIBSYM *
#ifdef __STDC__
addOTname(STRTAB *pNa, MIBSYM *pMod)
#else
addOTname(pNa, pMod)
    STRTAB *pNa;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pOid;


    /* check if item already defined */
    pSym = pNa->pSym;
    if ((pSym != NULL) && (pSym->pMod == pMod)) {
        /* item already defined */
        /* check if forward reference */
        if (pSym->usType != (MIBSYMFR|MIBSYMoid)) {
            yyerror("\"%s\" already defined in current module",
                    pSym->pszName);
#ifdef OLD
            yyterm();
#endif
            return(NULL);
        }
        pOid = pSym;
    } else {
        /* item not defined */
        /* allocate new symbol */
        pOid = newOID(pNa, pMod);
    }

    /* mark as OID, forward reference, and being defined */
    pOid->usType |= MIBSYMoid | MIBSYMFR | MIBSYMDF;

    return(pOid);

} /* addOTname */


/** addENUMitem - add enumerated item to syntax
*
* call with:
*   pNa - name of value
*   ulValue - value
*   pSyn - syntax to add item
*
* returns:
*   TRUE - no error
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
addENUMitem(STRTAB *pNa, ULONG ulVal, MIBSYN *pSyn)
#else
addENUMitem(pNa, ulVal, pSyn)
    STRTAB *pNa;
    ULONG ulVal;
    MIBSYN *pSyn;
#endif /* __STDC__ */
{
    MIBENUM *pT;
    MIBENUM *pOT;
    MIBENUM *pEnum;
    USHORT i;
    USHORT rt;


    /* check if any available */
    if (pEnumAvail == NULL) {
        /* none avail - so allocate some */
        pEnumAvail = (MIBENUM *)malloc(sizeof(MIBENUM)*MIBENUMACT);
        if (pEnumAvail == NULL) {
            yyerror("addENUMitem: out of heap memory");
            yystats();
            yyterm();
            return(FALSE);
        }
        cEnumAlloc += MIBENUMACT;
        /* put items in list */
        for (pT = pEnumAvail, i = 1; i < MIBENUMACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get enumerated item from list */
    pEnum = pEnumAvail;
    pEnumAvail = pEnumAvail->pNext;
    cEnumUsed++;

    /* init fields */
    pEnum->pszName = pNa->pszVal;
    pEnum->ulVal = ulVal;
    pEnum->pNext = NULL;
    pSyn->cEnum++;

    /* check for duplicate name or value */
    for (pOT = NULL, pT = pSyn->usi.pEnumL; pT != NULL;
            pOT = pT, pT = pT->pNext) {
        if ((rt = strcmp(pEnum->pszName, pT->pszName)) == 0) {
            /* duplicate name */
            yyerror("duplicate name \"%s\" in ENUMERATION",
                    pEnum->pszName);
#ifdef OLD
            yyterm();
            return(FALSE);
#endif
        }
        if (ulVal == pT->ulVal) {
            /* duplicate value */
            yyerror("duplicate value of %lu for \"%s\" and \"%s\" in ENUMERATION",
                    ulVal, pEnum->pszName, pT->pszName);
#ifdef OLD
            yyterm();
            return(FALSE);
#endif
        }
    }
    if (pOT == NULL) {
        /* insert at beginning of list */
        pSyn->usi.pEnumL = pEnum;
    } else {
        /* insert in list */
        pOT->pNext = pEnum;
    }

    /* check if name in lowercase */
    if (!islower(*(pEnum->pszName))) {
        yywarning("Name of enumerated value \"%s\" must start with a lowercase letter",
                pEnum->pszName);
    }

    return(TRUE);

} /* addENUMitem */


/** newINDX - allocate an Index item
*
* call with:
*   pOid - row object
*   usItype - index type
*   pOidIndx - index item (or NULL)
*   ulLow - low for range (or size for fixed length string)
*   ulHigh - high for range
*
* returns:
*   ptr to new item or NULL if error
*/
    MIBINDX *
#ifdef __STDC__
newINDX(MIBSYM *pOid, USHORT usItype, MIBSYM *pOidIndx,
        ULONG ulLow, ULONG ulHigh)
#else
newINDX(pOid, usItype, pOidIndx, ulLow, ulHigh)
    MIBSYM *pOid;
    USHORT usItype;
    MIBSYM *pOidIndx;
    ULONG ulLow;
    ULONG ulHigh;
#endif /* __STDC__ */
{
    MIBINDX *pT;
    MIBINDX *pOT;
    MIBINDX *pIndx;
    USHORT i;
    SHORT rt;
    BOOL fCkNa;


    /* check if any available */
    if (pIndxAvail == NULL) {
        /* none avail - so allocate some */
        pIndxAvail = (MIBINDX *)malloc(sizeof(MIBINDX)*MIBINDXACT);
        if (pIndxAvail == NULL) {
            yyerror("newINDX: out of heap memory");
            yystats();
            yyterm();
            return(NULL);
        }
        cIndxAlloc += MIBINDXACT;
        /* put items in list */
        for (pT = pIndxAvail, i = 1; i < MIBINDXACT; i++, pT++) {
            pT->pNext = pT+1;
        }
        pT->pNext = NULL;
    }

    /* get index item from list */
    pIndx = pIndxAvail;
    pIndxAvail = pIndxAvail->pNext;
    cIndxUsed++;
    ++(pOid->ut.oid.cIndx);

    /* init fields */
    pIndx->usItype = usItype;
    pIndx->pOid = pOidIndx;
    pIndx->ulLow = ulLow;
    pIndx->ulHigh = ulHigh;

    /* insert at end of list after checking for dups */
    fCkNa = (usItype == MIBITobj) || (usItype == MIBITnlobj);
    for (pOT = NULL, pT = pOid->ut.oid.pIndxL; pT != NULL;
            pOT = pT, pT = pT->pNext) {
        if (fCkNa && (pT->pOid != NULL) && ((rt = strcmp(pOidIndx->pszName,
                (pT->pOid)->pszName)) == 0)) {
            /* duplicate name */
            yyerror("duplicate name \"%s\" in INDEX list",
                    pOidIndx->pszName);
#ifdef OLD
            yyterm();
            return(NULL);
#endif
        }
    }
    if (pOT == NULL) {
        /* insert at beginning of list */
        pOid->ut.oid.pIndxL = pIndx;
    } else {
        /* insert in list */
        pOT->pNext = pIndx;
    }
    pIndx->pNext = pT;

    return(pIndx);

} /* newINDX */


/** addINDXitem - add index item to row (or leaf) object
*
* NOTE: can not determine if item is the proper type
*       at this point in time.
*       MUST check later!!
*
* call with:
*   pOid - row object
*   usItype - index type
*   pNa - index item
*   ulLow - low for range/size
*   ulHigh - high for range/size
*   pMod - module
*
* returns:
*   TRUE - no error
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
addINDXitem(MIBSYM *pOid, USHORT usItype, STRTAB *pNa,
            ULONG ulLow, ULONG ulHigh, MIBSYM *pMod)
#else
addINDXitem(pOid, usItype, pNa, ulLow, ulHigh, pMod)
    MIBSYM *pOid;
    USHORT usItype;
    STRTAB *pNa;
    ULONG ulLow;
    ULONG ulHigh;
    MIBSYM *pMod;
#endif /* __STDC__ */
{
    MIBSYM *pSym;
    MIBSYM *pOidIndx;
    USHORT usType;
    USHORT usItypeSav;

    pOidIndx = NULL;

    /* handle special cases */
    switch(usItype) {
    case MIBITobj:
    case MIBITnlobj:
        /* object name (or TC name) given */
        /* check if item defined */
        pSym = pNa->pSym;
        if ((pSym != NULL) && (pSym->pMod == pMod)) {
            /* item defined */
            /* check if import */
            if (pSym->usType == MIBSYMimp) {
                pSym->ut.imp.cUse++;
                pSym = pSym->ut.imp.pImpSym;
            }
            /* check if Textual convention */
            if ((usType = (pSym->usType & MIBSYMmask)) == MIBSYMtc) {
                /* textual convention */
                pSym->ut.tc.cUse++;
                /* get resolved syntax */
                usItypeSav = usItype;
                switch (pSym->ut.tc.rsyn.usSyntax) {
                case MIBSYNnu:      /* not used */
                case MIBSYNbad:     /* invalid syntax specified */
                case MIBSYNnull:    /* NULL */
                case MIBSYNopaque:  /* opaque */
                case MIBSYNseqOf:   /* sequence of (table) (pSeq pts to type) */
                case MIBSYNseq:     /* sequence (row)  (pSeq pts to type) */
                case MIBSYNtc:      /* textual convention (pTC points to item) */
                default:
                    yyerror("addINDXitem: bad syntax type of %d for \"%s\"",
                            pSym->ut.tc.rsyn.usSyntax, pSym->pszName);
                    usItype = MIBITbad; /* bad index item */
                    break;

                case MIBSYNint:     /* integer */
                case MIBSYNcntr:    /* counter */
                case MIBSYNgauge:   /* gauge */
                case MIBSYNticks:   /* timer tics */
                case MIBSYNenum:    /* enumerated */
                    usItype = MIBITint; /* INTEGER */
                    if (pSym->ut.tc.rsyn.usSizeRange == MIBSRno) {
                        /* no range - make max */
                        ulHigh = 0xffffffffL;
                    } else if (pSym->ut.tc.rsyn.usSizeRange == MIBSRpp) {
                        /* non-negative to positive */
                        ulLow  = pSym->ut.tc.rsyn.usr.ulRange[0];
                        ulHigh = pSym->ut.tc.rsyn.usr.ulRange[1];
                    } else {
                        /* a problem - make make max */
                        /*?? put more tests in */
                        ulHigh = 0xffffffffL;
                    }
                    break;

                case MIBSYNoctstr:  /* octet string */
                    if (pSym->ut.tc.rsyn.usSizeRange == MIBSRfs) {
                        usItype = MIBITfloct; /* fixed length OCTET STRING */
                        ulLow = pSym->ut.tc.rsyn.usr.usSize[0];
                    } else if (pSym->ut.tc.rsyn.usSizeRange != MIBSRbad) {
                        usItype = (usItypeSav == MIBITnlobj) ?
                                    MIBITnloct : MIBIToct;
                        ulLow = pSym->ut.tc.rsyn.usr.usSize[0];
                        ulHigh = pSym->ut.tc.rsyn.usr.usSize[1];
                    } else
                        usItype = MIBSRbad;
                    break;

                case MIBSYNoid:     /* object identifier */
                    usItype = (usItypeSav == MIBITnlobj) ?
                                    MIBITnloid : MIBIToid;
                    break;

                case MIBSYNnaddr:   /* network address */
                    usItype = MIBITnaddr;
                    break;

                case MIBSYNipaddr:  /* IP address */
                    usItype = MIBITipaddr;
                    break;

                }
                /* check for illegal NOLENGTH clause */
                if ((usItypeSav == MIBITnlobj) &&
                        (usItype != MIBITnloid) &&
                        (usItype != MIBITnloct))
                    yyerror("Invalid NOLENGTH qualifier for \"%s\" ignored",
                                pSym->pszName);
                break;
            }

            /* check if OID type (ignore real checks) */
            if (usType != MIBSYMoid) {
                /* not OID type */
                yyerror("\"%s\" is not valid type for index item",
                        pSym->pszName);
#ifdef OLD
                yyterm();
                return(FALSE);
#endif
            }
            pOidIndx = pSym;
        } else {
            /* item not defined */
            /* allocate new symbol */
            pOidIndx = newOID(pNa, pMod);
        }
        break;

    case MIBITint:
        /* integer */
        /* check that range is valid */
        if (ulLow > ulHigh) {
            yyerror("Invalid range given, low %lu greater than high %lu",
                    ulLow, ulHigh);
        }
        break;

    case MIBIToct:
    case MIBITnloct:
        /* octet string or nolength octet string */
        /* check that size is valid */
        if (ulLow > ulHigh) {
            yyerror("Invalid size given, low %lu greater than high %lu",
                    ulLow, ulHigh);
        }
        break;

    case MIBITfloct:
    case MIBIToid:
    case MIBITnloid:
    case MIBITnaddr:
    case MIBITipaddr:
        /* nothing special to do */
        break;

    default:
        yyerror("addINDXitem: bad index type of %d", usItype);
        break;
    }

    /* allocate new index item */
    newINDX(pOid, usItype, pOidIndx, ulLow, ulHigh);

    return(TRUE);

} /* addINDXitem */


/** addDV - add default value
*
* NOTE: can not determine if value valid for this object
*       at this point in time.
*       MUST check later!!
*
* call with:
*   pOid - object
*   usDevVal - type of value
*   pVal - string value
*   ulVal - integer value
*   pComp - oid value
*
* returns:
*   TRUE - no error
*   FALSE - an error
*/
    BOOL
#ifdef __STDC__
addDV(MIBSYM *pOid, USHORT usDefVal, STRTAB *pVal, ULONG ulVal, MIBSYM *pComp)
#else
addDV(pOid, usDefVal, pVal, ulVal, pComp)
    MIBSYM *pOid;
    USHORT usDefVal;
    STRTAB *pVal;
    ULONG ulVal;
    MIBSYM *pComp;
#endif /* __STDC__ */
{
    pOid->ut.oid.usDefVal = usDefVal;
    switch (usDefVal) {
    case MIBDFVno:
        /* no default value */
        pOid->ut.oid.udv.ulDefVal = 0L;
        break;

    case MIBDFVint:
    case MIBDFVneg:
    case MIBDFVip:
        /* integer type */
        pOid->ut.oid.udv.ulDefVal = ulVal;
        break;

    case MIBDFVstr:
    case MIBDFVbstr:
    case MIBDFVhstr:
    case MIBDFVna:
        /* string type */
        pOid->ut.oid.udv.pszDefVal = pVal->pszVal;
        break;

    case MIBDFVoid:
        /* oid type */
        pOid->ut.oid.udv.pOid = pComp;
        break;

    default:
        yyerror("addDV: bad value type of %d", usDefVal);
        return(FALSE);
        break;
    }

    return(TRUE);

} /* addDV */


/* end of SMPARCOM.C */
