
/********************************************************
 *  pingd - ping and maintain node state information	*
 *	Keep track of nodes that responded 		*
 *	Maintain cache					*
 ********************************************************/
#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <string.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <sys/errno.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netdb.h>
#include <errno.h>
#include <malloc.h>
#include "pathnames.h"
#include "CommonDefs.h"
#include "hostfile.h"
#include "problem.h"

#ifdef S_IRWXU
#define S_RWALL (S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH)
#else
#define S_RWALL 0000666
#endif

#define DEFAULT_PACKET_SIZE       32

/************************************************************************
 *	Standard Rover Files						*
 ************************************************************************/
static char *DefaultPingkyDir = DEFAULT_PINGKY_DIR;
static char Checking_File[MAXPATHLEN];	/* What node are we Checking_File?   */
static char pingdcycletime[MAXPATHLEN];	/* How long did pingky take to cycle */
static char logfile[MAXPATHLEN];	/* Where do we log things            */
static char Problem_File[MAXPATHLEN];	/* Where do we Add/Delete Problems   */
static char hostfile[MAXPATHLEN];	/* List of nodes we are Testing      */
static char pidfile[MAXPATHLEN];	/* Our pid goes there */
static char roverstatusfile[MAXPATHLEN];	/* Network Status File */

static char err_buf[BUFSIZ];		/* For diagnostics */
static char hostname[MAXHOSTNAMELEN];	/* Hostname on which we are running  */

char *progname;

static int packet_size = DEFAULT_PACKET_SIZE;
static char *ping_packet;

static int DefaultWait = 125;		/* Interpacket delay time 	*/
static int DefaultStateCacheTimeout = 120; /* How long to stay in state	*/

/*#define TESTMODE /* */
#ifdef TESTMODE
static int DefaultMaxRetries = 2;	/* How many retries b4 called NR*/
static int DefaultSecsB4Retry = 5;	/* How long to wait between send*/
#else
static int DefaultMaxRetries = 5;	/* How many retries b4 called NR*/
static int DefaultSecsB4Retry = 20;	/* How long to wait between send*/
#endif
static int FakePingTransmits=0;		/* For testing - do we Fake send? */

/***************************************************************************
 * Internal NodeStatusType Structure - the main structure that we maintain *
 ***************************************************************************/
static struct NodeStatusType {
	struct sockaddr whereto;/* Socket information for host */
	time_t TimeStamp;	/* Time the node went into the state */
	char *NodeName;		/* Name of Node */
	char *Address;		/* Node Address */
	int State;		/* Node State (UP or NR) */
	time_t NextPingAction;	/* Time at which next pings fly	*/
	int CurrentRetryCounter;/* Number of pings left to send b4 nodeNR*/
	/**********************************************/
	/* The following structure entries are static */
	/**********************************************/
	int StateCacheTimeout;	/* State Cache Timeout in seconds */
	int MaxRetries;		/* Number of pings to send b4 the node is NR */
	int SecsB4Retry;	/* Time to wait before retrying */
	struct icmp packet;	/* Ping packet header */
	unsigned char unusedflags[2];
} *NodeStatusArray[MAXNODES];

static int NumNodeStatusEntries = 0;	/* Count of entries we have */
static int s;                  /* Socket file descriptor */

int verbose;

static int stackindex = 0;
struct stacktype {
	int index;	/* Index of this node */
	int state;	/* node state NR,UP */
};
static struct stacktype stack[MAXNODES];

extern int exit();

/************************************************************************
 *  FindNodeStatusEntry() - Search through our structures for this node	*
 ************************************************************************/
FindNodeStatusEntry(Address)
char *Address;
{
	int i;

	for(i = 0; i < NumNodeStatusEntries; i++)
		if (strcmp(Address, NodeStatusArray[i]->Address) == 0)
			return(i);
	return(-1);
}

/************************************************************************
 * AddNode() - Add a node structure to the array			*
 ************************************************************************/
void AddNode(NodeName, Address, state, TimeStamp, TimeOut, maxretries, delay)
char *NodeName;
char *Address;
int state;
time_t TimeStamp;
int TimeOut;
int maxretries;
int delay;
{
	char *p;
	time_t TimeNow;
	int rc,index;
	extern int MakePINGPacket();
	extern time_t time();

	TimeNow = time(&TimeNow);
	if ((index = FindNodeStatusEntry(Address)) != -1) {
		if (state != -1) {	/* Known Node & Known State-update it*/
			NodeStatusArray[index]->State = state;
			NodeStatusArray[index]->TimeStamp = TimeStamp;
			NodeStatusArray[index]->TimeStamp = TimeStamp + NodeStatusArray[index]->StateCacheTimeout;
		}
	} else {
		index = NumNodeStatusEntries;
		if ((NodeStatusArray[index] = (struct NodeStatusType *) malloc( sizeof(struct NodeStatusType))) == NULL) {
			fprintf(stderr, "AddNode(): malloc failed\n");
			exit(1);
		}
		if (Address == NULL) 
			return;
		if ((p = malloc(strlen(Address) + 1)) == NULL) {
			fprintf(stderr,"AddNode(): malloc failed\n");
			exit(1);
		}
		strcpy(p, Address);
		NodeStatusArray[index]->Address = p;
		if (NodeName != NULL) {
			if ((p = malloc(strlen(NodeName) + 1)) == NULL) {
				fprintf(stderr,"AddNode(): malloc failed\n");
				exit(1);
			}
			strcpy(p, NodeName);
		}
		NodeStatusArray[index]->NodeName = p;
		NodeStatusArray[index]->TimeStamp = TimeNow;
		NodeStatusArray[index]->NextPingAction = TimeNow;
		if (state == -1) 
			NodeStatusArray[index]->State = NR;
		rc = MakePINGPacket(NodeStatusArray[index]->Address, index + 1, 
		    packet_size, &(NodeStatusArray[index]->packet),
		    &NodeStatusArray[index]->whereto);
		if (rc < 0) {
			fprintf(stderr, "Error making PING packet for %s (%s) rc=%d\n", NodeStatusArray[index]->Address, NodeName, rc);
			free(NodeStatusArray[index]->Address);
			free(NodeStatusArray[index]);
		}
		else 
			NumNodeStatusEntries++;
	}

	/****   Set up Polling Defaults - timeouts, retries, CacheTimeout ****/
	if (TimeOut < 0) 
		TimeOut = DefaultStateCacheTimeout;
	if (maxretries < 0) 
		maxretries = DefaultMaxRetries;
	if (delay < 0) 
		delay = DefaultSecsB4Retry;
	NodeStatusArray[index]->StateCacheTimeout = TimeOut;
	NodeStatusArray[index]->MaxRetries = maxretries;
	NodeStatusArray[index]->CurrentRetryCounter = maxretries;
	NodeStatusArray[index]->SecsB4Retry = delay;
}

/************************************************************************
 * FreeNodeStatusEntries() - Free all NodeStatusArray entries - usually	*
 *			for a re-read of the network status file	*
 ************************************************************************/
FreeNodeStatusEntries()
{
	register int i;

	for (i = 0; i < MAXNODES; i++) {
		if (NodeStatusArray[i] != NULL) {
			if (NodeStatusArray[i]->NodeName != NULL)
				free(NodeStatusArray[i]->NodeName);
			if (NodeStatusArray[i]->Address != NULL)
				free(NodeStatusArray[i]->NodeName);
			free(NodeStatusArray[i]);
		}
	}
	NumNodeStatusEntries = 0;
}

char *PrintState(i)
int i;
{
	switch (i) {
	case NR: 
		return("NR");
	case UP: 
		return("UP");
	default: 
		return("??");
	}
}

/************************************************************************
 *									*
 * This should really be done using the Network Status File I/O routines*
 *									*
 ************************************************************************/
WriteNetworkStatus(pingdfilename)
char *pingdfilename;
{
	FILE *fp;
	register int i;
	time_t TimeNow = time(&TimeNow);
	extern char *PrintState();

	if ((fp = fopen(pingdfilename, "w")) == NULL) {
		sprintf(err_buf, "Can't open pingd STATUS file: %s\n",
						pingdfilename);
		fprintf(stderr, err_buf);
		Log(err_buf, (char *) NULL);
		return;
	}

	fprintf(fp, "#\n# pingd STATUS as of (%ld) %s#\n",
				TimeNow,ctime(&TimeNow));
	for(i = 0; i < NumNodeStatusEntries; i++)
		fprintf(fp,"%lu NODE NSS\t%s\t%s\t%s\n",
		    NodeStatusArray[i]->TimeStamp,
		    NodeStatusArray[i]->NodeName,
		    NodeStatusArray[i]->Address,
		    PrintState(NodeStatusArray[i]->State));
	fclose(fp);
	chmod(pingdfilename, S_RWALL);
}

ReadNetworkStatusFile(pingdfilename)
char *pingdfilename;
{
	FILE *fp;
	int Line = 0, State,Valid;
	char *TimeStamp, *NodeName, 
		*Address, *PState;
	char inbuff[140];

	if ((fp = fopen(pingdfilename,"r")) == NULL) {
		Empty_File(pingdfilename);
		if ((fp = fopen(pingdfilename,"r")) == NULL) {
			sprintf(err_buf,"Can't open pingd STATUS file: %s\n",
							pingdfilename);
			fprintf(stderr, err_buf);
			Log(err_buf, (char *) NULL);
			return;
		}
	}

	while(fgets(inbuff, sizeof(inbuff), fp)) {
		Line++;
		if (inbuff[0] == '#') 
			continue;
		for (;;) {
			Valid = FALSE;
			if ((TimeStamp = strtok(inbuff, DELIMITERS)) == NULL) {
				sprintf(err_buf, "No TimeStamp Field\n"); 
				break; 
			}
			if (strtok((char *) NULL, DELIMITERS) == NULL) {
				sprintf(err_buf, "No LineType Field\n"); 
				break; 
			}
			if (strtok((char *) NULL, DELIMITERS) == NULL)  {
				sprintf(err_buf, "No Type Field\n"); 
				break; 
			}
			if ((NodeName = strtok((char *) NULL, DELIMITERS)) == NULL) {
				sprintf(err_buf, "No Name/Address1 Field\n"); 
				break; 
			}
			if ((Address = strtok((char *) NULL, DELIMITERS)) == NULL) {
				sprintf(err_buf, "No Address2 Field\n"); 
				break; 
			}
			if ((PState = strtok((char *) NULL, DELIMITERS)) == NULL) {
				sprintf(err_buf, "No State Field\n"); 
				break; 
			}
			if ((strcmp(PState, "UP")) == 0) {
				State = UP; 
			}
			if ((strcmp(PState, "NR")) == 0) {
				State = NR; 
			}
			Valid = TRUE;
			break;
		}
		if (!Valid)
			printf("ReadConfig(): Error on Line #%d(Ignoring Line): %s", Line, err_buf);
		else
			AddNode(NodeName, Address, State, 
					(time_t) atol(TimeStamp), -1, -1, -1 );
	}
	fclose(fp);
}

PushNodeMigration(index, state)
int index, state;
{
	if (stackindex == MAXNODES) {
		fprintf(stderr, "Node Migration Stack overflow\n");
		return(-1);   /* This should be an error */
	}
	stack[stackindex].index = index;
	stack[stackindex].state = state;
	stackindex++;
	return(stackindex);
}

PopNodeMigration(statep)
int *statep;
{
	if (stackindex == 0) 
		return(-1);
	stackindex--;
	*statep = stack[stackindex].state;
	return(stack[stackindex].index);
}

/************************************************************************
 * ProcessNodeMigration - For each state change that occured, batch up	*
 *			the problem adds/deletes, and then flush them	*
 *			into the ProblemManager.			*
 ************************************************************************/
ProcessNodeMigration()
{
	int index, state;

	if ( stackindex == 0 ) return;	/* Quick out */
	/*if ( stackindex > 2 ) printf("ProcessNodeMigration(): many problems: flush count=%d\n",stackindex);*/
	while ((index = PopNodeMigration(&state)) != -1) {
		if ((index < 0 ) || (index > NumNodeStatusEntries)) {
			continue;
		}
		
		if (state == NR) {
			/*printf("Adding problem %s\n", NodeStatusArray[index]->Address);*/
			Problem_Manager(ADD_BATCH_PRIMARY_PROBLEM, "PING", 
				NodeStatusArray[index]->NodeName, 
				NodeStatusArray[index]->Address, NULL);
		}
		else if (state == UP) {
			/*printf("Deleting problem %s\n", NodeStatusArray[index]->Address);*/
			Problem_Manager(DELETE_BATCH_PRIMARY_PROBLEM, "PING", 
				NodeStatusArray[index]->NodeName, 
				NodeStatusArray[index]->Address, NULL);
		}
	}
	/*printf("Flushing ......\n");*/
	Problem_Manager(FLUSH_BATCH_PRIMARY_PROBLEM, NULL, NULL, NULL, NULL );  /* varoom! */
}

/************************************************************************
 * MigrateNode() - Migrate a nodes' state - do whatever state transition*
 *		action is required.					*
 *		Also, reset the next action time and retry counter	*
 ************************************************************************/
void MigrateNode(index, state)
int index;
int state;
{
time_t TimeNow = time(&TimeNow);

	if ((index < 0 ) || (index > NumNodeStatusEntries)) {
		printf("icmp sequence or index out of spec=%d\n", index);
		return;
	}

	NodeStatusArray[index]->TimeStamp = TimeNow;
	NodeStatusArray[index]->NextPingAction = TimeNow + NodeStatusArray[index]->StateCacheTimeout;
	if (state == NR) /* If Down Don't use cache - keep trying */
		NodeStatusArray[index]->NextPingAction = TimeNow + NodeStatusArray[index]->SecsB4Retry;
	NodeStatusArray[index]->CurrentRetryCounter = NodeStatusArray[index]->MaxRetries;
	NodeStatusArray[index]->State = state;
}

/************************************************************************
 * ReceiveLoop() - A major percentage of time is spent in this module	*
 *		we just wait for a ping reply and process it.		*
 *		We'll use the supporting routines to print icmp errs.	*
 ************************************************************************/
ReceiveLoop()
{
	struct sockaddr_in from;
	int len, fromlen, cc, index;
	char *packet;

	/*
	 *  Get a buffer large enough to hold the packets we are sending
	 *  out.
	 */
	if ((packet = malloc((unsigned) packet_size)) == NULL) {
		perror("malloc");
		exit(-1);
	}
	memset(packet, 0, packet_size);

	/*
	 *  Now, catch the pings.
	 */
	for (;;) {
		len = packet_size;
		fromlen = sizeof(from);
		if ((cc = recvfrom(s, packet, len, 0, &from, &fromlen)) < 0) {
			if(errno == EINTR)
				continue;
		}
		/******************
		if (verbose) {
			register int i;
			printf("Got something (len = %d)\n", cc);
			for (i = 0; i < cc; i++) {
				printf("0x%-2x ", packet[i]);
				if (((i + 1) % 16) == 0)
					printf("\n");
			}
			printf("\n");
		}
		******************/
		index = ParsePINGResponse((unsigned char *) packet, cc, &from);
		if (index) {
			index--;
			if (index > NumNodeStatusEntries)
				return;   /* NO SUCH NODE CURRENT */
			if (verbose) 
				printf("responding node index=%d NodeStatusArray[%d]->Address=%s\n", index, index, NodeStatusArray[index]->Address);

			MigrateNode(index, UP);		/* Update Structures */
			PushNodeMigration(index, UP); 	/* Delete Problem    */
		}
	}
}

/************************************************************************
 * ReadHostfile() - Read the hostfile for a list of nodes to ping 	*
 ************************************************************************/
ReadHostfile( hostfile )
char *hostfile;
{
	int i, MaxRetries, SecsB4Retry, StateCacheTimeout;
	struct HostFileNodeType *np;
	time_t TimeNow = time(&TimeNow);
	extern struct HostFileNodeType * HostfileManager();

	FreeNodeStatusEntries();
	np = HostfileManager(READHOSTFILE, hostfile);
	for(i = 0; i < NumNodes; i++) {
		if (np[i].Test[0].argv[0]!=NULL && strcmp(np[i].Test[0].argv[0], "PING") == 0) {
			MaxRetries = -1;
			SecsB4Retry = -1;
			StateCacheTimeout = -1;
			if (np[i].Test[0].argv[1] != NULL)
				MaxRetries = atoi(np[i].Test[0].argv[1]);
			if (np[i].Test[0].argv[2] != NULL)
				SecsB4Retry = atoi(np[i].Test[0].argv[2]);
			if (np[i].Test[0].argv[3] != NULL)
				StateCacheTimeout = atoi(np[i].Test[0].argv[3]);
			AddNode(np[i].NodeName, np[i].UniqueID, -1, TimeNow, 
				StateCacheTimeout, MaxRetries, SecsB4Retry);
		}
	}
	if (NumNodes == 0) {
		fprintf(stderr,"No entries in the hostfile: %s - severe error ... exitting\n", hostfile);
		exit(1);
	}
}

/**************************************************************************
 *  main() - 	Process cmd line arguments, and start pinger and rcv loop *
 **************************************************************************/
main(argc,argv)
int argc;
char *argv[];
{
	extern SIG_FN FastPing();
	int c;
	struct stat buf;
	FILE *fp;
	char oldpid[10], buffer[100];
	char *pingkydir, *getenv();
	extern char Version[], *optarg;

	/*
	 *  Set the program name and then gather the args.
	 */
	progname = argv[0];
	while ((c = getopt(argc, argv, "r:s:t:vfw:")) != EOF) {
		switch (c) {
		case 'v':       
			verbose = 1;
			break;
		case 'r':
			DefaultMaxRetries = atoi(optarg);
			break;
		case 's':
			packet_size = atoi(optarg);
			if (packet_size < sizeof(struct icmp)) {
				fprintf(stderr, "packet size > %d\n",
						sizeof(struct icmp));
				fprintf(stderr, "using %d instead\n",
						DEFAULT_PACKET_SIZE);
				packet_size = DEFAULT_PACKET_SIZE;
			}
			break;
		case 't':
			DefaultSecsB4Retry = atoi(optarg);
			break;
		case 'w':
			DefaultWait = atoi(optarg);
			break;
		case 'f':
			FakePingTransmits=1;
			break;
		default:
			Usage();
			break;
		}
	}

	/*
	 *  Set up the proper pathnames.
	 */
	if ((pingkydir = getenv("PINGKYDIR")) == NULL) 
		pingkydir = DefaultPingkyDir;
	sprintf(Checking_File, "%s/%s", pingkydir, CHECK_FILE);
	sprintf(pingdcycletime, "%s/%s", pingkydir, PINGD_CYCLE);
	sprintf(logfile, "%s/%s", pingkydir, PROBLEM_LOG);
	sprintf(hostfile, "%s/%s", pingkydir, HOST_FILE);
	sprintf(roverstatusfile, "%s/%s", pingkydir, PINGD_STATUS);
	sprintf(Problem_File, "%s/%s", pingkydir, PROBLEM_FILE);

	/*
	 *  Check the pid file.  If a pingd process is running, then we
	 *  should just exit here.  Otherwise, tuck away our pid.
	 */
	sprintf(pidfile, "%s/%s", pingkydir, PINGD_PID);
	if (stat(pidfile, &buf) < 0) {
		if (errno == ENOENT) {
			if ((fp = fopen(pidfile, "a+")) == NULL) {
				perror("fopen");
				exit(-1);
				/*NOTREACHED*/
			}
			fprintf(fp, "%d\n", getpid());
			fclose(fp);
		}
		else {
			perror("stat");
			exit(-1);
			/*NOTREACHED*/
		}
	}
	else {
		if ((fp = fopen(pidfile, "r+")) == NULL) {
			perror("fopen");
			exit(-1);
			/*NOTREACHED*/
		}
		(void) fgets(oldpid, 10, fp);
		if (kill((pid_t) atol(oldpid), 0) < 0) {
			if (errno == ESRCH) {
				(void) rewind(fp);
				fprintf(fp, "%d\n", getpid());
				(void) fclose(fp);
			}
			else {
				perror("kill");
				exit(-1);
				/*NOTREACHED*/
			}
		}
		else {
			fprintf(stderr, 
				"pingd is already running: pid = %s", oldpid);
			exit(-1);
			/*NOTREACHED*/
		}
	}
	if (chmod(pidfile, S_RWALL) < 0)
		syserr("chmod");


	if ((ping_packet = malloc((unsigned) packet_size)) == NULL) {
		perror("malloc");
		exit(1);
	}
	if (getenv("USER") == NULL) {
		fprintf( stderr, "Unknown user: no USER environmental var\n");
		exit(1);
	}

	/*
	 *  Set the host name and log that we are starting.
	 */
	gethostname(hostname, sizeof(hostname));
	strtok(hostname, ". \t\n");
	sprintf(buffer, "Starting %s %s (packetsize = %d)\n", hostname, 
					Version, packet_size);
	printf(buffer);
	Log(buffer, logfile);

#ifdef DISKPIG
	/************* Log an empty cycle time ********************/
	if ((fp = fopen(pingdcycletime,"w")) == NULL)
		syserr("fopen pingd.cycle");
	fprintf(fp, "++:++");
	if (fclose(fp) < 0) 
		syserr("fclose pingky cycle");
	chmod(pingdcycletime , S_RWALL);
#endif
	ReadHostfile(hostfile);
	ReadNetworkStatusFile(roverstatusfile);
	if (NumNodeStatusEntries == 0) 
		Usage();

	printf("%s Starting %d Nodes StateCacheTimeout=%d DefaultMaxRetries=%d DefaultTime2WaitB4Retry=%d wait=%d\n",
		progname, NumNodeStatusEntries, DefaultStateCacheTimeout, 
		DefaultMaxRetries, DefaultSecsB4Retry, DefaultWait );
	s = MakePINGSocket();
	FastPing();
	ReceiveLoop();
	exit(0);
	/*NOTREACHED*/
}

Usage()
{
	fprintf(stderr, "\nUsage: %s [-v] [-r retries] [-t delay before retry] [-f pingfile] [-w wait time between pkts] [-s packet size] host1 [host2] . . . [hostN]\n\n", progname);
	exit(-1);
	/*NOTREACHED*/
}

/************************************************************************
 *  FastPing() - Ping nodes only enough to maintain cache information	*
 *----------------------------------------------------------------------*
 *	Search through list for a node with an expiring next ping action* 
 *		if retry count is 0, migrate node to NR			*
 *		else							*
 *			send ping packet				*
 *			mark our place					*
 *			set interval timer for interpacket delay time	*
 *			( Call our selves when time for next packet )	*
 *			return						*
 *    									*
 ************************************************************************/
SIG_FN FastPing()
{
	int i, rc;
	static int LastOne = 0;	/* Last node we left off pinging */
	static int cycle = 1;
	static struct itimerval value,ovalue;
	time_t TimeNow = time(&TimeNow), CurrentNext;
	static time_t FastPingStartTime, FastPingEndTime;
	FILE *fp;
	static NodesPinged;
	extern SIG_FN FastPing();

	signal(SIGALRM, SIG_IGN);
	if (LastOne == 0) {
		FastPingStartTime = time(&FastPingStartTime);	/* for stats */
		NodesPinged = 0;
	}
	ProcessNodeMigration();	/* Deal with any ping responses */
	if (verbose) 
		printf("Top of loop, LastOne=%d NumNodeStatusEntries=%d\n",
					LastOne,NumNodeStatusEntries);
	for (i = LastOne; i < NumNodeStatusEntries; i++) {
		if (verbose)
			printf("Inside Loop i=%d Node=%s NextPingAction=%lu TimeNow=%lu\n",i, NodeStatusArray[i]->Address, NodeStatusArray[i]->NextPingAction,TimeNow);
#ifdef DISKPING
		if ( LastOne % 200 == 0 ) {	/* I'm alive! */
			if ((fp = fopen(pingdcycletime,"w")) == NULL)
				syserr("fopen pingd.cycle");
			fprintf(fp,"??:??");
			if (fclose(fp) == -1) 
				syserr("fclose pingky cycle");
		}
#endif
		if (NodeStatusArray[i]->NextPingAction <= TimeNow) {
			if (verbose) 
				printf("Performing ping on %s\n",
						NodeStatusArray[i]->Address);
			if (--NodeStatusArray[i]->CurrentRetryCounter <= 0 ) {
				/* Node is Not Reachable - no response yet */
				/*printf("Node didn't respond : calling PushNodeMigration(%s, NR)\n",NodeStatusArray[i]->Address);*/
				MigrateNode(i, NR);	/* Update Structures */
				PushNodeMigration(i, NR); /* Add Problem */
				continue;
			}

			/*
			 *  Set the time for the next ping.
			 *
			 *  Copy the ping packet header into the ping packet,
			 *  and then ship it out.
			 */
			NodeStatusArray[i]->NextPingAction = TimeNow + 
					NodeStatusArray[i]->SecsB4Retry;
			memcpy(ping_packet, &(NodeStatusArray[i]->packet), 
							sizeof(struct icmp));
				/******************************
			if (verbose) {
				register int i;
				printf("sending packet (len = %d)\n", 
								packet_size);
				for (i = 0; i < packet_size; i++) {
					printf("0x%-2x ", ping_packet[i]);
					if (((i + 1) % 16) == 0)
						printf("\n");
				}
				printf("\n");
			}
				******************************/
			if ( FakePingTransmits ) rc=packet_size;
			else
				rc = sendto(s, ping_packet, packet_size, 0, 
					&NodeStatusArray[i]->whereto, 
					sizeof(struct sockaddr));
			if ((rc < 0) || (rc != packet_size)) {
				if (rc < 0 )  
					perror("sendto");
				printf("FastPing wrote %s %d chars, ret=%d\n",
				    NodeStatusArray[i]->Address, packet_size, rc );
				fflush(stdout);
			}
			LastOne = i + 1;	/* Mark our place */
			NodesPinged++;

			if (verbose) {
				printf("Sent packet to %s\n",
						NodeStatusArray[i]->Address);
				fflush(stdout);
			}
			value.it_interval.tv_sec = 0;
			value.it_interval.tv_usec = 0;
			value.it_value.tv_sec = 0;
			value.it_value.tv_usec = DefaultWait * 1000;
			signal(SIGALRM, FastPing);
			setitimer(ITIMER_REAL, &value, &ovalue);
			return;	/******* We have sent a ping - return ********/
		}
	}
	FastPingEndTime = time(&FastPingEndTime);
	if (verbose) 
		printf("**** This Cycle Through Time took %d secs Pinged %d Nodes ****\n", (int) (FastPingEndTime - FastPingStartTime), NodesPinged);

#ifdef DISKPIG
	if ((fp = fopen(pingdcycletime,"w")) == NULL)
		syserr("fopen pingd.cycle");
	fprintf(fp,"%2.2d:%2.2d",
	    (int) (FastPingEndTime - FastPingStartTime) / 60, (int) (FastPingEndTime - FastPingStartTime) % 60);
	if (fclose(fp) == -1) 
		syserr("fclose pingky cycle");
	chmod(pingdcycletime , S_RWALL);
#endif
	ProcessNodeMigration();	/* Deal with any ping responses */

	/******************************************************** 
	 *	OK , we've gotten to the bottom of the list,	* 
	 *		pinging nodes that needed it		*
	 *	set alarm for earliest NextPingAction 		*
	 ********************************************************/
	if (verbose)
		printf("Setting up for next action\n");
	CurrentNext = TimeNow + 5 * 60;	/* Wait at most 5 minutes */
	for(i = 0; i < NumNodeStatusEntries; i++ )
		if (NodeStatusArray[i]->NextPingAction < CurrentNext)
			CurrentNext = NodeStatusArray[i]->NextPingAction;
	if (HostfileManager(READCHECKHOSTFILE, hostfile) != NULL)
		ReadHostfile(hostfile);
	WriteNetworkStatus(roverstatusfile);

	TimeNow = time(&TimeNow);
	if (TimeNow >= CurrentNext)
		CurrentNext = TimeNow + 2;
	if (verbose) 
		printf("Done cycle \n");
	LastOne = 0;			/* Start at the beginning now */
	if (verbose) {
		printf("TimeNow=%s",ctime(&TimeNow));
		printf("___________Sleeping until %s",ctime(&CurrentNext));
	}

	value.it_interval.tv_sec = 0;
	value.it_interval.tv_usec = 0;
	value.it_value.tv_sec = CurrentNext - TimeNow;
	value.it_value.tv_usec = 0;
	signal(SIGALRM, FastPing);
	setitimer(ITIMER_REAL, &value, &ovalue);
	return;
}
