/*
 * Copyright (c) 1992 Regents of the University of Michigan.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that this notice is preserved and that due credit is given
 * to the University of Michigan at Ann Arbor. The name of the University
 * may not be used to endorse or promote products derived from this
 * software without specific prior written permission. This software
 * is provided ``as is'' without express or implied warranty.
 *
 *			dsuchk.c - Check DSU Counts
 *						
 *	Use fastsnmp API to get DSU interface indexes for our nodes	
 *	For each DSU Interface on each node				
 *								
 *		Build VarList of 			
 *			CurrentErrorCounter[if#] - Current Bucket	
 *			ErrorCounter[if#][1]	- Previous 15min sample
 *			ErrorCounter[if#][2]	- Previous 15min bucket
 *								
 *		Blast Out Queries to nodes		
 *		Process Received Table			
 *						
 * This program expects the LinkDetail file to contain accurate info	
 *	We are sending out one packet for each DSU...could be optimized
 *	later.							
 */

#include <stdio.h>
#include <sys/wait.h>
#include <sys/file.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/stat.h>
#include "CommonDefs.h"
#include "pathnames.h"
#include "snmptable2.h"
#include "Linkfileio.h"
#include "mib.h"

#ifdef PROFILING
#include <mon.h>
#endif

/*
 *  If a local node and a remote node are in the same POP, then their
 *  IP addresses will have the same set of 4 bits -xxxx--- in the third
 *  octet.
 */
#define POP_MASK	0x00007800

/* Add max retries =1 since snmp_next does own retries */

char *progname;

void Usage()
{
	extern void exit();

	fprintf(stderr,"Usage: %s [-r numretries ] [ -t timebetweenretires ] [-v] -c <community>\n",progname);
	exit(1);
}

/************************************************************************
 *	DSUVarBuckets[] - Variables we query.  It is important that 	*
 *			this and the DSUVarCurrent[] table grows	*
 *			together..					*
 ************************************************************************/
static char *DSUVarBuckets[]={
                        "ds3IntervalESs", /* ESs ( Errored Seconds )         */
                        "ds3IntervalSESs",/* SESs (Severely Errored Seconds )*/
                        "ds3IntervalCSSs",/* CSSs (Control Slip Seconds )    */
                        "ds3IntervalSEFSs",/* SEFs (Severly Errored Frames)  */
                        /*"ds3IntervalUASs",/* UASs (Unavailable Seconds)    */
};

static char *DSUVarCurrent[]={
                        "ds3CurrentESs",  /* ESs ( Errored Seconds ) 	     */
                        "ds3CurrentSESs", /* SESs (Severely Errored Seconds )*/	
                        "ds3CurrentCSSs", /* CSSs (Control Slip Seconds )    */
                        "ds3CurrentSEFSs", /* SEFs (Severly Errored Frames)  */
                        /*"ds3CurrentUASs", /* UASs (Unavailable Seconds)    */
}; 

/************************************************************************
 *	DSUThresholds[] - Two adjacent 15 minute buckets can not exceed	*
 *			these measures.					*
 ************************************************************************/
static int DSUThresholds[]={
#ifdef DEBUG_TEST_USING_ZERO_THRESHOLD
	-1,0,0,0,0,0,0,0,
#endif
                        /* ESs ( Errored Seconds )              */      0,
                        /* SESs (Severely Errored Seconds )     */      16,
                        /* CSSs (Control Slip Seconds )         */      60,
                        /* SEFSs (Severly Errored Frames)       */      6,
                        /* UASs (Unavailable Seconds)           */      60,
			/* Future growth			*/	0,
			/* Future growth			*/	0,
			/* Future growth			*/	0,
			/* Future growth			*/	0,
									0
                        };

#ifdef ORIGINAL
/***********************************************************************
 *	ThreshExceeded() - Returns whether Threshold exceeded for this *
 *			DSU Variable based on the DSUThresholds[].     *
 ***********************************************************************/
ThreshExceeded(Var, bucket1, bucket2, bucket3)
char *Var;
int bucket1;
int bucket2;
int bucket3;
{
	register int i;

	for (i = 0; i < ((sizeof(DSUVarCurrent)) / (sizeof(char *))); i++ ) {
		if ((strcmp(DSUVarCurrent[i], Var) == 0) ||
		    (strcmp(DSUVarBuckets[i], Var) == 0)) {
			if ((bucket1 > DSUThresholds[i]) && 
			    (bucket2 > DSUThresholds[i])) 
				return(DSUThresholds[i]);
			if ((bucket2 > DSUThresholds[i]) && 
			    (bucket3 > DSUThresholds[i])) 
				return(DSUThresholds[i]);
			return(0);
		}
	}
	fprintf(stderr, "ThreshExceeded(): Couldn't find variable: %s\n", Var);
	return(0);
}
#else
/***********************************************************************
 *	ThreshExceeded() - Returns whether Threshold exceeded for this *
 *			DSU Variable based on the DSUThresholds[].     *
 ***********************************************************************/
ThreshExceeded(Var, bucket1, bucket2, bucket3)
char *Var;
int bucket1;
int bucket2;
int bucket3;
{
	register int i;

	for (i = 0; i < ((sizeof(DSUVarCurrent)) / (sizeof(char *))); i++ ) {
		if ((strcmp(DSUVarCurrent[i], Var) == 0) ||
		    (strcmp(DSUVarBuckets[i], Var) == 0)) {
			if ((bucket1 > DSUThresholds[i]) || 
			    (bucket2 > DSUThresholds[i]) ||
			    (bucket3 > DSUThresholds[i])) 
				return(DSUThresholds[i]);
			else return(0);		/* Within threshold */
		}
	}
	fprintf(stderr, "ThreshExceeded(): Couldn't find variable: %s\n", Var);
	return(0);
}
#endif

/************************************************************************
 *	AddDSUData2Query() - Create VarList for this DSU interface	*
 ************************************************************************/
struct VarList * AddDSUData2Query(head, dsu)
struct VarList *head;
int dsu;
{
	register struct VarList *vp;
	register int i;
	char DSUIndex[40], Var[50];

	sprintf(DSUIndex, "%d", dsu);
	for (i = 0; i < ((sizeof(DSUVarBuckets)) / (sizeof(char *))); i++) {
		/*
		 *  Get the 2nd Interval.
		 */
		strcpy( Var, DSUVarBuckets[i] );
		head = AddVar(head, Var);
		for (vp = head; vp->Next != NULL; vp = vp->Next )
			;
		strcat( vp->Prefix, DSUIndex );
		strcat( vp->Prefix, ".2");

		/*
		 *  Get the 1st Interval.
		 */
		strcpy( Var, DSUVarBuckets[i] );
		head = AddVar( head , Var );
		for( vp = head; vp->Next != NULL; vp = vp->Next )
			;
		strcat( vp->Prefix, DSUIndex );
		strcat( vp->Prefix, ".1");

		/*
		 *  Get the Current Interval.
		 */
		strcpy( Var, DSUVarCurrent[i] );
		head=AddVar( head , Var );
		for ( vp = head; vp->Next != NULL; vp = vp->Next )
			;
		strcat( vp->Prefix, DSUIndex );
	}
	return( head );
}

/************************************************************************
 *	MakeDSUAddressList() - Make Address List with appropriate 	*
 *				VarList for the DSUs this box has	*
 *	Note that this creates one packet per DSU - could be optimized  *
 ************************************************************************/
struct AddressListType * MakeDSUAddressList(new)
struct AddressListType *new;
{
	struct AddressListType *ap;
	register struct LinkDetail *lp;
	extern struct LinkDetail *LinkDetailHead;
	struct in_addr remote, local;

	for (lp = LinkDetailHead; lp != NULL; lp = lp->Next) {

		/* ignore T1 link */
		if (lp->LinkType != MIB_IFTYPE_PROPPOINTTOPOINTSERIAL)
			continue;

		/*
		 *  If the local and remote addresses are in the
		 *  same POP, then we don't want to do this.
		 */
		local.s_addr = inet_addr(lp->LocalNode);
		remote.s_addr = inet_addr(lp->RemoteNode);
		if ((local.s_addr & POP_MASK) == (remote.s_addr & POP_MASK))
			continue;

		/*
		 *  Local Address and DSU 
		 */
		new = AddAddress(new, lp->LocalNode);
		for (ap = new; ap->Next != NULL; ap=ap->Next)
			;
		if (verbose) 
			printf("Adding %s DSU #%d\n", lp->LocalNode, 
							lp->LocalDSUNumber);
		ap->QueryListHead = AddDSUData2Query(ap->QueryListHead, 
							lp->LocalDSUNumber);
		/*
		 *  Remote Address and DSU
		 */
		new = AddAddress(new, lp->RemoteNode);
		for (ap = new; ap->Next != NULL; ap = ap->Next)
			;

		if (verbose) 
			printf("Adding %s DSU #%d\n", lp->RemoteNode, 
						lp->RemoteDSUNumber);
		ap->QueryListHead = AddDSUData2Query(ap->QueryListHead, 
							lp->RemoteDSUNumber);
	}
	return(new);
}

/**************************************************************************
 *	PrintReport() - Print Report on the Variables Each DSU returned.  *
 **************************************************************************/
void PrintReport(AddressListHead)
struct AddressListType *AddressListHead;
{
	register struct AddressListType *A;
	register struct VarList *V;
	struct bindings *b;
	char *lookup_SNMP_name();
	int dsunum, bucket1, bucket2, bucket3, t;
	char buffer[100],temp[40], dataline[300], *s;
	extern void WriteDataLine(), DealWithDirtyLinks();

	for (A = AddressListHead; A != NULL; A = A->Next) {
		if (A->ResponseListHead == NULL) { 
			fprintf(stderr, "%s did not respond to dsu query\n",
								A->Address);
			continue;
		}
		if (A->QueryListHead != NULL)
			printf("%s did not completely answer DSU Query\n",
								A->Address);
		
		for (V = A->ResponseListHead; V != NULL; V = V->Next) {
			if ((b=V->Head) == NULL) {
				printf("%s %s (%s) did NOT respond\n",
					A->Address, V->VarName, V->Prefix);
				break;
			}

			/* first bucket */
			s = b->instance+strlen(lookup_SNMP_name(V->VarName));
			sprintf(buffer,"%s\t%15.15s.%1.1s\t%s", 
			  A->Address, V->VarName, s, b->value);
			strcpy( dataline, buffer );
			dsunum = atoi(s);
			bucket1 = atoi(b->value);

			/* second bucket */
			V = V->Next; 
			if ( V==NULL || ((b=V->Head)==NULL) ) {
				printf("Second bucket pointer was NULL!\n");
				break;
			}
			sprintf(buffer,"\t%-7s\t", b->value);
			strcat( dataline, buffer );
			bucket2 = atoi(b->value);

			/* third bucket */
			V = V->Next; 
			if ( V==NULL || ((b=V->Head)==NULL) ) {
				printf("Third bucket pointer was NULL!\n");
				break;
			}
			sprintf(buffer,"%-7s", b->value);
			strcat( dataline, buffer );
			bucket3=atoi(b->value);
			strcpy( temp, "" );
			if ((t = ThreshExceeded( V->VarName, bucket1, bucket2, 
							bucket3 )) != 0) {
				sprintf(temp," [ %d ]",t);
				fprintf( stderr,"Attn: %s%s\n",dataline,temp);
				MarkLinkDirty( A->Address, dsunum );
			}
			strcat( dataline, temp);
			WriteDataLine( A->Address, dsunum, dataline );
			UpdateLinkTimeStamp( A->Address, dsunum );

			/* should be NULL now */
			if ((b = b->Next) != NULL)
				printf("Junk after third bucket!\n");
		}
	}
}

main(argc, argv)
int argc;
char *argv[];
{
	char *community = NULL, *workdir;
	struct AddressListType *NewAddressListHead = NULL;
	int c;
	extern int io_debug, DebugOnFailure;
	extern char *optarg, *getenv();
	extern void ClearLastSampleFiles();
	char *network="nsfnett3";

#ifdef PROFILING
	monitor((caddr_t) NULL);
#endif
	progname = argv[0];
	while ((c = getopt(argc, argv, "n:c:dr:t:v")) != EOF)
		switch (c) {

                case 'd':
			io_debug = 1;
			/* now fall through */

                case 'v':       
			fprintf(stderr, "Verbose Mode = ON\n");
			DebugOnFailure = 1;
			verbose = 1;
			break;

                case 'c':
                        community = optarg;
                        break;

                case 'n':
                        network = optarg;
                        break;

                case 'r':
                        MaxSNMPRetries = atoi(optarg);
                        break;

                case 't':
                        MaxSecsB4Retry = atoi(optarg);
                        break;

                default:
			Usage();
			/*NOTREACHED*/
                }
	if (community == NULL) 
		Usage();
	
	if ((workdir = getenv("PINGKYDIR")) == NULL)
		workdir = DEFAULT_PINGKY_DIR;

	ClearLastSampleFiles();
	Read_LinkDetail_File( network );
	NewAddressListHead = MakeDSUAddressList( NewAddressListHead );
	NewAddressListHead=Get( NewAddressListHead, community, NULL );
	PrintReport( NewAddressListHead );
	DealWithDirtyLinks(workdir);
	NewAddressListHead = FreeAddressList( NewAddressListHead );
	Write_LinkDetail_File();
	return(0);
}

