/************************************************************************
 *			Display.c	Display Current Problems 	*
 *									*
 *	   This program displays the problems in a format useful to	*
 *	the operators.  There is a date and time field for when the	*
 *	problem occured, and an Update command so that a problem	*
 *	status can be carried across operator shifts.  Each Update	*
 * 	appends a timestamped event to the pingky.log file as well.	*
 *									*
 *	   This Display program may be modified to the installations	*
 *	specifications without modifying the background data collection.*
 *									*
 *	   I encourage programmers to make additions to this code and	*
 *	forward them to me for inclusion in the next release.		*
 *									*
 ************************************************************************
 * Author:   WB Norton	4/23/89		Merit Computer Network		*
 *									*
 *	Version 1.0   Author: WB Norton, Merit Computer Network         *
 *									*
 * Modification History:						*
 * Written 5/18/89 Bill Norton, Merit Computer Network                  *
 *									*
 ************************************************************************/
/*
  * Copyright 1989
  * The Regents of the University of Michigan
  * All Rights Reserved
  *
  * Permission to use, copy, modify, and distribute this software and its
  * documentation for any purpose and without fee is hereby granted, provided
  * that the above copyright notice and this permission notice appear in
  * all copies of the software and derivative works or modified versions
  * thereof, and that both that copyright notice and this permission
  * notice appear in supporting documentation.
  *
  * The software is provided "as is" and the University of Michigan
  * disclaims all warranties with regard to this software, including
  * all implied warranties of merchantability and fitness.  In no event
  * shall the University of Michigan be liable for any special, direct,
  * indirect, or consequential damages or any damages whatsoever
  * resulting from loss of use, data or profits, whether in an action of
  * contract, negligence or other tortious action, arising out of or in
  * connection with the use or performance of this software.
  */


#include "../INCLUDE/defs.h"
#include "../INCLUDE/fileio.h"
#include <curses.h>
#include <signal.h>
#include <ctype.h>
#include <string.h>
#include <sys/file.h>

int version=1;
int subversion=2;

char DefaultPingkyDir[]=".";
char hostfile[100];			/* List of nodes we are checking */
char Problem_File[100];			/* PROBLEM FILE path */
char Checking_File[100];		/* What node are we checking? */
char cycletime[100];			/* How long did pingky take to cycle */
char logfile[100];			/* Where do we log things */

		/* Update Interval : How often do we check to see if the */
		/*                   problem file changed.		 */
#define UPDATEINTERVAL 30 /* Check for PROBLEM.FILE changes every 30 secs*/

char *delimiters=" \t\n";               /* A list of separator delimiters    */

char Updating=0;	/* Updating flag */
#ifdef NOC
char UpdateAble=1;	/* Allow updating by default */
#else
char UpdateAble=1;	/* Don't let others update */
#endif

struct NodeType Node[MAXNODES]; /* In Core version of node file */
int NumNodes=0;                 /* Number of nodes we are to keep trackof */

struct ProblemType Problem[MAXPROBLEMS];/*In Coreversion of existing problems*/
int NumProblems=0;

long CycleLastUpdated;         	/* Last time pingky cycle time was Updated */
long ProblemFileLastUpdated;    /*   Last time PROBLEM.FILE was updated    */
long NodeFileLastUpdated;       /*   Last Time the node file was updated   */


/********************************************************
 * PrintTime:  Print the current time on the screen.	*
 ********************************************************/
PrintTime()		/* Print the current time on the screen */
{
char mybuf[100]; 

    mvaddstr(0,63,GetTime(mybuf)); 
}

/**********************************************************
 *  Message: Put a message to the user on the bottom line * 
 **********************************************************/
void Message(msg)
char *msg;
{
   standout();
   mvaddstr(LINES-1,0,msg);
   standend();
   clrtoeol();
   refresh();
}

/********************************************************
 * DrawFrame: Draw a frame in which problems get logged	*
 ********************************************************/
DrawFrame()
{
char TopLine[80];

    clear();
    sprintf(TopLine,"Merit Internet Rover v%d.%d          NOC Watch",version,subversion);
    mvaddstr(0,0,TopLine);
    PrintTime(); 
    /*mvaddstr(3,0,"# Reported                Problem                        Status");*/
    mvaddstr(3,0,"# Reported                Problem                                 Status");
    /*mvaddstr(4,0,"  ========= ---------------------------------- ---------------------------------");*/
      mvaddstr(4,0,"  ========= -----------------------------------------  -------------------------");
    move(20,0);
}

/**************************************************************************
 *  PrintCycleTime: Print the current cycle time of the background pingky * 
 **************************************************************************/
PrintCycleTime()
{
FILE *stream;
char buffer[10];

   mvaddstr(1,0,"Cycle Time:");
   if ((stream=fopen(cycletime,"r"))==NULL) mvaddstr(1,12,"--:--");
   else {
      if (fgets(buffer,10,stream)==NULL) mvaddstr(1,12,"--:--");
      else mvaddstr(1,12,buffer);
      fclose(stream);
   }
}

/******************************************************************
 * ReadChecking:  See what node the background pingky is querying *
 ******************************************************************/
ReadChecking()
{
FILE *stream;
char buffer[MAXLINE];

   if ((stream=fopen(Checking_File,"r"))==NULL) {
      perror("Checking_File NOT FOUND! Error.");
   }
   else {
      fgets(buffer,MAXLINE,stream);
      mvaddstr(LINES-1,0,"Checking: ");
      mvaddstr(LINES-1,10,buffer);
      clrtoeol();
      fclose(stream);
   }
}

/*****************************************************************
 * ReadProblems:  Read Problems and Display on the screen nicely *
 *****************************************************************/
ReadProblems()
{
char buffer[10];

   NumProblems=ReadProblemFile(Problem,Problem_File);
   sprintf(buffer,"[%2d]",NumProblems);
   mvaddstr(1,75,buffer);
   DrawProblems();
}

/****************************************************************
 * DrawProblems:  Draw Problems on the screen in a nice format	*
 ****************************************************************/
DrawProblems()
{
int i,y=5;
char buffer[100];
 
   if (NumProblems==0) { 
      move(y,0);clrtoeol();
      mvaddstr(y++,30,"**** No Problems ****");
   }
   else
      for(i=0; i<NumProblems && y<LINES-2; i++,y++) {
         sprintf(buffer,"%d %s %s %s %s %s",i+1,Problem[i].Day,Problem[i].Time,
                 Problem[i].Name,Problem[i].Addr,Problem[i].NetProto);
         mvaddstr(y,0,buffer);
	 clrtoeol();
         mvaddstr(y,STATUSX,Problem[i].Status);
         /* if (Problem[i].Status[0] == '\0')   */
         /*   printf("%c",7);			*/
      } 
   if (i<NumProblems) mvaddstr(2,65,"[ MORE... ]");
   else {
	mvaddstr(2,65,"           ");
      for(; y<LINES-2; y++)
         for(i=0; i<COLS; i++)
            mvaddch(y,i,' '); 
   }
}

/****************************************************************
 * GotoCmdLine:  Got the Command Prompt 			*
 ****************************************************************/
GotoCmdLine()
{
   move(LINES-2,0); clrtoeol();
   standout();
   addstr("Command:");
   standend();  
   move(LINES-2,9);
}

/****************************************************************
 * RedrawScreen:  Force Update of the user's screen		*
 *		Call this routine
 ****************************************************************/
RedrawScreen()
{
      if (FileChanged(Problem_File,&ProblemFileLastUpdated)) {
         ReadProblems();
         DrawProblems();
      }
      ReadChecking();
      if (FileChanged(cycletime,&CycleLastUpdated)) PrintCycleTime();
      PrintTime();
      GotoCmdLine();
      refresh();
}

/****************************************************************
 * RefreshScreen:  Force Update of the user's screen		*
 *		   DO NOT CALL THIS ROUTINE!!!!			*
 *		   The user callable version is RedrawScreen	*
 *		   RefreshScreen should only be called 		*
 *		   by alarm clock because it restarts the alarm *
 *		   and we don't want multiple alarms to go off. *  
 *		   once every UPDATEINTERVAL seconds.		*
 ****************************************************************/
RefreshScreen()
{
   if (!Updating) {
	RedrawScreen();	
   }
   alarm(UPDATEINTERVAL);
}

/****************************************************************
 * ClearMessage:  Clear the message line			*
 ****************************************************************/
void ClearMessage()
{
   move(LINES-1,0);
   clrtoeol();
   refresh();
}

/***********************************************************************
 * GetUpdateList: get the user's request to update a list of problems. *
 ***********************************************************************/
int GetUpdateList()
{
char problemID[100];

   getstr(problemID);		/* 	get the user's string 	*/
   return(atoi(problemID));
}

/****************************************************************
 *		Update - Update the status of a problem		*
 ****************************************************************/
void Update(NumProblems,Problem_File)
int NumProblems;
char *Problem_File;
{
char ProblemList[100];	/* array of nodes to be updated at once */
int i;
char buffer[150];

   if (  access(Problem_File,W_OK) ) {
      Message("    You do not have permission to update    "); 
      sleep(5);
      return;
   } 
   for(i=0; i<100; i++) ProblemList[i]=0;   
   if (NumProblems==0) {
      Message("  There are no problems to update, Bonehead!");
      sleep(5);
      ClearMessage();
      return;
   }
   if (UpdateAble) {
      Updating=1;
      Message("   Enter the problem # that you want to Update   ");
      mvaddstr(LINES-2,10,"pdate Problem # ");
      refresh();
      if ((i=GetUpdateList())==0) {
         ClearMessage();
         Updating=0;
         return;
      }
      if (i>0 && i<=NumProblems) {
         Message("  Enter the Status as you want it to appear   ");
         move(4+i,STATUSX);
         clrtoeol(); 
         refresh();
         getstr(buffer);
	 buffer[MAXSTATUS-1]='\0';
	 strcpy(Problem[i-1].Status,buffer);
         sprintf(buffer,"%s New Status: %s\n",Problem[i-1].Name,Problem[i-1].Status);
         Log(buffer,logfile);
         WriteProblemFile(Problem,Problem_File,NumProblems);
      }
      else { 
         Message("   Illegal Problem Number   ");
         sleep(5);
         ClearMessage();
      } 
      Updating=0;
   }
}

/****************************************************************
 * Loop:  main loop of program.  				*
 *		Set an alarm to go off in 30 seconds		*
 *			and provide address of alarm routine	*
 *		Read the Problem List, Checking file		*
 *		Update the screen and display the current time	*
 *		Forever						*
 *			Get a key stroke and handle user cmd.	*	
 ****************************************************************/
Loop( Problem_File )
char *Problem_File;
{

   signal(SIGALRM,RefreshScreen);	/*  Invoke RefreshScreen on alarm  */
   alarm(UPDATEINTERVAL);		/* Alarm in UPDATEINTERVAL seconds */
   while(1) {
     ReadProblems();
     ReadChecking();
     PrintTime();
     PrintCycleTime();
     GotoCmdLine();
     refresh();
     switch(getch()) {
    case 'u':
    case 'U': 
             Update(NumProblems,Problem_File);
             break;
    case 'q':
    case 'Q':
              addstr("uit");
              move(LINES-1,0);
              clrtoeol();
              refresh();
              return;
              break;
    case 'h':
    case 'H':
              	addstr("elp ");
		GetHelp();		
              break;
    case 'a':
    case 'A':
		addstr("sk ");
		GetQuery();
		break;
    case 'p':
    case 'P':
		addstr("ing ");
		Ping();
		break;
    case '!':
		addstr(" ");
		/* Here put password protection and a shell &*/
		break;
		
		
    case '?':
	      Message("   Valid Commands: (U)pdate Problem (H)elp (P)ing (Q)uit   ");
	      sleep(5);
              ClearMessage();
	      break;
    default:  clear();
            DrawFrame(); 
            break;
    } 
   }
}

/**********************************************************************
 * GetQuery() - Allow user to Query any node for any SNMP variable    *
 **********************************************************************/
GetQuery()
{
char cmd[100],nodename[100],var[100];

   if (  access(Problem_File,W_OK) ) {
      Message("    You do not have permission to ask for anything");
      sleep(5);
      return;
   }

	Updating=1;
	Message("  Enter the node name that you want Query");	
	move(LINES-2,13);
	refresh();
	getstr(nodename);
	Message("  Enter the variable that you want ");	
	move(LINES-2,14+strlen(nodename));
	refresh();
	getstr(var);
	clear();
	Message("                       Type Q to quit                    "); 
        sprintf(cmd,"ask %s %s | more",nodename,var);
        printf("\nQuery Cmd=%s\nWaiting for Answer:",cmd);
        system(cmd);
	Updating=0;
	DrawFrame();
	RedrawScreen();
}

/************************************************************************
 * Ping() - Allow user to Ping any node 				*
 ************************************************************************/
Ping()
{
char cmd[100],nodename[100];

   if (  access(Problem_File,W_OK) ) {
      Message("    You do not have permission to ping anything");
      sleep(5);
      return;
   }

	Updating=1;
	Message("  Enter the node name that you want Ping");	
	move(LINES-2,14);
	refresh();
	getstr(nodename);
	clear();
	Message("                       Type Q to quit                    "); 
        sprintf(cmd,"ping `cat hostfile | grep %s | awk '{ print $2 }'` | more",nodename);
        printf("\nWaiting for Answer:",cmd);
        system(cmd);
	Updating=0;
	DrawFrame();
	RedrawScreen();
}


/************************************************************************
 *  GetHelp:	Display Help screen for a particular problem node	*
 ************************************************************************/
GetHelp()
{
char cmd[100];
char nodename[100];

   if (  access(Problem_File,W_OK) ) {
      Message("    You do not have permission to ask for help"); 
      sleep(5);
      return;
   } 
   if (FileChanged(hostfile,&NodeFileLastUpdated)) /*  if modified   */
      NumNodes=ReadNodeFile(Node,hostfile);       /* reread hostfile */

	Updating=1;
	Message("  Enter the node name that you want help with  ");	
	move(LINES-2,14);
	refresh();
	getstr(nodename);
	clear();
	Message("                       Type Q to quit                    "); 
	sprintf(cmd,"more `grep -i '%s' %s |tee /dev/tty| awk '{ print $%d }'`",
		nodename,hostfile,HELPFILE); 
	system(cmd);
	Updating=0;
	DrawFrame();
	RedrawScreen();
}

/************************************************************************
 *   Main Program:     Start up the curses screen package and start 	*
 *				command processing loop			*
 ************************************************************************/
main(argc,argv)
int argc;
char *argv[];
{
char *pingkydir,*getenv();

    if ( argc != 1 ) {
	printf("Usage: %s\n",argv[0]);
	exit(1);
    }	
    if ((pingkydir=getenv("PINGKYDIR"))==NULL) pingkydir=DefaultPingkyDir;
    strcat(pingkydir,"/");

    strcpy(Problem_File,pingkydir); strcat(Problem_File	,"PROBLEM.FILE");
    strcpy(Checking_File,pingkydir); strcat(Checking_File,"CHECKING");
    strcpy(cycletime,pingkydir); strcat(cycletime,"pingky.cycle");
    strcpy(logfile,pingkydir); strcat(logfile,"pingky.log");
    strcpy(hostfile,pingkydir); strcat(hostfile,"hostfile");

    initscr();			/* Open Up the curses environment	  */
    DrawFrame();		/* Draw the main Problem Screen Framework */
    Loop( Problem_File );	/* Call the main command handling loop 	  */
    endwin(); 			/* Close down the curses environment 	  */
}
