/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       mgt.c
**     SYSTEM   NAME:       SNMP Management Module
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     1
**     CREATION DATE:       1990/11/22
**
** DESCRIPTION: SNMP Management Module
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.0  $
** WORKFILE:    $Workfile:   mgt.c  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/SNMP/SNMP/VCS/MGT.C_V  $
**
**                 Rev 1.0   26 Nov 1990 16:38:02   etstjan
**              ipdate
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/SNMP/SNMP/VCS/MGT.C_V   1.0   26 Nov 1990 16:38:02   etstjan  $";
#endif
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include "snmp.h"
#include "err.h"
#include "pkt.h"


/**************************************************************
** NAME:        MgtAdr
** SYNOPSIS:    long unsigned
**                  MgtAdr
**                  (
**                      char        *Hst
**                  )
** DESCRIPTION: Translates hostname or address to IP address.
**              Parameters:
**              Hst: IP hostname or address in string.
** RETURNS:     0xFFFFFFFF --> error
**              else           IP address
**************************************************************/
unsigned long MgtAdr (Hst)
char    *Hst;
{
    struct hostent
        *Ent;
    long unsigned
        Adr;

    Ent = gethostbyname (Hst);
    if (Ent != NULL)
        Adr = *((long unsigned *)(Ent->h_addr_list [0]));
    else
        Adr = inet_addr (Hst);
    return (Adr);
}

char *MgtHst (Adr)
unsigned long Adr;
{
    struct hostent
        *Ent;
    static char 
	Hst[256];
    struct in_addr jan;

    Ent = gethostbyaddr ((char *)&Adr,sizeof(struct sockaddr_in),AF_INET);
    if (Ent != NULL)
        strcpy(Hst, Ent->h_name);
    else
    {
	jan.s_addr = (long) &Adr;
        strcpy(Hst, inet_ntoa(jan));
    }
    return (Hst);
}


/**************************************************************
** NAME:        MgtOpn                                    [API]
** SYNOPSIS:    mgt_sck
**                  *MgtOpn
**                  (
**                      short       Prt,
**                      long        Adr
**                  )
** DESCRIPTION: Opens a managent station.
**              Parameters:
**              Prt: IP Port.
**              Adr: IP Address.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
mgt_sck 
	*MgtOpn 
	(
		short Prt,
		long  Adr
	)
{
    IPS_IPA
        Sin;
    mgt_sck
        *Sck;

    if ((Sck = (mgt_sck *)malloc (sizeof (mgt_sck))) == NULL)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Could not allocate buffer\n");
        return (NULL);
    }
    Sin.sin_family = AF_INET;
    Sin.sin_port = Prt;
    Sin.sin_addr.s_addr = Adr;
    if ((Sck->SckBsd = socket (AF_INET, SOCK_DGRAM, 0)) == IPS_ERR)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "No socket\n");
        return (NULL);
    }
    if (connect (Sck->SckBsd, (IPS_ADR *)&Sin, sizeof (IPS_IPA)) < 0)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Cannot connect socket\n");
        return (NULL);
    }
    Sck->SckAdr = Adr;
    Sck->SckRid = 0;
    return (Sck);
}

/**************************************************************
** NAME:        MgtRqs                                    [API]
** SYNOPSIS:    int
**                  MgtRqs
**                  (
**                      mgt_sck         *Sck,     
**                      pkt_msg         *Msg,
**                      long unsigned   Tio,   
**                      unsigned        Rtr
**                  )
** DESCRIPTION: Sends a request.
**              Parameters:
**              Sck: Socket created with MgtOpn.
**              Msg: Request message.
**              Tio: Time out in micro sec.
**              Rtr: Retries.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MgtRqs (Sck,Msg,Tio,Rtr)                             
mgt_sck         *Sck;
pkt_msg         *Msg;
long unsigned   Tio;
unsigned        Rtr;
{
    pkt_ror
        *Ror;
    char
        Rqs [MGT_SZEPKT], Rsp [MGT_SZEPKT], *RqsBeg, *RspBeg;
    int
        RqsLen, RspLen, Ret;
    unsigned long
        Len;
    fd_set fdvar;
    struct timeval to;

    Ror = &Msg->MsgPdu.PduRor;
    Ror->RorRid = ++(Sck->SckRid);
    Ror->RorErrSts = 0;
    Ror->RorErrInd = 0;
    RqsBeg = Rqs;
    RqsLen = MGT_SZEPKT;
    if (PktEnc (Sck->SckAdr, Msg, &RqsBeg, (unsigned *)&RqsLen) < 0)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Cannot encode request\n");
        return (-1);
    }

    while (Rtr-- > 0)
    {
        if (send (Sck->SckBsd, RqsBeg, RqsLen, 0) < 0)
	{
	    perror("Send failed\n");
	    continue;
	}

	/* JvO replace ioctl by select call */
	to.tv_sec = Tio / 1000000L;
	to.tv_usec = Tio % 1000000L;
	
        Ret=-1;
	FD_ZERO(&fdvar);
	FD_SET(Sck->SckBsd,&fdvar);
	while (Ret<0)
        {
		Ret=select(Sck->SckBsd+1,(int *) &fdvar,NULL,NULL,&to);
                if (Ret==EBADF || Ret==EFAULT || Ret==EINVAL)
	        {
                     perror("Select failed\n");
                     return (-3);
    	        }
	        FD_ZERO(&fdvar);
	        FD_SET(Sck->SckBsd,&fdvar);
        }

        while ((ioctl (Sck->SckBsd, FIONREAD, (char *)&Len), (int)Len) > 0)
        {
	    RspBeg = Rsp;
            RspLen = MGT_SZEPKT;
            if ((RspLen = recv (Sck->SckBsd, RspBeg, RspLen, 0)) < 0)
            {
		perror("Receive failed\n");
	        continue;
            }
	    if (PktDec (Sck->SckAdr, Msg, RspBeg, RspLen) < 0)
    	    {
        	ERR_ERR (ERR_MGT, ERR_WARNING, "Cannot decode request\n");
	        continue;
    	    }
	    if (Ror->RorRid != Sck->SckRid)
	        continue;
	    return (0);
        }
    }
    return (-2);
}

/**************************************************************
** NAME:        MgtCls                                    [API]
** SYNOPSIS:    int
**                  MgtCls
**                  (
**                      mgt_sck *Sck
**                  )
** DESCRIPTION: Closes a management station.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int MgtCls
(
    mgt_sck *Sck
)
{
    IPS_CLS (Sck->SckBsd);
    free (Sck);
    return (0);
}


int MgtSnd
(
    IPS_SCK	    Bsd,
    pkt_msg         *Msg,
    unsigned short  Prt,
    unsigned long   Adr
)
{
    IPS_IPA
        Sin;
    int
        RqsLen, SinLen;
    pkt_ror
        *Ror;
    char
        Rqs [MGT_SZEPKT], *RqsBeg;

    Ror = &Msg->MsgPdu.PduRor;
    Ror->RorErrSts = 0;
    Ror->RorErrInd = 0;
    RqsBeg = Rqs;
    RqsLen = MGT_SZEPKT;
    if (PktEnc (Adr, Msg, &RqsBeg, (unsigned *)&RqsLen) < 0)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Cannot encode request\n");
        return (-1);
    }
    Sin.sin_family = AF_INET;
    Sin.sin_port = Prt;
    Sin.sin_addr.s_addr = Adr;
    SinLen = sizeof (IPS_IPA);
    if (sendto (Bsd, RqsBeg, RqsLen, 0, 
               (struct sockaddr *)&Sin, SinLen) < 0)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Send failed\n");
        return (-2);
    }
    return 0;
}


int MgtRcv
(
    IPS_SCK	    Bsd,
    pkt_msg         *Msg,
    unsigned short  *Prt,
    unsigned long   *Adr
)
{
    IPS_IPA
        Sin;
    int
        RspLen, SinLen;
    char
        Rsp [MGT_SZEPKT], *RspBeg;

    RspBeg = Rsp;
    RspLen = MGT_SZEPKT;
    Sin.sin_family = AF_INET;
    SinLen = sizeof (IPS_IPA);
    if ((RspLen = recvfrom (Bsd, RspBeg, RspLen, 0, 
                       (struct sockaddr *)&Sin, &SinLen)) < 0)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Receive failed\n");
        return (-1);
    }
    *Prt = Sin.sin_port;
    *Adr = Sin.sin_addr.s_addr;
    if (PktDec (*Adr, Msg, RspBeg, RspLen) < 0)
    {
        ERR_ERR (ERR_MGT, ERR_CRITICAL, "Cannot decode request\n");
        return (-2);
    }
    return 0;
}

int MgtLenRcv
(
    IPS_SCK         Bsd
)
{
    unsigned long Len;

    ioctl (Bsd, FIONREAD, (char *)&Len);
    return (int)Len;
}

