/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       agt.c
**     SYSTEM   NAME:       SNMP Agent Module
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     1
**     CREATION DATE:       1990/11/28
**
** DESCRIPTION: SNMP Agent Module
**
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header$";
#endif
#include <err.h>
#include <snmp.h>
#include "pkt.h"
#include "agt.h"


agt_sck
    AgtSck =
{
    0,
    0,
    IPS_ERR,
    {AF_INET, 0, 0},
    {AF_INET, 0, 0},
    IPS_ERR,
    {AF_INET, 0, 0},
    "trap",
    4,
    {1, 3, 6, 1, 4, 9999},
    6,
    0L,
    0,
    0
};

/**************************************************************
** NAME:        AgtIni                                    [API]
** SYNOPSIS:    int
**                  AgtIni
**                  (
**                      short       Prt,
**                      unsigned    *Oji,
**                      unsigned    OjiLen,
**                      int         TrpEna,
**                      short       TrpPrt,
**                      char        *TrpHst,
**                      char        *TrpCom,
**                      unsigned    TrpComLen
**                  )
** DESCRIPTION: Initializes the agent station.
**              Parameters:
**              Prt: IP Port for receiving requests.
**              Oji: Object Identifier for agent.
**              OjiLen: Length of Object Identifier.
**              TrpEna: Enable, Disable (1, !1) traps.
**              TrpPrt: IP Port,
**              TrpHst: IP Host,
**              TrpCom: Community to send traps to.
**              TrpComLen: Length of community in characters.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AgtIni
    (
        short       Prt,
        unsigned    *Oji,
        unsigned    OjiLen,
        int         TrpEna,
        short       TrpPrt,
        char        *TrpHst,
        char        *TrpCom,
        unsigned    TrpComLen
    )
{
    IPS_IPA
        *Sin;

    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Initializing agent\n");
    AgtSck.SckIni = 1;
    if ((AgtSck.SckOjiLen = OjiLen) > AGT_SZEOJI)
        return (-1);
    memcpy (AgtSck.SckOji, Oji, OjiLen * sizeof (unsigned));
    Sin = &AgtSck.SckLocAdr;
    Sin->sin_family = AF_INET;
    Sin->sin_addr.s_addr = INADDR_ANY;
    Sin->sin_port = htons (Prt);
    if (!TrpEna)
        return (0);
    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Initializing trap\n");
    AgtSck.SckEnaTrp = 1;
    AgtSck.SckEnaAut = 1;
    Sin = &AgtSck.SckTrpAdr;
    Sin->sin_family = AF_INET;
    Sin->sin_addr.s_addr = inet_addr (TrpHst);
    Sin->sin_port = htons (TrpPrt);
    if ((AgtSck.SckComLen = TrpComLen) > AGT_SZECOM)
        return (-1);
    memcpy (AgtSck.SckCom, TrpCom, TrpComLen);
    return (0);
}


/**************************************************************
** NAME:        AgtOpn                                    [API]
** SYNOPSIS:    int
**                  AgtOpn
**                  (
**                      void
**                  )
** DESCRIPTION: Opens the agent station.
**              Agent must be closed.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AgtOpn
    (
        void
    )
{
    IPS_IPA
        Adr;
    unsigned
        AdrLen = sizeof (IPS_IPA);

    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Opening agent\n");
    if (AgtSck.SckOpn)
        return (-1);
    if ((AgtSck.SckRor = socket (AF_INET, SOCK_DGRAM, 0)) == IPS_ERR)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "No socket\n");
        return (-1);
    }
    if (bind (AgtSck.SckRor, (IPS_ADR *)&AgtSck.SckLocAdr, sizeof (IPS_IPA)) < 0)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "Cannot bind socket\n");
        return (-1);
    }
    if ((AgtSck.SckTrp = socket (AF_INET, SOCK_DGRAM, 0)) == IPS_ERR)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "No trap socket\n");
        return (-1);
    }
    if (connect (AgtSck.SckTrp, (IPS_ADR *)&AgtSck.SckTrpAdr, sizeof (IPS_IPA)) < 0)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "Cannot connect trap socket\n");
        return (-1);
    }
    if (getsockname (AgtSck.SckTrp, (IPS_ADR *)&Adr, &AdrLen) < 0)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "Cannot get socket name\n");
        return (-1);
    }
    AgtSck.SckIpa = Adr.sin_addr.s_addr;
    if (AgtSck.SckIni)
    {
        ERR_DEB (ERR_AGT, ERR_MESSAGE, "Sending coldstart\n");
        AgtSck.SckIni = 0;
        AgtSndTrp (PKT_COLDSTART, 0, NULL, 0);
    }
    else
    {
        ERR_DEB (ERR_AGT, ERR_MESSAGE, "Sending warmstart\n");
        AgtSndTrp (PKT_WARMSTART, 0, NULL, 0);
    }
    AgtSck.SckOpn = 1;
    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Agent ready\n");
    return (0);
}

/**************************************************************
** NAME:        AgtCls                                    [API]
** SYNOPSIS:    int
**                  AgtCls
**                  (
**                      void
**                  )
** DESCRIPTION: Closes the agent station.
**              Agent must be open.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AgtCls
    (
        void
    )
{
    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Closing agent\n");
    if (!AgtSck.SckOpn)
        return (-1);
    IPS_CLS (AgtSck.SckRor);
    IPS_CLS (AgtSck.SckTrp);
    AgtSck.SckOpn = 0;
    return (0);
}

/**************************************************************
** NAME:        AgtSndTrp                                 [API]
** SYNOPSIS:    int
**                  AgtSndTrp
**                  (
**                      unsigned    Gen,
**                      unsigned    Spe,
**                      pkt_obj     *Lst,
**                      unsigned    LstLen
**                  )
** DESCRIPTION: Sends a trap.
**              Parameters:
**              Gen: General trap.
**              Spe: Only specified if Enterprise Specific trap.
**              Lst: List of objects if necessary.
**              LstLen: Length of list.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AgtSndTrp
    (
        unsigned    Gen,
        unsigned    Spe,
        pkt_obj     *Lst,
        unsigned    LstLen
    )
{
    char
        *Pkt;
    int
        PktLen;
    pkt_msg
        *Msg;
    pkt_trp
        *Trp;

    if (!AgtSck.SckEnaTrp)
        return (0);
    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Sending Trap\n");
    Msg = &AgtSck.SckMsg;
    Trp = &Msg->MsgPdu.PduTrp;
    memcpy (Msg->MsgCom, AgtSck.SckCom, AgtSck.SckComLen);
    Msg->MsgComLen = AgtSck.SckComLen;
    Trp->TrpTyp = PKT_TRP;
    memcpy (Trp->TrpOji, AgtSck.SckOji, AgtSck.SckOjiLen * sizeof (int));
    Trp->TrpOjiLen = AgtSck.SckOjiLen;
    Trp->TrpIpa = AgtSck.SckIpa;
    Trp->TrpGen = Gen;
    Trp->TrpSpe = Spe;
    Trp->TrpTit = TitGet ();
    if ((Trp->TrpLstLen = LstLen) > PKT_SZETRPLST)
        return (-1);
    memcpy (Trp->TrpLst, Lst, LstLen * sizeof (pkt_obj));
    Pkt = AgtSck.SckPkt;
    PktLen = AGT_SZEPKT;
    if (PktEnc (AgtSck.SckTrpAdr.sin_addr.s_addr, &AgtSck.SckMsg, &Pkt, &PktLen) < 0)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "Cannot encode trap\n");
        return (-1);
    }
    if (send (AgtSck.SckTrp, Pkt, PktLen, 0) < 0)
    {
        ERR_ERR (ERR_AGT, ERR_CRITICAL, "Cannot send trap\n");
        return (-1);
    }
    ERR_DEB (ERR_AGT, ERR_MESSAGE, "Trap send\n");
    return (0);
}

/**************************************************************
** NAME:        AgtRcvRqs                                 [API]
** SYNOPSIS:    int
**                  AgtRcvRqs
**                  (
**                      char        *Com,
**                      unsigned    *ComLen,
**                      pkt_ror     *Rqs
**                  )
** DESCRIPTION: Receives a request.
**              Parameters:
**              Com: Community at least of size AGT_SZECOM.
**              ComLen: Length of community.
**              Rqs: Request.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AgtRcvRqs
    (
        char        *Com,
        unsigned    *ComLen,
        pkt_ror     *Rqs
    )
{
    char
        *Pkt;
    int
        PktLen,
        AdrLen;
    pkt_msg
        *Msg;
    pkt_ror
        *Ror;

    Pkt = AgtSck.SckPkt;
    PktLen = AGT_SZEPKT;
    AdrLen = sizeof (IPS_IPA);
    if ((PktLen = recvfrom (AgtSck.SckRor, Pkt, PktLen, 0,
        (IPS_ADR *)&AgtSck.SckRemAdr, &AdrLen)) < 0)
        return (-1);
    switch (PktDec (AgtSck.SckRemAdr.sin_addr.s_addr, &AgtSck.SckMsg, Pkt, PktLen))
    {
        case PKT_ERRAUT:
            if (AgtSck.SckEnaAut)
                AgtSndTrp (PKT_AUTFAILURE, 0, NULL, 0);
            return (-1);
        case PKT_ERRASN:
            return (-1);
        case PKT_ERROKY:
            break;
    }
    Msg = &AgtSck.SckMsg;
    Ror = &Msg->MsgPdu.PduRor;
    if (Ror->RorTyp > PKT_SETRQS)
        return (-1);
    *ComLen = Msg->MsgComLen;
    memcpy (Com, Msg->MsgCom, Msg->MsgComLen);
    memcpy (Rqs, Ror, sizeof (pkt_ror));
    return (0);
}

/**************************************************************
** NAME:        AgtSndRsp                                 [API]
** SYNOPSIS:    int
**                  AgtSndRsp
**                  (
**                      char        *Com,
**                      unsigned    ComLen,
**                      pkt_ror     *Rsp
**                  )
** DESCRIPTION: Sends a response.
**              Parameters:
**              Com: Community.
**              ComLen: Length of community at most AGT_SZECOM.
**              Rqs: Request.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    AgtSndRsp
    (
        char        *Com,
        unsigned    ComLen,
        pkt_ror     *Rsp
    )
{
    char
        *Pkt;
    int
        PktLen,
        AdrLen;
    pkt_msg
        *Msg;
    pkt_ror
        *Ror;

    Msg = &AgtSck.SckMsg;
    Ror = &Msg->MsgPdu.PduRor;
    Msg->MsgComLen = ComLen;
    memcpy (Msg->MsgCom, Com, ComLen);
    memcpy (Ror, Rsp, sizeof (pkt_ror));
    Ror->RorTyp = PKT_GETRSP;
    Pkt = AgtSck.SckPkt;
    PktLen = AGT_SZEPKT;
    AdrLen = sizeof (IPS_IPA);
    if (PktEnc (AgtSck.SckRemAdr.sin_addr.s_addr, &AgtSck.SckMsg, &Pkt, &PktLen) < 0)
        return  (-1);
    if (sendto (AgtSck.SckRor, Pkt, PktLen, 0, (IPS_ADR *)&AgtSck.SckRemAdr, AdrLen) < 0)
        return (-1);
    return (0);
}

